// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Linq;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Reflection;
using System.Windows.Input;
using System.Windows.Threading;

namespace HavokVisualDebugger
{
    public partial class VdbViewModelCommands
    {
        #region VdbViewModel - independent command initialization (available statically)
        static VdbViewModelCommands()
        {
            LoseFocusCommand = new VdbModelRelayCommand<System.Windows.IInputElement>(
                delegate (System.Windows.IInputElement o)
                {
                    System.Windows.IInputElement prevFocusedElement = Keyboard.FocusedElement;
                    if ((o == null) || (prevFocusedElement == o))
                    {
                        Keyboard.Focus(null);
                        System.Diagnostics.Debug.Assert(Keyboard.FocusedElement != prevFocusedElement);
                    }
                });
            AssignFocusCommand = new VdbModelRelayCommand<System.Windows.IInputElement>(
                delegate (System.Windows.IInputElement o)
                {
                    if (o != null)
                    {
                        bool focused = o.Focus();
                        System.Diagnostics.Debug.Assert(focused);
                    }
                    else
                    {
                        Keyboard.Focus(null);
                    }
                });
            AdvanceFocusCommand = new VdbModelRelayCommand<System.Windows.UIElement>(
                delegate (System.Windows.UIElement o)
                {
                    System.Windows.UIElement prevFocusedElement = ((o == null) || (Keyboard.FocusedElement == o)) ? (Keyboard.FocusedElement as System.Windows.UIElement) : null;
                    if (prevFocusedElement != null)
                    {
                        if (!prevFocusedElement.MoveFocus(new TraversalRequest(FocusNavigationDirection.Next)))
                        {
                            // Try moving down if "next" failed
                            prevFocusedElement.MoveFocus(new TraversalRequest(FocusNavigationDirection.Down));
                        }

                        System.Diagnostics.Debug.Assert(Keyboard.FocusedElement != prevFocusedElement);
                    }
                });
            CollapseCommand = new VdbModelRelayCommand<System.Windows.UIElement>(
                delegate (System.Windows.UIElement o)
                {
                    if (o != null)
                    {
                        o.Visibility = System.Windows.Visibility.Collapsed;
                    }
                });
            
            TogglePropertyCommand = new VdbModelRelayCommand<Tuple<object, PropertyInfo>>(
                delegate (Tuple<object, PropertyInfo> accessor)
                {
                    if (accessor != null)
                    {
                        try
                        {
                            bool? oldValue = (accessor.Item2.GetValue(accessor.Item1) as bool?);
                            if (oldValue.HasValue)
                            {
                                accessor.Item2.SetValue(accessor.Item1, !oldValue.Value);
                            }
                        }
                        catch { }
                    }
                });
        }
        public static ICommand LoseFocusCommand { get; private set; }
        public static ICommand AssignFocusCommand { get; private set; }
        public static ICommand AdvanceFocusCommand { get; private set; }
        public static ICommand CollapseCommand { get; private set; }
        public static ICommand TogglePropertyCommand { get; private set; }
        #endregion

        #region VdbViewModel - dependent command initialization (not available statically)
        public VdbViewModelCommands(VdbViewModel viewModel)
        {
            ViewModel = viewModel;
            CommandShortcuts = new ObservableCollection<VdbShortcutViewModel>();
            InitCommands();
        }
        private void InitCommands()
        {
            // Setup commands which require ViewModel (aka not initialized in static ctor)
            // ICOMMAND = new ViewModelDelegateCommand(
            //     // INotifyPropertyChanged object
            //     ViewModel,
            //     // Function the command will call on execute, takes object  COMMAND_PARAMETER arg
            //     EXECUTION_FUNC,
            //     // List of properties on INotifyPropertyChanged object which affect
            //     // the commands ability to execute itself. These will be listened
            //     // to for changes and fwd on to CanExecuteChanged listeners.
            //     new List<string> {
            //         nameof(PROPERTY)
            //     },
            //     new List<Func<object, bool>> {
            //     delegate(object COMMAND_PARAMETER) {
            //         return BOOL_EXPRESSION;
            //     }});
            {
                UnselectAllCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.UnselectAll
                    // Below is technically more correct, but might be a perf hit, so not gonna do it.
                    // We'll just always allow this cmd to be exeuctable.
                    //, new List<string> {
                    //    nameof(ViewModel.Selection.SelectedObjectsCount),
                    //    nameof(ViewModel.Selection.HighlightedObjectsCount),
                    //    nameof(ViewModel.Selection.SelectedRenderObjectsCount),
                    //    nameof(ViewModel.Selection.HighlightedRenderObjectsCount)
                    //},
                    //new List<Func<object, bool>> {
                    //delegate(object o) {
                    //    return
                    //        (ViewModel.Selection.SelectedObjectsCount > 0) ||
                    //        (ViewModel.Selection.HighlightedObjectsCount > 0) ||
                    //        (ViewModel.Selection.SelectedRenderObjectsCount > 0) ||
                    //        (ViewModel.Selection.HighlightedRenderObjectsCount > 0);
                    //}}
                    );
                ConnectToFileWithPromptCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ConnectToFileWithPrompt);
                ConnectToFileWithPromptAndSyncToConnectionCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ConnectToFileWithPromptAndSyncToConnection);
                ConnectToMachineOrDisconnectCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ConnectToMachineOrDisconnect);
                SelectDiscoveredServerCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.SelectDiscoveredServer);
                ToggleRecordingCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ToggleRecording,
                    new List<string> {
                        nameof(ViewModel.ConnectedSource),
                        nameof(ViewModel.IsConnected)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return (ViewModel.ConnectedSource == Havok.Vdb.ConnectedSource.Network) && ViewModel.IsConnected;
                    }});
                SaveReplayCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.SaveReplay,
                    new List<string> {
                        nameof(ViewModel.HasPlaybackData),
                        nameof(ViewModel.Capabilities)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return (ViewModel.HasPlaybackData && ((ViewModel.Capabilities & Havok.Vdb.Capabilities.SaveCommandInputBuffer) != 0));
                    }});
                SkipToStartCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.SkipToStart,
                    new List<string> {
                        nameof(ViewModel.HasPlaybackData)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return ViewModel.HasPlaybackData;
                    }});
                StepBwdCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.StepBwd,
                    new List<string> {
                        nameof(ViewModel.HasPlaybackData)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return ViewModel.HasPlaybackData;
                    }});
                PlayBwdCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.PlayBwd,
                    new List<string> {
                        nameof(ViewModel.HasPlaybackData),
                        nameof(ViewModel.PlaybackDirection)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return ViewModel.HasPlaybackData;
                    }});
                TogglePlayingCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.TogglePlaying,
                    new List<string> {
                        nameof(ViewModel.HasPlaybackData)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return ViewModel.HasPlaybackData;
                    }});
                PlayFwdCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.PlayFwd,
                    new List<string> {
                        nameof(ViewModel.HasPlaybackData),
                        nameof(ViewModel.PlaybackDirection)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return ViewModel.HasPlaybackData;
                    }});
                StepFwdCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.StepFwd,
                    new List<string> {
                        nameof(ViewModel.HasPlaybackData)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return ViewModel.HasPlaybackData;
                    }});
                AdvanceOneFwdCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.AdvanceOneFwd,
                    new List<string> {
                        nameof(ViewModel.HasPlaybackData)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return ViewModel.HasPlaybackData;
                    }});
                SkipToEndCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.SkipToEnd,
                    new List<string> {
                        nameof(ViewModel.HasPlaybackData)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return ViewModel.HasPlaybackData;
                    }});
                SyncToConnectionCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.SyncToConnection,
                    new List<string> {
                        nameof(ViewModel.IsConnected),
                        nameof(ViewModel.IsSyncedToConnection),
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return ViewModel.IsConnected && !ViewModel.IsSyncedToConnection;
                    }});
                ConnectToFileCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ConnectToFile);
                ConnectToMachineCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ConnectToMachine,
                    new List<string> {
                        nameof(ViewModel.ConnectedSource),
                        nameof(ViewModel.IsDisconnected)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return (ViewModel.ConnectedSource != Havok.Vdb.ConnectedSource.Network) || ViewModel.IsDisconnected;
                    }});
                DisconnectNetworkCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.DisconnectNetwork,
                    new List<string> {
                        nameof(ViewModel.IsDisconnected)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return !ViewModel.IsDisconnected;
                    }});
                PauseCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.Pause,
                    new List<string> {
                        nameof(ViewModel.HasPlaybackData)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return ViewModel.HasPlaybackData;
                    }});
                PlayCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.Play,
                    new List<string> {
                        nameof(ViewModel.HasPlaybackData)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return ViewModel.HasPlaybackData;
                    }});
                PlayToCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.PlayTo,
                    new List<string> {
                        nameof(ViewModel.HasPlaybackData)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        Int64? oAsI = (o as Int64?);
                        return oAsI.HasValue && ViewModel.HasPlaybackData;
                    }});
                StartRecordingCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.StartRecording,
                    new List<string> {
                        nameof(ViewModel.ConnectedSource),
                        nameof(ViewModel.IsConnected),
                        nameof(ViewModel.IsRecording)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return (ViewModel.ConnectedSource == Havok.Vdb.ConnectedSource.Network) && ViewModel.IsConnected && !ViewModel.IsRecording;
                    }});
                StopRecordingCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.StopRecording,
                    new List<string> {
                        nameof(ViewModel.ConnectedSource),
                        nameof(ViewModel.IsConnected),
                        nameof(ViewModel.IsRecording)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return (ViewModel.ConnectedSource == Havok.Vdb.ConnectedSource.Network) && ViewModel.IsConnected && ViewModel.IsRecording;
                    }});
                DefaultSelectionViewCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.DefaultSelectionView);
                FitSelectionCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.FitSelection);
                DefaultWorldViewCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.DefaultWorldView);
                FitWorldCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.FitWorld);
                PositiveXViewCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.PositiveXView);
                NegativeXViewCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.NegativeXView);
                PositiveYViewCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.PositiveYView);
                NegativeYViewCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.NegativeYView);
                PositiveZViewCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.PositiveZView);
                NegativeZViewCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.NegativeZView);
                GoToFrameCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.GoToFrame,
                    new List<string> {
                        nameof(ViewModel.HasPlaybackData)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return ViewModel.HasPlaybackData;
                    }});
                SetCameraXUpAxisCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.SetCameraXUpAxis);
                SetCameraYUpAxisCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.SetCameraYUpAxis);
                SetCameraZUpAxisCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.SetCameraZUpAxis);
                ToggleCameraIsLookingAtPOICommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ToggleCameraIsLookingAtPOI);
                ToggleCameraIsKeepingRelativeToPOICommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ToggleCameraIsKeepingRelativeToPOI);
                ToggleCameraIsLookingAtGeometryCenterCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ToggleCameraIsLookingAtGeometryCenter);
                ToggleCameraIsKeepingRelativeToGeometryCenterCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ToggleCameraIsKeepingRelativeToGeometryCenter);
                ToggleWireframeCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ToggleWireframe);
                ToggleOutlineFacesCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ToggleOutlineFaces);
                ToggleGridCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ToggleGrid);
                ToggleRandomizedColorsCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ToggleRandomizedColors);
                ToggleSelectionVisualizationCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ToggleSelectionVisualization);
                ToggleLockLightToCameraCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ToggleLockLightToCamera);
                ResetLightingCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ResetLighting);
                ToggleUnitAxisCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ToggleUnitAxis);
                ToggleStatBarGraphCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ToggleStatBarGraph);
                TogglePerspectiveModeCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.TogglePerspectiveMode);
                LogClearCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ClearLogs);
                LogCopyCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.CopyLogs);
                ToggleLogPanelShownCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ToggleLogPanelShown);
                ToggleInspectionPanelShownCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ToggleVdbObjectTreeViewShown);
                ResetViewersCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ResetViewers);
                ResetViewersToDefaultCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ResetViewersToDefault);
                QueryVdbObjectTreeCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.QueryVdbObjectTree);
                CollapseVdbObjectTreeCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.CollapseVdbObjectTree);
                TurnOnWorldInspectionViewerCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.TurnOnWorldInspectionViewer);
                GoToNextTreeObjectCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.GoToNextTreeObjectCommand);
                GoToPrevTreeObjectCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.GoToPrevTreeObjectCommand);
                ToggleDiagnosticsPanelShownCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.ToggleDiagnosticsPanelShown);
                DeselectAllTimersCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.DeselectAllTimers);
                SaveSelectedGeometriesCommand = new ViewModelDelegateCommand(
                    ViewModel.Selection,
                    ViewModel.SaveSelectedGeometries,
                    new List<string> {
                        nameof(ViewModel.Selection.SelectedRenderObjectsCount),
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return (ViewModel.Selection.SelectedRenderObjectsCount > 0);
                    }});
                SaveHighlightedGeometriesCommand = new ViewModelDelegateCommand(
                    ViewModel.Selection,
                    ViewModel.SaveHighlightedGeometries,
                    new List<string> {
                        nameof(ViewModel.Selection.HighlightedRenderObjectsCount)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return (ViewModel.Selection.HighlightedRenderObjectsCount > 0);
                    }});
                SaveAllGeometriesCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.SaveAllGeometries,
                    new List<string> {
                        nameof(ViewModel.HasRenderObjects)
                    },
                    new List<Func<object, bool>> {
                    delegate(object o) {
                        return
                            ViewModel.HasRenderObjects;
                    }});
                LaunchHelpDocsCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.LaunchHelpDocs);
                OpenRecentFileCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.OpenRecentFile);
                GoToLogEntryFrameCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.GoToLogEntryFrame);

                // Referenced statically, potentially dangerous if xaml initializes before this call
                DeleteQueryItemCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.DeleteQueryItem);
                DeleteMachineAddressItemCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.DeleteMachineAddressItem);
                
                
                DeleteMachinePortItemCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.DeleteMachinePortItem);
                DeleteMachineDiscoveryPortItemCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.DeleteMachineDiscoveryPortItem);
                UnlockCameraCommand = new ViewModelDelegateCommand(
                    ViewModel,
                    ViewModel.UnlockCamera);

                SwapToCameraCommands = new List<ICommand>
                {
                    new ViewModelDelegateCommand(ViewModel, ViewModel.SwapToCamera1),
                    new ViewModelDelegateCommand(ViewModel, ViewModel.SwapToCamera2),
                    new ViewModelDelegateCommand(ViewModel, ViewModel.SwapToCamera3),
                    new ViewModelDelegateCommand(ViewModel, ViewModel.SwapToCamera4),
                    new ViewModelDelegateCommand(ViewModel, ViewModel.SwapToCamera5)
                };
            }

            // Setup our bindings
            {
                CommandShortcuts.CollectionChanged += delegate (object sender, System.Collections.Specialized.NotifyCollectionChangedEventArgs e)
                {
                    if( e.Action == System.Collections.Specialized.NotifyCollectionChangedAction.Add  && e.NewItems != null )
                    {
                        foreach( VdbShortcutViewModel vm in e.NewItems )
                        {
                            IEnumerable<VdbShortcutViewModel> foundShortcuts = CommandShortcuts.Where(x => x.Shortcut == vm.Shortcut);
                            if (foundShortcuts != null && foundShortcuts.Count() > 1)
                            {
                                System.Diagnostics.Debug.Assert(false, string.Format(Properties.Resources.Commands_ShortcutExistsFoundDuplicate, vm.Shortcut, vm.ShortcutName, foundShortcuts.ElementAt(0).ShortcutName));
                                return;
                            }
                        }
                    }
                };

                // Playback/Connection shortcuts
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Connection_Disconnect, "Ctrl+Alt+D", DisconnectNetworkCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_DisconnectNetwork));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Connection_Connect, "Ctrl+Alt+C", ConnectToMachineCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_ConnectToMachine));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_StartStopRecording, "Ctrl+R", ToggleRecordingCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_ToggleRecording));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.MenuHeader_Save, "Ctrl+S", SaveReplayCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_SaveReplay));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.MenuHeader_Open, "Ctrl+O", ConnectToFileWithPromptCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_ConnectToFileWithPrompt));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_SkipToStart, "Ctrl+Alt+Left", SkipToStartCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_SkipToStart));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_StepBwd, "Ctrl+Shift+Tab", StepBwdCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_StepBwd));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_StepFwd, "Ctrl+Tab", StepFwdCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_StepFwd));
                
                
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_SkipToEnd, "Ctrl+Alt+Right", SkipToEndCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_SkipToEnd));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_StartStopPlayback, "Ctrl+P", TogglePlayingCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_TogglePlaying));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_GoToFrame, "Ctrl+G", GoToFrameCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_GoToFrame));

                // Camera shortcuts
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_DefaultSelectionView, "Alt+F", DefaultSelectionViewCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_DefaultSelectionView));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_FitSelection, "Ctrl+Alt+F", FitSelectionCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_FitSelection));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_DefaultWorldView, "Ctrl+E", DefaultWorldViewCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_DefaultWorldView));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_FitWorld, "Ctrl+Alt+E", FitWorldCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_FitWorld));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Camera_PositiveXView, "Ctrl+X", PositiveXViewCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_PositiveXView));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Camera_NegativeXView, "Ctrl+Shift+X", NegativeXViewCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_NegativeXView));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Camera_PositiveYView, "Ctrl+Y", PositiveYViewCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_PositiveYView));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Camera_NegativeYView, "Ctrl+Shift+Y", NegativeYViewCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_NegativeYView));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Camera_PositiveZView, "Ctrl+Z", PositiveZViewCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_PositiveZView));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Camera_NegativeZView, "Ctrl+Shift+Z", NegativeZViewCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_NegativeZView));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Camera_PerspectiveMode, "Ctrl+Shift+P", TogglePerspectiveModeCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_TogglePerspectiveMode));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_SetCameraXUpAxis, "Ctrl+Alt+X", SetCameraXUpAxisCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_SetCameraXUpAxis));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_SetCameraYUpAxis, "Ctrl+Alt+Y", SetCameraYUpAxisCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_SetCameraYUpAxis));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_SetCameraZUpAxis, "Ctrl+Alt+Z", SetCameraZUpAxisCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_SetCameraZUpAxis));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_ToggleCameraIsLookingAtPOI, "Shift+Alt+P", ToggleCameraIsLookingAtPOICommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_ToggleCameraIsLookingAtPOI));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_ToggleCameraIsKeepingRelativeToPOI, "Alt+P", ToggleCameraIsKeepingRelativeToPOICommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_ToggleCameraIsKeepingRelativeToPOI));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_ToggleCameraIsLookingAtGeometryCenter, "Shift+Alt+G", ToggleCameraIsLookingAtGeometryCenterCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_ToggleCameraIsLookingAtGeometryCenter));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_ToggleCameraIsKeepingRelativeToGeometryCenter, "Alt+G", ToggleCameraIsKeepingRelativeToGeometryCenterCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_ToggleCameraIsKeepingRelativeToGeometryCenter));

                CommandShortcuts.Add(new VdbShortcutViewModel(string.Format(Properties.Resources.Commands_SwapCameras, 1), "Ctrl+Alt+F1", SwapToCameraCommands[0] as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_SwapToCamera1));
                CommandShortcuts.Add(new VdbShortcutViewModel(string.Format(Properties.Resources.Commands_SwapCameras, 2), "Ctrl+Alt+F2", SwapToCameraCommands[1] as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_SwapToCamera2));
                CommandShortcuts.Add(new VdbShortcutViewModel(string.Format(Properties.Resources.Commands_SwapCameras, 3), "Ctrl+Alt+F3", SwapToCameraCommands[2] as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_SwapToCamera3));
                CommandShortcuts.Add(new VdbShortcutViewModel(string.Format(Properties.Resources.Commands_SwapCameras, 4), "Ctrl+Alt+F4", SwapToCameraCommands[3] as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_SwapToCamera4));
                CommandShortcuts.Add(new VdbShortcutViewModel(string.Format(Properties.Resources.Commands_SwapCameras, 5), "Ctrl+Alt+F5", SwapToCameraCommands[4] as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_SwapToCamera5));

                // Render shortcuts
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_ToggleWireframe, "Ctrl+Alt+W", ToggleWireframeCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_ToggleWireframe));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_ToggleOutlineFaces, "Ctrl+Alt+O", ToggleOutlineFacesCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_ToggleOutlineFaces));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_ToggleGrid, "Ctrl+Alt+G", ToggleGridCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_ToggleGrid));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_ToggleUnitAxis, "Ctrl+Alt+U", ToggleUnitAxisCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_ToggleUnitAxis));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_ToggleStatGraph, "Ctrl+Alt+B", ToggleStatBarGraphCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_ToggleStatBarGraph));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_ToggleRandomizedColors, "Ctrl+Alt+R", ToggleRandomizedColorsCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_ToggleRandomizedColors));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_ToggleSelectionVisualization, "Ctrl+Alt+V", ToggleSelectionVisualizationCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_ToggleSelectionVisualization));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_ToggleLockLightToCamera, "Ctrl+D", ToggleLockLightToCameraCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_ToggleLockLightToCamera));
                CommandShortcuts.Add(new VdbShortcutViewModel(Properties.Resources.Commands_ResetLighting, "Ctrl+L", ResetLightingCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_ResetLighting));

                
                
                //CommandShortcuts.Add(new VdbShortcutViewModel("W", MoveForwardCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_MoveForward));
                //CommandShortcuts.Add(new VdbShortcutViewModel("A", StrafeLeftCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_StrafeLeft));
                //CommandShortcuts.Add(new VdbShortcutViewModel("S", MoveBackwardCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_MoveBackward));
                //CommandShortcuts.Add(new VdbShortcutViewModel("D", StrafeRightCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_StrafeRight));
                //CommandShortcuts.Add(new VdbShortcutViewModel("Q", MoveUpCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_MoveUp));
                //CommandShortcuts.Add(new VdbShortcutViewModel("Z", MoveDownCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_MoveDown));
                //CommandShortcuts.Add(new VdbShortcutViewModel("Space", MousePickCommand as ViewModelDelegateCommand, Properties.Settings.Default.VdbShortcut_MousePick));
            }
        }

        private VdbViewModel ViewModel { get; set; }

        public ICommand UnselectAllCommand { get; private set; }
        public ICommand ConnectToMachineOrDisconnectCommand { get; private set; }
        public ICommand ConnectToMachineCommand { get; private set; }
        public ICommand ConnectToFileCommand { get; private set; }
        public ICommand ConnectToFileWithPromptCommand { get; private set; }
        public ICommand ConnectToFileWithPromptAndSyncToConnectionCommand { get; private set; }
        public ICommand DisconnectNetworkCommand { get; private set; }
		public ICommand SelectDiscoveredServerCommand { get; private set; }
		public ICommand ToggleRecordingCommand { get; private set; }
        public ICommand StartRecordingCommand { get; private set; }
        public ICommand StopRecordingCommand { get; private set; }
        public ICommand SaveReplayCommand { get; private set; }
        public ICommand SkipToStartCommand { get; private set; }
        public ICommand StepBwdCommand { get; private set; }
        public ICommand PlayBwdCommand { get; private set; }
        public ICommand TogglePlayingCommand { get; private set; }
        public ICommand PlayFwdCommand { get; private set; }
        public ICommand StepFwdCommand { get; private set; }
        public ICommand AdvanceOneFwdCommand { get; private set; }
        public ICommand SkipToEndCommand { get; private set; }
        public ICommand SyncToConnectionCommand { get; private set; }
        public ICommand PlayCommand { get; private set; }
        public ICommand PlayToCommand { get; private set; }
        public ICommand PauseCommand { get; private set; }
        public ICommand ToggleThrottlingToServerFrameRate { get; private set; }
        public ICommand SuspendProcessingCommand { get; private set; }
        public ICommand ResumeProcessingCommand { get; private set; }
        public ICommand DefaultSelectionViewCommand { get; private set; }
        public ICommand FitSelectionCommand { get; private set; }
        public ICommand DefaultWorldViewCommand { get; private set; }
		public ICommand FitWorldCommand { get; private set; }
        public ICommand PositiveXViewCommand { get; private set; }
		public ICommand NegativeXViewCommand { get; private set; }
		public ICommand PositiveYViewCommand { get; private set; }
		public ICommand NegativeYViewCommand { get; private set; }
		public ICommand PositiveZViewCommand { get; private set; }
		public ICommand NegativeZViewCommand { get; private set; }
        public ICommand TogglePerspectiveModeCommand { get; private set; }
        public ICommand ReconnectCommand { get; private set; }
        public ICommand GoToFrameCommand { get; private set; }
        public ICommand SetCameraXUpAxisCommand { get; private set; }
        public ICommand SetCameraYUpAxisCommand { get; private set; }
        public ICommand SetCameraZUpAxisCommand { get; private set; }
        public ICommand ToggleCameraIsLookingAtPOICommand { get; private set; }
        public ICommand ToggleCameraIsKeepingRelativeToPOICommand { get; private set; }
        public ICommand ToggleCameraIsLookingAtGeometryCenterCommand { get; private set; }
        public ICommand ToggleCameraIsKeepingRelativeToGeometryCenterCommand { get; private set; }
        public ICommand ToggleWireframeCommand { get; private set; }
        public ICommand ToggleOutlineFacesCommand { get; private set; }
        public ICommand ToggleGridCommand { get; private set; }
        public ICommand ToggleUnitAxisCommand { get; private set; }
        public ICommand ToggleStatBarGraphCommand { get; private set; }
        public ICommand ToggleRandomizedColorsCommand { get; private set; }
        public ICommand ToggleSelectionVisualizationCommand { get; private set; }
        public ICommand ToggleLockLightToCameraCommand { get; private set; }
        public ICommand ResetLightingCommand { get; private set; }
        public ICommand MoveForwardCommand { get; private set; }
        public ICommand StrafeLeftCommand { get; private set; }
        public ICommand MoveBackwardCommand { get; private set; }
        public ICommand StrafeRightCommand { get; private set; }
        public ICommand MoveUpCommand { get; private set; }
        public ICommand MoveDownCommand { get; private set; }
        public ICommand MousePickCommand { get; private set; }
        public ICommand ResetViewersCommand { get; private set; }
        public ICommand ResetViewersToDefaultCommand { get; private set; }
        public ICommand QueryVdbObjectTreeCommand { get; private set; }
        public ICommand CollapseVdbObjectTreeCommand { get; private set; }
        public ICommand ToggleInspectionPanelShownCommand { get; private set; }
        public ICommand TurnOnWorldInspectionViewerCommand { get; private set; }
        public ICommand GoToNextTreeObjectCommand { get; private set; }
        public ICommand GoToPrevTreeObjectCommand { get; private set; }
        public ICommand ToggleDiagnosticsPanelShownCommand { get; private set; }
        public ICommand DeselectAllTimersCommand { get; private set; }
        public ICommand UnlockCameraCommand { get; private set; }
        public ICommand SaveSelectedGeometriesCommand { get; private set; }
        public ICommand SaveHighlightedGeometriesCommand { get; private set; }
        public ICommand SaveAllGeometriesCommand { get; private set; }
        public ICommand LogClearCommand { get; private set; }
        public ICommand LogCopyCommand { get; private set; }
        public ICommand ToggleLogPanelShownCommand { get; private set; }
        public ICommand LaunchHelpDocsCommand { get; private set; }
        public ICommand OpenRecentFileCommand { get; private set; }
        public ICommand GoToLogEntryFrameCommand { get; private set; }

        public List<ICommand> SwapToCameraCommands { get; private set; }

        // Referenced statically, potentially dangerous if xaml initializes before the InitCommands() call
        public static ICommand DeleteQueryItemCommand { get; private set; }
        public static ICommand DeleteMachineAddressItemCommand { get; private set; }
        public static ICommand DeleteMachinePortItemCommand { get; private set; }
        public static ICommand DeleteMachineDiscoveryPortItemCommand { get; private set; }

        #endregion

        // Shortcuts for all commands with assignable shortcuts
        public ObservableCollection<VdbShortcutViewModel> CommandShortcuts { get; private set; }
    }

    
    
    
    public class ViewModelDelegateCommand : ICommand
    {
        public event EventHandler CanExecuteChanged = delegate { };

        public ViewModelDelegateCommand(
            INotifyPropertyChanged handler,
            Action<object> executeMethod,
            List<string> canExecutePropertiesToWatch = null,
            List<Func<object, bool>> canExecuteMethods = null)
        {
            

            handler.PropertyChanged += delegate (object sender, PropertyChangedEventArgs args)
            {
                if ((canExecutePropertiesToWatch != null) && canExecutePropertiesToWatch.Contains(args.PropertyName))
                {
                    CanExecuteChanged(sender, EventArgs.Empty);
                }
            };

            m_handler = handler;
            m_executeMethod = executeMethod;
            if (canExecuteMethods != null)
            {
                m_canExecuteMethods = canExecuteMethods;
            }
        }

        public bool CanExecute(object parameter)
        {
            return m_canExecuteMethods.TrueForAll(f => (f.Invoke(null) as bool?) == true);
        }

        public void Execute(object parameter)
        {
            m_executeMethod.Invoke(parameter);
        }

        public string Name { get { return m_executeMethod.GetMethodInfo().Name; } }

        private object m_handler;
        private Action<object> m_executeMethod;
        private List<Func<object, bool>> m_canExecuteMethods = new List<Func<object, bool>>(); // always non null
    }

    public class VdbShortcutViewModel : ViewModelNotifyPropertyChanged
    {
        
        

        #region Initialization
        public VdbShortcutViewModel(string shortcutName, string defaultShortcut, ViewModelDelegateCommand command, string shortcut="")
        {
            ShortcutName = shortcutName;
            Command = command;
            DefaultShortcut = defaultShortcut;

            // Set the current shortcut
            if (!string.IsNullOrEmpty(shortcut))
            {
                Shortcut = shortcut;
            }
            else
            {
                Shortcut = defaultShortcut;
            }

            // Initialize the "default" command
            ResetShortcutCommand = new ViewModelDelegateCommand(
                this,
                delegate (object o) {
                    Shortcut = DefaultShortcut;
                },
                new List<string> {
                    nameof(Shortcut)
                },
                new List<Func<object, bool>> {
                delegate(object o) {
                    return (Shortcut != DefaultShortcut);
                }});

            m_popupTimer = new DispatcherTimer();
            {
                m_popupTimer.Interval = TimeSpan.FromSeconds(5);
                m_popupTimer.Tick += (object sender, EventArgs e) =>
                {
                    m_popupTimer.Stop();
                    IsShortcutErrorShown = false;
                };
            }
        }

        #endregion

        #region View Commands and Properties

        public ViewModelDelegateCommand Command { get; private set; }
        public ICommand ResetShortcutCommand { get; private set; }
        public InputBinding InputBinding { get; private set; }
        public string ShortcutName { get; private set; }
        public override string ToString() { return ShortcutName; }

        public string Shortcut
        {
            get
            {
                return _Shortcut ?? DefaultShortcut;
            }
            set
            {
                if ( _Shortcut != value )
                {
                    InputGesture currentGesture = InputBinding?.Gesture;
                    try
                    {
                        // Single value
                        if (value.Length == 1)
                        {
                            ShortcutErrorText = Properties.Resources.Commands_ShortcutHasNoModifiers;
                            //// Convert value to be within Key range
                            //char val = (char)(value[0] - 21);
                            //if ((val >= (int)Key.A) && (val < (int)Key.Z))
                            //{
                            //    // Within range of A-Z
                            //    KeyBinding k = new KeyBinding(Command, Key.W, ModifierKeys.None);
                            //    KeyBinding keyBinding = new KeyBinding(Command, (Key)val, ModifierKeys.None);
                            //    InputBinding = keyBinding;
                            //}
                        }
                        else if (value.Length > 1)
                        {
                            KeyGestureConverter kgc = new KeyGestureConverter();
                            InputBinding.Gesture = kgc.ConvertFromString(value) as InputGesture;

                            _Shortcut = value;

                            // Only save setting if above succeeded (no exception thrown)
                            string settingName = "VdbShortcut_" + Command.Name.Replace(" ", "");
                            VdbSettings.SetValue(settingName, value);
                        }
                    }
                    catch(Exception e)
                    {
                        if( e is NotSupportedException )
                        {
                            ShortcutErrorText = Properties.Resources.Commands_ShortcutRequiresAdditionalModifiers;
                        }
                        else
                        {
                            ShortcutErrorText = e.Message;
                        }

                        // Failed to convert to gesture, revert
                        if (currentGesture != null)
                        {
                            InputBinding.Gesture = currentGesture;
                        }
                    }
                    finally
                    {
                        NotifyPropertyChanged();
                        NotifyPropertyChanged(nameof(InputBinding));
                    }
                }
            }
        }
        private string _Shortcut;

        private string _DefaultShortcut;
        public string DefaultShortcut
        {
            get
            {
                return _DefaultShortcut;
            }

            private set
            {
                try
                {
                    KeyGestureConverter kgc = new KeyGestureConverter();
                    InputGesture gesture = kgc.ConvertFromString(value) as InputGesture;
                    InputBinding = new InputBinding(Command, gesture);
                    _DefaultShortcut = value;
                }
                catch(Exception e)
                {
                    System.Diagnostics.Debug.Assert(false, "Failed to apply shortcut " + value + ": " + e.Message);
                }
            }
        }

        public bool IsShortcutErrorShown
        {
            get { return _IsShortcutErrorShown; }
            set
            {
                if (_IsShortcutErrorShown != value)
                {
                    _IsShortcutErrorShown = value;
                    NotifyPropertyChanged();
                }

                // Restart timer if trying to reshow
                if (value)
                {
                    m_popupTimer.Stop();
                    m_popupTimer.Start();
                }
            }
        }
        private bool _IsShortcutErrorShown;

        public string ShortcutErrorText
        {
            get { return _ShortcutErrorText; }
            set
            {
                if (_ShortcutErrorText != value)
                {
                    _ShortcutErrorText = value;
                    NotifyPropertyChanged();
                }

                // Trigger the error to show
                IsShortcutErrorShown = true;
            }
        }
        private string _ShortcutErrorText;

        private DispatcherTimer m_popupTimer;
        #endregion
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
