// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System.Collections.Generic;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;
using System.Windows.Media;

namespace HavokVisualDebugger
{
    public static class TreeViewitemFinder
    {
        public static TreeViewItem BringTreeViewItemIntoView(this TreeView treeView, VdbObjectViewModel objectVm)
        {
            if (objectVm == null)
            {
                return null;
            }

            ItemsControl parentContainer = treeView.BringTreeViewItemIntoView(objectVm.Parent);
            if (parentContainer == null)
            {
                parentContainer = treeView;
            }

            return parentContainer.BringItemIntoView(objectVm);
        }

        private static TreeViewItem BringItemIntoView(this ItemsControl container, VdbObjectViewModel item)
        {
            try
            {
                VirtualizingStackPanel vsp = container.FindVisualChild<VirtualizingStackPanel>();
                if (vsp == null)
                {
                    TreeViewItem treeViewItem = (TreeViewItem)container.ItemContainerGenerator.ContainerFromItem(item);
                    if (treeViewItem != null)
                    {
                        treeViewItem.BringIntoView();
                        return treeViewItem;
                    }
                    else
                    {
                        // Try updating container layout first to get the container to create virtualized items
                        container.UpdateLayout();

                        vsp = container.FindVisualChild<VirtualizingStackPanel>();
                        treeViewItem = container.ItemContainerGenerator.ContainerFromItem(item) as TreeViewItem;
                        if (treeViewItem != null)
                        {
                            treeViewItem.BringIntoView();
                        }
                        else if (vsp == null)
                        {
                            System.Diagnostics.Debug.Assert(false, "Virtualized item not able to be brought into view (" + item.DisplayName + ")");
                            return null;
                        }
                    }
                }

                if( container.Items.Count > 0 )
                {
                    VdbObjectViewModel parentVm = (container.Items[0] as VdbObjectViewModel)?.Parent;
                    int foundChildIndex = parentVm.Children.IndexOf(item);
                    if( foundChildIndex >= 0 )
                    {
                        bool isItemOnScreen = IsItemOnScreen(vsp, item);

                        // If the item is already on screen, just bring it into view directly
                        if (isItemOnScreen)
                        {
                            TreeViewItem itemOnScreen = (TreeViewItem)container.ItemContainerGenerator.ContainerFromIndex(foundChildIndex);
                            if ((itemOnScreen != null) && (itemOnScreen.DataContext == item))
                            {
                                itemOnScreen.BringIntoView();
                                return itemOnScreen;
                            }
                        }

                        TreeViewItem nextItem = (TreeViewItem)container.ItemContainerGenerator.ContainerFromIndex(foundChildIndex);
                        if (nextItem == null || !nextItem.HasItems)
                        {
                            // Bring the index into view and try to grab the item again
                            vsp.BringIndexIntoViewPublic(foundChildIndex);
                            nextItem = (TreeViewItem)container.ItemContainerGenerator.ContainerFromIndex(foundChildIndex);
                        }

                        // Try to bring the actual item into view if we've matched.
                        // If the item currently has focus, don't bring into view as the item should already be in view and doing this will scroll the panel.
                        if ((nextItem != null) && (nextItem.DataContext == item) && !nextItem.IsFocused)
                        {
                            nextItem.BringIntoView();
                            return nextItem;
                        }
                    }
                }
            }
            catch
            {
            }

            return null;
        }

        private static bool IsItemOnScreen(VirtualizingStackPanel vsp, VdbObjectViewModel item)
        {
            var layoutBounds = LayoutInformation.GetLayoutSlot(vsp);
            var onScreenChildren =
                (from visualChild in vsp.GetChildren()
                 let childBounds = LayoutInformation.GetLayoutSlot(visualChild)
                 where layoutBounds.Contains(childBounds) && layoutBounds.IntersectsWith(childBounds)
                 select visualChild.DataContext).Cast<VdbObjectViewModel>().ToList();

            return onScreenChildren.Contains(item);
        }

        private static IEnumerable<FrameworkElement> GetChildren(this DependencyObject dependencyObject)
        {
            var numberOfChildren = VisualTreeHelper.GetChildrenCount(dependencyObject);
            return (from index in Enumerable.Range(0, numberOfChildren)
                    select VisualTreeHelper.GetChild(dependencyObject, index)).Cast<FrameworkElement>();
        }

        private static T FindVisualChild<T>(this Visual visual) where T : Visual
        {
            for (int i = 0; i < VisualTreeHelper.GetChildrenCount(visual); i++)
            {
                var child = (Visual)VisualTreeHelper.GetChild(visual, i);
                if (child != null)
                {
                    var correctlyTyped = child as T;
                    if (correctlyTyped != null)
                        return correctlyTyped;
                    var descendent = FindVisualChild<T>(child);
                    if (descendent != null)
                        return descendent;
                }
            }
            return null;
        }
    }


    public static class TreeViewItemExtension
    {
        #region TreeViewSelectionVirtualization
        public static readonly DependencyProperty TreeViewSelectionVirtualizationProperty =
            DependencyProperty.RegisterAttached("TreeViewSelectionVirtualization", typeof(VdbObjectViewModel),
                typeof(TreeViewItemExtension), new PropertyMetadata(null, TreeViewSelectionVirtualizationChanged));

        private static void TreeViewSelectionVirtualizationChanged(object sender, DependencyPropertyChangedEventArgs e)
        {
            TreeView treeView = sender as TreeView;
            VdbObjectViewModel objectVm = e.NewValue as VdbObjectViewModel;
            if (treeView != null && objectVm != null)
            {
                // We expand the *parent* and that will percolate up to the root.
                // We *don't* expand the selected node as this will usually create a highly expanded
                // tree which starts to get slow, plus it doesn't really make sense, we are just trying
                // to ensure viewModel is visible.
                if (objectVm.Parent != null)
                {
                    objectVm.Parent.IsExpanded = true;
                }

                // Now bring the item into view *after* expanding
                // (since that will have created virtualized nodes)
                treeView.BringTreeViewItemIntoView(objectVm);
            }
        }

        public static VdbObjectViewModel GetTreeViewSelectionVirtualization(TreeView treeView)
        {
            if (treeView != null)
            {
                return (VdbObjectViewModel)treeView.GetValue(TreeViewSelectionVirtualizationProperty);
            }

            return null;
        }

        public static void SetTreeViewSelectionVirtualization(TreeView treeView, VdbObjectViewModel virtualizedObjectVm)
        {
            if (treeView != null)
            {
                treeView.SetValue(TreeViewSelectionVirtualizationProperty, virtualizedObjectVm);
            }
        }
        #endregion
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
