// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Input;

namespace HavokVisualDebugger.Controls
{
    /// <summary>
    /// Interaction logic for VdbTimerSelectionWidget.xaml
    /// </summary>
    public partial class VdbTimerSelectionWidget : UserControl
    {
        public VdbTimerSelectionWidget()
        {
            InitializeComponent();
        }

        private void Button_Click(object sender, System.Windows.RoutedEventArgs e)
        {
            Button btn = sender as Button;
            if (btn != null)
            {
                TimerNodeViewModel vm = btn.DataContext as TimerNodeViewModel;
                if (vm != null)
                {
                    vm.IsColorPressed = !vm.IsColorPressed;
                }
            }
        }

        private void UserControl_PreviewMouseWheel(object sender, MouseWheelEventArgs e)
        {
            if (!e.Handled)
            {
                e.Handled = true;
                var eventArg = new MouseWheelEventArgs(e.MouseDevice, e.Timestamp, e.Delta);
                eventArg.RoutedEvent = UIElement.MouseWheelEvent;
                eventArg.Source = sender;
                var parent = ((Control)sender).Parent as UIElement;
                parent.RaiseEvent(eventArg);
            }
        }

        private void ToggleButton_PreviewKeyUp(object sender, KeyEventArgs e)
        {
            if (!e.Handled)
            {
                e.Handled = true;
                var eventArg = new KeyEventArgs(e.KeyboardDevice, e.InputSource, e.Timestamp, e.Key);
                eventArg.RoutedEvent = UIElement.KeyUpEvent;
                eventArg.Source = sender;
                var parent = ((Control)sender).Parent as UIElement;
                parent.RaiseEvent(eventArg);
            }
        }

        private void Copy(object sender, ExecutedRoutedEventArgs e)
        {
            DataGrid dg = sender as DataGrid;
            if (dg != null)
            {
                IEnumerable entries = dg.SelectedItems;
                if (entries == null)
                {
                    return;
                }

                int maxNumThreads = VdbTimerNodeManager.MaxNumThreads;
                string clipText = "";
                foreach (TimerNodeViewModel timerEntry in entries)
                {
                    if (timerEntry == null)
                    {
                        continue;
                    }

                    string threadData = "";
                    for (int i = 0; i < maxNumThreads; i++)
                    {
                        if (i < timerEntry.ThreadData.Count)
                        {
                            threadData += timerEntry.ThreadData[i].ToString("0.0000") + "\t";
                        }
                        else
                        {
                            threadData += "0.0000\t";
                        }
                    }

                    clipText += string.Join("\t", new string[] { timerEntry.Path + timerEntry.Name, threadData, "\n" });
                }
                Clipboard.SetData(DataFormats.Text, clipText);
            }
        }

        private void CanCopy(object sender, CanExecuteRoutedEventArgs e)
        {
            e.CanExecute = true;
        }

        private void CanSelectAll(object sender, CanExecuteRoutedEventArgs e)
        {
            e.CanExecute = true;
        }

        private void SelectAll(object sender, ExecutedRoutedEventArgs e)
        {
            DataGrid dg = sender as DataGrid;
            if (dg != null)
            {
                dg.SelectedItems.Clear();
                dg.SelectAllCells();
            }
        }

        private void DataGrid_TargetUpdated(object sender, System.Windows.Data.DataTransferEventArgs e)
        {
            // Number of data grid columns, not including thread data
            const int numDataGridColumns = 4;

            ObservableCollection<TimerNodeViewModel> collection = TimerDataGrid.ItemsSource as ObservableCollection<TimerNodeViewModel>;
            if ( collection != null && collection.Count > 0 )
            {
                if (TimerDataGrid.Columns.Count < VdbTimerNodeManager.MaxNumThreads + numDataGridColumns)
                {
                    // Add thread columns
                    for (int i = TimerDataGrid.Columns.Count; i < VdbTimerNodeManager.MaxNumThreads + numDataGridColumns; i++)
                    {
                        int threadIndex = (i - numDataGridColumns);
                        string threadIndexStr = threadIndex.ToString();

                        DataGridTextColumn column = new DataGridTextColumn();
                        column.Header = "Thread " + threadIndexStr;
                        column.Width = 80;
                        column.IsReadOnly = true;
                        column.CellStyle = this.FindResource("VdbSimpleGridCell") as Style;

                        MultiBinding multiThreadBindings = new MultiBinding();
                        {
                            // Add binding for ThreadData
                            Binding threadBinding = new Binding("ThreadData");
                            threadBinding.Converter = this.FindResource("AccessFloatListFromIndexConverter") as IValueConverter;
                            threadBinding.ConverterParameter = threadIndexStr;
                            threadBinding.FallbackValue = "0.0000";
                            threadBinding.Mode = BindingMode.OneWay;

                            // Add binding for ThreadDataCount
                            Binding threadCountBinding = new Binding("ThreadDataCount");
                            threadCountBinding.Converter = this.FindResource("AccessUIntListFromIndexConverter") as IValueConverter;
                            threadCountBinding.ConverterParameter = threadIndexStr;
                            threadCountBinding.FallbackValue = "0";
                            threadCountBinding.Mode = BindingMode.OneWay;

                            // Add single bindings to MultiBinding
                            multiThreadBindings.Bindings.Add(threadBinding);
                            multiThreadBindings.Bindings.Add(threadCountBinding);

                            // Although StringFormats in XAML require a leading "{}" if the first element is a reference like {0},
                            // this explicitly does NOT work in code!
                            multiThreadBindings.StringFormat = "{0:0.0000} ({1:0})";
                        }
                        column.Binding = multiThreadBindings;

                        // Insert before the "Sum" column
                        TimerDataGrid.Columns.Insert(TimerDataGrid.Columns.Count - 1, column);
                    }
                }
                else if (TimerDataGrid.Columns.Count > VdbTimerNodeManager.MaxNumThreads + numDataGridColumns)
                {
                    // Remove thread columns
                    for (int i = TimerDataGrid.Columns.Count; i > VdbTimerNodeManager.MaxNumThreads + numDataGridColumns; i--)
                    {
                        // Subtract 2 to skip the "Sum" column
                        TimerDataGrid.Columns.RemoveAt(TimerDataGrid.Columns.Count - 2);
                    }
                }
            }
        }

        private void Row_PreviewDoubleClick(object sender, MouseButtonEventArgs e)
        {
            FrameworkElement senderElement = sender as FrameworkElement;
            TimerNodeViewModel vm = senderElement?.DataContext as TimerNodeViewModel;
            FrameworkElement mouseOverElement = e?.MouseDevice?.DirectlyOver as FrameworkElement;
            if ((vm != null) && (mouseOverElement != null) && (mouseOverElement.Name != "CheckMark"))
            {
                vm.IsExpanded = !vm.IsExpanded;
            }
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
