// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Globalization;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;
using System.Windows.Input;
using System.Windows.Media;

namespace HavokVisualDebugger.Controls
{
    public class VdbTickBar : TickBar
    {
        private static Typeface m_typeface = new Typeface("Segoe UI");
        private static Brush m_textBrush = Application.Current.FindResource("ToolForeground") as Brush;
        private static Pen m_tickPen = new Pen(m_textBrush, 1);
        private static int m_numTicks = 10;
        private double m_fontSize = 8;

        public VdbTickBar()
        {
            if (m_textBrush.CanFreeze)
            {
                m_textBrush.Freeze();
            }

            if (m_tickPen.CanFreeze)
            {
                m_tickPen.Freeze();
            }

            if (DataContext is VdbViewModel)
            {
                (DataContext as VdbViewModel).OnFontSizeChanged += delegate (double newSize)
                {
                    m_fontSize = Math.Ceiling((newSize * 2) / 3);
                };
            }

            this.LayoutUpdated += VdbTickBar_LayoutUpdated;
        }

        private void VdbTickBar_LayoutUpdated(object sender, EventArgs e)
        {
            Brush foregroundBrush = Application.Current.FindResource("ToolForeground") as Brush;
            if (foregroundBrush != m_textBrush)
            {
                m_textBrush = foregroundBrush;
                m_tickPen = new Pen(m_textBrush, 1);
                this.InvalidateVisual();
            }
        }

        protected override void OnRender(DrawingContext dc)
        {
            double numFrames = Maximum - Minimum;

            // Draw max of 10 ticks
            double numTicks = (numFrames < m_numTicks) ? numFrames : m_numTicks;

            // Early out if nothing to draw
            if ((numFrames == 0) || (numTicks == 0))
            {
                return;
            }

            string text = "";
            FormattedText formattedText = null;
            double tickFrequency = (numFrames / numTicks);

            // Per tick offset in control space.  Include control margin
            Double offsetControlSpace = ((ActualWidth - 25) / (numTicks));

            // Margin of control
            double marginOffset = 12.5;

            // Draw tick marks and text
            for (double i = 0; i <= numTicks; i++)
            {
                // Find current label text
                text = Convert.ToString(Convert.ToInt32(Minimum + (tickFrequency * i)));
                formattedText = new FormattedText(text, CultureInfo.CurrentCulture,
                    FlowDirection.LeftToRight, m_typeface, m_fontSize, m_textBrush);

                double textSizeOffset = formattedText.Width / 2;
                double tickLocation = (offsetControlSpace * i) + marginOffset;

                // Draw label (include text size offset to draw it in the middle of tick mark)
                dc.DrawText(formattedText, new Point(tickLocation - textSizeOffset, 3));

                // Draw tick mark
                dc.DrawLine(m_tickPen, new Point(tickLocation, 4), new Point(tickLocation, -3));
            }
        }
    }

    public class VdbPlaybackSlider : Slider
    {
        protected override void OnThumbDragStarted(DragStartedEventArgs e)
        {
            base.OnThumbDragStarted(e);
            VdbViewModel vm = DataContext as VdbViewModel;

            m_wasPaused = vm.IsPaused;

            if( !m_wasPaused )
            {
                vm.Pause(null);
            }
        }

        protected override void OnThumbDragCompleted(DragCompletedEventArgs e)
        {
            base.OnThumbDragCompleted(e);
            VdbViewModel vm = DataContext as VdbViewModel;

            if(!vm.PauseOnScrub && !m_wasPaused )
            {
                vm.Play(null);
                m_wasPaused = false;
            }
        }

        private bool m_wasPaused = false;
    }

    public class VdbSlider : Slider
    {
        protected override void OnInitialized(EventArgs e)
        {
            base.OnInitialized(e);

            ApplyTemplate();
            Thumb thumb = (Template.FindName("PART_Track", this) as Track).Thumb;
            if (thumb != null)
            {
                thumb.MouseEnter += Thumb_MouseEnter;
            }
        }

        private void Thumb_MouseEnter(object sender, System.Windows.Input.MouseEventArgs e)
        {
            Thumb thumb = (sender as Thumb);
            if (e.LeftButton == MouseButtonState.Pressed && thumb.IsMouseOver)
            {
                MouseButtonEventArgs args = new MouseButtonEventArgs(e.MouseDevice, e.Timestamp, MouseButton.Left);
                args.RoutedEvent = MouseLeftButtonDownEvent;
                thumb.RaiseEvent(args);
            }
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
