// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;
using System.Windows.Media.Imaging;

namespace HavokVisualDebugger.Controls
{
    /// <summary>
    /// Interaction logic for VdbColorPicker.xaml
    /// </summary>
    public partial class VdbColorPicker : UserControl
    {
        public VdbColorPickerViewModel ColorPickerVm { get; set; }
        public bool IsColorInitialized { get; set; }

        public VdbColorPicker()
        {
            InitializeComponent();

            ColorPickerVm = new VdbColorPickerViewModel(this, SaturationValueGrid.Width, SaturationValueGrid.Height, HueGrid.Width, HueGrid.Height, AlphaBitmap.Width, AlphaBitmap.Height);
            LayoutRoot.DataContext = this;
            IsColorInitialized = false;
        }

        private void SaturationValueGrid_MouseLeftButtonDown(object sender, MouseButtonEventArgs e)
        {
            // Capture mouse
            UIElement element = (UIElement)sender;
            element.CaptureMouse();

            // Set the position
            SetSvPosition(e);
        }

        private void SaturationValueGrid_MouseMove(object sender, MouseEventArgs e)
        {
            UIElement element = (UIElement)sender;
            if (e.LeftButton == MouseButtonState.Pressed && element.IsMouseCaptured)
            {
                SetSvPosition(e);
            }
        }

        private void SaturationValueGrid_MouseUp(object sender, MouseButtonEventArgs e)
        {
            UIElement element = (UIElement)sender;
            element.ReleaseMouseCapture();
        }

        public void SetSvPosition(MouseEventArgs e)
        {
            System.Windows.Point mousePosition = e.GetPosition(SaturationValueGrid);
            SetWidgetPosition(mousePosition);
        }

        public void SetWidgetPosition(System.Windows.Point position)
        {
            SetWidgetPositionInternal(position);

            // Set the result color box
            SetResultColor(position);
        }

        public void SetWidgetPositionAndColor(System.Windows.Point position, Color color)
        {
            SetWidgetPositionInternal(position);
            ColorPickerVm.SetPickColor(color);
        }

        private void SetWidgetPositionInternal(System.Windows.Point position)
        {
            // Force widget position to remain inside the canvas
            double xPos = hkMath.Clamp(position.X, 0, SvCanvas.Width);
            double yPos = hkMath.Clamp(position.Y, 0, SvCanvas.Height);

            // Move the widget
            Canvas.SetLeft(SvDragWidget, xPos);
            Canvas.SetTop(SvDragWidget, yPos);
        }
        
        public void UpdateHueWidget(System.Windows.Point position)
        {
            // Force widget position to remain inside the canvas
            double xPos = hkMath.Clamp(position.X, 0, HueCanvas.Width);
            double yPos = hkMath.Clamp(position.Y, 0, HueCanvas.Height);

            // Move the widget
            Canvas.SetLeft(HueDragWidget, xPos);
            Canvas.SetTop(HueDragWidget, yPos);
        }

        private void SetResultColor(System.Windows.Point mousePosition)
        {
            Color finalColor = ColorPickerVm.GetColorAtMousePosition(mousePosition);
            ColorPickerVm.SetPickColor(finalColor);
        }

        public void UpdateSelectedColor(Color color)
        {
            if (!m_updating && ColorPickerVm != null)
            {
                m_updating = true;

                if (!IsColorInitialized)
                {
                    ColorPickerVm.InitializeColor(color);
                }
                else
                {
                    ColorPickerVm.SetPickColor(color);
                }

                SelectedColor = hkColorUtils.ColorToArgb(color);
                m_updating = false;
            }
        }
        private bool m_updating;
        
        private void ToggleButton_Click(object sender, RoutedEventArgs e)
        {
            if (CanToggleExpand)
            {
                IsExpanded = !IsExpanded;
            }
        }

        private void ResetButton_Click(object sender, RoutedEventArgs e)
        {
            IsColorInitialized = false;
            UpdateSelectedColor(hkColorUtils.MediaColorToDrawingColor(hkColorUtils.ArgbToColor(ResetColor)));
            IsColorInitialized = true;
        }

        /// <summary>
        /// Registered Dependency Properties
        /// </summary>
        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// SelectedColor
        /// </summary>
        public int SelectedColor
        {
            get { return (int)GetValue(SelectedColorProperty); }
            set { SetValue(SelectedColorProperty, value); PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(SelectedColor))); }
        }

        // Using a DependencyProperty as the backing store for SelectedColor.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty SelectedColorProperty =
            DependencyProperty.Register(nameof(SelectedColor), typeof(int), typeof(VdbColorPicker),
                new FrameworkPropertyMetadata(0, FrameworkPropertyMetadataOptions.BindsTwoWayByDefault, SelectedColorPropertyChangedCb));
        
        public static void SelectedColorPropertyChangedCb( DependencyObject obj, DependencyPropertyChangedEventArgs args)
        {
            int color = (int)args.NewValue;
            VdbColorPicker picker = obj as VdbColorPicker;
            
            picker.UpdateSelectedColor(hkColorUtils.MediaColorToDrawingColor(hkColorUtils.ArgbToColor(color)));
            picker.IsColorInitialized = true;
        }

        /// <summary>
        /// IsExpanded
        /// </summary>
        public bool IsExpanded
        {
            get { return (bool)GetValue(IsExpandedProperty); }
            set { SetValue(IsExpandedProperty, value); }
        }

        // Using a DependencyProperty as the backing store for IsExpanded.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty IsExpandedProperty =
            DependencyProperty.Register(nameof(IsExpanded), typeof(bool), typeof(VdbColorPicker),
                new PropertyMetadata(false));

        /// <summary>
        /// ShowColorAsHex
        /// </summary>
        public bool ShowHexValue
        {
            get { return (bool)GetValue(ShowColorAsHexProperty); }
            set { SetValue(ShowColorAsHexProperty, value); }
        }

        // Using a DependencyProperty as the backing store for ShowColorAsHex.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty ShowColorAsHexProperty =
            DependencyProperty.Register(nameof(ShowHexValue), typeof(bool), typeof(VdbColorPicker),
                new PropertyMetadata(true));


        /// <summary>
        /// CanToggleExpand
        /// </summary>
        public bool CanToggleExpand
        {
            get { return (bool)GetValue(CanToggleExpandProperty); }
            set { SetValue(CanToggleExpandProperty, value); }
        }

        // Using a DependencyProperty as the backing store for CanToggleExpand.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty CanToggleExpandProperty =
            DependencyProperty.Register(nameof(CanToggleExpand), typeof(bool), typeof(VdbColorPicker), new PropertyMetadata(true));


        /// <summary>
        /// ResetColor
        /// </summary>
        public int ResetColor
        {
            get { return (int)GetValue(ResetColorProperty); }
            set { SetValue(ResetColorProperty, value); }
        }

        // Using a DependencyProperty as the backing store for ResetColor.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty ResetColorProperty =
            DependencyProperty.Register(nameof(ResetColor), typeof(int), typeof(VdbColorPicker), new FrameworkPropertyMetadata(0, ResetColorPropertyChangedCb));

        public static void ResetColorPropertyChangedCb(DependencyObject obj, DependencyPropertyChangedEventArgs args)
        {
            int color = (int)args.NewValue;
            VdbColorPicker picker = obj as VdbColorPicker;
            picker.ColorPickerVm.SetResetColor(hkColorUtils.MediaColorToDrawingColor(hkColorUtils.ArgbToColor(color)));
        }
        

        /// <summary>
        /// IsAlphaEnabled
        /// </summary>
        public bool IsAlphaEnabled
        {
            get { return (bool)GetValue(IsAlphaEnabledProperty); }
            set { SetValue(IsAlphaEnabledProperty, value); }
        }

        // Using a DependencyProperty as the backing store for IsAlphaEnabled.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty IsAlphaEnabledProperty =
            DependencyProperty.Register(nameof(IsAlphaEnabled), typeof(bool), typeof(VdbColorPicker), new FrameworkPropertyMetadata(false));


        /// <summary>
        /// AllowUnsetColor
        /// </summary>
        public bool AllowUnsetColor
        {
            get { return (bool)GetValue(AllowUnsetColorProperty); }
            set { SetValue(AllowUnsetColorProperty, value); }
        }

        // Using a DependencyProperty as the backing store for AllowUnsetColor.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty AllowUnsetColorProperty =
            DependencyProperty.Register(nameof(AllowUnsetColor), typeof(bool), typeof(VdbColorPicker), new FrameworkPropertyMetadata(false));


        /// <summary>
        /// IsColorSet
        /// </summary>
        public bool IsColorSet
        {
            get { return (bool)GetValue(IsColorSetProperty); }
            set { SetValue(IsColorSetProperty, value); PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(IsColorSet))); }
        }

        // Using a DependencyProperty as the backing store for IsColorSet.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty IsColorSetProperty =
            DependencyProperty.Register(nameof(IsColorSet), typeof(bool), typeof(VdbColorPicker),
                new FrameworkPropertyMetadata(true, FrameworkPropertyMetadataOptions.BindsTwoWayByDefault));
    }

    public class VdbColorPickerViewModel : ViewModelNotifyPropertyChanged
    {
        #region Initialization
        public VdbColorPickerViewModel()
        {
        }

        public VdbColorPickerViewModel(VdbColorPicker owner, double svWidth, double svHeight, double hueWidth, double hueHeight, double alphaWidth, double alphaHeight)
        {
            m_owner = owner;
            SvWidth = svWidth;
            SvHeight = svHeight;
            HueWidth = hueWidth;
            HueHeight = hueHeight;
            AlphaWidth = alphaWidth;
            AlphaHeight = alphaHeight;
            _AlphaValue = 255;
            
            SaturationValueWriteableBitmap = new WriteableBitmap(
                (int)SvWidth,
                (int)SvHeight,
                m_numPixels,
                m_numPixels,
                System.Windows.Media.PixelFormats.Bgr32,
                null);

            HueWriteableBitmap = new WriteableBitmap(
                (int)HueWidth,
                (int)HueHeight,
                m_numPixels,
                m_numPixels,
                System.Windows.Media.PixelFormats.Bgr32,
                null);

            AlphaWriteableBitmap = new WriteableBitmap(
                (int)AlphaWidth,
                (int)AlphaHeight,
                m_numPixels,
                m_numPixels,
                System.Windows.Media.PixelFormats.Bgr32,
                null);

            UnsetColorCommand = new ViewModelDelegateCommand(
                this,
                UnsetIsColorSet);

            SetColorCommand = new ViewModelDelegateCommand(
                this,
                SetIsColorSet);

            InitializeHueGrid();
            InitializeAlphaGrid();
        }

        private void InitializeAlphaGrid()
        {
            int width = (int)AlphaWidth;
            int height = (int)AlphaHeight;

            // Reserve the back buffer for updates.
            AlphaWriteableBitmap.Lock();
            {
                Array alphaGridValues = Array.CreateInstance(typeof(int), height, width);
                {
                    double side = height / 2.0f;
                    int whiteColor = hkColorUtils.ColorToArgb(Color.White);
                    int grayColor = hkColorUtils.ColorToArgb(Color.FromArgb(255, 211, 211, 211));

                    for (int i = 0; i < width; i++)
                    {
                        bool isFlipped = (Math.Floor(i / side) % 2) == 0;
                        int topColor = isFlipped ? whiteColor : grayColor;
                        int bottomColor = isFlipped ? grayColor : whiteColor;
                        for (int j = 0; j < side; j++)
                        {
                            alphaGridValues.SetValue(topColor, j, i);
                        }

                        for (int j = (int)side; j < height; j++)
                        {
                            alphaGridValues.SetValue(bottomColor, j, i);
                        }
                    }
                }

                int bytesPerPixel = (AlphaWriteableBitmap.Format.BitsPerPixel + 7) / 8;
                int stride = AlphaWriteableBitmap.PixelWidth * bytesPerPixel;

                Int32Rect rect = new Int32Rect(0, 0, width, height);
                AlphaWriteableBitmap.WritePixels(rect, alphaGridValues, stride, 0, 0);

                // Specify the area of the bitmap that changed.
                AlphaWriteableBitmap.AddDirtyRect(rect);
            }
            // Release the back buffer and make it available for display.
            AlphaWriteableBitmap.Unlock();
        }
        
        private void InitializeHueGrid()
        {
            int width = (int)HueWidth;
            int height = (int)HueHeight;

            // Reserve the back buffer for updates.
            HueWriteableBitmap.Lock();
            {
                Array hues = Array.CreateInstance(typeof(int), height, width);
                {
                    double scalar = 360.0 / height;

                    for (byte i = 0; i < height; i++)
                    {
                        double hue = (360.0 - (i * scalar));
                        Color hsvColor = hkColorUtils.ColorFromHSV(hue, 1.0, 1.0);
                        int argbColor = hkColorUtils.ColorToArgb(hsvColor);

                        for (byte j = 0; j < width; j++)
                        {
                            hues.SetValue(argbColor, i, j);
                        }
                    }
                }

                int bytesPerPixel = (HueWriteableBitmap.Format.BitsPerPixel + 7) / 8;
                int stride = HueWriteableBitmap.PixelWidth * bytesPerPixel;

                Int32Rect rect = new Int32Rect(0, 0, width, height);
                HueWriteableBitmap.WritePixels(rect, hues, stride, 0, 0);

                // Specify the area of the bitmap that changed.
                HueWriteableBitmap.AddDirtyRect(rect);
            }
            // Release the back buffer and make it available for display.
            HueWriteableBitmap.Unlock();
        }
        
        private void SetHue(int hue)
        {
            // Reserve the back buffer for updates.
            SaturationValueWriteableBitmap.Lock();
            {
                // Re-seed the saturation/value matrix with the new hue
                Array saturationValueArray = Array.CreateInstance(typeof(int), m_numPixels, m_numPixels);
                {
                    for (int i = 0; i < m_numPixels; i++)
                    {
                        for (int j = 0; j < m_numPixels; j++)
                        {
                            Color hsvColor = hkColorUtils.ColorFromHSV(hue, ((double)j / m_numPixels), (1.0 - ((double)i / m_numPixels)));
                            int argbColor = hkColorUtils.ColorToArgb(hsvColor);
                            saturationValueArray.SetValue(argbColor, i, j);
                        }
                    }
                }

                int bytesPerPixel = (SaturationValueWriteableBitmap.Format.BitsPerPixel + 7) / 8;
                int stride = SaturationValueWriteableBitmap.PixelWidth * bytesPerPixel;

                Int32Rect rect = new Int32Rect(0, 0, m_numPixels, m_numPixels);
                SaturationValueWriteableBitmap.WritePixels(rect, saturationValueArray, stride, 0, 0);

                // Specify the area of the bitmap that changed.
                SaturationValueWriteableBitmap.AddDirtyRect(rect);
            }
            // Release the back buffer and make it available for display.
            SaturationValueWriteableBitmap.Unlock();
        }

        public Color GetColorAtMousePosition(System.Windows.Point mousePosition)
        {
            // Calculate the x,y offset of where the mouse was clicked, in bitmap space
            double gridWidth = SvWidth;
            double gridHeight = SvHeight;
            double bitmapWidth = SaturationValueWriteableBitmap.PixelWidth;
            double bitmapHeight = SaturationValueWriteableBitmap.PixelHeight;
            int xOffset = (int)(mousePosition.X * bitmapWidth / gridWidth);
            int yOffset = (int)(mousePosition.Y * bitmapHeight / gridHeight);

            // Clamp the value
            yOffset = hkMath.Clamp(yOffset, 0, (int)gridWidth);
            xOffset = hkMath.Clamp(xOffset, 0, (int)gridHeight);

            // Get the pixel color at mouse location (note: this is not super accurate, but good enough for mouse picking)
            double sat = (double)xOffset / m_numPixels;
            double val = 1.0 - ((double)yOffset / m_numPixels);
            double hue = Hue;

            return hkColorUtils.ColorFromHSV(hue, sat, val, (int)AlphaValue);
        }
        
        public void SetPickColor(Color color)
        {
            PickedColor = new System.Windows.Media.SolidColorBrush(hkColorUtils.DrawingColorToMediaColor(color));
        }

        public void SetResetColor(Color color)
        {
            ResetColor = new System.Windows.Media.SolidColorBrush(hkColorUtils.DrawingColorToMediaColor(color));
        }

        public void InitializeColor(Color color)
        {
            SetPickColor(color);
            UpdatePickedColor(-1);
        }

        #endregion

        #region View Commands and Properties

        public double SvWidth { get; set; }
        public double SvHeight { get; set; }

        public double HueWidth { get; set; }
        public double HueHeight { get; set; }

        public double AlphaWidth { get; set; }
        public double AlphaHeight { get; set; }

        public ICommand UnsetColorCommand { get; private set; }
        public ICommand SetColorCommand { get; private set; }

        private void SetIsColorSet(object sender) { m_owner.IsColorSet = true; }
        private void UnsetIsColorSet(object sender) { m_owner.IsColorSet = false; }

        public System.Windows.Media.SolidColorBrush ResetColor
        {
            get { return _ResetColor; }
            set
            {
                _ResetColor = value;
                NotifyPropertyChanged();
                NotifyPropertyChanged(nameof(HasResetColor));
            }
        }
        private System.Windows.Media.SolidColorBrush _ResetColor;

        public bool HasResetColor
        {
            get { return _ResetColor != null && hkColorUtils.ColorToArgb(_ResetColor.Color) != 0; }
        }
        
        public bool IsColorSet
        {
            get { return m_owner.IsColorSet || !m_owner.AllowUnsetColor; }
            set
            {
                m_owner.IsColorSet = value;
                NotifyPropertyChanged();
            }
        }

        // RGB values valid from 0-255
        public double RedValue
        {
            get { return _RedValue; }
            set
            {
                value = hkMath.Clamp(value, 0, 255);
                if (_RedValue == value)
                    return;

                _RedValue = value;
                NotifyPropertyChanged();
                UpdatePickedColor(-1);
            }
        }
        private double _RedValue;

        public double GreenValue
        {
            get { return _GreenValue; }
            set
            {
                value = hkMath.Clamp(value, 0, 255);
                if (_GreenValue == value)
                    return;
                _GreenValue = value;
                NotifyPropertyChanged();
                UpdatePickedColor(-1);
            }
        }
        private double _GreenValue;

        public double BlueValue
        {
            get { return _BlueValue; }
            set
            {
                value = hkMath.Clamp(value, 0, 255);
                if (_BlueValue == value)
                    return;
                _BlueValue = value;
                NotifyPropertyChanged();
                UpdatePickedColor(-1);
            }
        }
        private double _BlueValue;

        public double AlphaValue
        {
            get { return _AlphaValue; }
            set
            {
                value = hkMath.Clamp(value, 0, 255);
                if (_AlphaValue == value)
                    return;
                _AlphaValue = value;
                NotifyPropertyChanged();
                UpdatePickedColor(-1);
            }
        }
        private double _AlphaValue;

        public System.Windows.Media.SolidColorBrush PickedColor
        {
            get { return _PickedColor; }

            set
            {
                _PickedColor = value;
                _RedValue = value.Color.R;
                _GreenValue = value.Color.G;
                _BlueValue = value.Color.B;

                if(m_owner.IsAlphaEnabled)
                {
                    _AlphaValue = value.Color.A;
                }

                NotifyColorPropertiesChanged();

                m_owner.UpdateSelectedColor(hkColorUtils.MediaColorToDrawingColor(_PickedColor.Color));
            }
        }
        private System.Windows.Media.SolidColorBrush _PickedColor;

        public string ColorAsHexString
        {
            get
            {
                System.Windows.Media.Color color = (_PickedColor != null) ? _PickedColor.Color : System.Windows.Media.Color.FromArgb(255,0,0,0);
                return hkColorUtils.ToHex(color, m_owner.IsAlphaEnabled);
            }

            set
            {
                try
                {
                    InitializeColor(hkColorUtils.MediaColorToDrawingColor((System.Windows.Media.Color)System.Windows.Media.ColorConverter.ConvertFromString(value)));
                }
                catch { }
            }
        }

        // Hue values valid from 0-360
        public int Hue
        {
            get { return _Hue; }
            set
            {
                if(_Hue != value || !m_owner.IsColorInitialized)
                {
                    _Hue = value;
                    SetHue(value);
                    UpdatePickedColor(value);

                    System.Windows.Point position = new System.Windows.Point(0, HueHeight * (1 - ((double)value / 360)));
                    m_owner.UpdateHueWidget(position);

                    NotifyPropertyChanged();
                }
            }
        }
        private int _Hue = 0;

        // The Hue bitmap.  This is a static image (hues are constant)
        public WriteableBitmap HueWriteableBitmap { get; set; }

        // The SV (Saturation and Value) bitmap.  This is a dynamic image, based on hue
        public WriteableBitmap SaturationValueWriteableBitmap { get; set; }

        // The alpha background bitmap.
        public WriteableBitmap AlphaWriteableBitmap { get; set; }

        #endregion

        #region Private Methods and Properties

        private static int m_numPixels = 100;
        private VdbColorPicker m_owner;

        private void NotifyColorPropertiesChanged()
        {
            NotifyPropertyChanged(nameof(RedValue));
            NotifyPropertyChanged(nameof(GreenValue));
            NotifyPropertyChanged(nameof(BlueValue));
            NotifyPropertyChanged(nameof(AlphaValue));
            NotifyPropertyChanged(nameof(PickedColor));
            NotifyPropertyChanged(nameof(ColorAsHexString));
        }

        private void UpdatePickedColor(double optionalComputedHue)
        {
            double hue, sat, val;
            hkColorUtils.ColorToHSV(Color.FromArgb((int)AlphaValue, (int)_RedValue, (int)_GreenValue, (int)_BlueValue), out hue, out sat, out val);

            // Set the hue.  This will seed the saturation/value grid.
            // Don't use optionalComputedHue if it's invalid
            if (optionalComputedHue < 0)
            {
                // Valid hue not passed in, potential new hue calculated - set Hue property!
                Hue = (int)hue;
            }
            else
            {
                // Valid hue passed in, Hue property already set.  Use this for further calculations.
                hue = optionalComputedHue;
            }

            // Set the saturation/value nearest locations
            double stepValue = 1.0 / m_numPixels;
            System.Windows.Point svPosition = new System.Windows.Point((sat / stepValue), ((1.0 - val) / stepValue));
            m_owner.SetWidgetPositionAndColor(svPosition, hkColorUtils.ColorFromHSV(hue, sat, val, (int)AlphaValue));
        }

        #endregion
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
