// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.ComponentModel;
using System.Timers;
using System.Windows;
using System.Windows.Threading;

namespace HavokVisualDebugger.Controls
{
    /// <summary>
    /// Interaction logic for VdbClientProgressModalWindow.xaml
    /// </summary>
    
    
    
    
    
    
    
    public partial class VdbClientProgressModalWindow : Window
    {
        static public void WaitForAccess(Havok.Vdb.Client client)
        {
            do
            {
                bool hasEntered = false;
                System.Threading.Monitor.TryEnter(client, ref hasEntered);
                if (hasEntered)
                {
                    break;
                }
                else
                {
                    WaitForProgress(client);
                }
            } while (true);
        }
        
        static public void WaitForProgress(Havok.Vdb.Client client)
        {
            // Unfortunately CurrentDispatcher getter can stall incorrectly AFAI can tell.
            //System.Diagnostics.Debug.Assert(Dispatcher.CurrentDispatcher == Application.Current.Dispatcher, "This is intended to be called from the UI thread");
            if (s_working && (s_instance == null))
            {
                ProgressViewModel progressVm = new ProgressViewModel(client);
                progressVm.Estimate = s_lastKnownEstimate;
                s_instance = new VdbClientProgressModalWindow(progressVm);
                s_instance.ShowDialog();
            }
        }

        static public void CancelProgress()
        {
            if (s_instance != null)
            {
                s_instance.Close();
            }
        }

        static public void UpdateProgress(Havok.Vdb.Client client, Havok.Vdb.WorkEstimate estimate)
        {
            System.Diagnostics.Debug.Assert(Dispatcher.CurrentDispatcher != Application.Current.Dispatcher, "This is intended to be called from a non-UI thread");
            if (estimate.EstimatedTimeRemaining != 0)
            {
                NotifyUIOfWorkProgress(client, estimate);
            }
            else
            {
                NotifyUIWorkCompleted();
            }
        }

        private VdbClientProgressModalWindow(ProgressViewModel progressVm)
        {
            InitializeComponent();
            {
                Owner = Application.Current.MainWindow;
                DataContext = progressVm;

                // Internal event references
                // (except ProgressViewModel_PropertyChanged which gets properly unhooked during Closed event)
                SourceInitialized += (s, e) =>
                {
                    if (progressVm != null)
                    {
                        hkControlUtils.SetSysMenuVisibility(this, progressVm.CanCancel);
                        progressVm.PropertyChanged += ProgressViewModel_PropertyChanged;
                    }
                };
                Closing += (s, e) =>
                {
                    if (s_instance == this)
                    {
                        // Handle user-cancellations.
                        if (progressVm != null)
                        {
                            if (progressVm.CanCancel)
                            {
                                
                                
                                
                                
                                
                                progressVm.ClearClient();
                            }
                            else
                            {
                                // Don't let the user cancel in this case.
                                // We should be effectively hiding the UI option in the !CanCancel case
                                // but users can still use Alt+F4.
                                e.Cancel = true;
                                return;
                            }
                        }
                        s_instance = null;
                    }
                    else
                    {
                        // Handle backend-completion (noted by s_instance == null)
                        if (progressVm != null)
                        {
                            progressVm.Cleanup();
                        }
                    }
                };
                Closed += (s, e) =>
                {
                    if (progressVm != null)
                    {
                        progressVm.PropertyChanged -= ProgressViewModel_PropertyChanged;
                    };
                };
            }
        }

        private void ProgressViewModel_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            ProgressViewModel progressVm = (sender as ProgressViewModel);
            if (progressVm != null)
            {
                if (e.PropertyName == nameof(ProgressViewModel.CanCancel))
                {
                    hkControlUtils.SetSysMenuVisibility(this, progressVm.CanCancel);
                }
            }
        }

        static private void NotifyUIOfWorkProgress(Havok.Vdb.Client client, Havok.Vdb.WorkEstimate estimate)
        {
            if (Dispatcher.CurrentDispatcher != Application.Current.Dispatcher)
            {
                s_lastKnownEstimate = estimate;

                // Launch dialog if needed
                if ((s_launchUIModalProgressWindowTask == null) ||
                    (s_launchUIModalProgressWindowTask.Status == DispatcherOperationStatus.Aborted))
                {
                    // Cancel any completion which may be scheduled
                    if (s_notifyUIWorkCompletedTask != null)
                    {
                        s_notifyUIWorkCompletedTask.Abort();
                        s_notifyUIWorkCompletedTask = null;
                    }

                    s_launchUIModalProgressWindowTask = Application.Current.Dispatcher.InvokeAsync(delegate
                    {
                        s_working = true;

                        // Force UI thread into modal dialog
                        WaitForProgress(client);

                        s_launchUIModalProgressWindowTask = null;
                    });
                }

                // Update on progress
                if ((s_notifyUIOfWorkProgessTask == null) ||
                    (s_notifyUIOfWorkProgessTask.Status == DispatcherOperationStatus.Aborted))
                {
                    s_notifyUIOfWorkProgessTask = Application.Current.Dispatcher.InvokeAsync(delegate
                    {
                        // Update our progress window if it exists
                        if (s_instance != null)
                        {
                            ProgressViewModel progressVm = s_instance.DataContext as ProgressViewModel;
                            progressVm.Estimate = s_lastKnownEstimate;
                        }

                        s_notifyUIOfWorkProgessTask = null;
                    });
                }
            }
            else
            {
                System.Diagnostics.Debug.Assert(false, "This is intended for calls from non-UI threads");
            }
        }

        static private void NotifyUIWorkCompleted()
        {
            if (Dispatcher.CurrentDispatcher != Application.Current.Dispatcher)
            {
                // Complete our work
                if ((s_notifyUIWorkCompletedTask == null) ||
                    (s_notifyUIWorkCompletedTask.Status == DispatcherOperationStatus.Aborted))
                {
                    // Cancel any launch which may be scheduled
                    if (s_launchUIModalProgressWindowTask != null)
                    {
                        s_launchUIModalProgressWindowTask.Abort();
                        s_launchUIModalProgressWindowTask = null;
                    }

                    s_notifyUIWorkCompletedTask = Application.Current.Dispatcher.InvokeAsync(delegate
                    {
                        s_working = false;

                        // Close the window
                        if (s_instance != null)
                        {
                            VdbClientProgressModalWindow toClose = s_instance;
                            s_instance = null; // indicates back-end closing of progress window
                            toClose.Close();
                        }

                        s_notifyUIWorkCompletedTask = null;
                    });
                }
            }
            else
            {
                System.Diagnostics.Debug.Assert(false, "This is intended for calls from non-UI threads");
            }
        }

        static private bool s_working;
        static private Havok.Vdb.WorkEstimate s_lastKnownEstimate;
        static private DispatcherOperation s_launchUIModalProgressWindowTask;
        static private DispatcherOperation s_notifyUIOfWorkProgessTask;
        static private DispatcherOperation s_notifyUIWorkCompletedTask;
        static private VdbClientProgressModalWindow s_instance;
    }

    public class ProgressViewModel : ViewModelNotifyPropertyChanged
    {
        #region Initialization
        public ProgressViewModel(Havok.Vdb.Client client)
        {
            m_client = client;
            m_timer = new Timer(700);
            m_timer.Elapsed += ProgressTimer_Elapsed;
            m_timer.Start();
        }

        private void ProgressTimer_Elapsed(object sender, ElapsedEventArgs e)
        {
            // Don't update if the last estimate was too long ago.  This is in place
            // in case the estimates stop coming in for a long period of time
            if( (DateTime.Now - m_lastEstimateUpdate).TotalSeconds < 3 )
            {
                m_workingDots = (m_workingDots + 1) % 3;
                NotifyPropertyChanged(nameof(StatusText));
            }
        }
        #endregion

        #region View Commands and Properties

        public bool IsItemsRemainingIndeterminate
        {
            get { return IsTimeRemainingIndeterminate && ((m_estimate == null) || (m_estimate.EstimatedWorkItems == m_estimate.CompletedWorkItems)); }
        }

        public bool IsTimeRemainingIndeterminate
        {
            get { return (m_estimate != null) ? (m_estimate.EstimatedTimeRemaining < 0) : true; }
        }

        public uint CompletedWorkItems
        {
            get { return (m_estimate != null) ? (m_estimate.CompletedWorkItems) : 0; }
        }

        public uint ProgressMaximum
        {
            get { return (m_estimate != null) ? Math.Max(m_estimate.EstimatedWorkItems, m_estimate.CompletedWorkItems) : 0; }
        }
        
        public string StatusText
        {
            get
            {
                if (m_estimate == null)
                {
                    return "";
                }

                string workMsg = String.IsNullOrWhiteSpace(m_estimate.Message) ? Properties.Resources.ProgressBar_Working : m_estimate.Message;
                return workMsg + new string('.', m_workingDots + 1);
            }
        }

        public string EstimatedItemsRemainingText
        {
            get
            {
                if (IsItemsRemainingIndeterminate)
                {
                    return $"{Properties.Resources.ProgressBar_ItemsRemaining} : {Properties.Resources.ProgressBar_Unknown}";
                }
                else
                {
                    uint itemsRemaining = ProgressMaximum - CompletedWorkItems;
                    return $"{Properties.Resources.ProgressBar_ItemsRemaining} : {itemsRemaining}";
                }
            }
        }

        public string EstimatedTimeRemainingText
        {
            get
            {
                if (IsTimeRemainingIndeterminate)
                {
                    return $"{Properties.Resources.ProgressBar_TimeRemaining} : {Properties.Resources.ProgressBar_Unknown}";
                }
                else
                {
                    string formattedTimeRemaining = String.Format(@"{0:m\:ss}", TimeSpan.FromSeconds(m_estimate.EstimatedTimeRemaining));
                    return $"{Properties.Resources.ProgressBar_TimeRemaining} : {formattedTimeRemaining}";
                }
            }
        }

        public bool CanCancel
        {
            get { return (m_estimate != null) && m_estimate.CanCancel; }
        }

        public void Cancel()
        {
            if (m_estimate != null)
            {
                m_estimate.Cancel();
                m_estimate = null;
            }
        }

        internal Havok.Vdb.WorkEstimate Estimate
        {
            get { return m_estimate; }
            set
            {
                bool prevIsItemsRemainingIndeterminate = IsItemsRemainingIndeterminate;
                bool prevIsTimeRemainingIndeterminate = IsTimeRemainingIndeterminate;
                uint prevCompletedWorkItems = CompletedWorkItems;
                uint prevProgressMaximum = ProgressMaximum;
                string prevStatusText = StatusText;
                string prevEstimatedItemsRemainingText = EstimatedItemsRemainingText;
                string prevEstimatedTimeRemainingText = EstimatedTimeRemainingText;
                bool prevCanCancel = CanCancel;

                m_estimate = value;
                m_lastEstimateUpdate = DateTime.Now;

                if (prevIsItemsRemainingIndeterminate != IsItemsRemainingIndeterminate)
                {
                    NotifyPropertyChanged(nameof(IsItemsRemainingIndeterminate));
                }
                if (prevIsTimeRemainingIndeterminate != IsTimeRemainingIndeterminate)
                {
                    NotifyPropertyChanged(nameof(IsTimeRemainingIndeterminate));
                }
                if (prevCompletedWorkItems != CompletedWorkItems)
                {
                    NotifyPropertyChanged(nameof(CompletedWorkItems));
                }
                if (prevProgressMaximum != ProgressMaximum)
                {
                    NotifyPropertyChanged(nameof(ProgressMaximum));
                }
                if (prevStatusText != StatusText)
                {
                    NotifyPropertyChanged(nameof(StatusText));
                }
                if (prevEstimatedItemsRemainingText != EstimatedItemsRemainingText)
                {
                    NotifyPropertyChanged(nameof(EstimatedItemsRemainingText));
                }
                if (prevEstimatedTimeRemainingText != EstimatedTimeRemainingText)
                {
                    NotifyPropertyChanged(nameof(EstimatedTimeRemainingText));
                }
                if (prevCanCancel != CanCancel)
                {
                    NotifyPropertyChanged(nameof(CanCancel));
                }
            }
        }

        public void Cleanup()
        {
            m_timer.Dispose();
        }

        public void ClearClient()
        {
            Cleanup();
            Cancel();
            m_client = null;
        }

        #endregion

        #region Private Methods and Properties
        private Havok.Vdb.Client m_client;
        private int m_workingDots;
        private Havok.Vdb.WorkEstimate m_estimate;
        private Timer m_timer;
        private DateTime m_lastEstimateUpdate;
        #endregion
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
