// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.ComponentModel;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Windows;

namespace HavokVisualDebugger
{
    /// <summary>
    /// Interaction logic for App.xaml
    /// </summary>
    public partial class App : Application
    {
        public readonly FieldInfo MenuDropAlignment;
        public static VdbViewModel ViewModel { get; set; }

        [DllImport("kernel32.dll")]
        static extern bool AttachConsole(int dwProcessId);

        [DllImport("kernel32.dll", SetLastError = true)]
        static extern bool FreeConsole();

        private const int ATTACH_PARENT_PROCESS = -1;

        public App()
        {
            // Create embedded dlls
            hkResourceUtils.EmbeddedResourceToFile("Libs", "d3dcompiler_47.dll");

            // Hook up exception handlers
            this.DispatcherUnhandledException += Dispatcher_UnhandledException;
            AppDomain.CurrentDomain.UnhandledException += CurrentDomain_UnhandledException;
            this.Startup += App_Startup;

            // Set MenuDropAlignment to false to keep pop-up alignment consistent across platforms (particularly laptops)
            {
                MenuDropAlignment = typeof(SystemParameters).GetField("_menuDropAlignment", BindingFlags.NonPublic | BindingFlags.Static);
                SetConsistentPopupAlignment();
                SystemParameters.StaticPropertyChanged += SystemParameters_StaticPropertyChanged;
            }

            VdbSettings.ForceLoad();
        }

        private void App_Startup(object sender, StartupEventArgs e)
        {
            // Attach to command line console.  If one doesn't exist this will do nothing
            AttachConsole(ATTACH_PARENT_PROCESS);

            string errString = hkCommandLineUtils.ParseCommandLineArgs(e.Args);

            if ((errString == null) || (errString != ""))
            {
                if (errString != null)
                {
                    // Command Line argument processing error, print the error and exit the app
                    hkCommandLineUtils.WriteStringToConsole(errString, true);
                }

                Application.Current.Shutdown((errString == null) ? 0 : 1);
            }

            FreeConsole();

            // Initialize ViewModel
            ViewModel = new VdbViewModel();

            // Initialize ViewModel app-resource dependent props
            ViewModel.InitializeResourceProperties();
        }

        private void SystemParameters_StaticPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            SetConsistentPopupAlignment();
        }

        /// <summary>
        /// This is needed because Windows has a parameter called "MenuDropAlignment" that is different between PCs
        /// and laptops and causes tooltips/popups to left-align instead of right-align.  This messes up where these
        /// controls end up showing, so making it consistent here by disabling the parameter.
        /// </summary>
        private void SetConsistentPopupAlignment()
        {
            // Set MenuDropAlignment to false if needed for consistency of popup alignment
            if (MenuDropAlignment != null && SystemParameters.MenuDropAlignment)
            {
                MenuDropAlignment.SetValue(null, false);
            }
        }

        private void CurrentDomain_UnhandledException(object sender, UnhandledExceptionEventArgs e)
        {
            ReportError(e.ToString());
            this.Shutdown();
        }

        private void Dispatcher_UnhandledException(object sender, System.Windows.Threading.DispatcherUnhandledExceptionEventArgs e)
        {
            string crashReport = "Crash!";
            crashReport += "\n\nException Message: " + e.Exception.Message;
            crashReport += "\n" + e.Exception.StackTrace;

            if (e.Exception != null && e.Exception.InnerException != null)
            {
                crashReport += "\n\nInner Exception:" + e.Exception.InnerException.Message;
                crashReport += "\n" + e.Exception.InnerException.StackTrace;
            }

            e.Handled = true;

            ReportError(crashReport);
            this.Shutdown();
        }

        private void ReportError(string error)
        {
#if false
            MessageBox.Show(error, "VDB Crash", MessageBoxButton.OK, MessageBoxImage.Error);
#else
            System.Diagnostics.Debug.WriteLine(error);
            System.Console.WriteLine(error);
            System.Diagnostics.Trace.WriteLine(error);
#endif
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
