// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/hkBase.h>
#include <Common/Base/Types/Color/hkColor.h>

class hkgViewport;
class hkgFont;

//////////////////////////////////////////////////////////////////////////
/// Various input types that can be combined.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkgVdbInputTypes : public hkFlagsEx<hkUint8>
{
    enum Bits
    {
        ///
        /// No input.
        NONE = 0,

        /// Mouse/Gamepad/Touch x,y and/or button clicks.
        MOUSE = ( 1 << 0 ),

        /// Any key press.
        KEYBOARD = ( 1 << 1 ),

        /// All Inputs.
        ALL = ( MOUSE | KEYBOARD )
    };

    HK_DECLARE_FLAGS_EX_CLASS( hkgVdbInputTypes, hkgVdbInputTypes::Bits );
};

//////////////////////////////////////////////////////////////////////////
/// Various input type capabilities that can be combined.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkgVdbInputCapabilities : public hkFlagsEx<hkUint8>
{
    enum Bits
    {
        ///
        /// No special input capabilities.
        NONE = 0,

        /// The widget can handle the mouse wrapping around the screen.
        MOUSE_WRAPPING = ( 1 << 0 ),
    };

    HK_DECLARE_FLAGS_EX_CLASS( hkgVdbInputCapabilities, hkgVdbInputCapabilities::Bits );
};

//////////////////////////////////////////////////////////////////////////
/// Non concrete hkgVdbPoints.
//////////////////////////////////////////////////////////////////////////
template<typename T = hkInt32>
struct HK_EXPORT_VDB hkgVdbPointBase
{
    HK_DECLARE_CLASS( hkgVdbPointBase, New );
    HK_DECLARE_POD_TYPE();

    hkgVdbPointBase( T x = 0, T y = 0 ) :
        m_x( x ),
        m_y( y )
    {}

    T m_x;
    T m_y;

    HK_INLINE void add( hkgVdbPointBase<T> other )
    {
        m_x += other.m_x;
        m_y += other.m_y;
    }

    HK_INLINE void mul( hkReal s )
    {
        m_x = T( m_x * s );
        m_y = T( m_y * s );
    }

    HK_INLINE hkReal distanceTo( hkgVdbPointBase<T> other )
    {
        T x = ( other.m_x > m_x ) ? other.m_x - m_x : m_x - other.m_x;
        T y = ( other.m_y > m_y ) ? other.m_y - m_y : m_y - other.m_y;
        return hkMath::sqrt( hkReal( x * x + y * y ) );
    }

    static hkgVdbPointBase<T> viewportCoordsFromInputCoords( T x, T y, const hkgViewport& viewport );
    static hkgVdbPointBase<T> windowCoordsFromInputCoords( T x, T y, const hkgViewport& viewport );
};

//////////////////////////////////////////////////////////////////////////
/// hkgVdbPoint impls.
//////////////////////////////////////////////////////////////////////////
using hkgVdbPoint = hkgVdbPointBase<>;
using hkgVdbPointF = hkgVdbPointBase<float>;
using hkgVdbPointD = hkgVdbPointBase<double>;


//////////////////////////////////////////////////////////////////////////
/// Describes the dimensions of something; supports fractional representation.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkgVdbDimensions : public hkgVdbPoint
{
    HK_DECLARE_CLASS( hkgVdbDimensions, New );
    HK_DECLARE_POD_TYPE();

    hkgVdbDimensions() :
        m_width( 0 ),
        m_height( 0 ),
        m_widthFraction( 1.0f ),
        m_heightFraction( 1.0f )
    {}

    /// The absolute width.
    hkInt32 m_width;

    /// The absolute height.
    hkInt32 m_height;

    /// The fractional width (used if m_width == 0).
    hkReal m_widthFraction;

    /// The fractional height (used if m_height == 0).
    hkReal m_heightFraction;
};

//////////////////////////////////////////////////////////////////////////
/// Bounds around some space.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkgVdbBounds
{
    HK_DECLARE_CLASS( hkgVdbBounds, New );
    HK_DECLARE_POD_TYPE();

    hkgVdbBounds(
        hkInt32 minX = HK_INT32_MAX,
        hkInt32 maxX = -HK_INT32_MAX,
        hkInt32 minY = HK_INT32_MAX,
        hkInt32 maxY = -HK_INT32_MAX ) :
        m_minX( minX ),
        m_maxX( maxX ),
        m_minY( minY ),
        m_maxY( maxY )
    {}

    hkInt32 m_minX;
    hkInt32 m_maxX;
    hkInt32 m_minY;
    hkInt32 m_maxY;

    hkBool32 set( const hkgViewport* v, const hkgVdbDimensions& dimensions );
    hkBool32 set( const hkgViewport* v, const hkgVdbDimensions& dimensions, hkgFont& font, const char* text );
    hkBool32 set( hkgFont& font, const char* text );
    HK_INLINE void setEmpty() { m_minX = m_maxX = m_minY = m_maxY = 0; }
    HK_INLINE hkBool32 isEmpty( hkReal tol = 0 ) const { return ( m_minX >= ( m_maxX - tol ) ) || ( m_minY >= ( m_maxY - tol ) ); }
    HK_INLINE hkInt32 getWidth() const { return ( m_maxX - m_minX ); }
    HK_INLINE hkInt32 getHeight() const { return ( m_maxY - m_minY ); }
    template<typename T> HK_INLINE hkBool32 isInside( hkgVdbPointBase<T> p ) const;
    HK_INLINE hkBool32 contains( const hkgVdbBounds& other ) const;
    //HK_INLINE hkResult tryScale( hkgVdbPoint center, hkgVdbRealPoint scales );
    template<typename T> HK_INLINE hkResult tryTranslate( hkgVdbPointBase<T> t );
    template<typename T1, typename T2> HK_INLINE void scale( hkgVdbPointBase<T1> center, hkgVdbPointBase<T2> scales );
    template<typename T> HK_INLINE void translate( hkgVdbPointBase<T> t );
    hkBool32 clamp( const hkgVdbBounds& innerBounds, const hkgVdbBounds& outerBounds );
    hkBool32 clamp( const hkgVdbBounds& other );
};

//////////////////////////////////////////////////////////////////////////
/// Class used for double, triple, etc. buffering schemes.
/// NUM_BUFFERS defines the number of buffers to keep around of TYPE.
/// NUM_CYCLING_BUFFERS defines the wrap around point for "cylceBuffers".
//////////////////////////////////////////////////////////////////////////
template<typename TYPE, int NUM_BUFFERS = 2, int NUM_CYCLING_BUFFERS = NUM_BUFFERS>
struct HK_EXPORT_VDB hkgVdbDataBuffer
{
    HK_DECLARE_CLASS( hkgVdbDataBuffer, New );
    hkgVdbDataBuffer() : m_writeBufferUpdated( false ), m_writeBufferIdx( 0 ), m_readBufferIdx( 1 ) {}

    /// Returns NUM_BUFFERS.
    HK_INLINE int getNumBuffers() const { return NUM_BUFFERS; }

    /// Gets a const reference to the current read buffer.
    HK_INLINE const TYPE& getReadBuffer() const { return m_data[m_readBufferIdx]; }

    /// Gets a reference to the write buffer.
    /// By getting access, you're indicating an intention to write to the buffer
    /// (impacts behavior of cycleBuffersIfWriteUpdated()).
    HK_INLINE TYPE& accessWriteBuffer() { return accessBuffer( m_writeBufferIdx ); }

    /// Gets a const reference to any buffer.
    HK_INLINE const TYPE& getBuffer( int idx ) const { return m_data[idx]; }

    /// Gets a reference to any buffer.
    /// If you get access to the write buffer, you're indicating an intention to write
    /// to that buffer (impacts behavior of cycleBuffersIfWriteUpdated()).
    HK_INLINE TYPE& accessBuffer( int idx )
    {
        m_writeBufferUpdated |= ( idx == m_writeBufferIdx );
        return m_data[idx];
    }

    /// Cycles the write/read buffers through, wrapping around at NUM_CYCLING_BUFFERS.
    HK_INLINE void cycleBuffers()
    {
        m_writeBufferIdx = ( ( m_writeBufferIdx + 1 ) % NUM_CYCLING_BUFFERS );
        m_readBufferIdx = ( ( m_writeBufferIdx + 1 ) % NUM_CYCLING_BUFFERS );
        m_writeBufferUpdated = false;
    }

    /// If the write buffer was accessed, cycle the buffers.
    HK_INLINE void cycleBuffersIfWriteUpdated() { if ( m_writeBufferUpdated ) cycleBuffers(); }

private:

    HK_COMPILE_TIME_ASSERT( NUM_BUFFERS >= 2 );
    HK_COMPILE_TIME_ASSERT( NUM_CYCLING_BUFFERS >= 2 );
    HK_COMPILE_TIME_ASSERT( NUM_CYCLING_BUFFERS <= NUM_BUFFERS );

    TYPE m_data[NUM_BUFFERS];
    hkBool32 m_writeBufferUpdated;
    int m_readBufferIdx;
    int m_writeBufferIdx;
};

#include <VisualDebugger/VdbDisplay/Hkg/System/hkgVdbTypes.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
