// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPlugin.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/hkgVdbTypes.h>

#include <Common/Base/Container/String/hkUtf8.h>

#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/Font/hkgFont.h>

namespace
{
    template<typename T>
    hkgVdbPointBase<T> viewportCoordsFromInputCoordsInternal( T x, T y, const hkgViewport& viewport )
    {
        int vx, vy;
        viewport.getLowerLeftCoord( vx, vy );
        const T mvx = T( x - vx );
        const T hkgmvy = T( y - vy );
        return hkgVdbPointBase<T>( mvx, hkgmvy );
    }

    template<typename T>
    hkgVdbPointBase<T> windowCoordsFromInputCoordsInternal( T x, T y, const hkgViewport& viewport )
    {
        int vx, vy;
        viewport.getLowerLeftCoord( vx, vy );
        const T mvx = T( x - vx );
        const T hkgmvy = T( viewport.getOwnerWindow()->getHeight() - ( y - vy ) );
        return hkgVdbPointBase<T>( mvx, hkgmvy );
    }
}

template<> hkgVdbPointBase<hkInt32> hkgVdbPointBase<hkInt32>::viewportCoordsFromInputCoords( hkInt32 x, hkInt32 y, const hkgViewport& viewport )
{
    return viewportCoordsFromInputCoordsInternal( x, y, viewport );
}

template<> hkgVdbPointBase<double> hkgVdbPointBase<double>::viewportCoordsFromInputCoords( double x, double y, const hkgViewport& viewport )
{
    return viewportCoordsFromInputCoordsInternal( x, y, viewport );
}

template<> hkgVdbPointBase<hkInt32> hkgVdbPointBase<hkInt32>::windowCoordsFromInputCoords( hkInt32 x, hkInt32 y, const hkgViewport& viewport )
{
    return windowCoordsFromInputCoordsInternal( x, y, viewport );
}

template<> hkgVdbPointBase<double> hkgVdbPointBase<double>::windowCoordsFromInputCoords( double x, double y, const hkgViewport& viewport )
{
    return windowCoordsFromInputCoordsInternal( x, y, viewport );
}

hkBool32 hkgVdbBounds::set( const hkgViewport* v, const hkgVdbDimensions& dimensions )
{
    hkInt32 w = 0;
    hkInt32 h = 0;

    // Get viewport properties
    if ( v )
    {
        w = v->getWidth();
        h = v->getHeight();
    }

    // Considered too small to draw
    if ( ( w <= 0 ) || ( h <= 5 ) )
    {
        setEmpty();
        return false;
    }

    // Compute our effective width and height.
    hkInt32 effW =
        dimensions.m_width ?
        dimensions.m_width :
        hkInt32( w * dimensions.m_widthFraction );
    hkInt32 effH =
        dimensions.m_height ?
        dimensions.m_height :
        hkInt32( h * dimensions.m_heightFraction );

    // Assign final values.
    m_minX = dimensions.m_x;
    m_minY = dimensions.m_y;
    m_maxX = ( m_minX + effW );
    m_maxY = ( m_minY + effH );

    return true;
}

hkBool32 hkgVdbBounds::set( const hkgViewport* v, const hkgVdbDimensions& dimensions, hkgFont& font, const char* text )
{
    hkgVdbBounds txtBounds;
    txtBounds.set( font, text );

    // Override width if dimensions specify
    hkInt32 maxWidth = txtBounds.getWidth();
    if ( dimensions.m_width != 0 )
    {
        maxWidth = dimensions.m_width;
    }
    else if ( dimensions.m_widthFraction != 0 )
    {
        // Auto-width based on window size
        if ( v )
        {
            maxWidth = hkInt32( v->getWidth() * dimensions.m_widthFraction );
        }
        else
        {
            maxWidth = HK_INT32_MAX;
        }
    }

    // Override height if dimensions specify
    hkInt32 maxHeight = txtBounds.getHeight();
    if ( dimensions.m_height != 0 )
    {
        maxHeight = dimensions.m_height;
    }
    else if ( dimensions.m_heightFraction != 0 )
    {
        // Auto-height based on window size
        if ( v )
        {
            maxHeight = hkInt32( v->getHeight() * dimensions.m_heightFraction );
        }
        else
        {
            maxHeight = HK_INT32_MAX;
        }
    }

    // Set from x,y
    m_minX = dimensions.m_x;
    m_maxX = dimensions.m_x + maxWidth;
    m_minY = dimensions.m_y;
    m_maxY = dimensions.m_y + maxHeight;

    return true;
}

hkBool32 hkgVdbBounds::set( hkgFont& font, const char* text )
{
    if ( !text || ( *text ) == '\0' )
    {
        setEmpty();
        return false;
    }

    float maxWidthP = 0;
    float widthP = 0;
    int lines = 1;
    for ( hkUtf8::Iterator iter( text ); iter.advance(); /**/ )
    {
        hkUtf8::CodePoint cp = iter.current();
        // Handle \n
        if ( cp == '\n' )
        {
            lines++;
            widthP = 0;
        }
        else if ( cp == '\t' )
        {
            int tab = ( int ) font.getTabStopSize();
            int cwidth = ( int ) font.getCharWidth();
            int w = hkMath::hkFloatToInt( widthP * cwidth );
            widthP = float( ( ( w + tab ) / tab ) * tab ) / cwidth; // next whole multiple of tab pixels, in units of font relative widths
        }
        else
        {
            float w = 0;
            font.preResolve( cp, w );
            widthP += w;
        }
        maxWidthP = maxWidthP < widthP ? widthP : maxWidthP;
    }

    // Auto width based on text
    hkReal cwidth = font.getCharWidth();
    hkInt32 maxWidth = hkUint32( maxWidthP * cwidth + 5 );
    hkReal cheight = font.getCharHeight();
    hkInt32 maxHeight = hkInt32( lines * cheight );

    // Set from 0,0
    m_minX = 0;
    m_maxX = maxWidth;
    m_minY = 0;
    m_maxY = maxHeight;

    return true;
}

hkBool32 hkgVdbBounds::clamp( const hkgVdbBounds& innerBounds, const hkgVdbBounds& outerBounds )
{
    if ( !outerBounds.contains( innerBounds ) )
    {
        HK_WARN_ONCE( 0x22440953, "Larger bounds must contain smaller bounds" );
        return false;
    }

    hkInt32 minX = hkMath::clamp( m_minX, outerBounds.m_minX, innerBounds.m_minX );
    hkInt32 maxX = hkMath::clamp( m_maxX, innerBounds.m_maxX, outerBounds.m_maxX );
    hkInt32 minY = hkMath::clamp( m_minY, outerBounds.m_minY, innerBounds.m_minY );
    hkInt32 maxY = hkMath::clamp( m_maxY, innerBounds.m_maxY, outerBounds.m_maxY );

    hkBool32 clamped =
        ( m_minX != minX ) ||
        ( m_maxX != maxX ) ||
        ( m_minY != minY ) ||
        ( m_maxY != maxY );

    m_minX = minX;
    m_maxX = maxX;
    m_minY = minY;
    m_maxY = maxY;

    return clamped;
}

hkBool32 hkgVdbBounds::clamp( const hkgVdbBounds& other )
{
    hkInt32 minX = hkMath::max2( m_minX, other.m_minX );
    hkInt32 maxX = hkMath::min2( m_maxX, other.m_maxX );
    hkInt32 minY = hkMath::max2( m_minY, other.m_minY );
    hkInt32 maxY = hkMath::min2( m_maxY, other.m_maxY );

    hkBool32 clamped =
        ( m_minX != minX ) ||
        ( m_maxX != maxX ) ||
        ( m_minY != minY ) ||
        ( m_maxY != maxY );

    m_minX = minX;
    m_maxX = maxX;
    m_minY = minY;
    m_maxY = maxY;

    return clamped;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
