// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Container/Hash/hkHashMap.h>

#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/Utils/hkgVdbTextBoxDrawer.h>

class hkVdbClient;
class hkgVdbPlugin;
class hkgVdbWidget;
class hkgVdb2dWidget;
class hkgVdb3dWidget;
class hkgViewport;
class hkgInputManager;

//////////////////////////////////////////////////////////////////////////
/// Flags defining behavior of widget layout.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkgVdbWidgetLayoutFlags : public hkFlagsEx<hkUint8>
{
    enum Bits
    {
        /// If enabled, the system will recompute positions of widgets as they are
        /// enabled/disabled.
        AUTO_LAYOUT_OF_BUILTIN_WIDGETS = ( 1 << 0 ),

        /// Default behavior.
        DEFAULT = ( AUTO_LAYOUT_OF_BUILTIN_WIDGETS )
    };

    HK_DECLARE_FLAGS_EX_CLASS( hkgVdbWidgetLayoutFlags, hkgVdbWidgetLayoutFlags::Bits );
};

//////////////////////////////////////////////////////////////////////////
/// Flags defining behavior related to widget focus.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkgVdbWidgetFocusFlags : public hkFlagsEx<hkUint8>
{
    enum Bits
    {
        
        //
        // Input handling (regardless of focus)
        //

        /// When the mouse hovers over a widget, it can respond to mouse events
        /// (even if it does not have focus) unless they are captured by another widget.
        ENABLE_MOUSE_INPUTS_ON_HOVER = ( 1 << 0 ),

        /// 3d widgets can always respond to inputs (even if it does not have focus)
        /// unless they are captured by another widget.
        ENABLE_ALL_INPUTS_FOR_3D_WIDGETS = ( 1 << 1 ),

        //
        // Auto focus handling
        //

        /// Any mouse inputs that would be enabled by ENABLE_MOUSE_INPUTS_ON_HOVER
        /// additionally result in widget acquiring focus for mouse input.
        AUTO_FOCUS_MOUSE_ON_HOVER = ( 1 << 2 ),

        /// When a widget acquires focus automatically (AUTO_FOCUS_MOUSE_ON_HOVER),
        /// it will also capture the input until the input is *released* (such as mouse up).
        /// The widget must still properly *capture* inputs it wants to hold after
        /// the input is released.
        AUTO_FOCUS_CAPTURES_INPUT = ( 1 << 3 ),

        /// Disable automatic focus handling, but still allow enabling inputs, etc.
        /// based on focus set from external method calls.
        DISABLE_AUTO_FOCUS = ( 1 << 4 ),

        //
        // General
        //

        /// Default behavior.
        DEFAULT = ( ENABLE_MOUSE_INPUTS_ON_HOVER | ENABLE_ALL_INPUTS_FOR_3D_WIDGETS | AUTO_FOCUS_MOUSE_ON_HOVER | AUTO_FOCUS_CAPTURES_INPUT ),

        /// Disable all focus and input management.
        DISABLED = ( 1 << ( sizeof( hkgVdbWidgetFocusFlags::StorageType ) * 8 - 1 ) )
        
    };

    HK_DECLARE_FLAGS_EX_CLASS( hkgVdbWidgetFocusFlags, hkgVdbWidgetFocusFlags::Bits );
};

//////////////////////////////////////////////////////////////////////////
/// Options for how the handler renders hkgVdbWidgets.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkgVdbWidgetRenderOptions
{
    HK_DECLARE_CLASS( hkgVdbWidgetRenderOptions, New );

    hkgVdbWidgetRenderOptions( hkgFont& font ) :
        m_focusBorderColor( hkColor::WHITE ),
        m_inputBorderColor( hkColor::YELLOW ),
        m_showToolTips( HKG_VDB_DEFAULT_SHOW_TOOLTIPS ),
        m_showVerboseToolTips( HKG_VDB_DEFAULT_SHOW_VERBOSE_TOOLTIPS ),
        m_showToolTipDelay( HKG_VDB_DEFAULT_SHOW_TOOLTIP_DELAY ),
        m_toolTipDrawOptions( font )
    {
        // We set x/y to max to aid in determining if user sets them explicitly
        m_toolTipDrawOptions.m_x = HK_INT32_MAX;
        m_toolTipDrawOptions.m_y = HK_INT32_MAX;

        // Default colors
        m_toolTipDrawOptions.m_upperLeftColor = hkColor::ANTIQUEWHITE;
        m_toolTipDrawOptions.m_lowerRightColor = hkColor::LIGHTGREY;
        m_toolTipDrawOptions.m_textColor = hkColor::BLACK;
        m_toolTipDrawOptions.m_borderColor = hkColor::BLACK;
    }

    /// The currently focused widget will have a border drawn around it's bounds in this color.
    /// Setting to NONE will disable.
    hkColor::Argb m_focusBorderColor;

    /// Widgets receiving inputs that do not have focus will have a border drawn around it's bounds in this color.
    /// Setting to NONE will disable.
    hkColor::Argb m_inputBorderColor;

    /// If true, widget-provided tool tips will be shown when hovering the mouse over them.
    hkBool m_showToolTips;

    /// If true, widgets may give more verbose tooltips.
    hkBool m_showVerboseToolTips;

    /// If m_displayToolTips is true, determines how long the mouse must not move over a widget
    /// before the widget-provided toolTip is shown.
    hkReal m_showToolTipDelay;

    /// Determines the style of the toolTips.
    hkgVdbTextBoxDrawerOptions m_toolTipDrawOptions;
};

//////////////////////////////////////////////////////////////////////////
/// Defines the installation ordering of the built-in widgets.
/// This is a fixed ordering to reduce complexities around diagnosing
/// issues that can arise when graphical state is changed by a widget
/// during render:
/// 1. First, all 3D widgets are rendered in order; custom 3D widgets are rendered last by installation order
/// 2. Next, all 2D widgets are rendered in order; custom 2D widgets are rendered last by installation order
/// 3. All 2D widget borders are rendered
/// 3. The tooltip (if any) is rendered
/// Please refer to the widget implementation class for more information
/// about each widget.
//////////////////////////////////////////////////////////////////////////

struct hkgVdbWidgetIndices
{
    enum Enum
    {
        //
        // 3D widgets
        //

        /// Handles viewport inputs.
        VIEWPORT,

        /// Handler selection inputs and rendering.
        SELECTION,

        /// Renders and camera orientation frame.
        ORIENTATION_FRAME,

        /// Renders an axis at the origin.
        ORIGIN_FRAME,

        /// Renders a grid for the scene.
        SCENE_GRID,

        // Note: Custom 3d widgets would get drawn here; in installation order.

        //
        // 2D widgets
        //

        /// A widget rendering stats text for the current frame.
        STAT_TEXT,

        /// A widget rendering various status text.
        STATUS_TEXT,

        /// A widget rendering stat bars for the current frame.
        /// (This has been *deprecated* in favor of a new implementation which allows for
        /// expanding/collapsing of timer bars).
        DEPRECATED_STAT_BAR_GRAPH,

        /// A widget rendering a stat line graph over time.
        STAT_LINE_GRAPH,

        /// A widget rendering expandable stat bars for the current frame.
        STAT_BAR_GRAPH,

        // Note: Custom 2d widgets would get drawn here; in installation order.

        /// The builtin widget count.
        NUM_BUILTIN_WIDGETS,

        /// The starting index for custom widgets.
        START_CUSTOM_WIDGETS = NUM_BUILTIN_WIDGETS
    };
};

//////////////////////////////////////////////////////////////////////////
/// A class for managing hkgVdbWidgets.
/// Widgets that want to react to input and participate in rendering should
/// be installed with the widget manager.
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkgVdbWidgetManager : public hkVdbDefaultErrorReporter
{
public:
    HK_DECLARE_CLASS( hkgVdbWidgetManager, New );
    hkgVdbWidgetManager( hkgVdbPlugin& owner );
    ~hkgVdbWidgetManager();

    //
    // Registration
    //

    hkResult registerSelf( hkgVdbPlugin& plugin, hkVdbClient& client );
    hkResult unregisterSelf( hkgVdbPlugin& plugin, hkVdbClient& client );

    //
    // Control
    //

    /// Update focus and call update on widgets.
    virtual hkResult update( hkgViewport& viewport );

    /// Call render on widgets (setting up some common graphics state).
    virtual void render( hkgViewport& viewport ) const;

    /// Access the current render options.
    HK_INLINE hkgVdbWidgetRenderOptions& accessOptions() { return m_options; }

    //
    // Widgets
    //

    /// Get the flags currently governing layout-related behavior.
    HK_INLINE hkgVdbWidgetLayoutFlags getLayoutFlags() const { return m_focusFlags; }

    /// Set the flags currently governing layout-related behavior.
    HK_INLINE void setLayoutFlags( hkgVdbWidgetLayoutFlags flags ) { m_focusFlags = flags; }

    /// Installs a widget.
    /// If widgetHandle is negative, then the widget is appended and the handle is updated.
    /// If widgetHandler is zero or positive, the widget will be installed at that location
    /// in the array of installed widgets (which also determines execution order), returns failure
    /// if that position is already occupied.
    /// Note: Default widgets are installed during initialization.
    virtual hkResult installWidget( hkgVdbWidget& widget, int& widgetHandleInOut );

    /// Called during initialization to install our builtin widgets.
    /// But can also be called later to reset installed widget state.
    virtual hkResult installBuiltinWidgets();

    /// Uninstalls a widget (if installed).
    virtual hkResult uninstallWidget( int widgetHandle );

    /// Uninstalls all widgets.
    virtual hkResult uninstallAllWidgets();

    /// Returns the widget at the specified point (based on widget's reported bounds);
    /// or null if none found.
    hkgVdb2dWidget* getWidgetAt( hkgVdbPoint p ) const;

    /// Get an installed widget.
    HK_INLINE hkgVdbWidget* getInstalledWidget( int widgetHandle ) const;

    /// Get an installed widget casted to type.
    template <typename T> HK_INLINE T* getInstalledWidget( int widgetHandle ) const
    { return hkDynCast<T>( getInstalledWidget( widgetHandle ) ); }

    /// Get an installed widget of type T (must add DECLARE_INSTALLED_WIDGET_GETTER for custom types).
    template <typename T> HK_INLINE T* getInstalledWidget() const;

    /// Const access to the currently installed widgets list.
    HK_INLINE const hkArray<hkRefPtr<hkgVdbWidget>>& getInstalledWidgets() const { return m_installedWidgets; }

    //
    // Focus
    // Note: This is a passive system. Changing the input focus may not take effect immediately
    // if another widget has currently captured the input for example.
    //

    /// Get the flags currently governing focus-related behavior.
    HK_INLINE hkgVdbWidgetFocusFlags getFocusFlags() const { return m_focusFlags; }

    /// Set the flags currently governing focus-related behavior.
    HK_INLINE void setFocusFlags( hkgVdbWidgetFocusFlags flags ) { m_focusFlags = flags; }

    /// Set the input focus for the provided types to any widgets at the provided point.
    /// Optionally set those inputs as captured by the widgets.
    virtual void setInputFocus( hkgVdbInputTypes inputTypes, hkgVdbPoint mp, hkBool32 captureInputs );

    /// Set the input focus for the provided types to the provided widget.
    /// Optionally set those inputs as captured by the widget.
    virtual void setInputFocus( hkgVdbInputTypes inputTypes, hkgVdbWidget* widget, hkBool32 captureInputs );

    /// Get the widget that currently has focus for the provided input type.
    virtual hkgVdbWidget* getInputFocus( hkgVdbInputTypes inputType ) const;

    /// Get input types that are captured.
    hkgVdbInputTypes getCapturedInputs() const;

protected:

    void renderTooltip() const;
    void updateLayout( hkgViewport& viewport );
    void updateFocus( hkgViewport& viewport, const hkgInputManager& input );
    void getCapturedInputsInternal( hkgVdbInputTypes& userCapturedInputs, hkgVdbInputTypes& autoCapturedInputs ) const;

    hkRefPtr<hkgFont> m_defaultFont;
    mutable hkgVdbWidgetRenderOptions m_options;

    hkViewPtr<hkgVdbPlugin> m_plugin;
    hkRefPtr<hkVdbClient> m_client;
    hkArray< hkRefPtr<hkgVdbWidget> > m_installedWidgets;
    hkViewPtr<hkgVdb2dWidget> m_widgetUnderMouse;
    hkUint64 m_showToolTipAtTicks;
    hkgVdbWidgetLayoutFlags m_layoutFlags;
    hkgVdbWidgetFocusFlags m_focusFlags;
    hkHashMap<hkgVdbInputTypes, hkgVdbWidget*> m_focussedWidgets;
    hkHashMap<hkgVdbInputTypes, hkgVdbWidget*> m_capturedInputWidgets;
};

hkgVdbWidget* hkgVdbWidgetManager::getInstalledWidget( int widgetHandle ) const
{
    if ( ( widgetHandle >= 0 ) && ( widgetHandle < m_installedWidgets.getSize() ) )
    {
        return m_installedWidgets[widgetHandle];
    }
    else
    {
        return HK_NULL;
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
