// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPlugin.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/hkgVdbWidgetManager.h>

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPluginApi.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdb2dWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/3d/hkgVdb3dWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbOrientationWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbStatTextWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbDeprecatedStatBarGraphWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbStatBarGraphWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbStatLineGraphWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbStatusWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/3d/hkgVdbGridWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/3d/hkgVdbOriginWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/3d/hkgVdbSelectionWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/3d/hkgVdbViewportWidget.h>

#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/Font/hkgFont.h>

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/hkVdbClient.h>

#define DEBUG_LOG_IDENTIFIER "vdb.hkg.System.Widget"
#include <Common/Base/System/Log/hkLog.hxx>

// Update FOREACH_BUILTIN_WIDGET if you added a builtin widget
HK_COMPILE_TIME_ASSERT( hkgVdbWidgetIndices::NUM_BUILTIN_WIDGETS == 10 );
#define FOREACH_BUILTIN_WIDGET( ACTION ) \
    ACTION( hkgVdbWidgetIndices::VIEWPORT, hkgVdbViewportWidget ) \
    ACTION( hkgVdbWidgetIndices::SELECTION, hkgVdbSelectionWidget ) \
    ACTION( hkgVdbWidgetIndices::ORIENTATION_FRAME, hkgVdbOrientationWidget ) \
    ACTION( hkgVdbWidgetIndices::ORIGIN_FRAME, hkgVdbOriginWidget ) \
    ACTION( hkgVdbWidgetIndices::SCENE_GRID, hkgVdbGridWidget ) \
    ACTION( hkgVdbWidgetIndices::STAT_TEXT, hkgVdbStatTextWidget ) \
    ACTION( hkgVdbWidgetIndices::STATUS_TEXT, hkgVdbStatusWidget ) \
    ACTION( hkgVdbWidgetIndices::DEPRECATED_STAT_BAR_GRAPH, hkgVdbDeprecatedStatBarGraphWidget ) \
    ACTION( hkgVdbWidgetIndices::STAT_LINE_GRAPH, hkgVdbStatLineGraphWidget ) \
    ACTION( hkgVdbWidgetIndices::STAT_BAR_GRAPH, hkgVdbStatBarGraphWidget )

#define DECLARE_INSTALLED_WIDGET_GETTER( IDX, TYPE ) \
    class TYPE; \
    template <> \
    TYPE* hkgVdbWidgetManager::getInstalledWidget() const \
    { \
        return reinterpret_cast< TYPE* >( m_installedWidgets[hkgVdbWidgetIndices::IDX].val() ); \
    }

FOREACH_BUILTIN_WIDGET( DECLARE_INSTALLED_WIDGET_GETTER );

namespace
{
    HK_INLINE void setCommonBorderRenderState( hkgDisplayContext& context )
    {
        // Set default border rendering state
        hkgViewport* ov = context.getOwner()->getWindowOrthoView();
        ov->setAsCurrent( &context );
        context.matchState(
            ( hkgVdb2dWidget::DEFAULT_2D_ENABLED_STATE ) ? hkgVdb2dWidget::DEFAULT_2D_ENABLED_STATE : context.getEnabledState(),
            ( hkgVdb2dWidget::DEFAULT_2D_CULLFACE_MODE ) ? hkgVdb2dWidget::DEFAULT_2D_CULLFACE_MODE : context.getCullfaceMode(),
            ( hkgVdb2dWidget::DEFAULT_2D_BLEND_MODE ) ? hkgVdb2dWidget::DEFAULT_2D_BLEND_MODE : context.getBlendMode(),
            ( hkgVdb2dWidget::DEFAULT_2D_SAMPLE_MODE ) ? hkgVdb2dWidget::DEFAULT_2D_SAMPLE_MODE : context.getAlphaSampleMode()
        );
        ov->setDesiredState( ov->getDesiredState() & ~HKG_ENABLED_TEXTURE2D );
    }
}

hkgVdbWidgetManager::hkgVdbWidgetManager( hkgVdbPlugin& owner ) :
    hkVdbDefaultErrorReporter( &s_debugLog ),
    m_defaultFont( hkRefNew<hkgFont>( hkgFont::create() ) ),
    m_options( *m_defaultFont ),
    m_plugin( &owner ),
    m_showToolTipAtTicks( 0 ),
    m_layoutFlags( hkgVdbWidgetLayoutFlags::DEFAULT ),
    m_focusFlags( hkgVdbWidgetFocusFlags::DEFAULT )
{}

hkgVdbWidgetManager::~hkgVdbWidgetManager()
{}

hkResult hkgVdbWidgetManager::registerSelf( hkgVdbPlugin& plugin, hkVdbClient& client )
{
    m_defaultFont = hkgVdbTextBoxDrawer::loadDefaultFont( *m_plugin->getWindow()->getContext(), int( HKG_VDB_WIDGET_DEFAULT_TEXT_SIZE * m_plugin->getWindow()->getDpiScale() ) );
    m_options.m_toolTipDrawOptions.m_font = m_defaultFont;

    m_client = &client;

    HK_VDB_FAIL_IF_OPERATION_FAILED( installBuiltinWidgets() );

    setInputFocus( hkgVdbInputTypes::ALL, m_installedWidgets[hkgVdbWidgetIndices::VIEWPORT], false );

    return HK_SUCCESS;
}

hkResult hkgVdbWidgetManager::unregisterSelf( hkgVdbPlugin& plugin, hkVdbClient& client )
{
    setInputFocus( hkgVdbInputTypes::ALL, m_installedWidgets[hkgVdbWidgetIndices::VIEWPORT], false );

    hkResult result = uninstallAllWidgets();

    m_client = HK_NULL;
    m_defaultFont = HK_NULL;
    m_options.m_toolTipDrawOptions.m_font = HK_NULL;

    return result;
}

hkResult hkgVdbWidgetManager::update( hkgViewport& viewport )
{
    if ( m_installedWidgets.getSize() == 0 )
    {
        return HK_SUCCESS;
    }

    hkgInputManager* input = viewport.getOwnerWindow()->getInputManager();
    HK_VDB_VERIFY_CONDITION_MSG(
        input,
        0xadb00019,
        hkVdbError::PLUGIN_SETUP_INVALID,
        "Can't update widgets without input manager" );

    // Note: There are merits to updating focus before and after.
    //  Before allows immediate response to mouse down (but will be a slightly stale bounds check).
    //  After allows immediate response to mouse up (from a widget releasing captured mouse).
    updateFocus( viewport, *input );

    // Note: There are merits to updating layout before and after.
    //  Before means that the x/y of the bounds will be reasonable after a resize.
    //  After means that the width/height of the bounds will be reasonable after a resize.
    // Before produces less jarring of a change as the widget doesn't flash from one location
    // to another, it just might start of smaller/bigger and grow/shrink in one frame.
    // Could do two passes, but seems unnessary.
    updateLayout( viewport );

    bool succeeded = true;
    for ( int i = 0; i < m_installedWidgets.getSize(); i++ )
    {
        hkgVdbWidget* widget = m_installedWidgets[i];
        if ( widget && widget->getEnabled() )
        {
            if ( widget->update( viewport, *input ).isFailure() )
            {
                HK_VDB_SIGNAL_REPORTER_ERROR( *widget, hkVdbError::PLUGIN_ERROR );
                succeeded = false;
            }
        }
    }
    return succeeded ? HK_SUCCESS : HK_FAILURE;
}

void hkgVdbWidgetManager::render( hkgViewport& viewport ) const
{
    hkgDisplayContext* context = m_plugin->getWindow()->getContext();
    context->lock();
    viewport.setAsCurrent( context );
    {
        hkgVdb3dWidget::setCommonRenderState( *context );
        {
            for ( int i = 0; i < m_installedWidgets.getSize(); ++i )
            {
                hkgVdb3dWidget* widget3d = hkDynCast( m_installedWidgets[i] );
                if ( widget3d && widget3d->getEnabled() )
                {
                    widget3d->render( *context );
                }
            }
        }

        hkgVdb2dWidget::setCommonRenderState( *context );
        {
            for ( int i = 0; i < m_installedWidgets.getSize(); ++i )
            {
                hkgVdb2dWidget* widget2d = hkDynCast( m_installedWidgets[i] );
                if ( widget2d && widget2d->getEnabled() )
                {
                    widget2d->render( *context );
                }
            }
        }

        setCommonBorderRenderState( *context );
        {
            for ( int i = 0; i < m_installedWidgets.getSize(); ++i )
            {
                hkgVdb2dWidget* widget2d = hkDynCast( m_installedWidgets[i] );
                if ( widget2d && widget2d->getEnabled() )
                {
                    hkColor::Argb color;
                    if ( widget2d->getEnabledInputs().anyIsSet( hkgVdbInputTypes::KEYBOARD ) )
                    {
                        color = m_options.m_focusBorderColor;
                    }
                    else if ( widget2d->getEnabledInputs() )
                    {
                        color = m_options.m_inputBorderColor;
                    }
                    else
                    {
                        continue;
                    }

                    hkgVdbBounds bounds = widget2d->getBounds();
                    if ( bounds.isEmpty() )
                    {
                        continue;
                    }

                    // Draw
                    {
                        float p[3]; p[2] = -0.01f; //depth (0..-2)
                        float tl[2]; float lr[2];
                        const int border = 5;

                        tl[0] = ( float ) bounds.m_minX - border;
                        tl[1] = ( float ) viewport.getHeight() - bounds.m_minY + border;
                        lr[0] = ( float ) bounds.m_maxX + border;
                        lr[1] = ( float ) viewport.getHeight() - bounds.m_maxY - border;

                        float borderColor[4] =
                        {
                            hkColor::getRedAsFloat( color ),
                            hkColor::getGreenAsFloat( color ),
                            hkColor::getBlueAsFloat( color ),
                            hkColor::getAlphaAsFloat( color )
                        };

                        if ( borderColor[3] != 0.0f )
                        {
                            context->setCurrentColor4( borderColor );
                            context->beginGroup( HKG_IMM_LINES );
                            p[0] = tl[0]; p[1] = tl[1];
                            context->setCurrentPosition( p );
                            p[0] = tl[0]; p[1] = lr[1];
                            context->setCurrentPosition( p );
                            context->setCurrentPosition( p );
                            p[0] = lr[0]; p[1] = lr[1];
                            context->setCurrentPosition( p );
                            context->setCurrentPosition( p );
                            p[0] = lr[0]; p[1] = tl[1];
                            context->setCurrentPosition( p );
                            context->setCurrentPosition( p );
                            p[0] = tl[0]; p[1] = tl[1];
                            context->setCurrentPosition( p );
                            context->endGroup();
                        }
                    }
                }
            }
        }

        renderTooltip();
    }
    context->unlock();
}

hkResult hkgVdbWidgetManager::installWidget( hkgVdbWidget& widget, int& widgetHandleInOut )
{
    HK_VDB_VERIFY_CONDITION_MSG(
        m_plugin->getWindow(),
        0xadb00057,
        hkVdbError::INVALID_STATE,
        "Can't install widgets when plugin is not initialized" );

    // Get default handle
    if ( widgetHandleInOut < 0 )
    {
        widgetHandleInOut = m_installedWidgets.getSize();
        m_installedWidgets.expandOne();
    }
    // Or expand handles
    else if ( widgetHandleInOut >= m_installedWidgets.getSize() )
    {
        m_installedWidgets.setSize( widgetHandleInOut + 1 );
    }

    HK_VDB_VERIFY_CONDITION_MSG(
        !m_installedWidgets[widgetHandleInOut],
        0xadb00058,
        hkVdbError::INVALID_ARGUMENTS,
        "There is already a widget installed at " << widgetHandleInOut );

    

    // Install/initialize new widget
    {
        hkResult result;
        hkgDisplayContext* context = m_plugin->getWindow()->getContext();
        context->lock();
        {
            widget.setContext( context );
            result = widget.initialize( *m_plugin, *m_client );
        }
        context->unlock();

        HK_VDB_VERIFY_REPORTER_OPERATION( result, widget, hkVdbError::PLUGIN_ERROR );
        m_installedWidgets[widgetHandleInOut] = &widget;
    }

    return HK_SUCCESS;
}

hkResult hkgVdbWidgetManager::installBuiltinWidgets()
{
    bool success = true;

    m_installedWidgets.setSize( hkgVdbWidgetIndices::NUM_BUILTIN_WIDGETS );

#define INSTALL_BUILTIN_WIDGET( IDX, TYPE ) \
    if ( !getInstalledWidget( IDX ) ) \
    { \
        int widgetHandle = IDX; \
        hkRefPtr<hkgVdbWidget> widget = hkRefNew<TYPE>( new TYPE() ); \
        success &= installWidget( *widget, widgetHandle ).isSuccess(); \
    }

    FOREACH_BUILTIN_WIDGET( INSTALL_BUILTIN_WIDGET );

#undef INSTALL_BUILTIN_WIDGET

    // Ensure viewport and selection are enabled
    {
        if ( hkgVdbViewportWidget* widget = getInstalledWidget<hkgVdbViewportWidget>() ) widget->setEnabled( true );
        if ( hkgVdbSelectionWidget* widget = getInstalledWidget<hkgVdbSelectionWidget>() ) widget->setEnabled( true );
    }

    return success ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkgVdbWidgetManager::uninstallWidget( int widgetHandle )
{
    HK_VDB_VERIFY_CONDITION_MSG(
        m_plugin->getWindow(),
        0xadb00059,
        hkVdbError::INVALID_STATE,
        "Can't uninstall widgets when plugin is not initialized" );

    if ( ( widgetHandle == hkgVdbWidgetIndices::VIEWPORT ) ||
        ( widgetHandle == hkgVdbWidgetIndices::SELECTION ) )
    {
        hkLog_Warning( s_debugLog, "Disabling viewport or selection widget will likely cause numerous issues" );
    }

    if ( ( widgetHandle >= 0 ) && ( widgetHandle < m_installedWidgets.getSize() ) )
    {
        if ( hkgVdbWidget* widget = m_installedWidgets[widgetHandle] )
        {
            bool success = true;
            hkgDisplayContext* context = m_plugin->getWindow()->getContext();
            context->lock();
            {
                success &= widget->setEnabled( false ).isSuccess();
                success &= widget->deinitialize( *m_plugin, *m_client ).isSuccess();
                widget->setContext( HK_NULL );
            }
            context->unlock();
            m_installedWidgets[widgetHandle] = HK_NULL;
            HK_VDB_VERIFY_REPORTER_OPERATION(
                success ? HK_SUCCESS : HK_FAILURE,
                *widget,
                hkVdbError::PLUGIN_ERROR );
        }
    }

    return HK_SUCCESS;
}

hkResult hkgVdbWidgetManager::uninstallAllWidgets()
{
    HK_VDB_VERIFY_CONDITION_MSG(
        m_plugin->getWindow(),
        0xadb00060,
        hkVdbError::INVALID_STATE,
        "Can't uninstall widgets when plugin is not initialized" );

    bool success = true;
    for ( int i = 0; i < m_installedWidgets.getSize(); i++ )
    {
        success &= uninstallWidget( i ).isSuccess();
    }
    return success ? HK_SUCCESS : HK_FAILURE;
}

hkgVdb2dWidget* hkgVdbWidgetManager::getWidgetAt( hkgVdbPoint p ) const
{
    for ( int i = 0; i < m_installedWidgets.getSize(); i++ )
    {
        hkgVdbWidget* widget = m_installedWidgets[i];
        if ( hkgVdb2dWidget* widget2d = hkDynCast( widget ) )
        {
            if ( widget2d->getEnabled() && widget2d->getBounds().isInside( p ) )
            {
                return widget2d;
            }
        }
    }
    return HK_NULL;
}

void hkgVdbWidgetManager::setInputFocus( hkgVdbInputTypes inputTypes, hkgVdbPoint mp, hkBool32 captureInputs )
{
    setInputFocus( inputTypes, getWidgetAt( mp ), captureInputs );
}

void hkgVdbWidgetManager::setInputFocus( hkgVdbInputTypes inputTypes, hkgVdbWidget* widget, hkBool32 captureInputs )
{
    for ( int bitIt = 0; bitIt < ( sizeof( hkgVdbInputTypes::StorageType ) * 8 ); bitIt++ )
    {
        hkgVdbInputTypes::StorageType bit = ( 1 << bitIt );
        if ( inputTypes.anyIsSet( bit ) )
        {
            // Clear previous input captures
            if ( captureInputs )
            {
                for ( int i = 0; i < m_installedWidgets.getSize(); i++ )
                {
                    if ( hkgVdbWidget* iwidget = m_installedWidgets[i] )
                    {
                        iwidget->m_capturedInputs.clear( bit );
                    }
                }
            }

            // Update focus info
            hkgVdbWidget* focusWidget = ( widget && widget->getSupportedInputs().anyIsSet( bit ) ) ? widget : HK_NULL;
            m_focussedWidgets.insert( bit, focusWidget );

            // Set new input captures
            if ( captureInputs )
            {
                if ( focusWidget ) focusWidget->m_capturedInputs.orWith( bit );
                m_capturedInputWidgets.insert( bit, focusWidget );
            }
        }
    }
}

hkgVdbWidget* hkgVdbWidgetManager::getInputFocus( hkgVdbInputTypes inputType ) const
{
    HK_ON_DEBUG( hkUint8 inputTypeStorage = inputType.get(); )
        HK_ASSERT(
            0x22441101,
            hkMath::countBitsSet8( inputTypeStorage ) <= 1,
            "inputType should be a single type" );
    return m_focussedWidgets.getWithDefault( inputType, HK_NULL );
}

hkgVdbInputTypes hkgVdbWidgetManager::getCapturedInputs() const
{
    hkgVdbInputTypes capturedInputs;
    getCapturedInputsInternal( capturedInputs, capturedInputs );
    return capturedInputs;
}

void hkgVdbWidgetManager::renderTooltip() const
{
    if ( ( m_options.m_showToolTips || m_options.m_showVerboseToolTips ) && m_widgetUnderMouse && ( m_showToolTipAtTicks != 0 ) )
    {
        if ( hkSystemClock::getTickCounter() >= m_showToolTipAtTicks )
        {
            hkStringBuf toolTip;
            m_widgetUnderMouse->getToolTip( toolTip, m_options.m_showVerboseToolTips );

            if ( toolTip.getLength() )
            {
                hkgDisplayContext* context = m_plugin->getWindow()->getContext();
                context->matchState( HKG_ENABLED_ALPHABLEND, 0, HKG_BLEND_MODULATE, HKG_ALPHA_SAMPLE_NORMAL );
                context->setCurrentColorPacked( hkColor::WHITE );

                const hkgMouse& mouse = context->getCurrentViewport()->getOwnerWindow()->getInputManager()->getMouse();
                hkgVdbPoint wmp = hkgVdbPoint::windowCoordsFromInputCoords( mouse.getPosX(), mouse.getPosY(), *context->getCurrentViewport() );
                hkUint32 userSetX = m_options.m_toolTipDrawOptions.m_x;
                hkUint32 userSetY = m_options.m_toolTipDrawOptions.m_y;
                if ( userSetX == HK_INT32_MAX ) m_options.m_toolTipDrawOptions.m_x = ( wmp.m_x + HKG_VDB_CURSOR_SIDE_PIXELS );
                if ( userSetY == HK_INT32_MAX ) m_options.m_toolTipDrawOptions.m_y = ( wmp.m_y + HKG_VDB_CURSOR_SIDE_PIXELS );
                hkgVdbTextBoxDrawer::drawTextBox( *context, toolTip, m_options.m_toolTipDrawOptions );
                m_options.m_toolTipDrawOptions.m_x = userSetX;
                m_options.m_toolTipDrawOptions.m_y = userSetY;
            }
        }
    }
}

void hkgVdbWidgetManager::updateLayout( hkgViewport& viewport )
{
    if ( m_layoutFlags.anyIsSet( hkgVdbWidgetLayoutFlags::AUTO_LAYOUT_OF_BUILTIN_WIDGETS ) )
    {
        //
        // First handle offsets from top left
        //
        hkBool32 barGraphAtTopLeft = false;
        {
            int maxY = 0;

            hkgVdbStatTextWidget* statTextWidget = getInstalledWidget<hkgVdbStatTextWidget>();
            if ( statTextWidget )
            {
                if ( statTextWidget->getEnabled() )
                {
                    maxY = hkMath::max2( statTextWidget->getBounds().m_maxY + 5, maxY );
                }
            }
            else
            {
                hkgVdbStatBarGraphWidget* barGraphWidget = getInstalledWidget<hkgVdbStatBarGraphWidget>();
                if ( barGraphWidget && barGraphWidget->getEnabled() )
                {
                    maxY = hkMath::max2( barGraphWidget->getBounds().m_maxY + 5, maxY );
                }
                barGraphAtTopLeft = true;
            }

            // Position just past maxY
            hkgVdbStatusWidget* statusWidget = getInstalledWidget<hkgVdbStatusWidget>();
            if ( statusWidget && statusWidget->getEnabled() )
            {
                statusWidget->accessOptions().m_y = maxY;
            }
        }

        //
        // Now offsets from lower left
        //
        {
            int maxY = viewport.getHeight();

            // Position just before minY
            hkgVdbStatLineGraphWidget* lineGraphWidget = getInstalledWidget<hkgVdbStatLineGraphWidget>();
            if ( lineGraphWidget && lineGraphWidget->getEnabled() )
            {
                hkgVdbBounds b = lineGraphWidget->getBounds();
                hkUint32 h = b.getHeight();
                maxY -= h;
                lineGraphWidget->accessOptions().m_y = maxY;
            }

            // Position just before minY
            hkgVdbDeprecatedStatBarGraphWidget* depStatGraphWidget = getInstalledWidget<hkgVdbDeprecatedStatBarGraphWidget>();
            if ( depStatGraphWidget && depStatGraphWidget->getEnabled() )
            {
                hkgVdbBounds b = depStatGraphWidget->getBounds();
                hkUint32 h = b.getHeight();
                maxY -= h;
                depStatGraphWidget->accessOptions().m_y = maxY;
            }

            // Position just before minY
            if ( !barGraphAtTopLeft )
            {
                hkgVdbStatBarGraphWidget* barGraphWidget = getInstalledWidget<hkgVdbStatBarGraphWidget>();
                if ( barGraphWidget && barGraphWidget->getEnabled() )
                {
                    hkgVdbBounds b = barGraphWidget->getBounds();
                    hkUint32 h = b.getHeight();
                    maxY -= h;
                    barGraphWidget->accessOptions().m_y = maxY;
                }
            }

            // Position just before minY
            hkgVdbOrientationWidget* orientationWidget = getInstalledWidget<hkgVdbOrientationWidget>();
            if ( orientationWidget && orientationWidget->getEnabled() )
            {
                hkgVdbBounds b = orientationWidget->getBounds();
                hkUint32 h = b.getHeight();
                maxY -= h;
                orientationWidget->accessOptions().m_yOffset = maxY;
            }
        }
    }
}

void hkgVdbWidgetManager::updateFocus( hkgViewport& viewport, const hkgInputManager& input )
{
    if ( m_focusFlags.anyIsSet( hkgVdbWidgetFocusFlags::DISABLED ) )
    {
        return;
    }

    const hkgMouse& mouse = input.getMouse();

    // Determine tool tip end tick if we haven't moved
    {
        if ( ( m_options.m_showToolTips || m_options.m_showVerboseToolTips ) &&
            ( mouse.getPosX() == mouse.getPrevPosX() ) &&
            ( mouse.getPosY() == mouse.getPrevPosY() ) &&
            ( mouse.getButtonState() == 0 ) &&
            ( mouse.getWheelNotchDelta() == 0 ) )
        {
            if ( m_showToolTipAtTicks == 0 )
            {
                m_showToolTipAtTicks = hkSystemClock::getTickCounter() + hkUint64( hkSystemClock::getTicksPerSecond() * hkMath::max2( m_options.m_showToolTipDelay, 0 ) );
            }
        }
        else
        {
            m_showToolTipAtTicks = 0;
        }
    }

    // Get global capture information
    hkgVdbInputTypes userCapturedInputs, capturedInputs;
    getCapturedInputsInternal( userCapturedInputs, capturedInputs );
    capturedInputs.orWith( userCapturedInputs );
    hkgVdbInputTypes newlyCapturedInputs = 0;

    // Update input focus
    {
        hkgVdbPoint wmp = hkgVdbPoint::windowCoordsFromInputCoords( mouse.getPosX(), mouse.getPosY(), viewport );
        m_widgetUnderMouse = getWidgetAt( wmp );
        if ( m_focusFlags.noneIsSet( hkgVdbWidgetFocusFlags::DISABLE_AUTO_FOCUS ) )
        {
            hkgVdbInputTypes focusInputs = 0;
            hkgVdb2dWidget* focusWidget = HK_NULL;

            // Focus based on button press
            if ( mouse.wasButtonPressed( HKG_MOUSE_ANY_BUTTON ) )
            {
                focusInputs = hkgVdbInputTypes::ALL;
                focusWidget = m_widgetUnderMouse;
            }
            // Focus based on button release
            else if ( mouse.wasButtonReleased( HKG_MOUSE_ANY_BUTTON ) )
            {
                // Only process mouse release if there's no other mouse buttons down
                if ( mouse.getButtonState() == 0 )
                {
                    // See if the widget under the mouse has captured any inputs from when mouse down occurred
                    if ( m_widgetUnderMouse )
                    {
                        for ( int bitIt = 0; bitIt < ( sizeof( hkgVdbInputTypes::StorageType ) * 8 ); bitIt++ )
                        {
                            hkgVdbInputTypes::StorageType bit = ( 1 << bitIt );
                            focusInputs.orWith( ( m_capturedInputWidgets.getWithDefault( bit, HK_NULL ) == m_widgetUnderMouse ) * bit );
                            focusWidget = m_widgetUnderMouse;
                        }
                    }

                    // If the widget under mouse hasn't captured anything while the mouse was down, then it's a clear
                    if ( focusInputs == 0 )
                    {
                        focusInputs = hkgVdbInputTypes::ALL;
                        focusWidget = HK_NULL;
                    }
                }
            }
            // Focus based on hover
            else if ( userCapturedInputs.noneIsSet( hkgVdbInputTypes::MOUSE ) && m_focusFlags.anyIsSet( hkgVdbWidgetFocusFlags::AUTO_FOCUS_MOUSE_ON_HOVER ) )
            {
                focusInputs = hkgVdbInputTypes::MOUSE;
                focusWidget = m_widgetUnderMouse;
            }

            // Apply input focus
            if ( focusInputs )
            {
                setInputFocus( focusInputs, focusWidget, m_focusFlags.anyIsSet( hkgVdbWidgetFocusFlags::AUTO_FOCUS_CAPTURES_INPUT ) );
            }
        }
    }

    // Get focus information
    hkHashMap<hkgVdbWidget*, hkgVdbInputTypes> widgetToFocusInputs;
    hkHashMap<hkgVdbWidget*, hkgVdbInputTypes> widgetToAutoCapturedInputs;
    {
        for ( hkHashMap<hkgVdbInputTypes, hkgVdbWidget*>::Iterator iter = m_focussedWidgets.getIterator();
            m_focussedWidgets.isValid( iter );
            iter = m_focussedWidgets.getNext( iter ) )
        {
            hkgVdbInputTypes focusType = m_focussedWidgets.getKey( iter );
            hkgVdbWidget* widget = m_focussedWidgets.getValue( iter );
            hkgVdbInputTypes& focusInputs = widgetToFocusInputs.getOrInsertKey( widget, 0 );
            focusInputs.orWith( focusType );
        }
        for ( hkHashMap<hkgVdbInputTypes, hkgVdbWidget*>::Iterator iter = m_capturedInputWidgets.getIterator();
            m_capturedInputWidgets.isValid( iter );
            iter = m_capturedInputWidgets.getNext( iter ) )
        {
            hkgVdbInputTypes focusType = m_capturedInputWidgets.getKey( iter );
            hkgVdbWidget* widget = m_capturedInputWidgets.getValue( iter );
            hkgVdbInputTypes& focusInputs = widgetToAutoCapturedInputs.getOrInsertKey( widget, 0 );
            focusInputs.orWith( focusType );
        }
    }

    // Update enabled inputs and retrieve new captured inputs
    {
        for ( int i = 0; i < m_installedWidgets.getSize(); i++ )
        {
            hkgVdbWidget* widget = m_installedWidgets[i];
            if ( widget && widget->getEnabled() )
            {
                // Determine the this widget's capabilities and captured inputs
                hkgVdbInputTypes capableInputsForWidget = widget->getSupportedInputs();
                hkgVdbInputTypes capturedInputsForWidget = widget->m_capturedInputs;
                hkgVdbInputTypes autoCapturedInputsForWidget = widgetToAutoCapturedInputs.getWithDefault( widget, 0 );

                // Determine the inputs that should be enabled for this widget
                hkgVdbInputTypes focusInputsForWidget = widgetToFocusInputs.getWithDefault( widget, 0 );
                hkgVdbInputTypes enabledInputsForWidget = focusInputsForWidget;
                {
                    if ( i == hkgVdbWidgetIndices::VIEWPORT )
                    {
                        enabledInputsForWidget.orWith( hkgVdbInputTypes::ALL );
                    }
                    else if ( hkDynCast<hkgVdb2dWidget>( widget ) )
                    {
                        enabledInputsForWidget.orWith(
                            // If we are sending inputs to the widget under the mouse
                            bool( m_focusFlags.anyIsSet( hkgVdbWidgetFocusFlags::ENABLE_MOUSE_INPUTS_ON_HOVER ) && ( widget == m_widgetUnderMouse ) && widget->getSupportedInputs().anyIsSet( hkgVdbInputTypes::MOUSE ) ) *
                            // orWith mouse type
                            ( hkgVdbInputTypes::MOUSE ) );
                    }
                    else if ( hkDynCast<hkgVdb3dWidget>( widget ) )
                    {
                        enabledInputsForWidget.orWith(
                            // If we are sending inputs to all 3d widgets
                            bool( m_focusFlags.anyIsSet( hkgVdbWidgetFocusFlags::ENABLE_ALL_INPUTS_FOR_3D_WIDGETS ) ) *
                            // orWith ALL types
                            hkgVdbInputTypes::ALL );
                    }
                    else
                    {
                        HK_WARN( 0x28a6ab60, "Unhandled widget class" );
                    }

                    enabledInputsForWidget.andNotWith( capturedInputs );
                    enabledInputsForWidget.orWith( capturedInputsForWidget | autoCapturedInputsForWidget );
                    enabledInputsForWidget.andWith( capableInputsForWidget );
                }

                // Notify widget of enabled inputs and retrieve any newly captured inputs
                hkgVdbInputTypes newlyCapturedInputsForWidget = widget->setEnabledInputs( viewport, input, m_widgetUnderMouse, enabledInputsForWidget );
                newlyCapturedInputsForWidget.orWith( autoCapturedInputsForWidget );

                // Update our understanding of captured inputs based on widgets new captured inputs
                capturedInputs.orWith( newlyCapturedInputsForWidget );
                newlyCapturedInputs.orWith( newlyCapturedInputsForWidget );
            }
        }
    }
}

void hkgVdbWidgetManager::getCapturedInputsInternal( hkgVdbInputTypes& userCapturedInputs, hkgVdbInputTypes& autoCapturedInputs ) const
{
    userCapturedInputs = 0;

    for ( int i = 0; i < m_installedWidgets.getSize(); i++ )
    {
        hkgVdbWidget* widget = m_installedWidgets[i];
        if ( widget && widget->getEnabled() )
        {
            userCapturedInputs.orWith( widget->m_capturedInputs );
        }
    }

    if ( &userCapturedInputs != &autoCapturedInputs )
    {
        autoCapturedInputs = 0;
    }

    for ( hkHashMap<hkgVdbInputTypes, hkgVdbWidget*>::Iterator iter = m_capturedInputWidgets.getIterator();
        m_capturedInputWidgets.isValid( iter );
        iter = m_capturedInputWidgets.getNext( iter ) )
    {
        hkgVdbInputTypes focusType = m_capturedInputWidgets.getKey( iter );
        autoCapturedInputs.orWith( bool( m_capturedInputWidgets.getValue( iter ) ) * focusType );
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
