// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPlugin.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/Utils/hkgVdbStatBarGraph.h>

#include <Common/Base/hkBase.h>
#include <Common/Base/Monitor/MonitorStreamAnalyzer/hkMonitorStreamParser.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>

namespace {

    struct TimerRange
    {
        hkUint16 threadNum;
        double displayStartTime;
        double displayEndTime;
        hkReal startX;
        hkReal startY;
        hkReal endX;
        hkUint16 maxLevels;
        hkUint32 pixelRange; // usually just feed main render target num pixels for this
    };

    hkColor::Argb findColor(const hkMonitorStreamParser::Node* node, const hkMonitorStreamColorTableCache* colorTable )
    {
        const hkMonitorStreamParser::Node* colorNode = node;
        bool cacheColor = false;
        hkColor::Argb c = 0xffffffff;
        while (colorNode && !colorTable->findColor(colorNode->m_name, c))
        {
            colorNode = colorNode->m_parent; // keep going up until we find a named node with a set color, otherwise we will end up with default color
            cacheColor = true;
        }

        if (cacheColor)
        {
            colorTable->cacheColor(node->m_name, c);
        }

        return c;
    }

    void findRange( const hkMonitorStreamParser::Node* node, int highlightMask, double& minTime, double& maxTime)
    {
        if ( ( (!highlightMask) || node->m_flags.allAreSet( (hkUint16)highlightMask) ) && (!node->m_flags.anyIsSet( hkMonitorStreamParser::Node::FLAGS_TIME_IS_NOT_ABSOLUTE)) )
        {
            double startTime = node->m_absoluteStartTime;
            float nodeDurationTime = node->m_value;
            double endTime = startTime + nodeDurationTime;
            if ( (startTime > 0.f) && (endTime > startTime))
            {
                minTime = hkMath::min2(startTime, minTime);
                maxTime = hkMath::max2(endTime, maxTime);
            }
        }

        for (int c=0; c < node->m_children.getSize(); ++c)
        {
            findRange( node->m_children[c], highlightMask, minTime, maxTime );
        }
    }

    void findRange( const hkArrayView<hkMonitorStreamParser::Tree*>& timers, double& totalStartTime, double& totalEndTime )
    {
        totalStartTime = HK_REAL_MAX;
        totalEndTime = 0.f;
        for (int n=0; n < timers.getSize(); ++n)
        {
            const hkMonitorStreamParser::Node* frameGraph = timers[n];
            for (int f=0; f < frameGraph->m_children.getSize(); ++f)
            {
                const hkMonitorStreamParser::Node* timerGraph = frameGraph->m_children[f];
                for (int c=0; c < timerGraph->m_children.getSize(); ++c)
                {
                    if ( !timerGraph->m_children[c]->m_flags.anyIsSet( hkMonitorStreamParser::Node::FLAGS_TIME_IS_NOT_ABSOLUTE ) && (timerGraph->m_children[c]->m_absoluteStartTime > 0.f) )
                    {
                        totalStartTime = hkMath::min2<double>( timerGraph->m_children[c]->m_absoluteStartTime, totalStartTime);
                        double endTime = timerGraph->m_children[c]->m_absoluteStartTime + timerGraph->m_children[c]->m_value;
                        totalEndTime = hkMath::max2<double>(totalEndTime, endTime );
                    }
                }
            }
        }
    }

    void findRange( const hkArrayView<hkMonitorStreamParser::Tree*>& cpuThreads, const hkArrayView<hkGpuTraceResult>& gpuTrace, int highlightMask, double& minTime, double& maxTime)
    {
        for (int c = 0; c < gpuTrace.getSize(); ++c)
        {
            const hkGpuTraceResult& gn = gpuTrace[c];

            if ( highlightMask )
            {
                hkLocalArray<hkMonitorStreamGpuHandleCache::Mapping*> cpuContext(4);
                const hkMonitorStreamParser::Node* cpuNode = HK_NULL;

                for (int thread = 0; thread < cpuThreads.getSize(); ++thread  )
                {
                    hkMonitorStreamGpuHandleCache::Mapping* ctx = cpuThreads[thread]->m_gpuHandleCache.get( gn.m_id, false );
                    if (ctx) cpuContext.pushBack(ctx);
                }

                // addHandle node is in the cpu timer in the task that triggered the draw call etc
                // The timerNode is usually just in the executeCommandBuffer task so not as interesting a CPU context
                // so check all for that first.
                for (int ci = 0; (!cpuNode) && (ci < cpuContext.getSize()); ++ci)
                {
                    cpuNode = cpuContext[ci]->m_addHandleNode;
                }
                for (int cj = 0; (!cpuNode) && (cj < cpuContext.getSize()); ++cj)
                {
                    cpuNode = cpuContext[cj]->m_timerNode;
                }

                if ( !cpuNode || !cpuNode->m_flags.allAreSet( (hkUint16)highlightMask) )
                {
                    continue;
                }
            }

            if ( (gn.m_gpuTimeBegin > 0) && (gn.m_gpuTimeEnd > gn.m_gpuTimeBegin))
            {
                minTime = hkMath::min2(minTime, gn.m_gpuTimeBegin);
                maxTime = hkMath::max2(maxTime, gn.m_gpuTimeEnd);
            }
        }
    }

    hkgVdbStatBarGraph::Quad& addQuad( const hkMonitorStreamParser::Node* node, hkColor::Argb color,
        const TimerRange& timerInfo, int currentNodeLevel,
        hkgVdbStatBarGraph::Thread& thread,
        double startTime,
        hkReal displayWidthTime,
        hkReal displayWidthPixels,
        hkReal pixelNodeWidth)
    {
        hkgVdbStatBarGraph::Quad& q = thread.m_quads.expandOne();
        q.m_srcNode = node;
        q.m_srcGpuResult = HK_NULL;

        q.m_time = node->m_value;
        q.m_threadNum = (hkUint16)timerInfo.threadNum;
        q.m_level = (hkUint16)currentNodeLevel;
        q.m_timerName = node->m_name;
        q.m_color = color;

        float startTimePercent = float(startTime - timerInfo.displayStartTime) / displayWidthTime;

        hkUint16 renderLevel = thread.m_collapsed ? 0 : q.m_level;
        q.m_rect.setFromMinMax
        (
            hkReal(timerInfo.startX + (startTimePercent * displayWidthPixels)),
            hkReal(renderLevel*hkgVdbStatBarGraph::PIXELS_PER_LEVEL + timerInfo.startY),
            hkReal(timerInfo.startX + (startTimePercent * displayWidthPixels + pixelNodeWidth)),
            hkReal((renderLevel + 1)*hkgVdbStatBarGraph::PIXELS_PER_LEVEL + timerInfo.startY)
        );
        return q;
    }

    void generateQuadsForChildren( const hkMonitorStreamParser::Node* node,
        const hkMonitorStreamColorTableCache* colorTable,
        const TimerRange& timerInfo,
        int currentNodeLevel,
        hkgVdbStatBarGraph::Thread& thread,
        int highlightMask)
    {
        bool hadChildren = false;
        const hkReal displayWidthPixels = hkReal(timerInfo.endX - timerInfo.startX);
        const float displayWidthTime = (float)(timerInfo.displayEndTime - timerInfo.displayStartTime);

        thread.m_height = hkMath::max2(thread.m_height, (currentNodeLevel) * hkgVdbStatBarGraph::PIXELS_PER_LEVEL);
        thread.m_quads.reserve( thread.m_quads.getSize() + node->m_children.getSize() );
        for (int c = 0; c < node->m_children.getSize(); ++c)
        {
            const hkMonitorStreamParser::Node* childNode = node->m_children[c];

            // when zooming etc, clamp to start pos
            float nodeDurationTime = childNode->m_value;
            double startTime = childNode->m_absoluteStartTime;
            if (startTime < timerInfo.displayStartTime)
            {
                float diff = float( timerInfo.displayStartTime - startTime );
                nodeDurationTime -= diff;
                startTime = timerInfo.displayStartTime;
            }
            double endTime = startTime + nodeDurationTime;

            if (childNode->m_absoluteStartTime > 0
                && childNode->m_absoluteStartTime <= timerInfo.displayEndTime
                && nodeDurationTime > 0
                && endTime >= timerInfo.displayStartTime
                && !childNode->m_flags.anyIsSet(hkMonitorStreamParser::Node::FLAGS_TIME_IS_NOT_ABSOLUTE))
            {
                hadChildren = true;

                float percentNodeWidth = nodeDurationTime / displayWidthTime;
                hkColor::Argb color = findColor(childNode, colorTable);
                if (highlightMask && !childNode->m_flags.allAreSet((hkUint16)highlightMask))
                {
                    // then darken and alpha out bit
                    color = 0x3FFFFFFF & hkColor::darken(color);
                }

                // node is big enough, we know we have room, so lets display it
                addQuad(childNode, color, timerInfo, currentNodeLevel, thread, startTime, displayWidthTime, displayWidthPixels, percentNodeWidth * displayWidthPixels);
            }
        }

        // see if no more room
        if ( (currentNodeLevel != 0) || hadChildren)
            ++currentNodeLevel;

        // otherwise continue on
        for (int c=0; c < node->m_children.getSize(); ++c)
        {
            generateQuadsForChildren( node->m_children[c], colorTable, timerInfo, currentNodeLevel, thread, highlightMask);
        }
    }
}

//
// hkgVdbStatBarGraph

hkgVdbStatBarGraph::hkgVdbStatBarGraph()
{
    m_requestedRect.setPosition(0,0);
    m_requestedRect.setSize(0,0);

    m_frameTime = 16.666f;

    m_colorTable.setAndDontIncrementRefCount( new hkMonitorStreamColorTable() );
    m_colorTable->setupDefaultColorTable();

    m_graphStartTime = m_graphEndTime = 0.f;
}

hkgVdbStatBarGraph::~hkgVdbStatBarGraph()
{
    m_threads.clear();
}

void hkgVdbStatBarGraph::init(const hkRectF& graphArea)
{
    m_requestedRect = graphArea;
}

hkInt32 hkgVdbStatBarGraph::getNumThreads() const
{
    return m_threads.getSize();
}

const hkgVdbStatBarGraph::Thread& hkgVdbStatBarGraph::getThread(int i) const
{
    return m_threads[i];
}

hkgVdbStatBarGraph::Thread& hkgVdbStatBarGraph::accessThread(int i)
{
    return m_threads[i];
}

float hkgVdbStatBarGraph::getGraphStartTime() const
{
    return m_graphStartTime;
}

float hkgVdbStatBarGraph::getGraphEndTime() const
{
    return m_graphEndTime;
}

void hkgVdbStatBarGraph::updateCpu(
    const hkArrayView<hkMonitorStreamParser::Tree*>& timers,
    int highlightNodeFlagMask)
{
    if (m_threads.getSize() != timers.getSize())
    {
        int startIdx = m_threads.getSize();
        m_threads.setSize(timers.getSize());
        for (int i = startIdx; i < m_threads.getSize(); ++i)
        {
            hkStringBuf buf;
            buf.printf("CPU Thread %i:", i);
            m_threads[i].m_name = buf;
            m_threads[i].m_collapsed = true;
        }
    }

    // Work out our full range (so max across all threads)
    double totalStartTime = HK_REAL_MAX;
    double totalEndTime = 0.f;
    findRange( timers, totalStartTime, totalEndTime);

    if (totalStartTime >= totalEndTime)
    {
        return;
    }

    double maxFrameTime = totalEndTime - totalStartTime;

    // round up to ms boundaries
    m_frameTime = ( int(maxFrameTime) + 1.f );

    //Clamp min to 60Hz frame
    m_frameTime = hkMath::max2(10.0f, m_frameTime);

    // See if we only want to get a sub range of that total time
    double displayStartTime = totalStartTime;
    double displayEndTime = totalStartTime + m_frameTime ;
    if (displayEndTime <= displayStartTime)
    {
        m_graphStartTime = m_graphEndTime = 0.f;
        return;
    }

    m_graphStartTime = 0;
    m_graphEndTime = m_frameTime;

    // generate the quads, starting at ignoreFirstNCalls deep, stopping when reach too high for this thread or < 1 pixel.
    hkUint16 pixelsPerThread = hkUint16( m_requestedRect.getHeight() / m_threads.getSize() );

    TimerRange tr;
    tr.displayStartTime = displayStartTime;
    tr.displayEndTime = displayEndTime;
    tr.maxLevels = pixelsPerThread / PIXELS_PER_LEVEL;
    tr.startX = 0;
    tr.endX = m_frameTime * PIXELS_PER_MILLISECOND;

    hkMonitorStreamColorTableCache cache;
    cache.m_colorTable = m_colorTable;

    hkReal startY = m_requestedRect.getY();
    m_actualRect.setInvalid();
    for (int t=0; t < m_threads.getSize(); ++t)
    {
        const hkMonitorStreamParser::Node* timerGraph = timers[t];

        tr.threadNum = (hkUint16) t;
        tr.startY = startY;

        m_threads[t].m_quads.clear();

        hkReal previousHeight = m_threads[t].m_height;
        m_threads[t].m_height = 0;
        generateQuadsForChildren( timerGraph, &cache, tr, 0, m_threads[t], highlightNodeFlagMask);

        if (m_threads[t].m_collapsed)
        {
            m_threads[t].m_height = PIXELS_PER_LEVEL * 2;
            startY += PIXELS_PER_LEVEL * 2;
        }
        else
        {
            // Grow (never shrink) thread stacks up to MAX_PER_THREAD_STABLE_DEPTH, then dynamically resize
            // stack depth beyond this threshold.
            m_threads[t].m_height = hkMath::max2(m_threads[t].m_height, hkMath::min2(previousHeight, MAX_PER_THREAD_STABLE_DEPTH * PIXELS_PER_LEVEL));
            startY += m_threads[t].m_height;
        }

        for (int jthQuad = 0; jthQuad < m_threads[t].m_quads.getSize(); ++jthQuad)
        {
            m_actualRect.includeRect(m_threads[t].m_quads[jthQuad].m_rect);
        }
    }
    m_actualRect.setHeight(startY - m_requestedRect.getY());
}

const hkgVdbStatBarGraph::Quad* hkgVdbStatBarGraph::getQuadUnderPos(hkReal x, hkReal y ) const
{
    for (int ithThread = 0; ithThread < m_threads.getSize(); ++ithThread)
    {
        const hkArray<Quad>& quads = m_threads[ithThread].m_quads;
        if (m_threads[ithThread].m_collapsed)
        {
            int bestQuadIndex = -1;
            for (int q = 0; q < quads.getSize(); ++q)
            {
                const Quad& quad = quads[q];
                if ((quad.m_rect.getX() <= x) && (quad.m_rect.getX2() >= x)
                    && (quad.m_rect.getY() <= y) && (quad.m_rect.getY2() >= y)
                    && (bestQuadIndex < 0 || quad.m_level > quads[bestQuadIndex].m_level))
                {
                    bestQuadIndex = q;
                }
            }

            if (bestQuadIndex >= 0)
            {
                return &quads[bestQuadIndex];
            }
        }
        else
        {
            for (int q = 0; q < quads.getSize(); ++q)
            {
                const Quad& quad = quads[q];
                if ((quad.m_rect.getX() <= x) && (quad.m_rect.getX2() >= x)
                    && (quad.m_rect.getY() <= y) && (quad.m_rect.getY2() >= y))
                {
                    return &quad;
                }
            }
        }
    }
    return HK_NULL;
}

void hkgVdbStatBarGraph::setColorTable(hkMonitorStreamColorTable* c)
{
    m_colorTable = c;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
