// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Visualize/hkDebugDisplayHandler.h>

#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>

#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/3d/hkgVdb3dWidget.h>


#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>

class hkVdbCmdOutput;
class hkgVdbGeometryControl;
class hkgMouse;

//////////////////////////////////////////////////////////////////////////
/// Flags governing the behavior of selection.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkgVdbSelectionFlags : public hkFlagsEx<hkUint8>
{
    enum Bits
    {
        /// If a highlight set exists, any selections that occur which are not
        /// contained in the highlight set are ignored.
        IGNORE_UNHIGHLIGHTED_SELECTIONS = ( 1 << 0 ),

        // If a highlight set exists, any selections that occur which are not
        // contained in the highlight set cause the highlight set to be cleared.
        
        //Needs more work, see cpp comments
        //CLEAR_ON_UNHIGHLIGHTED_SELECTIONS = ( 1 << 1 ),

        /// Allow grabbing/releasing of geometry to impact selection set.
        MODIFY_SELECTION_ON_GRABBING_GEOMETRY = ( 1 << 2 ),

        /// Update primary point of interest based on grabbing geometry.
        MODIFY_PRIMARY_POI_ON_GRABBING_GEOMETRY = ( 1 << 3 ),

        /// The default selection behavior.
        DEFAULT =
            ( IGNORE_UNHIGHLIGHTED_SELECTIONS |
            MODIFY_SELECTION_ON_GRABBING_GEOMETRY |
            MODIFY_PRIMARY_POI_ON_GRABBING_GEOMETRY )
    };

    HK_DECLARE_FLAGS_EX_CLASS( hkgVdbSelectionFlags, hkgVdbSelectionFlags::Bits );
};

//////////////////////////////////////////////////////////////////////////
/// A widget which determines selection behavior.
/// Also can draw some minimal indicators around the selection
/// as defined by flags. Most selection-centric drawing happens
/// in response to selection changed signals from this widget.
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkgVdbSelectionWidget : public hkgVdb3dWidget
{
public:
    HK_DECLARE_CLASS( hkgVdbSelectionWidget, New, ReflectIdentity );
    hkgVdbSelectionWidget();
    ~hkgVdbSelectionWidget();

    /// Type defining a set of selected things.
    
    struct SelectionSet;
    struct PointOfInterest;
    typedef hkVdbMap<hkUint64, PointOfInterest> PointsOfInterest;

    //
    // 3d Widget interface
    //

    virtual hkResult initialize( hkgVdbPlugin& plugin, hkVdbClient& client ) HK_OVERRIDE;
    virtual hkResult deinitialize( hkgVdbPlugin& plugin, hkVdbClient& client ) HK_OVERRIDE;
    virtual hkResult update( const hkgViewport& viewport, const hkgInputManager& input ) HK_OVERRIDE;
    virtual hkgVdbInputTypes getSupportedInputs() const HK_OVERRIDE { return hkgVdbInputTypes::ALL; }

    //
    // Control
    //

    /// Sets the current flags governing selection behavior.
    hkResult setSelectionFlags( hkgVdbSelectionFlags flags );

    /// Gets the current flags governing selection behavior.
    hkgVdbSelectionFlags getSelectionFlags() const { return m_selectionFlags; }

    //
    // Points of Interest
    //

    /// This id is interpreted as the most important POI for purposes of camera tracking, etc.
    static const hkUint64 PrimaryPointOfInterest = hkUint64( -2 );

    /// Sets a POI with the given unique POI id.
    HK_INLINE hkResult setPointOfInterest(
        hkUint64 id,
        hkVector4Parameter worldPosition,
        hkColor::Argb color,
        hk1PointDisplayStyle::Enum style = hk1PointDisplayStyle::DOT,
        hkReal scale = 0 );

    /// Sets a local POI with the given unique POI id.
    /// Local POIs track the geometry/sub-geometry associated with provided geometryId/subGeomIdx.
    HK_INLINE hkResult setLocalPointOfInterest(
        hkUint64 id,
        hkUint64 geometryId,
        int subGeomIdx,
        hkVector4Parameter localPosition,
        hkColor::Argb color,
        hk1PointDisplayStyle::Enum style = hk1PointDisplayStyle::DOT,
        hkReal scale = 0 );

    /// Clear POIs (optionally only those for specific ids).
    HK_INLINE bool clearPointsOfInterest( hkArrayView<hkUint64> ids = hkVdbValidity<hkArrayView<hkUint64>>::invalid() );

    /// Get a single POI for the id (if registered).
    HK_INLINE const PointOfInterest* getPointOfInterest( hkUint64 id ) const;

    /// Get our POIs.
    HK_INLINE const PointsOfInterest& getPointsOfInterest() const;

    //
    // Selection
    //

    
    
    
    
    

    /// Highlight geometry for the specified id.
    HK_INLINE hkResult highlightGeometry( hkUint64 id );

    /// Highlight geometries for the specified ids.
    HK_INLINE hkResult highlightGeometries( hkArrayView<hkUint64> ids );

    /// Get the current set of highlighted geometry ids.
    HK_INLINE const SelectionSet& getHighlightedGeometries() const;

    /// Clear highlighted geometries (optionally only those for specific ids).
    HK_INLINE bool clearHighlightedGeometries( hkArrayView<hkUint64> ids = hkVdbValidity<hkArrayView<hkUint64>>::invalid() );

    /// Select geometry for the specified id.
    HK_INLINE hkResult selectGeometry( hkUint64 id );

    /// Select geometries for the specified ids.
    HK_INLINE hkResult selectGeometries( hkArrayView<hkUint64> ids );

    /// Get the current set of selected geometry ids.
    HK_INLINE const SelectionSet& getSelectedGeometries() const;

    /// Clear selected geometries (optionally only those for specific ids).
    HK_INLINE bool clearSelectedGeometries( hkArrayView<hkUint64> ids = hkVdbValidity<hkArrayView<hkUint64>>::invalid() );

    /// Highlight immediate-mode-displays for the specified id.
    
    
    HK_INLINE hkResult highlightDisplay( hkUint64 id );

    /// Highlight immediate-mode-displays for the specified ids.
    HK_INLINE hkResult highlightDisplays( hkArrayView<hkUint64> ids );

    /// Get the current set of highlighted immediate-mode-display ids.
    HK_INLINE const SelectionSet& getHighlightedDisplays() const;

    /// Clear highlighted immediate-mode-displays (optionally only those for specific ids).
    HK_INLINE bool clearHighlightedDisplays( hkArrayView<hkUint64> ids = hkVdbValidity<hkArrayView<hkUint64>>::invalid() );

    /// Select immediate-mode-displays for the specified id.
    HK_INLINE hkResult selectDisplay( hkUint64 id );

    /// Select immediate-mode-displays for the specified ids.
    HK_INLINE hkResult selectDisplays( hkArrayView<hkUint64> ids );

    /// Get the current set of selected immediate-mode-display ids.
    HK_INLINE const SelectionSet& getSelectedDisplays() const;

    /// Clear selected immediate-mode-displays (optionally only those for specific ids).
    HK_INLINE bool clearSelectedDisplays( hkArrayView<hkUint64> ids = hkVdbValidity<hkArrayView<hkUint64>>::invalid() );

    //
    // Interaction
    // Can happen automatically based on flags.
    //

    /// Grab the geometry associated with the id at the provided position.
    hkResult grabGeometry( hkUint64 id, hkVector4Parameter worldPosition );

    /// Grab a sub-geometry (Eg. particle) associated with the id and index
    /// at the provided position.
    hkResult grabGeometry( hkUint64 id, int subGeomIndex, hkVector4Parameter worldPosition );

    /// Get currently grabbed geometry id.
    hkUint64 getGrabbedGeometry() const;

    /// Drag the currently grabbed geometry to a new position.
    void dragGeometry( hkVector4Parameter worldPosition );

    /// Release currently grabbed geometry.
    void releaseGeometry();

    //
    // Signals
    //

    /// Information about a PointOfInterest
    struct PointOfInterest
    {
        HK_DECLARE_CLASS( PointOfInterest, New );
        PointOfInterest() :
            // Indicates uninitialized POI
            m_scale( -1 )
        {}

        /// Position in local space.
        /// Valid only if m_geometryId != 0.
        hkVector4 m_localSpacePos;

        /// Position in world space.
        hkVector4 m_worldSpacePos;

        /// The geometry this POI is attached to.
        /// 0 if not used.
        hkUint64 m_geometryId;

        /// The subGeomIdx if geometryId is an instanced geometry.
        /// -1 if not used.
        int m_subGeomIdx;

        /// The color for the POI.
        hkColor::Argb m_color;

        /// The style for the POI.
        hk1PointDisplayStyle::Enum m_style;

        /// The scale for the POI.
        hkReal m_scale;
    };

    /// Information about how the POIs have changed.
    struct PointsOfInterestChangedInfo
    {
        HK_DECLARE_CLASS( PointsOfInterestChangedInfo, New );

        /// The ids removed from the global POIs.
        PointsOfInterest m_removedFromGlobal;

        /// The ids added to the global POIs.
        PointsOfInterest m_addedToGlobal;

        /// The ids removed from local POIs.
        PointsOfInterest m_removedFromLocal;

        /// The ids added to local POIs.
        PointsOfInterest m_addedToLocal;
    };

    /// A set of selection-related items.
    struct SelectionSet : hkVdbMap<hkUint64, hkRefPtr<hkgDisplayObject> >
    {
        HK_DECLARE_CLASS( SelectionSet, New );

        /// Returns the Aabb surrounding the set.
        void computeAabb( hkAabb& aabbOut ) const;
    };

    /// Multiple Selection Sets
    struct SelectionSets
    {
        HK_DECLARE_CLASS( SelectionSets, New );

        /// The set of selected ids.
        SelectionSet m_selected;

        /// The set of highlighted ids.
        SelectionSet m_highlighted;
    };

    /// Information about how a selection has changed.
    struct SelectionChangedInfo
    {
        HK_DECLARE_CLASS( SelectionChangedInfo, New );

        /// The ids removed from the selected set.
        SelectionSet m_removedFromSelected;

        /// The ids added to the selected set.
        SelectionSet m_addedToSelected;

        /// The ids removed from the highlighted set.
        SelectionSet m_removedFromHighlighted;

        /// The ids added to the highlighted set.
        SelectionSet m_addedToHighlighted;
    };

    HK_DECLARE_SIGNAL( PointsOfInterestChangedSignal, hkSignal2< const PointsOfInterest&, const PointsOfInterestChangedInfo& > );
    /// Fired when the set of POIs change.
    /// It is not called when the value changes, instead that should be grabbed
    /// fresh every render frame.
    PointsOfInterestChangedSignal m_pointOfInterestChanged;

    /// A signal type for any selection set.
    using SelectionChangedSignal = hkSignal2< const SelectionSets&, const SelectionChangedInfo& >;

    HK_DECLARE_SIGNAL( GeometrySelectionChangedSignal, hkSignal2< const SelectionSets&, const SelectionChangedInfo& > );
    /// Fired when the geometry selection/highlighted set changed.
    GeometrySelectionChangedSignal m_geometrySelectionChanged;

    HK_DECLARE_SIGNAL( DisplaySelectionChangedSignal, hkSignal2< const SelectionSets&, const SelectionChangedInfo& > );
    /// Fired when the immediate-mode-display selection/highlighted set changed.
    DisplaySelectionChangedSignal m_displaySelectionChanged;

    //
    // Internal use
    //

    void onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection );
    void onGeometryHiddenSignal( hkgDisplayObject& object );
    void onPlaybackInfoReceivedSignal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result );

protected:

    
    
    struct QueuedGrabbedGeomCommand
    {
        hkVdbCmdType::Enum m_type;
        hkVector4 m_position;
        hkUint64 m_id;
        int m_index;
        int m_updateId;
    };

    virtual hkResult enable() HK_OVERRIDE;
    virtual hkResult disable() HK_OVERRIDE;

    virtual hkgVdbInputTypes captureInputs(
        const hkgViewport& viewport,
        const hkgInputManager& input,
        hkgVdb2dWidget* widgetUnderMouse,
        hkgVdbInputTypes enabledInputs ) HK_OVERRIDE;

    hkResult setPOIInternal(
        hkUint64 id,
        hkUint64 geometryId,
        int subGeomIdx,
        hkVector4Parameter position,
        hkColor::Argb color,
        hk1PointDisplayStyle::Enum style,
        hkReal scale,
        PointsOfInterest& pois,
        PointsOfInterest& addedOut );

    bool clearPOIsInternal(
        hkArrayView<hkUint64> ids,
        PointsOfInterest& pois,
        PointsOfInterest& globalRemovedOut,
        PointsOfInterest& localRemovedOut );

    template<int IDX> HK_INLINE hkResult highlightSetInternal( hkArrayView<hkUint64> ids );
    template<int IDX> HK_INLINE const SelectionSet& getHighlightedSetInternal() const;
    template<int IDX> HK_INLINE bool clearHighlightedSetInternal( hkArrayView<hkUint64> ids = hkArrayView<hkUint64>() );
    template<int IDX> HK_INLINE hkResult selectSetInternal( hkArrayView<hkUint64> ids );
    template<int IDX> HK_INLINE const SelectionSet& getSelectedSetInternal() const;
    template<int IDX> HK_INLINE bool clearSelectedSetInternal( hkArrayView<hkUint64> ids = hkArrayView<hkUint64>() );

    template<int IDX> hkResult addToSetInternal(
        hkArrayView<hkUint64> ids,
        SelectionSet& set,
        SelectionSet& addedOut,
        SelectionSet* setFilter = HK_NULL,
        SelectionSet* setFilterAddedOut = HK_NULL );

    bool removeFromSetInternal(
        hkArrayView<hkUint64> ids,
        SelectionSet& set,
        SelectionSet& removedOut,
        SelectionSet* dependentSet = HK_NULL,
        SelectionSet* dependentRemovedOut = HK_NULL );

    hkgDisplayObject* getObjectFromId( hkUint64 id );
    void modifySelectionOnGrabbedGeometry( hkUint64 id, hkgDisplayObject* obj );

    template<int IDX> HK_INLINE void signalSelectionChangedInternal( SelectionChangedInfo& changedInfo );

    hkBool32 getGeometryTransform( hkUint64 geometryId, int subGeomIdx, hkTransform& transformOut ) const;
    hkBool32 computeLocalPointOfInterest( hkUint64 geometryId, int subGeomIdx, hkVector4Parameter worldPosition, hkVector4& localPositionOut ) const;
    hkBool32 computeGlobalPointOfInterest( hkUint64 geometryId, int subGeomIdx, hkVector4Parameter localPosition, hkVector4& worldPositionOut ) const;

    
    // 0 - Geometry
    // 1 - Display
    static const int GeometrySetIdx = 0;
    static const int DisplaySetIdx = 1;
    SelectionSets m_selectionSets[2];
    PointsOfInterest m_pointsOfInterest;

    hkRefPtr<hkVdbClient> m_client;
    hkRefPtr<hkVdbCmdOutput> m_output;
    hkRefPtr<hkgVdbGeometryControl> m_geometryControl;

    hkgVdbSelectionFlags m_selectionFlags;
    hkReal m_mouseDepth;
    hkUint64 m_grabbedId;
    int m_grabbedSubGeomIdx;
    hkArray<QueuedGrabbedGeomCommand> m_queuedCmdBuffers[2];
    int m_writeCmdBufferIndex;
    hkRefPtr<hkgDisplayObject> m_dummyObj;
};

#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/3d/hkgVdbSelectionWidget.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
