// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPlugin.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/3d/hkgVdbSelectionWidget.h>

#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/DisplayObject/hkgInstancedDisplayObject.h>

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPluginApi.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/hkgVdbPluginControl.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Control/hkgVdbGeometryControl.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdOutput.h>

#ifndef HK_VDB_USE_HASH_MAP
#include <Common/Base/Container/PointerMap/hkMap.h>
#include <Common/Base/Container/PointerMap/hkMap.hxx>
template class HK_EXPORT_COMMON hkMapBase<hkUint64, hkRefPtr<hkgDisplayObject> >;
template class HK_EXPORT_COMMON hkMap<hkUint64, hkRefPtr<hkgDisplayObject> >;
template class HK_EXPORT_COMMON hkMapBase<hkUint64, hkgVdbSelectionWidget::PointOfInterest>;
template class HK_EXPORT_COMMON hkMap<hkUint64, hkgVdbSelectionWidget::PointOfInterest>;
#endif

void hkgVdbSelectionWidget::SelectionSet::computeAabb( hkAabb& aabbOut ) const
{
    aabbOut.setEmpty();
    for ( auto iter = getIterator();
        isValid( iter );
        iter = getNext( iter ) )
    {
        if ( hkgDisplayObject* object = getValue( iter ) )
        {
            if ( !object->hasAABB() )
            {
                object->computeAABB();
            }

            if ( object->hasAABB() )
            {
                hkVector4 mmin;
                hkVector4 mmax;
                // Note: HKG is strange in that the AABB is actually not updated to be a true AABB...
                // rather it's the local AABB translated by the object transform.
                object->getAABB( mmin, mmax );
                aabbOut.m_min.setMin( aabbOut.m_min, mmin );
                aabbOut.m_max.setMax( aabbOut.m_max, mmax );
            }
        }
    }
}

hkgVdbSelectionWidget::hkgVdbSelectionWidget() :
    m_writeCmdBufferIndex( 0 ),
    m_selectionFlags( hkgVdbSelectionFlags::DEFAULT ),
    m_grabbedId( 0 ),
    m_grabbedSubGeomIdx( -1 ),
    m_dummyObj( hkRefNew<hkgDisplayObject>( hkgDisplayObject::create() ) )
{}
hkgVdbSelectionWidget::~hkgVdbSelectionWidget() {}

hkResult hkgVdbSelectionWidget::initialize( hkgVdbPlugin& plugin, hkVdbClient& client )
{
    m_geometryControl = &plugin.getControl()->getGeometryControl();
    HK_SUBSCRIBE_TO_SIGNAL( m_geometryControl->m_geometryHidden, this, hkgVdbSelectionWidget );
    m_client = &client;
    return HK_SUCCESS;
}

hkResult hkgVdbSelectionWidget::deinitialize( hkgVdbPlugin& plugin, hkVdbClient& client )
{
    m_geometryControl->m_geometryRemoved.unsubscribeAll( this );
    m_geometryControl = HK_NULL;
    m_client = HK_NULL;
    return HK_SUCCESS;
}

hkResult hkgVdbSelectionWidget::update( const hkgViewport& viewport, const hkgInputManager& input )
{
    // Update local POIs
    if ( m_pointsOfInterest.getSize() )
    {
        // We are accessing render transforms, need to lock
        viewport.getOwnerWindow()->getContext()->lock();

        for ( auto iter = m_pointsOfInterest.getIterator();
            m_pointsOfInterest.isValid( iter );
            iter = m_pointsOfInterest.getNext( iter ) )
        {
            PointOfInterest& poi = m_pointsOfInterest.getValue( iter );
            if ( poi.m_geometryId )
            {
                if ( HK_VERY_UNLIKELY( !computeGlobalPointOfInterest(
                    poi.m_geometryId,
                    poi.m_subGeomIdx,
                    poi.m_localSpacePos,
                    poi.m_worldSpacePos ) ) )
                {
                    
                    
                    
                    
                }
            }
        }

        viewport.getOwnerWindow()->getContext()->unlock();
    }

    if ( getEnabledInputs().anyIsSet( hkgVdbInputTypes::MOUSE ) )
    {
        const hkgMouse& mouse = input.getMouse();

        hkVector4 mousePosWorldSpace;
        mousePosWorldSpace.setZero4();

        hkgCamera* c = viewport.getCamera();
        hkgVdbPoint vmp = hkgVdbPoint::viewportCoordsFromInputCoords( mouse.getPosX(), mouse.getPosY(), viewport );
        c->unProject( vmp.m_x, vmp.m_y, float( m_mouseDepth ), viewport.getWidth(), viewport.getHeight(), mousePosWorldSpace );

        dragGeometry( mousePosWorldSpace );
    }

    return HK_SUCCESS;
}

hkResult hkgVdbSelectionWidget::setSelectionFlags( hkgVdbSelectionFlags flags )
{
    //HK_VDB_VERIFY_CONDITION_MSG(
    //  !flags.allAreSet( hkgVdbSelectionFlags::IGNORE_UNHIGHLIGHTED_SELECTIONS | hkgVdbSelectionFlags::CLEAR_ON_UNHIGHLIGHTED_SELECTIONS ),
    //  0xadb00026,
    //  hkVdbError::INVALID_ARGUMENTS,
    //  "The selection flags are not compatible with each other" );
    m_selectionFlags = flags;
    return HK_SUCCESS;
}

hkResult hkgVdbSelectionWidget::grabGeometry( hkUint64 id, hkVector4Parameter worldPosition )
{
    return grabGeometry( id, -1, worldPosition );
}

hkResult hkgVdbSelectionWidget::grabGeometry( hkUint64 id, int subGeomIndex, hkVector4Parameter worldPosition )
{
    hkResult result = HK_SUCCESS;
    hkgDisplayObject* object = m_geometryControl->getGeometryFromId( id );
    if ( object )
    {
        // For instanced objects, send back the update Id from the last UPDATE_PARTICLE_TRANSFORMS command
        int particleUpdateId = 0;
        if ( object->getStatusFlags() & HKG_DISPLAY_OBJECT_INSTANCED )
        {
            particleUpdateId = (int)(hkUint64)object->m_userPtr;
        }

        m_grabbedId = id;
        
        m_grabbedSubGeomIdx = subGeomIndex;
        m_queuedCmdBuffers[m_writeCmdBufferIndex].pushBack( {
            hkVdbCmdType::GRAB_GEOMETRY,
            worldPosition,
            hkVdbCmdId::extractStreamId( id ),
            subGeomIndex,
            particleUpdateId } );

        // Note: we do POI before modifying selection, in case selection callbacks are interested
        if ( m_selectionFlags.anyIsSet( hkgVdbSelectionFlags::MODIFY_PRIMARY_POI_ON_GRABBING_GEOMETRY ) )
        {
            hkVector4 localPointOfInterest;
            if ( computeLocalPointOfInterest(
                m_grabbedId,
                m_grabbedSubGeomIdx,
                worldPosition,
                localPointOfInterest ) )
            {
                setLocalPointOfInterest(
                    PrimaryPointOfInterest,
                    m_grabbedId,
                    m_grabbedSubGeomIdx,
                    localPointOfInterest,
                    hkColor::WHITE,
                    hk1PointDisplayStyle::STAR,
                    1.0f );
            }
            else
            {
                signalError(
                    0xadb00051,
                    hkVdbError::GEOMETRY_NOT_FOUND,
                    "Could not retrieve geometry transform" );
                result = HK_FAILURE;
            }
        }
    }
    else if ( id )
    {
        signalError( 0xadb00004, hkVdbError::GEOMETRY_NOT_FOUND, "Could not find geometry to grab" );
        result = HK_FAILURE;
    }

    // Now update selection, if requested.
    if ( m_selectionFlags.anyIsSet( hkgVdbSelectionFlags::MODIFY_SELECTION_ON_GRABBING_GEOMETRY ) )
    {
        modifySelectionOnGrabbedGeometry( id, object );
    }

    return result;
}

hkUint64 hkgVdbSelectionWidget::getGrabbedGeometry() const
{
    return m_grabbedId;
}

void hkgVdbSelectionWidget::dragGeometry( hkVector4Parameter worldPosition )
{
    if ( m_grabbedId )
    {
        m_queuedCmdBuffers[m_writeCmdBufferIndex].pushBack( { hkVdbCmdType::DRAG_GEOMETRY, worldPosition } );
    }
}

void hkgVdbSelectionWidget::releaseGeometry()
{
    if ( m_grabbedId )
    {
        m_grabbedId = 0;
        m_grabbedSubGeomIdx = -1;
        m_queuedCmdBuffers[m_writeCmdBufferIndex].pushBack( { hkVdbCmdType::RELEASE_GEOMETRY } );
    }
}

void hkgVdbSelectionWidget::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        int* idx = &m_writeCmdBufferIndex;
        int oldIdx = m_writeCmdBufferIndex;
        int newIdx = ( m_writeCmdBufferIndex + 1 ) % 2;
        HK_VERIFY( 0x22441106, hkAtomic::compareAndSwap( idx, oldIdx, newIdx ), "Shouldn't be multiple threads calling onConnectedSignal" );
        m_queuedCmdBuffers[oldIdx].clear();
        for ( int i = 0; i < 2; i++ )
        {
            m_selectionSets[i].~SelectionSets();
            new ( &m_selectionSets[i] ) SelectionSets();
        }
    }
}

void hkgVdbSelectionWidget::onGeometryHiddenSignal( hkgDisplayObject& object )
{
    // Update grabbed
    hkUint64 geometryId = object.m_id;
    if ( geometryId == m_grabbedId )
    {
        releaseGeometry(); // cant be current if it is not shown anymore.
    }

    // Update POIs
    
    hkLocalArray<hkUint64> poisToRemove( 10 );
    for ( auto iter = m_pointsOfInterest.getIterator();
        m_pointsOfInterest.isValid( iter );
        iter = m_pointsOfInterest.getNext( iter ) )
    {
        PointOfInterest& poi = m_pointsOfInterest.getValue( iter );
        if ( poi.m_geometryId == geometryId )
        {
            hkUint64 poiId = m_pointsOfInterest.getKey( iter );
            poisToRemove.pushBack( poiId );
        }
    }
    clearPointsOfInterest( poisToRemove );

    // Update Selection
    {
        SelectionChangedInfo changedInfo;
        acquireRenderLock();
        removeFromSetInternal(
            hkArrayViewT::fromSingleObject( geometryId ),
            m_selectionSets[GeometrySetIdx].m_selected,
            changedInfo.m_removedFromSelected );
        removeFromSetInternal(
            hkArrayViewT::fromSingleObject( geometryId ),
            m_selectionSets[GeometrySetIdx].m_highlighted,
            changedInfo.m_removedFromHighlighted );
        releaseRenderLock();

        if ( changedInfo.m_removedFromSelected.getSize() ||
            changedInfo.m_removedFromHighlighted.getSize() )
        {
            signalSelectionChangedInternal<GeometrySetIdx>( changedInfo );
        }
    }
}

void hkgVdbSelectionWidget::onPlaybackInfoReceivedSignal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result )
{
    // Note: during replay these should get ignored anyway
    if ( info.flagWasSet( hkVdbPlaybackFlags::FRAME_ENDED ) )
    {
        int* idx = &m_writeCmdBufferIndex;
        int oldIdx = m_writeCmdBufferIndex;
        int newIdx = ( m_writeCmdBufferIndex + 1 ) % 2;
        HK_VERIFY( 0x22441104, hkAtomic::compareAndSwap( idx, oldIdx, newIdx ), "Shouldn't be multiple threads calling onPlaybackInfoReceivedSignal" );
        for ( int i = 0; i < m_queuedCmdBuffers[oldIdx].getSize(); i++ )
        {
            QueuedGrabbedGeomCommand& cmd = m_queuedCmdBuffers[oldIdx][i];
            switch ( cmd.m_type )
            {
                case hkVdbCmdType::GRAB_GEOMETRY:
                {
                    if ( hkVdbScopedStreamWriter writer = m_output->startCmd( hkVdbCmdType::GRAB_GEOMETRY ) )
                    {
                        hkVdbOStream ostream( writer );
                        ostream.writeQuadVector4( cmd.m_position );
                        ostream.write64( cmd.m_id );
                        ostream.write32( cmd.m_index );
                        ostream.write32( cmd.m_updateId );
                    }
                    else
                    {
                        HK_VDB_SIGNAL_ERROR_MSG( 0xadb00003, hkVdbError::PLUGIN_WRITE_ERROR, "Could not write geometry grabbed cmd to server" );
                        result.signalError( *this );
                    }
                    break;
                }
                case hkVdbCmdType::DRAG_GEOMETRY:
                {
                    if ( hkVdbScopedStreamWriter writer = m_output->startCmd( hkVdbCmdType::DRAG_GEOMETRY ) )
                    {
                        hkVdbOStream ostream( writer );
                        ostream.writeQuadVector4( cmd.m_position );
                    }
                    else
                    {
                        HK_VDB_SIGNAL_ERROR_MSG( 0xadb00023, hkVdbError::PLUGIN_WRITE_ERROR, "Could not write geometry dragged cmd to server" );
                        result.signalError( *this );
                    }
                    break;
                }
                case hkVdbCmdType::RELEASE_GEOMETRY:
                {
                    if ( !m_output->startCmd( hkVdbCmdType::RELEASE_GEOMETRY ) )
                    {
                        HK_VDB_SIGNAL_ERROR_MSG( 0xadb00024, hkVdbError::PLUGIN_WRITE_ERROR, "Could not write geometry released cmd to server" );
                        result.signalError( *this );
                    }
                    break;
                }
                default:
                {
                    HK_ASSERT( 0x22441105, false, "Unknown grab geometry cmd type" );
                    HK_VDB_SIGNAL_ERROR_MSG( 0xadb00021, hkVdbError::PLUGIN_ERROR, "Unknown grab geometry cmd type" );
                    result.signalError( *this );
                    break;
                }
            }
        }
        m_queuedCmdBuffers[oldIdx].clear();
    }
}

hkResult hkgVdbSelectionWidget::enable()
{
    HK_VDB_VERIFY_CONDITION( m_client, 0xadb00043, hkVdbError::PLUGIN_ERROR );
    HK_SUBSCRIBE_TO_SIGNAL( m_client->m_connected, this, hkgVdbSelectionWidget );
    HK_SUBSCRIBE_TO_SIGNAL( m_client->getCmdHandler<hkVdbPlaybackHandler>()->m_playbackInfoReceived, this, hkgVdbSelectionWidget );
    m_output = &m_client->getCmdOutput();
    return HK_SUCCESS;
}

hkResult hkgVdbSelectionWidget::disable()
{
    HK_VDB_VERIFY_CONDITION( m_client, 0xadb00044, hkVdbError::PLUGIN_ERROR );
    m_client->m_connected.unsubscribeAll( this );
    m_client->getCmdHandler<hkVdbPlaybackHandler>()->m_playbackInfoReceived.unsubscribeAll( this );
    m_output = HK_NULL;
    return HK_SUCCESS;
}

hkgVdbInputTypes hkgVdbSelectionWidget::captureInputs(
    const hkgViewport& viewport,
    const hkgInputManager& input,
    hkgVdb2dWidget* widgetUnderMouse,
    hkgVdbInputTypes enabledInputs )
{
    if ( enabledInputs.anyIsSet( hkgVdbInputTypes::MOUSE ) )
    {
        // Note: ignoring captured keyboard states here intentionally
        const hkgKeyboard& keyboard = input.getKeyboard();
        if ( keyboard.wasKeyPressed( HKG_VKEY_SPACE ) )
        {
            const hkgMouse& mouse = input.getMouse();

            // Find id
            hkgViewportPickData pd;
            hkgDisplayWorld* world = &m_geometryControl->accessDisplayWorld();
            hkgVdbPoint vmp = hkgVdbPoint::viewportCoordsFromInputCoords( mouse.getPosX(), mouse.getPosY(), viewport );
            hkUint64 id = 0;
            int index = 0;
            hkVector4 mousePosWorldSpace;
            if ( viewport.pick( vmp.m_x, vmp.m_y, world, pd ) )
            {
                const int objectNum = pd.getClosestObjectIndex();
                mousePosWorldSpace.set( pd.getClosestObjectWorldPos()[0], pd.getClosestObjectWorldPos()[1], pd.getClosestObjectWorldPos()[2] );

                float result[3];
                viewport.getCamera()->project(
                    pd.getClosestObjectWorldPos()[0],
                    pd.getClosestObjectWorldPos()[1],
                    pd.getClosestObjectWorldPos()[2],
                    viewport.getWidth(),
                    viewport.getHeight(),
                    result );
                m_mouseDepth = result[2];

                if ( const hkgDisplayObject* dobject = world->getDisplayObject( objectNum ) )
                {
                    id = dobject->m_id;
                    index = pd.getClosestGeometryIndex();
                }
            }

            // Grab the geometry (default of 0 will end up doing a proper clear, etc.)
            if ( grabGeometry( id, index, mousePosWorldSpace ).isSuccess() )
            {
                return hkgVdbInputTypes::MOUSE;
            }
        }
        else if ( keyboard.getKeyState( HKG_VKEY_SPACE ) )
        {
            // Return if we already have the object
            if ( m_grabbedId )
            {
                return hkgVdbInputTypes::MOUSE;
            }
        }
    }

    // Shouldn't be dragging anything anymore
    releaseGeometry();

    // Release mouse
    return 0;
}

hkResult hkgVdbSelectionWidget::setPOIInternal(
    hkUint64 id,
    hkUint64 geometryId,
    int subGeomIdx,
    hkVector4Parameter position,
    hkColor::Argb color,
    hk1PointDisplayStyle::Enum style,
    hkReal scale,
    PointsOfInterest& pois,
    PointsOfInterest& addedOut )
{
    // NOTE: args are checked by callers.
    HK_ASSERT( 0x22441414, scale != -1, "Should have been rejected by caller" );

    auto iter = pois.findOrInsertKey( id, PointOfInterest() );
    PointOfInterest& poi = pois.getValue( iter );
    if ( poi.m_scale == -1 )
    {
        addedOut.insert( id, poi );
    }

    // Local
    if ( geometryId )
    {
        if ( computeGlobalPointOfInterest( geometryId, subGeomIdx, position, poi.m_worldSpacePos ) )
        {
            poi.m_localSpacePos = position;
            poi.m_geometryId = geometryId;
            poi.m_subGeomIdx = subGeomIdx;
            poi.m_color = color;
            poi.m_style = style;
            poi.m_scale = scale;
        }
        else
        {
            signalError(
                0xadb00055,
                hkVdbError::INVALID_ARGUMENTS,
                "The geometryId/subGeomIdx was not valid for adding a local POI and was therefore rejected" );
            pois.remove( iter );
            return HK_FAILURE;
        }
    }
    // Global
    else
    {
        poi.m_worldSpacePos = position;
        poi.m_geometryId = 0;
        poi.m_subGeomIdx = -1;
        poi.m_color = color;
        poi.m_style = style;
        poi.m_scale = scale;
    }

    return HK_SUCCESS;
}

bool hkgVdbSelectionWidget::clearPOIsInternal(
    hkArrayView<hkUint64> ids,
    PointsOfInterest& pois,
    PointsOfInterest& globalRemovedOut,
    PointsOfInterest& localRemovedOut )
{
    bool removed = false;
    if ( pois.getSize() )
    {
        if ( hkVdbValidity<hkArrayView<hkUint64>>::isValid( ids ) )
        {
            for ( int i = 0; i < ids.getSize(); i++ )
            {
                hkUint64 id = ids[i];
                auto iter = pois.findKey( id );
                if ( pois.isValid( iter ) )
                {
                    PointOfInterest& poi = pois.getValue( iter );
                    if ( poi.m_geometryId )
                    {
                        localRemovedOut.insert( id, poi );
                    }
                    else
                    {
                        globalRemovedOut.insert( id, poi );
                    }
                    pois.remove( iter );
                    removed = true;
                }
            }
        }
        else
        {
            removed = pois.getSize();
            for ( auto iter = pois.getIterator();
                pois.isValid( iter );
                iter = pois.getNext( iter ) )
            {
                hkUint64 id = pois.getKey( iter );
                PointOfInterest& poi = pois.getValue( iter );
                if ( poi.m_geometryId )
                {
                    localRemovedOut.insert( id, poi );
                }
                else
                {
                    globalRemovedOut.insert( id, poi );
                }
            }
            pois.clear();
        }
    }
    return removed;
}

bool hkgVdbSelectionWidget::removeFromSetInternal(
    hkArrayView<hkUint64> ids,
    SelectionSet& set,
    SelectionSet& removedOut,
    SelectionSet* dependentSet,
    SelectionSet* dependentRemovedOut )
{
    bool removed = false;
    if ( set.getSize() )
    {
        if ( hkVdbValidity<hkArrayView<hkUint64>>::isValid( ids ) )
        {
            for ( int i = 0; i < ids.getSize(); i++ )
            {
                hkUint64 id = ids[i];
                auto iter = set.findKey( id );
                if ( set.isValid( iter ) )
                {
                    hkgDisplayObject* obj = set.getValue( iter );
                    set.remove( iter );
                    removed = true;
                    removedOut.insert( id, obj );
                }

                
                if ( dependentSet &&
                    m_selectionFlags.anyIsSet( /*hkgVdbSelectionFlags::CLEAR_ON_UNHIGHLIGHTED_SELECTIONS | */hkgVdbSelectionFlags::IGNORE_UNHIGHLIGHTED_SELECTIONS ) )
                {
                    iter = dependentSet->findKey( id );
                    if ( dependentSet->isValid( iter ) )
                    {
                        hkgDisplayObject* obj = dependentSet->getValue( iter );
                        dependentSet->remove( iter );
                        removed = true;
                        dependentRemovedOut->insert( id, obj );
                    }
                }
            }
        }
        else
        {
            removed = set.getSize();
            hkVdbMap_copy( removedOut, set );
            if ( dependentSet &&
                m_selectionFlags.anyIsSet( /*hkgVdbSelectionFlags::CLEAR_ON_UNHIGHLIGHTED_SELECTIONS | */hkgVdbSelectionFlags::IGNORE_UNHIGHLIGHTED_SELECTIONS ) )
            {
                for ( auto iter = set.getIterator();
                    set.isValid( iter );
                    iter = set.getNext( iter ) )
                {
                    hkUint64 id = set.getKey( iter );
                    auto dependentIter = dependentSet->findKey( id );
                    if ( dependentSet->isValid( dependentIter ) )
                    {
                        hkgDisplayObject* obj = dependentSet->getValue( dependentIter );
                        dependentSet->remove( dependentIter );
                        dependentRemovedOut->insert( id, obj );
                    }
                }
            }
            set.clear();
        }
    }
    return removed;
}

hkgDisplayObject* hkgVdbSelectionWidget::getObjectFromId( hkUint64 id )
{
    return m_geometryControl->getGeometryFromId( id );
}

void hkgVdbSelectionWidget::modifySelectionOnGrabbedGeometry( hkUint64 id, hkgDisplayObject* obj )
{
    if ( id && obj )
    {
        
        
        
        
        clearSelectedGeometries();
        clearSelectedDisplays();
        if ( selectGeometry( id ).isSuccess() )
        {
            
            
        }
    }
    else
    {
        clearSelectedGeometries();
        clearSelectedDisplays();
    }
}

hkBool32 hkgVdbSelectionWidget::getGeometryTransform( hkUint64 geometryId, int subGeomIdx, hkTransform& transformOut ) const
{
    const float* hkgmat4 = HK_NULL;
    if ( hkgDisplayObject* object = m_geometryControl->getGeometryFromId( geometryId ) )
    {
        if ( object->getStatusFlags() & HKG_DISPLAY_OBJECT_INSTANCED )
        {
            hkgInstancedDisplayObject* instancedObject = static_cast< hkgInstancedDisplayObject* >( object );
            if ( ( subGeomIdx >= 0 ) && ( subGeomIdx < instancedObject->getNumObjects() ) )
            {
                hkgmat4 = instancedObject->getTransform( subGeomIdx );
            }
        }
        else
        {
            hkgmat4 = object->getTransform();
        }
    }

    if ( hkgmat4 )
    {
        HK_ALIGN16( float mat4[16] );
        hkgMat4Copy( mat4, hkgmat4 );
        transformOut.set4x4ColumnMajor( mat4 );
        return true;
    }
    else
    {
        return false;
    }
}

hkBool32 hkgVdbSelectionWidget::computeLocalPointOfInterest(
    hkUint64 geometryId,
    int subGeomIdx,
    hkVector4Parameter worldPosition,
    hkVector4& localPositionOut ) const
{
    hkTransform t;
    if ( getGeometryTransform( geometryId, subGeomIdx, t ) )
    {
        localPositionOut.setTransformedInversePos( t, worldPosition );
        return true;
    }
    else
    {
        return false;
    }
}

hkBool32 hkgVdbSelectionWidget::computeGlobalPointOfInterest(
    hkUint64 geometryId,
    int subGeomIdx,
    hkVector4Parameter localPosition,
    hkVector4& worldPositionOut ) const
{
    hkTransform t;
    if ( getGeometryTransform( geometryId, subGeomIdx, t ) )
    {
        hkVector4 pointOfInterestWS;
        worldPositionOut.setTransformedPos( t, localPosition );
        return true;
    }
    else
    {
        return false;
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
