// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPlugin.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/3d/hkgVdbGridWidget.h>

#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/Math/hkgMath.h>
#include <Graphics/Common/DisplayContext/hkgDisplayContext.h>
#include <Graphics/Common/Window/hkgViewport.h>
#include <Graphics/Common/Camera/hkgCamera.h>

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/hkVdbClient.h>

hkResult hkgVdbGridWidget::initialize( hkgVdbPlugin& plugin, hkVdbClient& client )
{
    hkVdbDisplayHandler* handler = client.getCmdHandler<hkVdbDisplayHandler>();
    HK_SUBSCRIBE_TO_SIGNAL( handler->m_displayOptionsSet, this, hkgVdbGridWidget );
    hkVdbSignalResult result;
    hkDebugDisplayHandler::Options options;
    if ( const hkDebugDisplayHandler::Options* handlerOptions = handler->getDisplayOptions() )
    {
        options = *handlerOptions;
    }
    
    
    onDisplayOptionsSetSignal( options, result );
    HK_VDB_VERIFY_SIGNAL_RESULT( result );
    return HK_SUCCESS;
}

hkResult hkgVdbGridWidget::deinitialize( hkgVdbPlugin& plugin, hkVdbClient& client )
{
    client.getCmdHandler<hkVdbDisplayHandler>()->m_displayOptionsSet.unsubscribeAll( this );
    return HK_SUCCESS;
}

void hkgVdbGridWidget::render( hkgDisplayContext& context ) const
{
    hkgViewport* v = context.getCurrentViewport();
    HK_ASSERT_NO_MSG( 0x6f2f828d, v != HK_NULL );

    hkgCamera* c = v->getCamera();

    int axis1 = ( m_options.m_upAxis + 2 ) % 3;
    int axis2 = ( m_options.m_upAxis + 1 ) % 3;
    int upAxis = ( m_options.m_upAxis ) % 3;

    float dir[3];
    hkgVec3Sub( dir, c->getToPtr(), c->getFromPtr() );

    int span = m_options.m_halfSpan;
    hkReal scale = m_options.m_scale;

    bool trackball = v->getNavigationMode() == HKG_CAMERA_NAV_TRACKBALL;
    if ( trackball ) // we go off the distance to the POI
    {
        float distSqrd = hkgVec3Dot( dir, dir );
        if ( distSqrd > ( span * span ) )
            scale *= 10;
    }
    // not trackball (so fly mode, so go off the height above plane)
    else if ( hkg_fabs( c->getFromPtr()[upAxis] ) > span )
    {
        scale *= 10;
    }

    int mult = int( span * scale );

    int jumpSize = int( m_options.m_numSubLines * scale );
    float gridOrigin[3]; gridOrigin[upAxis] = 0;
    // if ( trackball )
    {
        gridOrigin[axis1] = c->getToPtr()[axis1];
        gridOrigin[axis2] = c->getToPtr()[axis2];
    }
#if 0
    else
    {
        // determine intersection of view and the plane
        float toOnPlane[3];
        {
            hkgVec3Normalize( dir );
            float s = -c->getFromPtr()[upAxis] / dir[upAxis];
            hkgVec3Copy( toOnPlane, c->getFromPtr() );
            hkgVec3MultAdd( toOnPlane, dir, s );
        }
        // place offset at that intersection
        //gridOrigin[axis1] = toOnPlane[axis1];
        //gridOrigin[axis2] = toOnPlane[axis2];
        gridOrigin[axis1] = ( c->getFromPtr()[axis1] + toOnPlane[axis1] ) / 2;
        gridOrigin[axis2] = ( c->getFromPtr()[axis2] + toOnPlane[axis2] ) / 2;
    }
#endif

    // clamp
#if 0
    {
        float fromOnPlane[3];
        hkgVec3Copy( fromOnPlane, c->getFromPtr() );
        fromOnPlane[upAxis] = 0;
        float dirOnPlane[3];
        hkgVec3Sub( dirOnPlane, gridOrigin, fromOnPlane );
        if ( hkgVec3Length( dirOnPlane ) > m_options.m_maxProjDistanceToGrid )
        {
            hkgVec3Normalize( dirOnPlane );
            hkgVec3MultAdd( fromOnPlane, dirOnPlane, m_options.m_maxProjDistanceToGrid );
            hkgVec3Copy( gridOrigin, fromOnPlane );
        }
    }
#endif

    int a1 = ( int ) ( hkg_floor( gridOrigin[axis1] / jumpSize ) ) * jumpSize;
    int a2 = ( int ) ( hkg_floor( gridOrigin[axis2] / jumpSize ) ) * jumpSize;

    float gridOffset[3] = { 0.0f, 0.0f, 0.0f };
    gridOffset[axis1] = ( float ) a1;
    gridOffset[axis2] = ( float ) a2;

    float start[3] = { 0.0f, 0.0f, 0.0f };
    float end[3] = { 0.0f, 0.0f, 0.0f };

    context.beginGroup( HKG_IMM_LINES );
    for ( int i = -span; i <= span; i++ )
    {
        unsigned int color = m_options.m_subLineColor;
        if ( !( i % m_options.m_numSubLines ) )
        {
            color = m_options.m_color;
        }

        // not to good for ps2gl, but fine for everything else (need to def everytime for ps2gl
        // but that should be automatic in the hkg
        context.setCurrentColorPacked( color );

        // along x
        start[axis1] = ( float ) mult;
        end[axis1] = -( float ) mult;
        start[axis2] = float( i * scale );
        end[axis2] = start[axis2];

        hkgVec3Add( start, gridOffset );
        hkgVec3Add( end, gridOffset );
        context.setCurrentPosition( start );
        context.setCurrentPosition( end );

        // along z
        start[axis2] = float( mult );
        end[axis2] = -float( mult );
        start[axis1] = float( i * scale );
        end[axis1] = start[axis1];

        hkgVec3Add( start, gridOffset );
        hkgVec3Add( end, gridOffset );
        context.setCurrentPosition( start );
        context.setCurrentPosition( end );
    }
    context.endGroup();
}

void hkgVdbGridWidget::onDisplayOptionsSetSignal( const hkDebugDisplayHandler::Options& options, hkVdbSignalResult& result )
{
    // Determine canonical up
    {
        int maxAxisIdx = options.m_up.getIndexOfMaxAbsComponent<3>();
        hkReal maxAxisVal = options.m_up.getComponent( maxAxisIdx );
        m_options.m_upAxis =
            hkgVdbGridWidgetOptions::UpAxis(
                hkgVdbGridWidgetOptions::X * ( ( maxAxisIdx == 0 ) && ( maxAxisVal >= 0 ) ) +
                hkgVdbGridWidgetOptions::Y * ( ( maxAxisIdx == 1 ) && ( maxAxisVal >= 0 ) ) +
                hkgVdbGridWidgetOptions::Z * ( ( maxAxisIdx == 2 ) && ( maxAxisVal >= 0 ) ) +
                hkgVdbGridWidgetOptions::NEG_X * ( ( maxAxisIdx == 0 ) && ( maxAxisVal < 0 ) ) +
                hkgVdbGridWidgetOptions::NEG_Y * ( ( maxAxisIdx == 1 ) && ( maxAxisVal < 0 ) ) +
                hkgVdbGridWidgetOptions::NEG_Z * ( ( maxAxisIdx == 2 ) && ( maxAxisVal < 0 ) )
            );
    }

    // Determine halfspan
    {
        hkVector4 halfExtents;
        options.m_displayBounds.getHalfExtents( halfExtents );
        if ( halfExtents.isOk<3>() )
        {
            hkVector4 canonicalUpSelect;
            canonicalUpSelect.set(
                ( ( m_options.m_upAxis == hkgVdbGridWidgetOptions::X ) || ( m_options.m_upAxis == hkgVdbGridWidgetOptions::NEG_X ) ),
                ( ( m_options.m_upAxis == hkgVdbGridWidgetOptions::Y ) || ( m_options.m_upAxis == hkgVdbGridWidgetOptions::NEG_Y ) ),
                ( ( m_options.m_upAxis == hkgVdbGridWidgetOptions::Z ) || ( m_options.m_upAxis == hkgVdbGridWidgetOptions::NEG_Z ) ) );
            hkVector4 nonUpAxesSelect;
            nonUpAxesSelect.setSub( canonicalUpSelect, hkVector4::getConstant<HK_QUADREAL_1>() );
            nonUpAxesSelect.mul( -1 );
            halfExtents.mul( nonUpAxesSelect );
            hkSimdReal max = halfExtents.horizontalMax<3>();
            hkInt32 maxI = hkInt32( hkReal( max ) );
            // Make grid roughly half as big as world
            hkInt32 idealHalfSpan = ( maxI / 4 );
            if ( idealHalfSpan < HKG_VDB_MIN_GRID_HALF_SPAN ) idealHalfSpan = HKG_VDB_MIN_GRID_HALF_SPAN;
            // Ensure that we are divisible by our sublines so everything is neat
            m_options.m_halfSpan = ( idealHalfSpan % m_options.m_numSubLines ) * m_options.m_numSubLines;
        }
        else
        {
            m_options.m_halfSpan = HKG_VDB_DEFAULT_GRID_HALF_SPAN;
        }
    }

    
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
