// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPlugin.h>
#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPluginApi.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbStatusWidget.h>

#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/DisplayContext/hkgDisplayContext.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/Font/hkgFont.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdInput.h>

#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/Utils/hkgVdbTextBoxDrawer.h>

hkgVdbStatusWidget::hkgVdbStatusWidget() :
    m_window( HK_NULL ),
    m_defaultFont( hkRefNew<hkgFont>( hkgFont::create() ) ),
    m_options( *m_defaultFont ),
    m_spfCalc( .5f, 1.0f / 60 )
{
    m_options.m_textColor = hkColor::rgbFromFloats( 1.0f, 0.75f, 0.75f, 1.0f );
    m_options.m_upperLeftColor = hkColor::rgbFromChars( 0, 0, 0, 0 );
    m_options.m_lowerRightColor = hkColor::rgbFromChars( 0, 0, 0, 0 );
    m_options.m_borderColor = hkColor::rgbFromChars( 0, 0, 0, 0 );
}

hkResult hkgVdbStatusWidget::initialize( hkgVdbPlugin& plugin, hkVdbClient& client )
{
    m_defaultFont = hkgVdbTextBoxDrawer::loadDefaultFont( *plugin.getWindow()->getContext(), int( HKG_VDB_WIDGET_DEFAULT_TEXT_SIZE * plugin.getWindow()->getDpiScale() ) );
    m_options.m_font = m_defaultFont;
    m_bounds = hkgVdbTextBoxDrawer::getBounds( plugin.getWindow()->getCurrentViewport(), "", m_options );
    reset();
    m_client = &client;
    m_window = plugin.getWindow();
    return HK_SUCCESS;
}

hkResult hkgVdbStatusWidget::deinitialize( hkgVdbPlugin& plugin, hkVdbClient& client )
{
    m_window = HK_NULL;
    m_client = HK_NULL;
    return HK_SUCCESS;
}

hkResult hkgVdbStatusWidget::update( const hkgViewport& viewport, const hkgInputManager& input )
{
    acquireRenderLock();

    // Fps update
    {
        const float frameTime = m_spfStopwatch.getElapsedSeconds();
        m_spfStopwatch.reset(); m_spfStopwatch.start();
        m_spfCalc.pushNewValue( frameTime );

        const hkReal spf = m_spfCalc.getWindowedMean();
        const hkReal fps = ( spf > 0 ) ? ( 1.0f / spf ) : 0.0f;
        const hkReal ms = ( 1000.0f * spf );
        const hkReal serverStepMs = ( 1000.0f * ( ( m_serverStepFps > 0 ) ? ( 1.0f / m_serverStepFps ) : 0.0f ) );
        const hkReal serverVdbStepMs = ( 1000.0f * ( ( m_serverVdbStepFps > 0 ) ? ( 1.0f / m_serverVdbStepFps ) : 0.0f ) );
        m_serverInfo.printf( "%.1f fps (%.1f ms)\n%.1f server fps (%.1f ms)\n%.1f step fps (%.1f ms)\n", fps, ms, m_serverStepFps, serverStepMs, m_serverVdbStepFps, serverVdbStepMs );
    }

    // Frame info
    {
        float msec = ( 1000 * m_continuousStopwatch.getElapsedSeconds() );
        if ( msec < 0 ) msec = 0;
        const int msecsInAnHour = 1000 * 60 * 60;
        const int msecsInAMin = 1000 * 60;
        const int hours = ( int ) ( msec / msecsInAnHour );
        const int mins = ( int ) ( ( msec - ( hours * msecsInAnHour ) ) / msecsInAMin );
        const int secs = ( int ) ( ( msec - ( hours * msecsInAnHour ) - ( mins * msecsInAMin ) ) / 1000 );
        const int mmsecs = ( int ) ( ( msec - ( hours * msecsInAnHour ) - ( mins * msecsInAMin ) - ( secs * 1000 ) ) * 10 );
        m_clientInfo.printf(
            "Frame:%.4d Time:%.2dH:%.2dM:%.2dS:%.4d Bytes:%d\n",
            m_serverFrames,
            hours, mins, secs, mmsecs, m_serverFrameBytes );
    }

    // Camera update
    if ( m_window != HK_NULL )
    {
        if ( hkgViewport* v = m_window->getCurrentViewport() )
        {
            int llx, lly, urx, ury;
            v->getLowerLeftCoord( llx, lly );
            v->getUpperRightCoord( urx, ury );

            hkgCamera* c = v->getCamera();
            const float * f = c->getFromPtr();
            const float * t = c->getToPtr();

            m_cameraInfo.printf(
                "Cam  POI[%.1f %.1f %.1f]\n"
                "Cam From[%.1f %.1f %.1f]\n",
                t[0], t[1], t[2],
                f[0], f[1], f[2] );
        }
    }

    releaseRenderLock();

    return HK_SUCCESS;
}

void hkgVdbStatusWidget::render( hkgDisplayContext& context ) const
{
    hkStringBuf status;
    status.append( m_serverInfo );
    status.append( m_clientInfo );
    status.append( m_cameraInfo );
    m_bounds = hkgVdbTextBoxDrawer::drawTextBox( context, status, m_options );
}

void hkgVdbStatusWidget::onFrameStatsUpdatedSignal( const hkVdbStatsHandler::FrameStats& frameStats )
{
    acquireRenderLock();
    if ( !m_continuousStopwatch.isRunning() ) m_continuousStopwatch.start();
    hkReal spf = frameStats.getStepStats().getWindowedMean();
    m_serverStepFps = ( spf > 0 ) ? ( 1.0f / spf ) : 0.0f;
    spf = frameStats.getVdbStepStats().getWindowedMean();
    m_serverVdbStepFps = ( spf > 0 ) ? ( 1.0f / spf ) : 0.0f;
    m_serverFrames = frameStats.getNumFrames();
    releaseRenderLock();
}

void hkgVdbStatusWidget::onStepCompletedSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd )
{
    acquireRenderLock();
    m_serverFrameBytes = frame.getCmdBytes();
    releaseRenderLock();
}

void hkgVdbStatusWidget::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        reset();
    }
}

void hkgVdbStatusWidget::onDisconnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        acquireRenderLock();
        m_serverFrameBytes = 0;
        if ( m_continuousStopwatch.isRunning() ) m_continuousStopwatch.stop();
        releaseRenderLock();
    }
}

hkResult hkgVdbStatusWidget::enable()
{
    HK_VDB_VERIFY_CONDITION( m_client, 0xadb00035, hkVdbError::PLUGIN_ERROR );
    HK_SUBSCRIBE_TO_SIGNAL( m_client->m_connected, this, hkgVdbStatTextWidget );
    HK_SUBSCRIBE_TO_SIGNAL( m_client->m_disconnected, this, hkgVdbStatTextWidget );
    HK_SUBSCRIBE_TO_SIGNAL( m_client->getCmdHandler<hkVdbStatsHandler>()->m_frameStatsUpdated, this, hkgVdbStatTextWidget );
    HK_SUBSCRIBE_TO_SIGNAL( m_client->m_stepCompleted, this, hkgVdbStatTextWidget );
    return HK_SUCCESS;
}

hkResult hkgVdbStatusWidget::disable()
{
    HK_VDB_VERIFY_CONDITION( m_client, 0xadb00036, hkVdbError::PLUGIN_ERROR );
    m_client->m_connected.unsubscribeAll( this );
    m_client->m_disconnected.unsubscribeAll( this );
    m_client->getCmdHandler<hkVdbStatsHandler>()->m_frameStatsUpdated.unsubscribeAll( this );
    m_client->m_stepCompleted.unsubscribeAll( this );
    return HK_SUCCESS;
}

void hkgVdbStatusWidget::reset()
{
    acquireRenderLock();
    m_continuousStopwatch.reset();
    m_spfStopwatch.reset();
    m_spfCalc.reset();
    m_serverStepFps = 0;
    m_serverVdbStepFps = 0;
    m_serverFrames = 0;
    m_serverFrameBytes = 0;
    releaseRenderLock();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
