// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPlugin.h>
#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPluginApi.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbStatTextWidget.h>

#include <Common/Base/System/Hardware/hkHardwareInfo.h>
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>

#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/DisplayContext/hkgDisplayContext.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/Font/hkgFont.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>

#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/Utils/hkgVdbTextBoxDrawer.h>

#if defined(HK_PLATFORM_WIN32)
#include <windows.h>
#include <WinUser.h>
#endif

hkgVdbStatTextWidget::hkgVdbStatTextWidget() :
    m_defaultFont( hkRefNew<hkgFont>( hkgFont::create() ) ),
    m_options( *m_defaultFont ),
    m_activeNode( HK_NULL ),
    m_activeNodeToRestore( HK_NULL ),
    m_numThreads( 0 )
{}

hkgVdbStatTextWidget::~hkgVdbStatTextWidget()
{
    hkMonitorStreamAnalyzer::deallocateLocalStrings( m_statsLocalStrings );
}

hkResult hkgVdbStatTextWidget::initialize( hkgVdbPlugin& plugin, hkVdbClient& client )
{
    m_defaultFont = hkgVdbTextBoxDrawer::loadDefaultFont( *plugin.getWindow()->getContext(), int( HKG_VDB_WIDGET_DEFAULT_TEXT_SIZE * plugin.getWindow()->getDpiScale() ) );
    m_options.m_font = m_defaultFont;
    hkArray<char> buf;
    writeStatText( buf );
    m_bounds = hkgVdbTextBoxDrawer::getBounds( plugin.getWindow()->getContext()->getCurrentViewport(), buf.begin(), m_options );
    m_client = &client;
    return HK_SUCCESS;
}

hkResult hkgVdbStatTextWidget::deinitialize( hkgVdbPlugin& plugin, hkVdbClient& client )
{
    m_client = HK_NULL;
    return HK_SUCCESS;
}

hkResult hkgVdbStatTextWidget::update( const hkgViewport& viewport, const hkgInputManager& input )
{
    acquireRenderLock();

    // Do we have enough stats to init the active node?
    // As the sims can start off with no active node (no children in stats, esp multithread demos)
    if ( !m_activeNode && ( m_statsTree.m_children.getSize() > 0 ) )
    {
        m_statsTree.m_userFlags |= 1;
        m_activeNode = m_statsTree.m_children[0];
    }

    // No active node is a no-op
    if ( !m_activeNode )
    {
        releaseRenderLock();
        return HK_SUCCESS;
    }

    const hkgMouse& mouse = input.getMouse();
    const hkgKeyboard& keyboard = input.getKeyboard();
    const hkgPad& gamepad = input.getGamePad( 0 );
    hkMonitorStreamAnalyzer::CursorKeys keys;

    // Buttons
    if ( getEnabledInputs().anyIsSet( hkgVdbInputTypes::KEYBOARD ) || getEnabledInputs().anyIsSet( hkgVdbInputTypes::MOUSE ) )
    {
        // Copy
        if ( keyboard.wasKeyPressed( 'C' ) )
        {
            copyStatTextToClipboard();
        }

        // Normal button interactions
        keys.m_upPressed =
            keyboard.wasKeyPressed( HKG_VKEY_UP ) ||
            keyboard.isKeyHeld( HKG_VKEY_UP, HKG_VDB_KEY_HOLD_START, HKG_VDB_KEY_HOLD_PERIOD ) ||
            gamepad.wasButtonPressed( HKG_PAD_DPAD_UP );
        keys.m_downPressed =
            keyboard.wasKeyPressed( HKG_VKEY_DOWN ) ||
            keyboard.isKeyHeld( HKG_VKEY_DOWN, HKG_VDB_KEY_HOLD_START, HKG_VDB_KEY_HOLD_PERIOD ) ||
            gamepad.wasButtonPressed( HKG_PAD_DPAD_DOWN );
        keys.m_leftPressed =
            keyboard.wasKeyPressed( HKG_VKEY_LEFT ) ||
            keyboard.isKeyHeld( HKG_VKEY_LEFT, HKG_VDB_KEY_HOLD_START, HKG_VDB_KEY_HOLD_PERIOD ) ||
            gamepad.wasButtonPressed( HKG_PAD_DPAD_LEFT ) ||
            mouse.wasButtonPressed( HKG_MOUSE_RIGHT_BUTTON );
        keys.m_rightPressed =
            keyboard.wasKeyPressed( HKG_VKEY_RIGHT ) ||
            keyboard.isKeyHeld( HKG_VKEY_RIGHT, HKG_VDB_KEY_HOLD_START, HKG_VDB_KEY_HOLD_PERIOD ) ||
            gamepad.wasButtonPressed( HKG_PAD_DPAD_RIGHT ) ||
            mouse.wasButtonPressed( HKG_MOUSE_LEFT_BUTTON );

        // Check for save/restore
        if ( mouse.wasButtonPressed( HKG_MOUSE_MIDDLE_BUTTON ) )
        {
            if ( m_activeNodeToRestore )
            {
                m_activeNode = m_activeNodeToRestore;
                hkMonitorStreamAnalyzer::expandMonitors( m_activeNode );
                // We also restore all the root level timers since they are part of a root directory which doesn't collapse
                for ( int i = 0; i < m_rootNodesToRestore.getSize(); i++ )
                {
                    hkMonitorStreamAnalyzer::Node* rootNode = m_rootNodesToRestore[i];
                    rootNode->m_userFlags |= 1;
                }
                m_activeNodeToRestore = HK_NULL;
                m_rootNodesToRestore.clear();
            }
            else
            {
                m_activeNodeToRestore = m_activeNode;
                hkMonitorStreamAnalyzer::collapseMonitors( m_activeNodeToRestore );
                // We also collapse all the root level timers since they are part of a root directory which doesn't collapse
                for ( int i = 0; i < m_statsTree.m_children.getSize(); i++ )
                {
                    hkMonitorStreamAnalyzer::Node* rootNode = m_statsTree.m_children[i];
                    if ( rootNode->m_userFlags & 1 )
                    {
                        rootNode->m_userFlags &= ~1;
                        m_rootNodesToRestore.pushBack( rootNode );
                    }
                }
                m_activeNode = HK_NULL;
            }
        }
        // Else, apply normal button interactions
        else
        {
            hkMonitorStreamAnalyzer::Node* formerActiveNode = m_activeNode;
            hkUint32 formerExpansionState = m_activeNode->m_userFlags;
            m_activeNode = hkMonitorStreamAnalyzer::navigateMonitors( keys, m_activeNode );

            // If we changed expanded/collapsed state or the active node via a descent/ascent, then clear our restoration data
            if ( ( ( formerActiveNode != m_activeNode ) || ( formerExpansionState != m_activeNode->m_userFlags ) ) &&
                ( keys.m_leftPressed || keys.m_rightPressed ) )
            {
                m_activeNodeToRestore = HK_NULL;
            }
        }
    }

    // Mouse Wheel
    if ( getEnabledInputs().anyIsSet( hkgVdbInputTypes::MOUSE ) )
    {
        
        hkgVdbPoint wmp = hkgVdbPoint::windowCoordsFromInputCoords( mouse.getPosX(), mouse.getPosY(), viewport );
        if ( m_bounds.isInside( wmp ) )
        {
            keys.m_upPressed = false;
            keys.m_downPressed = false;
            keys.m_leftPressed = false;
            keys.m_rightPressed = false;
            int clicks = mouse.getWheelNotchDelta();
            while ( clicks )
            {
                keys.m_downPressed = bool( clicks < 0 );
                keys.m_upPressed = bool( clicks > 0 );
                m_activeNode = hkMonitorStreamAnalyzer::navigateMonitors( keys, m_activeNode );
                clicks += ( keys.m_downPressed - keys.m_upPressed );
            }
        }
    }

    releaseRenderLock();
    return HK_SUCCESS;
}

void hkgVdbStatTextWidget::render( hkgDisplayContext& context ) const
{
    hkArray<char> buf;
    writeStatText( buf );
    m_bounds = hkgVdbTextBoxDrawer::drawTextBox( context, buf.begin(), m_options );
}

void hkgVdbStatTextWidget::getToolTip( hkStringBuf& toolTipOut, bool verbose ) const
{
    toolTipOut =

        ( verbose ) ?

        "\x14/\x15 or Mouse Wheel : Up / Down\n"
        "\x16/\x17 or Click Right Mouse Button / Left Mouse Button : Collapse / Expand\n"
        "Click Middle Mouse Button : Toggle Collapse-All / Restore\n"
        "C : Copy Contents to Clipboard" :

        "\x14/\x15 or Wheel : Up/Down\n"
        "\x16/\x17 or Click RMB/LMB : Collapse/Expand\n"
        "Click MMB : Toggle Collapse-All/Restore\n"
        "C : Copy Contents to Clipboard";
}

void hkgVdbStatTextWidget::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        acquireRenderLock();
        resetStats();
        hkMonitorStreamAnalyzer::deallocateLocalStrings( m_statsLocalStrings );
        releaseRenderLock();
    }
}

void hkgVdbStatTextWidget::onPerfStatsReceivedSignal( const hkVdbStatsHandler::PerfStats& perfStats, hkVdbSignalResult& result )
{
    acquireRenderLock();
    {
        // If number of threads changes, clear our blended tree data
        if ( m_numThreads != perfStats.m_threadRootNodes.getSize() )
        {
            resetStats();
            m_numThreads = perfStats.m_threadRootNodes.getSize();
        }

        // Copy the tree since our process tick and render tick can be on different threads
        hkMonitorStreamAnalyzer::blendTree( &m_statsTree, &perfStats.m_statsTree, 0 );
        hkMonitorStreamAnalyzer::makeStringsLocal( &m_statsTree, m_statsLocalStrings );
    }
    releaseRenderLock();
}

hkResult hkgVdbStatTextWidget::enable()
{
    HK_VDB_VERIFY_CONDITION( m_client, 0xadb00037, hkVdbError::PLUGIN_ERROR );
    HK_SUBSCRIBE_TO_SIGNAL( m_client->m_connected, this, hkgVdbStatTextWidget );
    HK_SUBSCRIBE_TO_SIGNAL( m_client->getCmdHandler<hkVdbStatsHandler>()->m_perfStatsReceived, this, hkgVdbStatTextWidget );
    return HK_SUCCESS;
}

hkResult hkgVdbStatTextWidget::disable()
{
    HK_VDB_VERIFY_CONDITION( m_client, 0xadb00038, hkVdbError::PLUGIN_ERROR );
    m_client->m_connected.unsubscribeAll( this );
    m_client->getCmdHandler<hkVdbStatsHandler>()->m_perfStatsReceived.unsubscribeAll( this );
    return HK_SUCCESS;
}

hkBool32 hkgVdbStatTextWidget::expandNode()
{
    hkMonitorStreamAnalyzer::CursorKeys keys;
    keys.m_upPressed = false;
    keys.m_downPressed = false;
    keys.m_leftPressed = false;
    keys.m_rightPressed = true;

    if ( m_activeNode )
    {
        hkBool32 wasEnabled = ( ( m_activeNode->m_userFlags & 1 ) != 0 );
        m_activeNode = hkMonitorStreamAnalyzer::navigateMonitors( keys, m_activeNode );
        return !wasEnabled && ( ( m_activeNode->m_userFlags & 1 ) != 0 );
    }
    return false;
}

hkBool32 hkgVdbStatTextWidget::collapseNode()
{
    hkMonitorStreamAnalyzer::CursorKeys keys;
    keys.m_upPressed = false;
    keys.m_downPressed = false;
    keys.m_leftPressed = true;
    keys.m_rightPressed = false;

    if ( m_activeNode )
    {
        hkBool32 wasEnabled = ( ( m_activeNode->m_userFlags & 1 ) != 0 );
        m_activeNode = hkMonitorStreamAnalyzer::navigateMonitors( keys, m_activeNode );
        return wasEnabled && ( ( m_activeNode->m_userFlags & 1 ) == 0 );
    }
    return false;
}

hkBool32 hkgVdbStatTextWidget::selectPreviousNode()
{
    hkMonitorStreamAnalyzer::CursorKeys keys;
    keys.m_upPressed = true;
    keys.m_downPressed = false;
    keys.m_leftPressed = false;
    keys.m_rightPressed = false;

    hkMonitorStreamAnalyzer::Node* lastActiveNode = m_activeNode;
    m_activeNode = hkMonitorStreamAnalyzer::navigateMonitors( keys, m_activeNode );
    return ( lastActiveNode != m_activeNode );
}

hkBool32 hkgVdbStatTextWidget::selectNextNode()
{
    hkMonitorStreamAnalyzer::CursorKeys keys;
    keys.m_upPressed = false;
    keys.m_downPressed = true;
    keys.m_leftPressed = false;
    keys.m_rightPressed = false;

    hkMonitorStreamAnalyzer::Node* lastActiveNode = m_activeNode;
    m_activeNode = hkMonitorStreamAnalyzer::navigateMonitors( keys, m_activeNode );
    return ( lastActiveNode != m_activeNode );
}

void hkgVdbStatTextWidget::appendStatText( hkStringBuf& statTextOut ) const
{
    hkArray<char> buf;
    writeStatText( buf );
    statTextOut.append( buf.begin() );
}

hkResult hkgVdbStatTextWidget::copyStatTextToClipboard() const
{
#if defined(HK_PLATFORM_WIN32)
    acquireRenderLock();
    hkArray<char> buf;
    writeStatText( buf );
    HGLOBAL hMem = GlobalAlloc( GMEM_MOVEABLE, buf.getSize() + 1 );
    if ( !hMem ) { releaseRenderLock(); return HK_FAILURE; }
    memcpy( GlobalLock( hMem ), buf.begin(), buf.getSize() + 1 );
    GlobalUnlock( hMem );
    OpenClipboard( NULL );
    EmptyClipboard();
    SetClipboardData( CF_TEXT, hMem );
    CloseClipboard();
    releaseRenderLock();
    return HK_SUCCESS;
#else
    return HK_FAILURE;
#endif
}

void hkgVdbStatTextWidget::resetStats()
{
    m_activeNode = HK_NULL;
    m_activeNodeToRestore = HK_NULL;
    m_rootNodesToRestore.clear();
    m_statsTree.~Tree();
    new ( &m_statsTree ) hkMonitorStreamAnalyzer::Tree();
}

void hkgVdbStatTextWidget::writeStatText( hkArray<char>& bufOut ) const
{
    hkMonitorStreamAnalyzer::CombinedThreadSummaryOptions options;
    options.m_activeNode = m_activeNode;
    options.m_displayPartialTree = true;
    options.m_downArrowChar = HKG_FONT_BUTTON_DOWN;
    options.m_rightArrowChar = HKG_FONT_BUTTON_RIGHT;
    options.m_indentationToFirstTimerValue = -1;
    options.m_timerColumnWidth = 14;
    options.m_tabSpacingForTimerNames = 1;
    options.m_tabSpacingForTimerValues = 1;
    options.m_useTabsNotSpacesForColumns = false;

    hkOstream os( bufOut );
    if ( m_numThreads == 0 )
    {
        // If we've received no data, give us a reasonable look with some position holders for threads
        hkMonitorStreamAnalyzer::Tree dummyTree;
        hkLocalBuffer<hkMonitorStreamAnalyzer::Tree*> rootNodes( hkHardwareInfo::calcNumHardwareThreads() - 1 );
        hkMonitorStream::CommandStreamConfig config;
        hkMonitorStreamFrameInfo frameInfo;
        const char* startEndRef = "";
        for ( int i = 0; i < rootNodes.getSize(); i++ )
        {
            rootNodes[i] = hkMonitorStreamAnalyzer::makeStatisticsTreeForSingleFrame(
                config,
                startEndRef,
                startEndRef,
                frameInfo );

            hkMonitorStreamAnalyzer::mergeTreesForCombinedThreadSummary( &dummyTree, rootNodes[i], i, 0 );
        }

        hkMonitorStreamAnalyzer::showCombinedThreadSummaryForSingleFrame( &dummyTree, rootNodes.getSize(), os, options );

        for ( int i = 0; i < rootNodes.getSize(); i++ )
        {
            delete rootNodes[i];
        }
    }
    else
    {
        hkMonitorStreamAnalyzer::showCombinedThreadSummaryForSingleFrame( &m_statsTree, m_numThreads, os, options );
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
