// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdb2dWidget.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbStatsHandler.h>

#include <Graphics/Bridge/hkGraphicsBridge.h>
#include <Graphics/Bridge/SeriesGraph/hkgTimerGraph.h>

// This is usually not that interesting and is extra work.
//#define HK_VDB_ENABLE_AVERAGE_LINE_GRAPH
#ifdef HK_VDB_ENABLE_AVERAGE_LINE_GRAPH
#define HK_VDB_NUM_LINE_GRAPHS 3
#else
#define HK_VDB_NUM_LINE_GRAPHS 2
#endif

class hkgDisplayContext;

//////////////////////////////////////////////////////////////////////////
/// Options for the hkgVdbStatLineGraphWidget.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkgVdbStatLineGraphWidgetOptions : public hkgVdbDimensions
{
    HK_DECLARE_CLASS( hkgVdbStatLineGraphWidgetOptions, New );
    hkgVdbStatLineGraphWidgetOptions( hkgTimerGraph ( &timerGraphs )[HK_VDB_NUM_LINE_GRAPHS] );

    /// A format for how the line graph will be drawn.
    
    enum Format
    {
        //
        // Timer Name Matching
        // These determine how the timer names are interpreted/matched
        //

        /// Matches root-most timer(s) by name.
        FLAT = ( 1 << 0 ),

        /// Matches a specific timer by path using '/' as separators.
        HIERARCHICAL = ( 1 << 1 ),

        /// Mask for timer name matching options
        TIMER_NAME_FORMAT_MASK = ( 0x3 ),

        //
        // Timer Unit Display
        // These determine how the units are displayed and correspond to hkgSeriesGraph::Format
        //

        /// No decimal part
        INT_VALUES = ( 1 << 2 ),

        /// Thousands are converted to k, millions to M, etc
        SI_PREFIX = ( 1 << 3 ),

        /// 3 digits
        FLOAT_VALUES = ( 1 << 4 ),

        /// Mask for timer unit display options
        TIMER_UNIT_FORMAT_MASK = ( 0x1c ),

        //
        // Timer Rendering
        // These determine how the timers are rendered and correspond to hkgTimerGraph::Format.
        // Only one of these is possible at any time.
        //

        /// All threads are rendered
        THREADS = (1 << 5),

        /// Timers are summed across threads and displayed
        SUM = (1 << 6),

#ifdef HK_VDB_ENABLE_AVERAGE_LINE_GRAPH
        /// Timers are averaged across threads and displayed
        AVERAGE = (1 << 7),

        /// Mask for timer render options
        TIMER_RENDER_FORMAT_MASK = ( 0xe0 )
#else
        /// Mask for timer render options
        TIMER_RENDER_FORMAT_MASK = ( 0x60 )
#endif
    };

    /// Sets the range for the line graph.
    /// If min > max, range will be disabled.
    
    
    HK_INLINE void setRangeY( hkReal minY, hkReal maxY );

    /// Get the current format used to draw the line graph.
    HK_INLINE Format getFormat() const { return m_format; }

    /// Set the format and optionally units for drawing line graph.
    void setFormat( Format format, const char* unit = HK_NULL );

    /// Add a timer with the given color.
    HK_INLINE void addTimer( const char* timerName, hkColor::Argb color = hkColor::BLACK );

    /// Remove a timer.
    HK_INLINE void removeTimer( const char* timerName );

    /// Set a previously added timer's color.
    HK_INLINE void setTimerColor( const char* timerName, hkColor::Argb newColor );

    /// Returns true if the timer has been added.
    HK_INLINE bool doesTimerExist( const char* timerName );

protected:

    hkRefPtr<hkgDisplayContext> m_context;
    Format m_format;
    hkgTimerGraph ( &m_timerGraphs )[HK_VDB_NUM_LINE_GRAPHS];
    friend class hkgVdbStatLineGraphWidget;
};

//////////////////////////////////////////////////////////////////////////
/// A widget which draws a line graph for stats over time.
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkgVdbStatLineGraphWidget : public hkgVdb2dWidget
{
public:
    HK_DECLARE_CLASS( hkgVdbStatLineGraphWidget, New, ReflectIdentity );
    hkgVdbStatLineGraphWidget();

    //
    // 2d Widget interface
    //

    virtual hkResult initialize( hkgVdbPlugin& plugin, hkVdbClient& client ) HK_OVERRIDE;
    virtual hkResult deinitialize( hkgVdbPlugin& plugin, hkVdbClient& client ) HK_OVERRIDE;
    virtual hkResult update( const hkgViewport& viewport, const hkgInputManager& input ) HK_OVERRIDE;
    virtual void render( hkgDisplayContext& context ) const HK_OVERRIDE;
    virtual void getToolTip( hkStringBuf& toolTipOut, bool verbose ) const HK_OVERRIDE;
    virtual hkgVdbInputTypes getSupportedInputs() const HK_OVERRIDE { return hkgVdbInputTypes::MOUSE; }
    virtual hkgVdbBounds getBounds() const HK_OVERRIDE { return m_bounds; }

    //
    // Control
    //

    /// Access the current options.
    hkgVdbStatLineGraphWidgetOptions& accessOptions()
        { HK_ASSERT( 0x22441467, m_options.m_context, "Not safe to access before initialization"); return m_options; }

    //
    // Internal use
    //

    void onPerfStatsReceivedSignal( const hkVdbStatsHandler::PerfStats& perfStats, hkVdbSignalResult& result );

protected:

    virtual hkResult enable() HK_OVERRIDE;
    virtual hkResult disable() HK_OVERRIDE;

    void adjustForContext( hkgDisplayContext& context );

    hkRefPtr<hkVdbClient> m_client;
    mutable hkgTimerGraph m_timerGraphs[HK_VDB_NUM_LINE_GRAPHS];
    hkgVdbStatLineGraphWidgetOptions m_options;
    mutable hkgVdbBounds m_bounds;
};

HK_INLINE void hkgVdbStatLineGraphWidgetOptions::setRangeY( hkReal minY, hkReal maxY )
{
    if ( m_context ) m_context->lock();
    {
        for ( int i = 0; i < HK_COUNT_OF( m_timerGraphs ); i++ )
        {
            m_timerGraphs[i].setRangeY( minY, maxY );
        }
    }
    if ( m_context ) m_context->unlock();
}

HK_INLINE void hkgVdbStatLineGraphWidgetOptions::addTimer( const char* timerName, hkColor::Argb color )
{
    if ( m_context ) m_context->lock();
    {
        for ( int i = 0; i < HK_COUNT_OF( m_timerGraphs ); i++ )
        {
            m_timerGraphs[i].addSeries( timerName, color );
        }
    }
    if ( m_context ) m_context->unlock();
}

HK_INLINE void hkgVdbStatLineGraphWidgetOptions::removeTimer( const char* timerName )
{
    if ( m_context ) m_context->lock();
    {
        for ( int i = 0; i < HK_COUNT_OF( m_timerGraphs ); i++ )
        {
            m_timerGraphs[i].removeSeries( timerName );
        }
    }
    if ( m_context ) m_context->unlock();
}

HK_INLINE void hkgVdbStatLineGraphWidgetOptions::setTimerColor( const char* timerName, hkColor::Argb newColor )
{
    if ( m_context ) m_context->lock();
    {
        for ( int i = 0; i < HK_COUNT_OF( m_timerGraphs ); i++ )
        {
            m_timerGraphs[i].setSeriesColor( timerName, newColor );
        }
    }
    if ( m_context ) m_context->unlock();
}

HK_INLINE bool hkgVdbStatLineGraphWidgetOptions::doesTimerExist( const char* timerName )
{
    bool result;
    if ( m_context ) m_context->lock();
    {
        result = ( HK_COUNT_OF( m_timerGraphs ) > 0 ) ? m_timerGraphs[0].doesSeriesExist( timerName ) : false;
    }
    if ( m_context ) m_context->unlock();
    return result;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
