// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPlugin.h>
#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPluginApi.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbStatLineGraphWidget.h>

#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/Window/hkgWindow.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>

#define CONVERT_FORMAT_FROM( VAL, ENUM, TO_CLASS ) \
    ( bool( VAL & hkgVdbStatLineGraphWidgetOptions::ENUM ) * TO_CLASS::ENUM )

// Add more options to our format
HK_COMPILE_TIME_ASSERT( hkgSeriesGraph::NUM_FORMATS == 3 && hkgTimerGraph::NUM_FORMATS == 3 );

hkgVdbStatLineGraphWidgetOptions::hkgVdbStatLineGraphWidgetOptions(
    hkgTimerGraph( &timerGraphs )[HK_VDB_NUM_LINE_GRAPHS] ) :
    m_timerGraphs( timerGraphs )
{
    int format = HIERARCHICAL | FLOAT_VALUES;
    format |= ( hkgTimerGraph::DEFAULT_FORMAT == hkgTimerGraph::THREADS ) * THREADS;
    format |= ( hkgTimerGraph::DEFAULT_FORMAT == hkgTimerGraph::SUM ) * SUM;
#ifdef HK_VDB_ENABLE_AVERAGE_LINE_GRAPH
    format |= ( hkgTimerGraph::DEFAULT_FORMAT == hkgTimerGraph::AVERAGE ) * AVERAGE;
#else
    format |= ( hkgTimerGraph::DEFAULT_FORMAT == hkgTimerGraph::AVERAGE ) * SUM;
#endif
    m_format = Format( format );
    for ( int i = 0; i < HK_COUNT_OF( m_timerGraphs ); i++ )
    {
        m_timerGraphs[i].setFormat( hkgSeriesGraph::FLOAT_VALUES, "ms" );
    }
}

void hkgVdbStatLineGraphWidgetOptions::setFormat( Format format, const char* unit )
{
    if ( m_context ) m_context->lock();
    {
        HK_ASSERT(
            0x22441314,
            ( ( TIMER_NAME_FORMAT_MASK & format ) != TIMER_NAME_FORMAT_MASK ) &&
            ( ( TIMER_UNIT_FORMAT_MASK & format ) != TIMER_UNIT_FORMAT_MASK ) &&
            ( ( TIMER_RENDER_FORMAT_MASK & format ) != TIMER_RENDER_FORMAT_MASK ),
            "Setting mutually exclusive flags" );

        if ( TIMER_UNIT_FORMAT_MASK & format )
        {
            hkgSeriesGraph::Format sgFormat = hkgSeriesGraph::Format(
                CONVERT_FORMAT_FROM( format, INT_VALUES, hkgSeriesGraph ) +
                CONVERT_FORMAT_FROM( format, SI_PREFIX, hkgSeriesGraph ) +
                CONVERT_FORMAT_FROM( format, FLOAT_VALUES, hkgSeriesGraph ) );
            for ( int i = 0; i < HK_COUNT_OF( m_timerGraphs ); i++ ) m_timerGraphs[i].setFormat( sgFormat, unit );
        }

        m_format = format;
    }
    if ( m_context ) m_context->unlock();
}

hkgVdbStatLineGraphWidget::hkgVdbStatLineGraphWidget() :
    m_options( m_timerGraphs )
{
    m_options.m_heightFraction = 1 / 5.f;

#if 0
    // Test FLAT configuration, matches demo framework
    m_options.setFormat(
        hkgVdbStatLineGraphWidgetOptions::Format(
            hkgVdbStatLineGraphWidgetOptions::FLAT |
            hkgVdbStatLineGraphWidgetOptions::FLOAT_VALUES |
            hkgVdbStatLineGraphWidgetOptions::THREADS ),
        "ms" );
    m_options.addTimer( "Physics", hkColor::WHITE );
    m_options.addTimer( "BroadPhase" );
    m_options.addTimer( "NarrowPhase" );
    m_options.addTimer( "Solve" );
#endif
}

hkResult hkgVdbStatLineGraphWidget::initialize( hkgVdbPlugin& plugin, hkVdbClient& client )
{
    m_options.m_context = plugin.getWindow()->getContext();
    adjustForContext( *plugin.getWindow()->getContext() );
    m_client = &client;
    return HK_SUCCESS;
}

hkResult hkgVdbStatLineGraphWidget::deinitialize( hkgVdbPlugin& plugin, hkVdbClient& client )
{
    m_client = HK_NULL;
    m_options.m_context = HK_NULL;
    return HK_SUCCESS;
}

hkResult hkgVdbStatLineGraphWidget::update( const hkgViewport& viewport, const hkgInputManager& input )
{
    // Update based on shared resources
    acquireRenderLock();
    {
        adjustForContext( *viewport.getOwnerWindow()->getContext() );
    }
    releaseRenderLock();

    if ( getEnabledInputs().anyIsSet( hkgVdbInputTypes::MOUSE ) )
    {
        const hkgMouse& mouse = input.getMouse();

        // Compute a bump to the render format (we cycle through these)
        int formatBump =
            // Add 1 if we pushed any button
            ( mouse.wasButtonPressed( HKG_MOUSE_ANY_BUTTON ) ? 1 :
                // Otherwise check for mouse wheel clicks
                bool( m_bounds.isInside( hkgVdbPoint::windowCoordsFromInputCoords( mouse.getPosX(), mouse.getPosY(), viewport ) ) ) *
                mouse.getWheelNotchDelta() );

        // Apply the bump
        int renderFormat = ( m_options.m_format & hkgVdbStatLineGraphWidgetOptions::TIMER_RENDER_FORMAT_MASK );
        {
            // Clear
            m_options.m_format = hkgVdbStatLineGraphWidgetOptions::Format( m_options.m_format & ~hkgVdbStatLineGraphWidgetOptions::TIMER_RENDER_FORMAT_MASK );
            int wrapAroundVal;
            if ( formatBump >= 0 )
            {
                // Shift by bump
                renderFormat = ( renderFormat << formatBump );
                wrapAroundVal = ~( hkgVdbStatLineGraphWidgetOptions::TIMER_RENDER_FORMAT_MASK << 1 ) & hkgVdbStatLineGraphWidgetOptions::TIMER_RENDER_FORMAT_MASK;
            }
            else
            {
                // Shift by bump
                renderFormat = ( renderFormat >> -formatBump );
                wrapAroundVal = ~( hkgVdbStatLineGraphWidgetOptions::TIMER_RENDER_FORMAT_MASK >> 1 ) & hkgVdbStatLineGraphWidgetOptions::TIMER_RENDER_FORMAT_MASK;
            }
            // Handle wrap-around
            renderFormat |= !bool( renderFormat & hkgVdbStatLineGraphWidgetOptions::TIMER_RENDER_FORMAT_MASK ) * wrapAroundVal;
            // Set
            m_options.m_format = hkgVdbStatLineGraphWidgetOptions::Format( m_options.m_format | renderFormat );
        }
    }
    return HK_SUCCESS;
}

void hkgVdbStatLineGraphWidget::render( hkgDisplayContext& context ) const
{
    if ( !m_bounds.isEmpty() )
    {
        int y = ( context.getCurrentViewport()->getHeight() - m_bounds.m_maxY );

        if ( m_options.m_format & hkgVdbStatLineGraphWidgetOptions::THREADS )
        {
            m_timerGraphs[0].render( m_options.m_x, y, false );
        }
        else if ( m_options.m_format & hkgVdbStatLineGraphWidgetOptions::SUM )
        {
            m_timerGraphs[1].render( m_options.m_x, y, false );
        }
#ifdef HK_VDB_ENABLE_AVERAGE_LINE_GRAPH
        else if ( m_options.m_format & hkgVdbStatLineGraphWidgetOptions::AVERAGE )
        {
            m_timerGraphs[2].render( m_options.m_x, y, false );
        }
#endif
        else
        {
            HK_ASSERT_NOT_IMPLEMENTED( 0x22441315 );
        }
    }
}

void hkgVdbStatLineGraphWidget::getToolTip( hkStringBuf& toolTipOut, bool verbose ) const
{
    toolTipOut =

        ( verbose ) ?

        "Mouse Wheel : Cycle Display\n"
        "Click Any Mouse Button : Cycle Display" :

        "Wheel : Cycle Display\n"
        "Click MB : Cycle Display";
}

void hkgVdbStatLineGraphWidget::onPerfStatsReceivedSignal( const hkVdbStatsHandler::PerfStats& perfStats, hkVdbSignalResult& result )
{
    // Create a frameinfo for interpreting monitor stream in ms
    hkMonitorStreamFrameInfo frameInfo = perfStats.m_frameInfo;
    {
        //don't support another timer
        frameInfo.m_indexOfTimer1 = -1;
        //ensure we are in ms
        frameInfo.m_timerFactor0 = 1e3f / float( hkStopwatch::getTicksPerSecond() );
    }

    // Convert to hkTimerDatas
    
    hkArray<hkTimerData> timerDatas( perfStats.m_monitorStreams.getSize() );
    {
        for ( int i = 0; i < timerDatas.getSize(); ++i )
        {
            hkTimerData& timerData = timerDatas[i];
            timerData.m_streamBegin = perfStats.m_monitorStreams[i].begin();
            timerData.m_streamEnd = perfStats.m_monitorStreams[i].end();
        }
    }

    // Note: normally, we would store off data and update our internal graphs during ::update().
    // However, the streamBegin/Ends and statsTree are only valid during this callback, so we can't do that
    // unless we copied the entire data, which seems like overkill for a little perf back in the process tick.
    acquireRenderLock();
    {
        if ( m_options.m_format & hkgVdbStatLineGraphWidgetOptions::FLAT )
        {
            m_timerGraphs[0].update(
                frameInfo, timerDatas.begin(), timerDatas.getSize(), &perfStats.m_streamConfig,
                hkgTimerGraph::THREADS );
            m_timerGraphs[1].update(
                frameInfo, timerDatas.begin(), timerDatas.getSize(), &perfStats.m_streamConfig,
                hkgTimerGraph::SUM );
#ifdef HK_VDB_ENABLE_AVERAGE_LINE_GRAPH
            m_timerGraphs[2].update(
                frameInfo, timerDatas.begin(), timerDatas.getSize(), &perfStats.m_streamConfig,
                hkgTimerGraph::AVERAGE );
#endif
        }
        else
        {
            m_timerGraphs[0].update(
                &perfStats.m_statsTree,
                perfStats.m_threadRootNodes.getSize(),
                hkgTimerGraph::THREADS );
            m_timerGraphs[1].update(
                &perfStats.m_statsTree,
                perfStats.m_threadRootNodes.getSize(),
                hkgTimerGraph::SUM );
#ifdef HK_VDB_ENABLE_AVERAGE_LINE_GRAPH
            m_timerGraphs[2].update(
                &perfStats.m_statsTree,
                perfStats.m_threadRootNodes.getSize(),
                hkgTimerGraph::AVERAGE );
#endif
        }
    }
    releaseRenderLock();
}

hkResult hkgVdbStatLineGraphWidget::enable()
{
    HK_VDB_VERIFY_CONDITION( m_client, 0xadb00033, hkVdbError::PLUGIN_ERROR );
    HK_SUBSCRIBE_TO_SIGNAL( m_client->getCmdHandler<hkVdbStatsHandler>()->m_perfStatsReceived, this, hkgVdbStatLineGraphWidget );
    return HK_SUCCESS;
}

hkResult hkgVdbStatLineGraphWidget::disable()
{
    HK_VDB_VERIFY_CONDITION( m_client, 0xadb00034, hkVdbError::PLUGIN_ERROR );
    m_client->getCmdHandler<hkVdbStatsHandler>()->m_perfStatsReceived.unsubscribeAll( this );
    return HK_SUCCESS;
}

void hkgVdbStatLineGraphWidget::adjustForContext( hkgDisplayContext& context )
{
    // Update our bounds
    hkgViewport* v = context.getCurrentViewport();
    m_bounds.set( v, m_options );

    // Adjust our timer graphs based on bounds
    {
        int newW = m_bounds.getWidth();
        int newH = m_bounds.getHeight();
        for ( int i = 0; i < HK_COUNT_OF( m_timerGraphs ); i++ )
        {
            int currentW = m_timerGraphs[i].getDisplayWidth();
            int currentH = m_timerGraphs[i].getDisplayHeight();
            if ( ( currentW != newW ) || ( currentH != newH ) )
            {
                m_timerGraphs[i].setDisplayContext( &context, newW, newH );
            }
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
