// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPlugin.h>
#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPluginApi.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbStatBarGraphWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/Utils/hkgVdbTextBoxDrawer.h>

#include <Graphics/Common/Font/hkgFont.h>
#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/Texture/hkgTexture.h>

#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdb2dWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbScene2d.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>

#define TIMER_SELECTED_FLAG (hkMonitorStreamParser::Node::FLAGS_USER_0 | hkMonitorStreamParser::Node::FLAGS_USER_1)
#define TIMER_TREE_SELECTED_FLAG (hkMonitorStreamParser::Node::FLAGS_USER_0)

int hkgVdbStatBarGraphWidgetOptions::getHiddenTimers( hkArray<const char*>& hiddenTimersOut ) const
{
    const int prevCount = hiddenTimersOut.getSize();

    hiddenTimersOut.reserve( hiddenTimersOut.getSize() + m_hiddenTimers.getSize() );
    for ( hkHashSet<hkStringPtr>::Iterator iter = m_hiddenTimers.getIterator();
        m_hiddenTimers.isValid( iter );
        iter = m_hiddenTimers.getNext( iter ) )
    {
        hiddenTimersOut.pushBack( m_hiddenTimers.getElement( iter ) );
    }

    return ( hiddenTimersOut.getSize() - prevCount );
}

hkgVdbStatBarGraphWidget::hkgVdbStatBarGraphWidget()
{
    m_mouseDragState = 0x0;
    m_mouseCurrentPosition.set( 0, 0 );
    m_barGraph = hkRefNew<hkgVdbStatBarGraph>( new hkgVdbStatBarGraph() );
}

hkgVdbStatBarGraphWidget::~hkgVdbStatBarGraphWidget()
{
    for ( int b = 0; b < m_timerData.getNumBuffers(); b++ )
    {
        hkArray<hkMonitorStreamParser::Tree*>& timerData = m_timerData.accessBuffer( b );
        for ( int i = 0; i < timerData.getSize(); ++i )
        {
            delete timerData[i];
        }
        m_timerData.cycleBuffers();
    }
}

hkResult hkgVdbStatBarGraphWidget::initialize( hkgVdbPlugin& plugin, hkVdbClient& client )
{
    m_bounds.set( plugin.getWindow()->getContext()->getCurrentViewport(), m_options );
    m_dpiScaling = plugin.getWindow()->getDpiScale();
    m_defaultFont = hkgVdbTextBoxDrawer::loadDefaultFont( *plugin.getWindow()->getContext(), int( HKG_VDB_WIDGET_DEFAULT_TEXT_SIZE * m_dpiScaling ) );
    m_client = &client;
    return HK_SUCCESS;
}

hkResult hkgVdbStatBarGraphWidget::deinitialize( hkgVdbPlugin& plugin, hkVdbClient& client )
{
    m_client = HK_NULL;
    return HK_SUCCESS;
}

hkResult hkgVdbStatBarGraphWidget::update( const hkgViewport& viewport, const hkgInputManager& input )
{
    // Update based on shared resources
    acquireRenderLock();
        m_timerData.cycleBuffersIfWriteUpdated();
        const hkArray<hkMonitorStreamParser::Tree*>& timerData = m_timerData.getReadBuffer();
    releaseRenderLock();

    // Adjust our bar graph based on bounds computed from viewport(always in a normalized 0 -> w/h space)
    m_bounds.set( &viewport, m_options );
    hkVector2 boundsDimensions;
    {
        hkReal newW = hkReal( m_bounds.getWidth() );
        hkReal newH = hkReal( m_bounds.getHeight() );
        boundsDimensions.set( newW, newH );
        m_barGraph->init( hkRectF( 0, 0, newW, newH ) );
    }

    // Update the graph's timer quads with CPU performance values
    hkRectF previousRect = m_barGraph->getActualRect();
    m_barGraph->updateCpu( timerData, m_options.getSearchString() ? TIMER_TREE_SELECTED_FLAG : 0 );
    hkRectF currentRect = m_barGraph->getActualRect();

    // Recompute scene/bounds
    // Note: we set width so that we at least span the screen, the height clamp to what's needed by bar graph
    hkBool32 sceneUpdated = false;
    {
        hkRectF sceneRect(
            hkReal( m_bounds.m_minX ),
            hkReal( m_bounds.m_minY ),
            hkMath::max2( currentRect.getWidth(), m_bounds.getWidth() ),
            hkMath::max2( currentRect.getHeight(), hkgVdbStatBarGraph::PIXELS_PER_LEVEL + ( m_options.m_threadTitleMargin.m_y * 2 ) ) );
        if ( ( m_barGraphScene == HK_NULL )
            || !sceneRect.equals( m_barGraphScene->getBounds() ) )
        {
            updateScene( sceneRect );
            sceneUpdated = true;
        }
    }

    // Current and target transforms
    hkgOrthoTransform2d currentRootTransformLS = m_rootCanvasNode->getLocalTransform();
    hkgOrthoTransform2d targetRootTransformLS = currentRootTransformLS;
    hkgOrthoTransform2d currentTimelineTransformLS = m_timelineCanvasNode->getLocalTransform();
    hkgOrthoTransform2d targetTimelineTransformLS = currentTimelineTransformLS;
    hkRectF timelineViewportDimensionsGS;
    {
        timelineViewportDimensionsGS.setFromMinMax(
            m_timelineContainerNode->getGlobalTransform().m_translation.x,
            m_timelineContainerNode->getGlobalTransform().m_translation.y,
            boundsDimensions.x,
            boundsDimensions.y );
    }
    hkRectF timelineViewportDimensionsLS;
    {
        hkVector2 min; min.set( timelineViewportDimensionsGS.getX(), timelineViewportDimensionsGS.getY() );
        hkVector2 max; max.set( timelineViewportDimensionsGS.getX2(), timelineViewportDimensionsGS.getY2() );
        hkgOrthoTransform2d inverseTimelineTransform;
        inverseTimelineTransform.setInverse( m_timelineCanvasNode->getGlobalTransform() );
        hkgOrthoTransform2d::transform( min, inverseTimelineTransform, min );
        hkgOrthoTransform2d::transform( max, inverseTimelineTransform, max );
        timelineViewportDimensionsLS.setFromMinMax( min.x, min.y, max.x, max.y );
    }
    hkgOrthoTransform2d perfectTimelineTransformLS;
    {
        perfectTimelineTransformLS.setIdentity();
        perfectTimelineTransformLS.m_scale.set(
            timelineViewportDimensionsGS.getWidth()
            / m_timelineCanvasNode->getDimensions().x, 1 );
    }

    // Handle input
    const hkgMouse& mouse = input.getMouse();
    m_mousePreviousPosition = m_mouseCurrentPosition;
    m_mouseCurrentPosition.set( hkReal( mouse.getPosX() ), hkReal( mouse.getPosY() ) );
    hkVector2 mouseDeltaGS; mouseDeltaGS.setSub( m_mouseCurrentPosition, m_mousePreviousPosition );
    // skip screen wrap frame for jump > 2/3 window width
    if ( mouseDeltaGS.lengthSquared() > ( viewport.getOwnerWindow()->getWidth() * viewport.getOwnerWindow()->getWidth() * 0.44f ) ) return HK_SUCCESS;
    if ( getEnabledInputs().anyIsSet( hkgVdbInputTypes::MOUSE ) )
    {
        // Calculate mouse vectors
        hkVector2 mousePosTimelineLS;
        {
            // Calculate the mouse position in the timeline's frame of reference.
            hkgOrthoTransform2d inverseTimelineTransform;
            inverseTimelineTransform.setInverse( m_timelineCanvasNode->getGlobalTransform() );
            hkgOrthoTransform2d::transform( m_mouseCurrentPosition, inverseTimelineTransform, mousePosTimelineLS );

            // Clip the result into visible timers.
            hkVector2 min; min.set( timelineViewportDimensionsLS.getX(), timelineViewportDimensionsLS.getY() );
            hkVector2 max; max.set( timelineViewportDimensionsLS.getX2(), timelineViewportDimensionsLS.getY2() );
            mousePosTimelineLS.setMax( mousePosTimelineLS, min );
            mousePosTimelineLS.setMin( mousePosTimelineLS, max );
        }

        // The mouse wheel scales the graph centered on the mouse cursor
        int mouseScrollDelta = mouse.getWheelNotchDelta();
        if ( mouseScrollDelta != 0 )
        {
            hkVector2 scalePoint;
            if ( mousePosTimelineLS.x > m_timelineCanvasNode->getDimensions().x )
            {
                scalePoint.setZero();
            }
            else
            {
                scalePoint = mousePosTimelineLS;
            }

            hkReal scalingFactor = ( 1.0f + ( HKG_VDB_WHEEL_ZOOM_FACTOR * mouseScrollDelta ) );

            hkgOrthoTransform2d toOrigin;
            toOrigin.setIdentity();
            toOrigin.m_translation.set( -scalePoint.x, -scalePoint.y );

            hkgOrthoTransform2d deltaScale;
            deltaScale.setIdentity();
            deltaScale.m_scale.set( scalingFactor, 1 );

            hkgOrthoTransform2d inverseToOrigin;
            inverseToOrigin.setIdentity();
            inverseToOrigin.m_translation.set( scalePoint.x, scalePoint.y );

            // Transform to the origin, apply scale, then transform back
            targetTimelineTransformLS.setMul( deltaScale, toOrigin );
            hkgOrthoTransform2d::mul( inverseToOrigin, targetTimelineTransformLS, targetTimelineTransformLS );
            hkgOrthoTransform2d::mul( currentTimelineTransformLS, targetTimelineTransformLS, targetTimelineTransformLS );
        }

        // The middle mouse button strafes the timeline.
        if ( mouse.getButtonState() & HKG_MOUSE_MIDDLE_BUTTON )
        {
            if ( !( m_mouseDragState & HKG_MOUSE_MIDDLE_BUTTON ) )
            {
                m_mouseDragState |= HKG_MOUSE_MIDDLE_BUTTON;
            }

            targetRootTransformLS.m_translation.set(
                currentRootTransformLS.m_translation.x,
                currentRootTransformLS.m_translation.y - mouseDeltaGS.y );

            targetTimelineTransformLS.m_translation.set(
                currentTimelineTransformLS.m_translation.x + mouseDeltaGS.x,
                currentTimelineTransformLS.m_translation.y );
        }
        else if ( m_mouseDragState & HKG_MOUSE_MIDDLE_BUTTON )
        {
            m_mouseDragState &= ~HKG_MOUSE_MIDDLE_BUTTON;
        }

        // Left clicking collapses and expands individual threads
        if ( mouse.getButtonState() & HKG_MOUSE_LEFT_BUTTON )
        {
            if ( !( m_mouseDragState & HKG_MOUSE_LEFT_BUTTON ) )
            {
                m_mouseDragState |= HKG_MOUSE_LEFT_BUTTON;
            }
        }
        else if ( m_mouseDragState & HKG_MOUSE_LEFT_BUTTON )
        {
            for ( int ithThread = 0; ithThread < m_barGraph->getNumThreads(); ++ithThread )
            {
                hkRectF threadGlobalBound;
                m_threadNodes[ithThread]->getGlobalBounds( threadGlobalBound );
                if ( threadGlobalBound.contains( m_mouseCurrentPosition.x, m_mouseCurrentPosition.y ) )
                {
                    m_barGraph->accessThread( ithThread ).m_collapsed = !m_barGraph->getThread( ithThread ).m_collapsed;
                    break;
                }
            }

            m_mouseDragStart.set( 0, 0 );
            m_mouseDragState &= ~HKG_MOUSE_LEFT_BUTTON;
        }

        // Right click and drag draws/commits selection window, click resets
        if ( mouse.getButtonState() & HKG_MOUSE_RIGHT_BUTTON )
        {
            if ( !( m_mouseDragState & HKG_MOUSE_RIGHT_BUTTON ) )
            {
                m_mouseDragStart = mousePosTimelineLS;
                m_mouseDragState |= HKG_MOUSE_RIGHT_BUTTON;
            }
            hkRectF mouseSelectionBox; mouseSelectionBox.setInvalid();
            mouseSelectionBox.includePoint( m_mouseDragStart.x, m_mouseDragStart.y );
            mouseSelectionBox.includePoint( mousePosTimelineLS.x, mousePosTimelineLS.y );

            if ( mouseSelectionBox.isValid() )
            {
                m_selectionBoxNode->setLocalTranslation( mouseSelectionBox.getX(), 0 );
                m_selectionBoxNode->setDimensions( mouseSelectionBox.getWidth(), m_timelineCanvasNode->getDimensions().y );
            }
            else
            {
                m_selectionBoxNode->setDimensions( 0, 0 );
            }
        }
    }

    // Confirm any mouse selection
    if (// If we've released the RMB while over the timeline *or* someone else has nabbed the mouse input.
        ( !( mouse.getButtonState() & HKG_MOUSE_RIGHT_BUTTON ) || getEnabledInputs().noneIsSet( hkgVdbInputTypes::MOUSE ) )
        // And we were dragging the RMB button
        && ( m_mouseDragState & HKG_MOUSE_RIGHT_BUTTON ) )
    {
        hkRectF selectionBoundsLS; m_selectionBoxNode->getLocalBounds( selectionBoundsLS );

        // If no region was selected, reset all transforms
        // We consider a selection less than 3px to be no selection.
        if ( ( selectionBoundsLS.getWidth() * currentTimelineTransformLS.m_scale.x ) < 3 )
        {
            hkgOrthoTransform2d identity; identity.setIdentity();
            targetRootTransformLS = identity;
            targetTimelineTransformLS = perfectTimelineTransformLS;
        }
        // Otherwise, zoom to the selected region
        else
        {
            // Translate to put min selected position at origin, and scale to put position max position at the far edge
            targetTimelineTransformLS.m_scale.set(
                ( timelineViewportDimensionsLS.getWidth() * currentTimelineTransformLS.m_scale.x )
                / selectionBoundsLS.getWidth(), 1 );
            targetTimelineTransformLS.m_translation.set( -selectionBoundsLS.getX() * targetTimelineTransformLS.m_scale.x, 0 );
        }
        m_mouseDragStart.set( 0, 0 );
        m_mouseDragState &= ~HKG_MOUSE_RIGHT_BUTTON;
        m_selectionBoxNode->setDimensions( 0, 0 );
    }

    // Apply transforms
    
    {
        // Anchor left/top
        {
            targetRootTransformLS.m_translation.y = hkMath::min2( 0.0f, targetRootTransformLS.m_translation.y );
            targetTimelineTransformLS.m_translation.x = hkMath::min2( 0.0f, targetTimelineTransformLS.m_translation.x );
        }

#if 0
        // Auto zoom only if the user "reset"
        if ( targetTimelineTransformLS )
        {
            m_options.m_autoZoomRate
            targetTimelineTransformLS.m_translation.setInterpolate(
                targetTimelineTransformLS.m_translation,
                perfectTimelineTransformLS.m_translation,
                )
        }
#endif

        // Clamp
        {
            // Max width
            hkFloat32 maxTimelineViewportWidth =
                // Abribrary number of ms
                // An alternative would be to compute/retain the "max" timestep from the circ buffer
                
                
                
                ( hkgVdbStatBarGraph::PIXELS_PER_MILLISECOND * HKG_VDB_DEFAULT_TIMER_MS_VISIBLE );

            // Note: y scale isn't touched atm, should stay the same
            targetRootTransformLS.m_scale.y = currentRootTransformLS.m_scale.y;

            // x scale is clamped such that we don't expose *more* of the timeline than is needed
            // Note: it matters this is applied before translation calculation
            if ( targetTimelineTransformLS.m_scale.x != currentTimelineTransformLS.m_scale.x )
            {
                hkFloat32 maxScaleWidth =
                    timelineViewportDimensionsGS.getWidth()
                    / maxTimelineViewportWidth;

                targetTimelineTransformLS.m_scale.x =
                    hkMath::max2(
                        targetTimelineTransformLS.m_scale.x,
                        maxScaleWidth );
            }

            // y translation is clamped so that at least one thread is showing
            if ( ( targetRootTransformLS.m_translation.y != currentRootTransformLS.m_translation.y ) || sceneUpdated )
            {
                hkReal threadsThroughNMinusOneHeight = 0;
                for ( int i = 0; i < m_threadNodes.getSize() - 1; i++ ) threadsThroughNMinusOneHeight += m_threadNodes[i]->getDimensions().y; // scale? not used atm
                threadsThroughNMinusOneHeight = hkMath::max2( threadsThroughNMinusOneHeight, hkgVdbStatBarGraph::PIXELS_PER_LEVEL ); // always show at least one row
                targetRootTransformLS.m_translation.y = hkMath::max2( -threadsThroughNMinusOneHeight, targetRootTransformLS.m_translation.y );
            }

            // x translation is clamped such that we don't expose *more* of the timeline than is needed
            if ( targetTimelineTransformLS.m_translation.x != currentTimelineTransformLS.m_translation.x )
            {
                if ( targetTimelineTransformLS.m_scale.x > perfectTimelineTransformLS.m_scale.x )
                {
                    hkFloat32 targetTimelineViewportDimensionsLS_width = timelineViewportDimensionsLS.getWidth();
                    targetTimelineViewportDimensionsLS_width *= currentTimelineTransformLS.m_scale.x;
                    targetTimelineViewportDimensionsLS_width /= targetTimelineTransformLS.m_scale.x;
                    targetTimelineTransformLS.m_translation.x =
                        hkMath::max2(
                            targetTimelineTransformLS.m_translation.x,
                            -( m_timelineCanvasNode->getDimensions().x - targetTimelineViewportDimensionsLS_width )
                            * targetTimelineTransformLS.m_scale.x );
                }
                else
                {
                    targetTimelineTransformLS.m_translation.x = 0;
                }
            }
        }

        // Apply to canvases
        {
            m_rootCanvasNode->setLocalTransform( targetRootTransformLS );
            m_timelineCanvasNode->setLocalTransform( targetTimelineTransformLS );
        }

        // Update our bounds
        {
            m_bounds.m_minY = hkInt32( m_barGraphScene->getBounds().getY() );
            // Don't include "empty" space at the end which we aren't drawing to
            m_bounds.m_maxY = hkInt32( m_bounds.m_minY + ( m_rootCanvasNode->getDimensions().y + targetRootTransformLS.m_translation.y ) );
        }
    }

    return HK_SUCCESS;
}

void hkgVdbStatBarGraphWidget::render( hkgDisplayContext& context ) const
{
    if ( !m_bounds.isEmpty() )
    {
        hkgViewport* currentView = context.getCurrentViewport();
        if ( ( m_barGraphScene != HK_NULL ) && ( m_barGraphNode != HK_NULL ) )
        {
            // Render the scene
            m_barGraphScene->render( context );

            hkRectF widgetBounds;
            m_barGraphScene->getRoot()->getGlobalBounds( widgetBounds );

            hkRectF timelineBounds;
            m_timelineContainerNode->getGlobalBounds( timelineBounds );

            // Draw timer data on mouse hover
            if ( widgetBounds.containsPoint( m_mouseCurrentPosition.x, m_mouseCurrentPosition.y )
                && timelineBounds.containsPoint( m_mouseCurrentPosition.x, m_mouseCurrentPosition.y ) )
            {
                // Enable GPU state for text rendering
                context.setTexture2DState( true );
                {
                    hkColor::Argb textColorArgb = m_options.m_textColor;
                    float textColor[4] = { hkColor::getRedAsFloat( textColorArgb ), hkColor::getGreenAsFloat( textColorArgb ), hkColor::getBlueAsFloat( textColorArgb ), 1.0f };

                    hkgOrthoTransform2d timerTransform = m_barGraphNode->getGlobalTransform();

                    hkgOrthoTransform2d inverseTimerTransform;
                    inverseTimerTransform.setInverse( timerTransform );

                    hkVector2 mousePositionGraph;
                    mousePositionGraph.set( m_mouseCurrentPosition.x, m_mouseCurrentPosition.y );
                    hkgOrthoTransform2d::transform( mousePositionGraph, inverseTimerTransform, mousePositionGraph );

                    const hkgVdbStatBarGraph::Quad* currentQuad = m_barGraph->getQuadUnderPos( mousePositionGraph.x, mousePositionGraph.y );
                    if ( currentQuad != HK_NULL )
                    {
                        hkStringBuf buf;
                        if ( currentQuad->m_srcNode != HK_NULL )
                        {
                            buf.printf(
                                "%s/%s:%.3fms",
                                currentQuad->m_srcNode->m_parent ? currentQuad->m_srcNode->m_parent->m_name : "",
                                currentQuad->m_srcNode->m_name,
                                currentQuad->m_time );
                        }
                        else
                        {
                            buf.printf(
                                "%s:%.3fms",
                                currentQuad->m_timerName.cString(),
                                currentQuad->m_time );
                        }

                        m_defaultFont->render( &context, buf, m_mouseCurrentPosition.x, m_mouseCurrentPosition.y, textColor );
                    }
                }
            }
        }

        // Restore viewport
        currentView->setAsCurrent( &context );
    }
}

void hkgVdbStatBarGraphWidget::getToolTip( hkStringBuf& toolTipOut, bool verbose ) const
{
    if ( verbose )
    {
        toolTipOut.format(
            "Click Right Mouse Button : Restore default view\n"
            "Drag Right Mouse Button : Drag to zoom on selected region\n"
            "Drag Middle Mouse Button : Strafe timeline\n"
            "Mouse Wheel : Zoom In / Out\n"
            "Click Left Mouse Button : Expand/Collapse thread timelines\n" );
    }
    else
    {
        toolTipOut.format(
            "Click RMB : Default View\n"
            "Drag RMB : Zoom to Selected\n"
            "Drag MMB : Strafe\n"
            "Wheel : Zoom\n"
            "Click LMB : Collapse threads\n" );
    }
}

hkResult hkgVdbStatBarGraphWidget::enable()
{
    HK_VDB_VERIFY_CONDITION( m_client, 0xadb00041, hkVdbError::PLUGIN_ERROR );
    HK_SUBSCRIBE_TO_SIGNAL( m_client->getCmdHandler<hkVdbStatsHandler>()->m_perfStatsReceived, this, hkgVdbStatBarGraphWidget );
    return HK_SUCCESS;
}

hkResult hkgVdbStatBarGraphWidget::disable()
{
    HK_VDB_VERIFY_CONDITION( m_client, 0xadb00042, hkVdbError::PLUGIN_ERROR );
    m_client->getCmdHandler<hkVdbStatsHandler>()->m_perfStatsReceived.unsubscribeAll( this );
    return HK_SUCCESS;
}

void hkgVdbStatBarGraphWidget::onPerfStatsReceivedSignal( const hkVdbStatsHandler::PerfStats& perfStats, hkVdbSignalResult& result )
{
    
    
    
    
    // Get timer factor
    hkReal timerFactor = perfStats.m_frameInfo.m_timerFactor0;

    // Build up new timer trees
    
    hkArray<hkMonitorStreamParser::Tree*> newTimerData( perfStats.m_monitorStreams.getSize() );
    {
        hkArray<const char*> hiddenTimers;
        m_options.getHiddenTimers( hiddenTimers );

        for ( int i = 0; i < newTimerData.getSize(); ++i )
        {
            hkTimerData timerData;
            timerData.m_streamBegin = perfStats.m_monitorStreams[i].begin();
            timerData.m_streamEnd = perfStats.m_monitorStreams[i].end();

            newTimerData[i] = hkMonitorStreamParser::makeTree(
                perfStats.m_streamConfig,
                timerData,
                timerFactor,
                "/",
                false /* no reuse */,
                false /*do draw calls*/,
                hiddenTimers );

            if ( m_options.getSearchString() )
            {
                hkMonitorStreamParserUtil::setFlagOnNodesByName(
                    newTimerData[i],
                    m_options.getSearchString(),
                    TIMER_SELECTED_FLAG,
                    TIMER_TREE_SELECTED_FLAG );
            }
        }
    }

    acquireRenderLock();
    {
        // Swap old data for new and then cleanup the old data
        m_timerData.accessWriteBuffer().swap( newTimerData );
        for ( int i = 0; i < newTimerData.getSize(); ++i )
        {
            delete newTimerData[i];
        }
    }
    releaseRenderLock();
}

void hkgVdbStatBarGraphWidget::updateScene( const hkRectF& bounds )
{
    // Get transforms to restore if needed.
    hkgOrthoTransform2d rootCanvasTransform; rootCanvasTransform.setIdentity();
    if ( m_rootCanvasNode ) { rootCanvasTransform = m_rootCanvasNode->getLocalTransform(); }
    hkgOrthoTransform2d timelineTransform; timelineTransform.setIdentity();
    if ( m_timelineCanvasNode ) { timelineTransform = m_timelineCanvasNode->getLocalTransform(); }

    // Create scene
    m_barGraphScene = hkRefNew<hkgVdbScene2d>( new hkgVdbScene2d( bounds ) );
    hkRefPtr<hkgVdbNode2d> root = m_barGraphScene->getRoot();

    // The root's bounds provide a window into the (much larger) root canvas, which contains the scene.
    m_rootCanvasNode = hkRefNew<hkgVdbNode2d>( new hkgVdbNode2d( "RootCanvas" ) );
    {
        m_rootCanvasNode->setLocalTransform( rootCanvasTransform );
        m_rootCanvasNode->setDimensions( root->getDimensions().x, root->getDimensions().y );
        root->attachChild( m_rootCanvasNode );

        // Add per-thread horizontal boxes
        hkReal y = 0;
        hkReal threadLabelColumnWidth = 0;
        int availableThreads = m_barGraph->getNumThreads();
        int numThreadsToDraw = hkMath::max2( availableThreads, 1 );
        m_threadNodes.setSize( numThreadsToDraw, HK_NULL );
        m_threadLabelBackgrounds.setSize( numThreadsToDraw, HK_NULL );
        m_threadLabels.setSize( numThreadsToDraw, HK_NULL );
        for ( int ithThread = 0; ithThread < numThreadsToDraw; ++ithThread )
        {
            // Compute bounds from text
            hkStringBuf buf;
            buf.printf( "Thread %i", ithThread );
            hkgVdbBounds titleTextBounds;
            titleTextBounds.set( *m_defaultFont, buf );

            // Determine the height (width is set to the whole root)
            
            // We'd have to support this in the drawing of the thread quads to support a vertical margin
            hkReal yMargin = 0; // m_options.m_threadTitleMargin.m_y
            hkReal height = //hkMath::max2(
                ( ithThread < availableThreads ) ? m_barGraph->getThread( ithThread ).m_height : hkgVdbStatBarGraph::PIXELS_PER_LEVEL * 2;// ,
                // titleTextBounds.getHeight() + ( 2 * yMargin ) );

            // Create a scene node representing the entire thread (no color, so not drawn, just for bounds checks).
            hkRefPtr<hkgVdbNode2d> threadNode = hkRefNew<hkgVdbNode2d>( new hkgVdbNode2d( "ThreadBounds" ) );
            m_threadNodes[ithThread] = threadNode;
            {
                threadNode->setColor( 0 );
                threadNode->setLocalTranslation( 0, y );
                threadNode->setDimensions( root->getDimensions().x, height );
                m_rootCanvasNode->attachChild( threadNode );
            }

            // Add a label background
            hkRefPtr<hkgVdbNode2d> threadLableBackgroundNode = hkRefNew<hkgVdbNode2d>( new hkgVdbNode2d( "ThreadLabelBackground" ) );
            m_threadLabelBackgrounds[ithThread] = threadLableBackgroundNode;
            {
                threadLableBackgroundNode->setColor( m_options.m_threadTitleColumnColor );
                threadLableBackgroundNode->setDimensions( 0, height ); // width adjusted after this loop
                threadNode->attachChild( threadLableBackgroundNode );
            }

            // Add a label
            hkRefPtr<hkgVdbTextNode2d> threadLabelNode = hkRefNew<hkgVdbTextNode2d>( new hkgVdbTextNode2d( buf.cString(), m_defaultFont ) );
            m_threadLabels[ithThread] = threadLabelNode;
            {
                threadLabelNode->setColor( m_options.m_textColor );
                threadLabelNode->setLocalTranslation(
                    m_options.m_threadTitleMargin.m_x,
                    yMargin );
                threadLabelNode->setDimensions(
                    hkReal( titleTextBounds.getWidth() + m_options.m_threadTitleMargin.m_x ),
                    height - ( 2 * yMargin ) );
                threadLableBackgroundNode->attachChild( threadLabelNode );
                threadLabelColumnWidth = hkMath::max2( threadLabelColumnWidth, threadLabelNode->getDimensions().x );
            }

            // Inc total y
            y += threadNode->getDimensions().y;
        }

        // Apply column width to the threadNodes
        for ( int ithThread = 0; ithThread < numThreadsToDraw; ++ithThread )
        {
            m_threadLabelBackgrounds[ithThread]->setDimensions(
                threadLabelColumnWidth,
                m_threadLabelBackgrounds[ithThread]->getDimensions().y );
        }

        // The timer container defines a window into the (much larger) timer canvas, which contains the actual timeline data.
        m_timelineContainerNode = hkRefNew<hkgVdbNode2d>( new hkgVdbNode2d( "TimerContainer" ) );
        {
            m_timelineContainerNode->setDimensions( root->getDimensions().x - threadLabelColumnWidth, root->getDimensions().y );
            m_timelineContainerNode->setLocalTranslation( threadLabelColumnWidth, 0 );
            m_rootCanvasNode->attachChild( m_timelineContainerNode );
        }

        // A large canvas whose view is controlled by the timeline container above
        m_timelineCanvasNode = hkRefNew<hkgVdbNode2d>( new hkgVdbNode2d( "TimelineCanvas" ) );
        {
            m_timelineCanvasNode->setLocalTransform( timelineTransform );
            m_timelineCanvasNode->setDimensions( m_timelineContainerNode->getDimensions().x, m_timelineContainerNode->getDimensions().y );
            m_timelineContainerNode->attachChild( m_timelineCanvasNode );

            // Vertical strips that define millisecond boundaries and boundary labels
            hkReal x = 0;
            hkInt32 bands = hkInt32(
                hkMath::ceil(
                    hkMath::max2( m_barGraph->getActualRect().getWidth(), m_barGraph->getGraphRect().getWidth() ) / hkReal( hkgVdbStatBarGraph::PIXELS_PER_MILLISECOND ) ) );
            for ( int i = 0; i < bands; ++i )
            {
                // Band
                hkRefPtr<hkgVdbNode2d> ithMillisecondNode;
                ithMillisecondNode = hkRefNew<hkgVdbNode2d>( new hkgVdbNode2d( "MillisecondBounds" ) );
                ithMillisecondNode->setColor( ( i % 2 ) ? m_options.m_oddMsColor : m_options.m_evenMsColor );
                ithMillisecondNode->setDimensions( hkReal( hkgVdbStatBarGraph::PIXELS_PER_MILLISECOND ), m_timelineCanvasNode->getDimensions().y );
                ithMillisecondNode->setLocalTranslation( x, 0 );
                m_timelineCanvasNode->attachChild( ithMillisecondNode );

                // Label
                hkStringBuf buf;
                buf.printf( "%i", i + 1 );
                hkgVdbBounds textBounds;
                textBounds.set( *m_defaultFont, buf );
                hkRefPtr<hkgVdbTextNode2d> ithMillisecondNodeLabel =
                    hkRefNew<hkgVdbTextNode2d>( new hkgVdbTextNode2d( buf.cString(), m_defaultFont ) );
                ithMillisecondNodeLabel->setColor( m_options.m_textColor );
                ithMillisecondNodeLabel->setDimensions(
                    hkReal( textBounds.getWidth() ) + 2,
                    hkReal( textBounds.getHeight() ) + 2 );
                ithMillisecondNodeLabel->setLocalTranslation( x, 0 );
                m_timelineCanvasNode->attachChild( ithMillisecondNodeLabel );

                x += ithMillisecondNode->getDimensions().x;
            }

            // Add the node which renders the bar graph timers
            m_barGraphNode = hkRefNew<hkgVdbStatBarGraphNode2d>( new hkgVdbStatBarGraphNode2d( m_barGraph ) );
            {
                m_barGraphNode->setDimensions( m_timelineCanvasNode->getDimensions().x, m_timelineCanvasNode->getDimensions().y );
                m_timelineCanvasNode->attachChild( m_barGraphNode );
            }

            // Add a selection box, which draws both the cursor and the timeline box selection highlight
            m_selectionBoxNode = hkRefNew<hkgVdbNode2d>( new hkgVdbNode2d( "SelectionBox" ) );
            {
                // Highlight with a saturated, transparent pink
                m_selectionBoxNode->setColor( m_options.m_selectionColor );
                m_timelineCanvasNode->attachChild( m_selectionBoxNode );
            }
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
