// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbScene2d.h>

#include <Graphics/Common/Font/hkgFont.h>
#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/Texture/hkgTexture.h>

hkgVdbScene2d::hkgVdbScene2d(const hkRectF& inBounds)
{
    m_bounds = inBounds;
    m_root.setAndDontIncrementRefCount(new hkgVdbNode2d("root"));
    m_root->setDimensions(getBounds().getWidth(), getBounds().getHeight());
}

void hkgVdbScene2d::render(hkgDisplayContext& context)
{
    hkgWindow* window = context.getOwner();
    hkgViewport* orthoView = window->getWindowOrthoView();

    hkgOrthoTransform2d viewportTransform;
    viewportTransform.m_scale.set(1, -1);
    viewportTransform.m_translation.set(0, hkReal(context.getCurrentViewport()->getHeight()));

    hkgOrthoTransform2d boundsTransform;
    boundsTransform.setIdentity();
    boundsTransform.m_translation.set(getBounds().getX(), getBounds().getY());

    hkgOrthoTransform2d rootTransform;
    rootTransform.setMul(viewportTransform, boundsTransform);
    m_root->setDimensions(getBounds().getWidth(), getBounds().getHeight());
    m_root->setLocalTransform(rootTransform);

    // Set graphics state
    {
        orthoView->setAsCurrent(&context);
        context.setBlendState(true);
        context.setBlendMode(HKG_BLEND_MODULATE_OPAQUE_DEST);
        context.setDepthReadState(false);
        context.setDepthWriteState(false);
        context.setCullFaceState(false);
        context.setLightingState(false);
        context.setCurrentSoleTexture(HK_NULL, HKG_TEXTURE_MODULATE); // for 360, incase we have to modify the font texture for extra chars
        context.setTexture2DState(false);
    }

    // Build and render a triangle list
    context.beginGroup(HKG_IMM_TRIANGLE_LIST);
    {
        if (m_root != HK_NULL)
        {
            hkRectF rootBounds;
            m_root->getGlobalBounds(rootBounds);
            m_root->render(context, rootBounds);
        }
    }
    context.endGroup();
}

hkgVdbNode2d::hkgVdbNode2d(const char* inName)
{
    m_parent = HK_NULL;
    m_name = inName;
    m_dimensions.setZero();
    setColor(hkColor::NONE);

    m_localTransform.setIdentity();
    m_globalTransform.setIdentity();
    m_globalTransformDirty = false;
}

void hkgVdbNode2d::setLocalTransform(const hkgOrthoTransform2d& value)
{
    if (value.m_translation.x != m_localTransform.m_translation.x
     || value.m_translation.y != m_localTransform.m_translation.y
     || value.m_scale.x != m_localTransform.m_scale.x
     || value.m_scale.y != m_localTransform.m_scale.y)
    {
        m_localTransform = value;
        markGlobalTransformDirty();
    }
}

void hkgVdbNode2d::setLocalTranslation(hkReal x, hkReal y)
{
    if (x != m_localTransform.m_translation.x
     || y != m_localTransform.m_translation.y)
    {
        m_localTransform.m_translation.set(x, y);
        markGlobalTransformDirty();
    }
}

void hkgVdbNode2d::setLocalScale(hkReal xScale, hkReal yScale)
{
    if (xScale != m_localTransform.m_scale.x
     || yScale != m_localTransform.m_scale.y)
    {
        m_localTransform.m_scale.set(xScale, yScale);
        markGlobalTransformDirty();
    }
}

const hkgOrthoTransform2d& hkgVdbNode2d::getGlobalTransform() const
{
    if (m_globalTransformDirty)
    {
        if (m_parent == HK_NULL)
        {
            m_globalTransform = m_localTransform;
        }
        else
        {
            m_globalTransform.setMul(m_parent->getGlobalTransform(), m_localTransform);
        }
        m_globalTransformDirty = false;
    }
    return m_globalTransform;
}

void hkgVdbNode2d::getLocalBounds(hkRectF& result) const
{
    const hkgOrthoTransform2d& localTransform = getLocalTransform();
    result.setFromPositionSize(0, 0, m_dimensions.x, m_dimensions.y);
    result.transform(localTransform.m_scale.x, localTransform.m_scale.y, localTransform.m_translation.x, localTransform.m_translation.y);
}

void hkgVdbNode2d::getGlobalBounds(hkRectF& result) const
{
    const hkgOrthoTransform2d& globalTransform = getGlobalTransform();
    result.setFromPositionSize(0, 0, m_dimensions.x, m_dimensions.y);
    result.transform(globalTransform.m_scale.x, globalTransform.m_scale.y, globalTransform.m_translation.x, globalTransform.m_translation.y);
}

void hkgVdbNode2d::attachChild(hkgVdbNode2d* child)
{
    if (child->getParent() != HK_NULL)
    {
        child->getParent()->detachChild(child);
    }

    child->markGlobalTransformDirty();
    child->m_parent = this;

    m_children.pushBack(child);
}

void hkgVdbNode2d::detachChild(hkgVdbNode2d* child)
{
    for (int i = 0; i < m_children.getSize(); ++i)
    {
        if (m_children[i] == child)
        {
            m_children.removeAt(i);
            child->m_parent = HK_NULL;
            child->markGlobalTransformDirty();
            return;
        }
    }
}

void hkgVdbNode2d::render(hkgDisplayContext& context, const hkRectF& parentBounds)
{
    hkRectF globalBounds;
    getGlobalBounds(globalBounds);

    if (globalBounds.clip(parentBounds))
    {
        renderInternal(context, globalBounds);
        for (int i = 0; i < m_children.getSize(); ++i)
        {
            m_children[i]->render(context, globalBounds);
        }
    }
}

void hkgVdbNode2d::markGlobalTransformDirty()
{
    if (!m_globalTransformDirty)
    {
        m_globalTransformDirty = true;
        for (int i = 0; i < m_children.getSize(); ++i)
        {
            m_children[i]->markGlobalTransformDirty();
        }
    }
}

void hkgVdbNode2d::renderInternal(hkgDisplayContext& context, const hkRectF& rectangle)
{
    if (hkColor::getAlphaAsChar(m_color) != 0)
    {
        renderSolidRectangle(context, rectangle, m_color, -0.01f);
    }
}

void hkgVdbNode2d::renderSolidRectangle(hkgDisplayContext& context, const hkRectF& rectangle, hkColor::Argb color, float depth)
{
        hkColor::Argb darkColor = hkColor::darken(color);
        float fc[4] =
        {
            hkColor::getRedAsFloat(color),
            hkColor::getGreenAsFloat(color),
            hkColor::getBlueAsFloat(color),
            hkColor::getAlphaAsFloat(color)
        };
        float fdc[4] =
        {
            hkColor::getRedAsFloat(darkColor),
            hkColor::getGreenAsFloat(darkColor),
            hkColor::getBlueAsFloat(darkColor),
            hkColor::getAlphaAsFloat(darkColor)
        };

        float p[3];
        p[2] = depth;

        p[0] = rectangle.getX(); p[1] = rectangle.getY();
        context.setCurrentColor4(fdc);
        context.setCurrentPosition(p);
        p[0] = rectangle.getX(); p[1] = rectangle.getY2();
        context.setCurrentColor4(fc);
        context.setCurrentPosition(p);
        p[0] = rectangle.getX2(); p[1] = rectangle.getY2();
        context.setCurrentColor4(fc);
        context.setCurrentPosition(p);

        p[0] = rectangle.getX2(); p[1] = rectangle.getY2();
        context.setCurrentColor4(fc);
        context.setCurrentPosition(p);
        p[0] = rectangle.getX2(); p[1] = rectangle.getY();
        context.setCurrentColor4(fdc);
        context.setCurrentPosition(p);
        p[0] = rectangle.getX(); p[1] = rectangle.getY();
        context.setCurrentColor4(fdc);
        context.setCurrentPosition(p);
}

hkgVdbTextNode2d::hkgVdbTextNode2d(const char* inText, hkgFont* inFont)
    : hkgVdbNode2d(inText)
{
    m_font = inFont;
}

void hkgVdbTextNode2d::renderInternal(hkgDisplayContext& context, const hkRectF& clipBounds)
{
    if (clipBounds.containsPoint(getGlobalTransform().m_translation.x, getGlobalTransform().m_translation.y))
    {
        context.setTexture2DState(true);
        float colorF[4] =
        {
            hkColor::getRedAsFloat(m_color),
            hkColor::getGreenAsFloat(m_color),
            hkColor::getBlueAsFloat(m_color),
            hkColor::getAlphaAsFloat(m_color)
        };

        m_font->render(&context, m_name, getGlobalTransform().m_translation.x, getGlobalTransform().m_translation.y - m_font->getCharHeight(), colorF);
        context.setTexture2DState(false);
    }
}

void hkgVdbStatBarGraphNode2d::renderInternal(hkgDisplayContext& context, const hkRectF& clipBounds)
{
    hkgOrthoTransform2d globalTransform = getGlobalTransform();

    // Draw timer data
    for (int ithThread = 0; ithThread < m_barGraph->getNumThreads(); ++ithThread)
    {
        const hkArray<hkgVdbStatBarGraph::Quad>& quads = m_barGraph->getThread(ithThread).m_quads;
        for (int jthQuad = 0; jthQuad < quads.getSize(); ++jthQuad)
        {
            const hkgVdbStatBarGraph::Quad& quad = quads[jthQuad];

            hkRectF rectangle(quad.m_rect.getX(), quad.m_rect.getY(), hkReal(quad.m_rect.getWidth()), hkReal(quad.m_rect.getHeight()));
            rectangle.transform(globalTransform.m_scale.x, globalTransform.m_scale.y, globalTransform.m_translation.x, globalTransform.m_translation.y);
            if (rectangle.clip(clipBounds))
            {
                renderSolidRectangle(context, rectangle, quad.m_color, -0.01f);
            }
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
