// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPlugin.h>
#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPluginApi.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbDeprecatedStatBarGraphWidget.h>

#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/Font/hkgFont.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>

hkgVdbDeprecatedStatBarGraphWidget::hkgVdbDeprecatedStatBarGraphWidget() :
    m_timerNodeUnderMouse( HK_NULL ),
    m_defaultFont( hkRefNew<hkgFont>( hkgFont::create() ) ),
    m_options( *m_defaultFont ),
    m_state( NONE )
{
    m_zoomExtentsStart.m_x = HK_INT32_MAX;
    m_lastZooming.m_x = HK_INT32_MAX;
    m_lastPanning.m_x = HK_INT32_MAX;
    m_options.m_heightFraction = 1 / 5.f;
}

hkgVdbDeprecatedStatBarGraphWidget::~hkgVdbDeprecatedStatBarGraphWidget()
{
    for ( int b = 0; b < m_timerData.getNumBuffers(); b++ )
    {
        hkArray<hkMonitorStreamAnalyzer::ThreadRootNodes>& threadRoots = m_timerData.accessBuffer( b ).m_threadRoots;
        for ( int i = 0; i < threadRoots.getSize(); ++i )
        {
            delete threadRoots[i].m_node;
        }
        m_timerData.cycleBuffers();
    }
}

hkResult hkgVdbDeprecatedStatBarGraphWidget::initialize( hkgVdbPlugin& plugin, hkVdbClient& client )
{
    adjustForContext( *plugin.getWindow()->getContext() );
    m_defaultFont = hkgVdbTextBoxDrawer::loadDefaultFont( *plugin.getWindow()->getContext(), int( HKG_VDB_WIDGET_DEFAULT_TEXT_SIZE * plugin.getWindow()->getDpiScale() ) );
    m_options.m_onHoverTimerDisplayOptions.m_font = m_defaultFont;
    m_client = &client;
    return HK_SUCCESS;
}

hkResult hkgVdbDeprecatedStatBarGraphWidget::deinitialize( hkgVdbPlugin& plugin, hkVdbClient& client )
{
    m_client = HK_NULL;
    return HK_SUCCESS;
}

hkResult hkgVdbDeprecatedStatBarGraphWidget::update( const hkgViewport& viewport, const hkgInputManager& input )
{
    // Update based on shared resources
    acquireRenderLock();
        adjustForContext( *viewport.getOwnerWindow()->getContext() );
        m_timerData.cycleBuffersIfWriteUpdated();
        const TimerData& timerData = m_timerData.getReadBuffer();
        hkArray<hkMonitorStreamAnalyzer::ThreadRootNodes>& threadRoots =
            const_cast< hkArray<hkMonitorStreamAnalyzer::ThreadRootNodes>& >( timerData.m_threadRoots );
    releaseRenderLock();

    m_timerNodeUnderMouse = HK_NULL;
    const hkgMouse& mouse = input.getMouse();
    hkgVdbPoint wmp = hkgVdbPoint::windowCoordsFromInputCoords( mouse.getPosX(), mouse.getPosY(), viewport );
    hkgVdbPoint vmp = hkgVdbPoint::viewportCoordsFromInputCoords( mouse.getPosX(), mouse.getPosY(), viewport );

    // Handle input
    if ( getEnabledInputs().anyIsSet( hkgVdbInputTypes::MOUSE ) )
    {
        const hkgKeyboard& keyboard = input.getKeyboard();

        // Process mouse wheel.
        {
            if ( int delta = mouse.getWheelNotchDelta() )
            {
                bool doPan = keyboard.getKeyState( HKG_VDB_WIDGET_PAN_KEY );
                bool doZoom = keyboard.getKeyState( HKG_VDB_WIDGET_ZOOM_KEY );

                if ( doZoom )
                {
                    const double scaleFactor = ( 1.0 + ( delta * HKG_VDB_WHEEL_ZOOM_FACTOR ) );
                    if ( applyZoom( vmp, scaleFactor ) )
                    {
                        m_zoomedBounds = m_zoomingBounds;
                    }
                }
                else if ( doPan )
                {
                    if ( applyTranslate( delta * HKG_VDB_WHEEL_SCROLL_PIXELS ) )
                    {
                        m_zoomedBounds = m_zoomingBounds;
                    }
                }
            }
        }

        // Process mouse drags/interactive updates.
        {
            // Update our state
            hkgStatGraph::StatGraphUserActions statGraphActions;
            if ( mouse.getButtonState() )
            {
                statGraphActions =
                    ( mouse.getButtonState() & HKG_MOUSE_LEFT_BUTTON ) ?
                    m_statGraph.userAction( vmp.m_x, vmp.m_y ) :
                    hkgStatGraph::HKG_STAT_GRAPH_ACTION_NONE;
                bool doPan =
                    bool( mouse.getButtonState() & HKG_MOUSE_MIDDLE_BUTTON ) ||
                    ( bool( mouse.getButtonState() & HKG_MOUSE_LEFT_BUTTON ) && keyboard.getKeyState( HKG_VDB_WIDGET_PAN_KEY ) );
                bool doZoom =
                    bool( mouse.getButtonState() & HKG_MOUSE_RIGHT_BUTTON ) ||
                    ( bool( mouse.getButtonState() & HKG_MOUSE_LEFT_BUTTON ) && keyboard.getKeyState( HKG_VDB_WIDGET_ZOOM_KEY ) );

                m_state = InteractionState( m_state | ( doPan * PANNING ) );
                m_state = InteractionState( m_state | ( doZoom * ZOOMING ) );
                m_state = InteractionState( m_state | ( bool( !statGraphActions && ( m_state == NONE ) && ( mouse.getButtonState() & HKG_MOUSE_LEFT_BUTTON ) ) * ZOOM_EXTENTS ) );
            }
            else
            {
                m_state = NONE;
                statGraphActions = hkgStatGraph::HKG_STAT_GRAPH_ACTION_NONE;
            }

            if ( ( m_state == NONE ) && ( statGraphActions == hkgStatGraph::HKG_STAT_GRAPH_ACTION_NONE ) )
            {
                m_timerNodeUnderMouse = m_statGraph.findTimerNodeAtSample( threadRoots, timerData.m_frameInfo, timerData.m_streamConfig, vmp.m_x, vmp.m_y );
            }
            else
            {
                // Handle stat graph actions (from button clicks)
                if ( statGraphActions != hkgStatGraph::HKG_STAT_GRAPH_ACTION_NONE )
                {
                    // Handle reset button click
                    if ( statGraphActions == hkgStatGraph::HKG_STAT_GRAPH_ACTION_RESET_ZOOM )
                    {
                        resetBounds();
                    }

                    // Handle pan button click
                    if ( int delta =
                        ( statGraphActions == hkgStatGraph::HKG_STAT_GRAPH_ACTION_PAN_NEGX ) * -5 +
                        ( statGraphActions == hkgStatGraph::HKG_STAT_GRAPH_ACTION_PAN_POSX ) * +5 )
                    {
                        if ( applyTranslate( delta ) )
                        {
                            m_zoomedBounds = m_zoomingBounds;
                        }
                    }
                }
                // Handle zoom extents (no other interactions allowed in this mode)
                else if ( m_state & ZOOM_EXTENTS )
                {
                    if ( m_zoomExtentsStart.m_x == HK_INT32_MAX )
                    {
                        m_zoomExtentsStart.m_x = vmp.m_x;
                        m_zoomExtentsStart.m_y = vmp.m_y;
                    }
                }
                // Handle interactions that can happen simultaneously
                else
                {
                    // Handle zoom
                    if ( m_state & ZOOMING )
                    {
                        // Get starting info
                        if ( m_lastZooming.m_x == HK_INT32_MAX )
                        {
                            m_lastZooming = vmp;
                        }

                        const double scaleFactor = ( 1.0 + ( ( m_lastZooming.m_y - vmp.m_y ) * HKG_VDB_PIXEL_ZOOM_FACTOR ) );
                        if ( applyZoom( m_lastZooming, scaleFactor ) )
                        {
                            m_zoomedBounds = m_zoomingBounds;
                        }

                        // Update last
                        m_lastZooming.m_y = vmp.m_y;
                    }

                    // Handle mouse pan
                    if ( m_state & PANNING )
                    {
                        // Get starting info
                        if ( m_lastPanning.m_x == HK_INT32_MAX )
                        {
                            m_lastPanning = vmp;
                        }

                        if ( applyTranslate( m_lastPanning.m_x - vmp.m_x ) )
                        {
                            m_zoomedBounds = m_zoomingBounds;
                        }

                        // Update last
                        m_lastPanning = vmp;
                    }
                }
            }
        }
    }
    else
    {
        m_state = NONE;
    }

    // Finish interactions
    {
        if ( !( m_state & ZOOM_EXTENTS ) && ( m_zoomExtentsStart.m_x != HK_INT32_MAX ) )
        {
            if ( m_zoomExtentsStart.distanceTo( vmp ) > 10 )
            {
                m_zoomingBounds.m_minX = hkMath::min2( m_zoomExtentsStart.m_x, vmp.m_x );
                m_zoomingBounds.m_minY = hkMath::min2( m_zoomExtentsStart.m_y, vmp.m_y );
                m_zoomingBounds.m_maxX = hkMath::max2( m_zoomExtentsStart.m_x, vmp.m_x );
                m_zoomingBounds.m_maxY = hkMath::max2( m_zoomExtentsStart.m_y, vmp.m_y );
                m_zoomingBounds.clamp( m_statGraphTimerBounds );
                m_zoomedBounds = m_zoomingBounds;
            }
            m_zoomExtentsStart.m_x = HK_INT32_MAX;
        }

        if ( !( m_state & ZOOMING ) && ( m_lastZooming.m_x != HK_INT32_MAX ) )
        {
            m_zoomedBounds = m_zoomingBounds;
            m_lastZooming.m_x = HK_INT32_MAX;
        }

        if ( !( m_state & PANNING ) && ( m_lastPanning.m_x != HK_INT32_MAX ) )
        {
            m_zoomedBounds = m_zoomingBounds;
            m_lastPanning.m_x = HK_INT32_MAX;
        }
    }

    if ( threadRoots.getSize() )
    {
        acquireRenderLock();
        // Reset our zoom since our bounds are relative to entire stat graph
        m_statGraph.resetZoom();
        m_statGraph.update( threadRoots, timerData.m_frameInfo, timerData.m_streamConfig ); // required for proper sizing :(

                                                                                            // Apply our bounds
        m_statGraph.zoom(
            m_zoomingBounds.m_minX,
            m_zoomingBounds.m_minY,
            m_zoomingBounds.m_maxX,
            m_zoomingBounds.m_maxY );

        // Now update with our set zoom
        
        m_statGraph.update( threadRoots, timerData.m_frameInfo, timerData.m_streamConfig );
        releaseRenderLock();
    }

    return HK_SUCCESS;
}

void hkgVdbDeprecatedStatBarGraphWidget::render( hkgDisplayContext& context ) const
{
    if ( !m_bounds.isEmpty() )
    {
        const hkgWindow* window = context.getOwner();

        // Display the graph
        int displayY = ( context.getCurrentViewport()->getHeight() - m_bounds.m_maxY );
        m_statGraph.display( m_bounds.m_minX, displayY, false );

        // Display zoom extents if we are in that mode
        if ( m_state & ZOOM_EXTENTS )
        {
            const hkgMouse& mouse = window->getInputManager()->getMouse();
            hkgVdbPoint vmp = hkgVdbPoint::viewportCoordsFromInputCoords( mouse.getPosX(), mouse.getPosY(), *context.getCurrentViewport() );
            m_statGraph.displayZoomExtents( m_zoomExtentsStart.m_x, m_zoomExtentsStart.m_y, vmp.m_x, vmp.m_y );
        }

        // Restore 2D graphics state modified by calling hkgStatGraph::display()
        hkgVdb2dWidget::setCommonRenderState( context );

        // Display timers we are hovering over
        if ( m_timerNodeUnderMouse )
        {
            const hkgMouse& mouse = window->getInputManager()->getMouse();
            hkgVdbPoint wmp = hkgVdbPoint::windowCoordsFromInputCoords( mouse.getPosX(), mouse.getPosY(), *context.getCurrentViewport() );

            hkStringBuf buf;
            buf.printf(
                "%s/%s:%.3fms",
                m_timerNodeUnderMouse->m_parent ? m_timerNodeUnderMouse->m_parent->m_name : "",
                m_timerNodeUnderMouse->m_name,
                m_timerNodeUnderMouse->m_threadData[m_timerData.getReadBuffer().m_frameInfo.m_indexOfTimer0].m_value * 0.001f );

            hkUint32 userSetX = m_options.m_onHoverTimerDisplayOptions.m_x;
            hkUint32 userSetY = m_options.m_onHoverTimerDisplayOptions.m_y;
            if ( userSetX == HK_INT32_MAX ) m_options.m_onHoverTimerDisplayOptions.m_x = wmp.m_x;
            if ( userSetY == HK_INT32_MAX ) m_options.m_onHoverTimerDisplayOptions.m_y = wmp.m_y;
            m_options.m_onHoverTimerDisplayOptions.m_y -= hkInt32( m_options.m_onHoverTimerDisplayOptions.m_font->getCharHeight() );

            hkgVdbTextBoxDrawer::drawTextBox( context, buf, m_options.m_onHoverTimerDisplayOptions );
            m_options.m_onHoverTimerDisplayOptions.m_x = userSetX;
            m_options.m_onHoverTimerDisplayOptions.m_y = userSetY;
        }
    }
}

void hkgVdbDeprecatedStatBarGraphWidget::getToolTip( hkStringBuf& toolTipOut, bool verbose ) const
{
    toolTipOut.format(

        ( verbose ) ?

        "Drag Left Mouse Button : Create Zoom Extents\n"
        "Drag ({0} + Left Mouse Button), ({0} + Mouse Wheel), or Right Mouse Button Up / Down : Zoom In / Out\n"
        "Drag ({1} + Left Mouse Button), ({1} + Mouse Wheel), or Middle Mouse Button Left / Right : Pan Left / Right\n"
        "Magnifying Glass Icon : Reset" :

        "Drag LMB : Zoom Extents\n"
        "Drag ({0} + LMB), ({0} + Wheel), or RMB : Zoom\n"
        "Drag ({1} + LMB), ({1} + Wheel), or MMB : Pan\n"
        "Magnifying Glass Icon - Reset",

        hkgKeyboard::nameOfVkey( HKG_VDB_WIDGET_ZOOM_KEY ),
        hkgKeyboard::nameOfVkey( HKG_VDB_WIDGET_PAN_KEY ) );
}

void hkgVdbDeprecatedStatBarGraphWidget::onPerfStatsReceivedSignal( const hkVdbStatsHandler::PerfStats& perfStats, hkVdbSignalResult& result )
{
    
    

    // Create a frameinfo for interpreting monitor stream in us
    hkMonitorStreamFrameInfo frameInfo = perfStats.m_frameInfo;
    {
        //don't support another timer
        frameInfo.m_indexOfTimer1 = -1;
        //convert to us from ms for stat bar graph, the ms marker lines assume this
        frameInfo.m_timerFactor0 *= 1e3f;
    }

    // Create a directory node per thread
    hkArray<hkMonitorStreamAnalyzer::ThreadRootNodes> newThreadRoots;
    newThreadRoots.reserve( perfStats.m_monitorStreams.getSize() );
    {
        for ( int i = 0; i < perfStats.m_monitorStreams.getSize(); ++i )
        {
            const hkArray<char>& monitorStream = perfStats.m_monitorStreams[i];
            if ( hkMonitorStreamAnalyzer::Node* currentTree = hkMonitorStreamAnalyzer::makeStatisticsTreeForSingleFrame(
                perfStats.m_streamConfig,
                monitorStream.begin(),
                monitorStream.end(),
                frameInfo,
                "/",
                false ) )
            {
                hkMonitorStreamAnalyzer::Node* perThreadPerFrame = new hkMonitorStreamAnalyzer::Node( HK_NULL, "/", hkMonitorStreamAnalyzer::Node::NODE_TYPE_DIRECTORY );
                perThreadPerFrame->m_children.pushBack( currentTree );

                hkMonitorStreamAnalyzer::ThreadRootNodes& nodeThreadPair = newThreadRoots.expandOne();
                nodeThreadPair.m_node = perThreadPerFrame;
            }
        }
    }

    acquireRenderLock();
    {
        // Swap old data for new and then cleanup the old data
        TimerData& timerData = m_timerData.accessWriteBuffer();
        timerData.m_threadRoots.swap( newThreadRoots );
        for ( int i = 0; i < newThreadRoots.getSize(); ++i )
        {
            delete newThreadRoots[i].m_node;
        }
        timerData.m_frameInfo = frameInfo;
        timerData.m_streamConfig = perfStats.m_streamConfig;
    }
    releaseRenderLock();
}

hkResult hkgVdbDeprecatedStatBarGraphWidget::enable()
{
    HK_VDB_VERIFY_CONDITION( m_client, 0xadb00039, hkVdbError::PLUGIN_ERROR );
    HK_SUBSCRIBE_TO_SIGNAL( m_client->getCmdHandler<hkVdbStatsHandler>()->m_perfStatsReceived, this, hkgVdbDeprecatedStatBarGraphWidget );
    return HK_SUCCESS;
}

hkResult hkgVdbDeprecatedStatBarGraphWidget::disable()
{
    HK_VDB_VERIFY_CONDITION( m_client, 0xadb00040, hkVdbError::PLUGIN_ERROR );
    m_client->getCmdHandler<hkVdbStatsHandler>()->m_perfStatsReceived.unsubscribeAll( this );
    return HK_SUCCESS;
}

hkBool32 hkgVdbDeprecatedStatBarGraphWidget::applyZoom( hkgVdbPoint center, double scale ) const
{
    // Apply the scale (relative to total bounds)
    hkgVdbBounds zoomingBounds = m_zoomedBounds;
    zoomingBounds.scale( translateFromZoomedSpace( center ), hkgVdbPointD( scale, 1.0 ) );
    m_zoomingBounds = zoomingBounds;
    return true;
}

hkBool32 hkgVdbDeprecatedStatBarGraphWidget::applyTranslate( int delta ) const
{
    // Move our bounds
    hkgVdbBounds zoomingBounds = m_zoomingBounds;
    if ( zoomingBounds.tryTranslate( hkgVdbPoint( delta, 0 ) ).isSuccess() )
    {
        if ( !zoomingBounds.clamp( m_statGraphTimerBounds ) )
        {
            m_zoomingBounds = zoomingBounds;
            return true;
        }
    }
    return false;
}

void hkgVdbDeprecatedStatBarGraphWidget::adjustForContext( hkgDisplayContext& context )
{
    // Update our bounds
    hkgViewport* v = context.getCurrentViewport();
    m_bounds.set( v, m_options );

    // Adjust our stat graph based on bounds
    {
        int currentW = m_statGraph.getDisplayWidth();
        int currentH = m_statGraph.getDisplayHeight();
        int newW = m_bounds.getWidth();
        int newH = m_bounds.getHeight();
        if ( ( currentW != newW ) || ( currentH != newH ) ||
            ( m_bounds.m_minX != m_options.m_x ) || ( m_bounds.m_minY != m_options.m_y ) )
        {
            m_statGraph.init( HKG_STAT_GRAPH_BAR, &context, newW, newH );
            int displayY = ( v->getHeight() - ( m_options.m_y + newH ) );
            m_statGraph.display( m_options.m_x, displayY, false );

            // Ensure any zoom operations are reset
            m_zoomExtentsStart.m_x = HK_INT32_MAX;
            m_lastZooming.m_x = HK_INT32_MAX;
            m_lastPanning.m_x = HK_INT32_MAX;

            // Reset our timer bounds
            resetBounds();
        }
    }
}

hkgVdbPoint hkgVdbDeprecatedStatBarGraphWidget::translateFromZoomedSpace( hkgVdbPoint mp ) const
{
    // Note: m_statGraphTimerBounds has our absolute pix counts for the window
    double percentXInZoom = double( mp.m_x - m_statGraphTimerBounds.m_minX ) / ( m_statGraphTimerBounds.m_maxX - m_statGraphTimerBounds.m_minX );
    double percentYInZoom = double( mp.m_y - m_statGraphTimerBounds.m_minY ) / ( m_statGraphTimerBounds.m_maxY - m_statGraphTimerBounds.m_minY );

    hkInt32 absX = hkInt32( m_zoomedBounds.m_minX + ( percentXInZoom * ( m_zoomedBounds.m_maxX - m_zoomedBounds.m_minX ) ) );
    hkInt32 absY = hkInt32( m_zoomedBounds.m_minY + ( percentYInZoom * ( m_zoomedBounds.m_maxY - m_zoomedBounds.m_minY ) ) );

    return hkgVdbPoint( absX, absY );
}

void hkgVdbDeprecatedStatBarGraphWidget::resetBounds() const
{
    m_statGraphTimerBounds.m_minX = ( m_statGraph.getDisplayLastX() + m_statGraph.getStatDisplayOffsetX() );
    m_statGraphTimerBounds.m_minY = ( m_statGraph.getDisplayLastY() + m_statGraph.getStatDisplayOffsetY() );
    m_statGraphTimerBounds.m_maxX = ( m_statGraphTimerBounds.m_minX + m_statGraph.getStatDisplayWidth() );
    m_statGraphTimerBounds.m_maxY = ( m_statGraphTimerBounds.m_minY + m_statGraph.getStatDisplayHeight() );
    m_zoomingBounds = m_statGraphTimerBounds;
    m_zoomedBounds = m_statGraphTimerBounds;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
