// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPlugin.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Utils/hkgVdbObjectConversionUtil.h>

#include <Graphics/Common/hkgraphics.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/Geometry/hkgGeometry.h>
#include <Graphics/Common/Geometry/hkgMaterialFaceSet.h>
#include <Graphics/Common/Geometry/FaceSet/hkgFaceSet.h>
#include <Graphics/Common/Geometry/VertexSet/hkgVertexSet.h>
#include <Graphics/Common/Material/hkgMaterial.h>

// Scene Data
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/SceneData/Scene/hkxSceneUtils.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Mesh/hkxMeshSection.h>
#include <Common/SceneData/graph/hkxNode.h>
#include <Common/SceneData/Mesh/hkxVertexBuffer.h>
#include <Common/SceneData/Mesh/hkxIndexBuffer.h>
#include <Common/SceneData/Mesh/hkxVertexDescription.h>
#include <Common/SceneData/Material/hkxMaterial.h>

// Display Stuff
#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Visualize/Shape/hkDisplayGeometry.h>

#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>

namespace hkgVdbObjectConversionUtil
{
    hkRefNew<hkxVertexBuffer> createHkxVertexBuffer( const hkArrayView<LitVert> vertices )
    {
        const int numVertices = vertices.getSize();

        // Describe what we are going to export
        hkxVertexDescription desiredDesc;
        desiredDesc.m_decls.pushBack( hkxVertexDescription::ElementDecl( hkxVertexDescription::HKX_DU_POSITION, hkxVertexDescription::HKX_DT_FLOAT, 3 ) );
        desiredDesc.m_decls.pushBack( hkxVertexDescription::ElementDecl( hkxVertexDescription::HKX_DU_NORMAL, hkxVertexDescription::HKX_DT_FLOAT, 3 ) );

        // Allocate buffer
        hkxVertexBuffer* vertexBuffer = new hkxVertexBuffer;
        vertexBuffer->setNumVertices( numVertices, desiredDesc );

        const hkxVertexDescription& actualDesc = vertexBuffer->getVertexDesc();
        const hkxVertexDescription::ElementDecl* posDecl = actualDesc.getElementDecl( hkxVertexDescription::HKX_DU_POSITION, 0 );
        const hkxVertexDescription::ElementDecl* normDecl = actualDesc.getElementDecl( hkxVertexDescription::HKX_DU_NORMAL, 0 );

        char* posBuf = static_cast< char* >( vertexBuffer->getVertexDataPtr( *posDecl ) );
        char* normBuf = static_cast< char* >( vertexBuffer->getVertexDataPtr( *normDecl ) );
        int posStride = posDecl->m_byteStride;
        int normStride = normDecl->m_byteStride;

        for ( int i = 0; i < numVertices; ++i )
        {
            // Set position
            {
                hkFloat32* _pos = ( hkFloat32* ) ( posBuf );
                vertices[i].m_position.store<3, HK_IO_NATIVE_ALIGNED>( _pos );
            }

            // Set normal
            {
                hkFloat32* _normal = ( hkFloat32* ) ( normBuf );
                vertices[i].m_normal.store<3, HK_IO_NATIVE_ALIGNED>( _normal );
            }

            // next
            posBuf += posStride;
            normBuf += normStride;
        }

        return vertexBuffer;
    }

    hkRefNew<hkxIndexBuffer> createHkxIndexBuffer( const hkArrayView<hkUint32> indices, bool largeIndices )
    {
        const int numIndices = indices.getSize();

        hkxIndexBuffer* indexBuffer = new hkxIndexBuffer;
        indexBuffer->m_indexType = hkxIndexBuffer::INDEX_TYPE_TRI_LIST;
        indexBuffer->m_vertexBaseOffset = 0;

        if ( largeIndices )
        {
            indexBuffer->m_indices32.setSize( numIndices );
            hkString::memCpy( indexBuffer->m_indices32.begin(), indices.begin(), sizeof( hkUint32 ) * numIndices );
        }
        else
        {
            indexBuffer->m_indices16.setSize( numIndices );
            for ( int i = 0; i < numIndices; ++i )
            {
                indexBuffer->m_indices16[i] = ( hkUint16 ) indices[i];
            }
        }

        indexBuffer->m_length = numIndices;

        return indexBuffer;
    }

    hkRefNew<hkxMaterial> createHkxMaterial( hkReal cr, hkReal cg, hkReal cb, hkReal ca )
    {
        hkxMaterial* material = new hkxMaterial;

        material->m_name = "material";

        // Set only diffuse color now
        material->m_diffuseColor.set( cr, cg, cb, ca );
        material->m_ambientColor.setZero();
        material->m_specularColor.setZero();
        material->m_emissiveColor.setZero();

        return material;
    }

    static void _createHkxMeshSectionsFromGeometry( const hkgGeometry* geom, hkArray<hkRefPtr<hkxMeshSection>>& sections, hkPointerMap< int, hkxMaterial* >& materialMap )
    {
        hkReal cr, cg, cb, ca;
        cr = cg = cb = ca = 1.0f;

        int mgfs = geom->getNumMaterialFaceSets();

        for ( int m = 0; m < mgfs; ++m )
        {
            const hkgMaterialFaceSet* mfs = geom->getMaterialFaceSet( m );
            int nfs = mfs->getNumFaceSets();

            // Vertices and Indices
            hkArray< LitVert > vertices;
            int offset = 0;
            bool largeIndices = false;
            hkArray<hkUint32> indices;

            for ( int f = 0; f < nfs; ++f )
            {
                const hkgFaceSet* faceSet = mfs->getFaceSet( f );

                const hkgVertexSet* vertexSet = faceSet->getVertexSet(); //added hreyn
                const_cast<hkgVertexSet*>( vertexSet )->lock( HKG_LOCK_READONLY ); // added hreyn
                {
                    vertices.reserve( vertices.getSize() + vertexSet->getNumVerts() );
                    for ( int i = 0; i < vertexSet->getNumVerts(); ++i )
                    {
                        LitVert v;
                        const hkFloat32* pos = reinterpret_cast< const hkFloat32* > ( vertexSet->getVertexComponentData( HKG_VERTEX_COMPONENT_POS, i ) );
                        v.m_position.load<3, HK_IO_NATIVE_ALIGNED>( pos ); v.m_position.zeroComponent<3>();
                        const hkFloat32* normal = reinterpret_cast< const hkFloat32* > ( vertexSet->getVertexComponentData( HKG_VERTEX_COMPONENT_NORMAL, i ) );
                        v.m_normal.load<3, HK_IO_NATIVE_ALIGNED>( normal ); v.m_normal.zeroComponent<3>();

                        vertices.pushBack( v );
                    }
                }
                const_cast<hkgVertexSet*>( vertexSet )->unlock();

                // indeces
                int ioffset = offset;
                offset += vertexSet->getNumVerts();

                indices.reserve( indices.getSize() + ( faceSet->getNumPrimitives() * 3 ) );
                for ( int p = 0; p < faceSet->getNumPrimitives(); ++p )
                {
                    const hkgFaceSetPrimitive* prim = faceSet->getPrimitive( p );

                    largeIndices |= ( offset > 0xffff ) || ( prim->getIndexSize() == HKG_INDICES_UINT32 );

                    int len = prim->getLength();
                    int poffset = prim->getVertexBufferStartIndex();
                    const hkUint16* pindices16 = prim->getIndices16();
                    const hkUint32* pindices32 = prim->getIndices32();

                    HK_ASSERT_DEV( 0xb720ff6a, prim->getType() == HKG_TRIANGLE_STRIP || prim->getType() == HKG_TRIANGLE_LIST, "_createHkxMeshSectionsFromGeometry() only supports triangle strip and list geometry" );
                    if ( prim->getType() == HKG_TRIANGLE_STRIP )
                    {
                        if ( pindices16 )
                        {
                            indices.pushBack( pindices16[0] + ioffset );
                            indices.pushBack( pindices16[1] + ioffset );
                            indices.pushBack( pindices16[2] + ioffset );
                        }
                        else if ( pindices32 )
                        {
                            indices.pushBack( pindices32[0] + ioffset );
                            indices.pushBack( pindices32[1] + ioffset );
                            indices.pushBack( pindices32[2] + ioffset );
                        }
                        else
                        {
                            indices.pushBack( poffset + ioffset );
                            indices.pushBack( poffset + ioffset + 1 );
                            indices.pushBack( poffset + ioffset + 2 );
                        }

                        for ( int s = 3; s < len; ++s )
                        {
                            const int flip = s % 2;
                            if ( pindices16 )
                            {
                                indices.pushBack( pindices16[s - ( 2 - flip )] + ioffset );
                                indices.pushBack( pindices16[s - ( 1 + flip )] + ioffset );
                                indices.pushBack( pindices16[s] + ioffset );
                            }
                            else if ( pindices32 )
                            {
                                indices.pushBack( pindices32[s - ( 2 - flip )] + ioffset );
                                indices.pushBack( pindices32[s - ( 1 + flip )] + ioffset );
                                indices.pushBack( pindices32[s] + ioffset );
                            }
                            else
                            {
                                indices.pushBack( poffset + s - ( 2 - flip ) + ioffset );
                                indices.pushBack( poffset + s - ( 1 + flip ) + ioffset );
                                indices.pushBack( poffset + s + ioffset );
                            }
                        }
                    }
                    else if ( prim->getType() == HKG_TRIANGLE_LIST )
                    {
                        for ( int s = 0; s < len; )
                        {
                            if ( pindices16 )
                            {
                                indices.pushBack( pindices16[s++] + ioffset );
                                indices.pushBack( pindices16[s++] + ioffset );
                                indices.pushBack( pindices16[s++] + ioffset );
                            }
                            else if ( pindices32 )
                            {
                                indices.pushBack( pindices32[s++] + ioffset );
                                indices.pushBack( pindices32[s++] + ioffset );
                                indices.pushBack( pindices32[s++] + ioffset );
                            }
                            else
                            {
                                indices.pushBack( poffset + s + ioffset ); s++;
                                indices.pushBack( poffset + s + ioffset ); s++;
                                indices.pushBack( poffset + s + ioffset ); s++;
                            }
                        }
                    }
                }
            }

            // Create hkxMaterial
            const hkgMaterial* hkgmat = mfs->getMaterial();
            hkgmat->getDiffuseColor( cr, cb, cg, ca );
            int color = hkColor::rgbFromFloats( cr, cb, cg, ca );
            // Since color is used below as a map index, we ensure that it can't be -1.
            if ( color == -1 )
            {
                color = hkColor::rgbFromChars( 0xfe, 0xfe, 0xfe, 0xff );
            }
            hkRefPtr<hkxMaterial> hkxmat = materialMap.getWithDefault( color, HK_NULL );
            if ( !hkxmat )
            {
                hkxmat = createHkxMaterial( cr, cb, cg, ca );
                materialMap.insert( color, hkxmat );
            }

            // Create new set of hkxMeshSections
            {
                hkRefPtr<hkxMeshSection> section = hkRefNew<hkxMeshSection>( new hkxMeshSection );
                section->m_material = hkxmat;

                // Create vertexBuffer
                {
                    hkRefPtr<hkxVertexBuffer> newVertexBuffer = createHkxVertexBuffer( vertices );
                    section->m_vertexBuffer = newVertexBuffer;
                }

                // Create indexBuffers
                {
                    hkRefPtr<hkxIndexBuffer> newIndexBuffer = createHkxIndexBuffer( indices, largeIndices );
                    section->m_indexBuffers.pushBack( newIndexBuffer );
                }

                sections.pushBack( section );
            }
        }
    }

    hkRefNew<hkxMesh> createHkxMesh( const hkgDisplayObject* displayObject, hkPointerMap< int, hkxMaterial* >& materialMap )
    {
        // Create mesh
        hkxMesh* mesh = new hkxMesh;

        // Add geometry
        const int numGeom = displayObject->getNumGeometry();
        for ( int g = 0; g < numGeom; ++g )
        {
            const hkgGeometry* geom = displayObject->getGeometry( g );
            _createHkxMeshSectionsFromGeometry( geom, mesh->m_sections, materialMap );
        }

        return mesh;
    }

    hkRefNew<hkxMesh> createHkxMesh( const hkArrayView<hkgGeometry*> geometries, hkPointerMap< int, hkxMaterial* >& materialMap )
    {
        // Create mesh
        hkxMesh* mesh = new hkxMesh;

        // Add geometry
        const int numGeom = geometries.getSize();
        for ( int g = 0; g < numGeom; ++g )
        {
            _createHkxMeshSectionsFromGeometry( geometries[g], mesh->m_sections, materialMap );
        }

        return mesh;
    }

    hkRefNew<hkxNode> createHkxNode( char* name, const int numChildren, const hkxMesh* mesh, const hkMatrix4& trmat )
    {
        hkxNode* node = new hkxNode;
        node->m_object.set( const_cast< hkxMesh* >( mesh ) );
        node->m_name = name;
        node->m_keyFrames.expandOne();
        node->m_keyFrames[0] = trmat;
        node->m_children.setSize( ( numChildren > 0 ) ? numChildren : 0 );
        node->m_selected = true;
        return node;
    }

    hkRefNew<hkxScene> createHkxScene( const hkArrayView<hkRefPtr<hkxMesh>> meshes, const hkArrayView<hkMatrix4> tmats )
    {
        const int numMeshes = meshes.getSize();
        HK_ASSERT( 0x5828eef0, numMeshes == tmats.getSize(), "Difference in number of meshes and transform matrices" );

        hkMatrix4 itrmat; itrmat.setIdentity();

        // Create root node
        hkRefPtr<hkxNode> rootNode = createHkxNode( "SCENE_NODE", numMeshes, HK_NULL, itrmat );

        // Create child nodes
        for ( int i = 0; i < numMeshes; i++ )
        {
            rootNode->m_children[i] = createHkxNode( "OBJECT_NODE", 0, meshes[i], tmats[i] );
        }

        // Create hkxScene
        hkxScene* scene = new hkxScene;
        {
            scene->m_rootNode = rootNode;
            scene->m_modeller = "VDB Export";
            scene->m_asset = HK_NULL;
            scene->m_sceneLength = 0;
            scene->m_appliedTransform.setIdentity();
        }

        return scene;
    }

    hkResult writeHkxScene( hkStreamWriter& writer, const hkxScene* scene )
    {
        hkRootLevelContainer rootLevelContainer;
        {
            hkRootLevelContainer::NamedVariant genericData(
                hkReflect::getType<hkxScene>()->getName(),
                scene );
            rootLevelContainer.m_namedVariants.pushBack( genericData );
        }
        return hkSerializeUtil::saveTagfile( &rootLevelContainer, &writer );
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
