// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbDisplayHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/3d/hkgVdbSelectionWidget.h>

class hkgWindow;
class hkgViewport;
class hkgDisplayContext;
class hkgDisplayWorld;
class hkgDisplayObject;
class hkVdbClient;
class hkgMouse;
class hkgDisplayObjectExInterface;
class hkgVdbPlugin;

//////////////////////////////////////////////////////////////////////////
/// Flags which control how we render the world.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkgVdbRenderFlags : public hkFlagsEx<hkUint8>
{
    enum Bits
    {
        
        /// Geometry will be rendered with random colors.
        RANDOMIZED_COLORS   = ( 1 << 0 ),

        /// Geometry will be rendered as wireframe (not solid/filled).
        WIREFRAME           = ( 1 << 1 ),

        /// Geometry will be renered with back face culling enabled.
        BACKFACE_CULLING    = ( 1 << 2 ),

        /// In conjunction with BACKFACE_CULLING, determines if CCW winding is used.
        CULLMODE_CCW        = ( 1 << 3 ),

        /// Geometry will be rendered as solid but also have it's faced outlined.
        /// This conceptually similar to layering WIREFRAME on top of normal rendering.
        OUTLINE_FACES       = ( 1 << 4 ),

        /// The system will apply a visualization to the current geometry selection/highlights.
        VISUALIZE_GEOMETRY_SELECTION = ( 1 << 5 ),

        /// All geometry will be hidden unless the specific geometry has ALWAYS_VISIBLE specified.
        HIDE_ALL            = ( 1 << 6 ),

        /// Default render flags.
        DEFAULT = ( BACKFACE_CULLING | CULLMODE_CCW | VISUALIZE_GEOMETRY_SELECTION )
        
    };

    HK_DECLARE_FLAGS_EX_CLASS( hkgVdbRenderFlags, hkgVdbRenderFlags::Bits );
};

//////////////////////////////////////////////////////////////////////////
/// Options for how to display an individual Geometry.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkgVdbGeometryDisplayOptions
{
    
    enum VisibilityMode
    {
        /// The object will never be visible.
        NEVER_VISIBLE,

        /// The object is visible unless the render flag HIDE_ALL is enabled.
        HIDEABLE,

        /// The object will always be visible regardless of the render flag HIDE_ALL.
        ALWAYS_VISIBLE
    };

    HK_DECLARE_CLASS( hkgVdbGeometryDisplayOptions, New );
    hkgVdbGeometryDisplayOptions() :
        m_visibility( HIDEABLE ),
        m_userColor( hkColor::NONE ),
        m_enableUserColor( false ),
        m_randomColor( hkColor::NONE ),
        m_serverColor( hkColor::NONE )
    {}

    /// In conjunction with global render flags, influences object visibility.
    VisibilityMode m_visibility;

    /// The color of the object is initially determined by the server,
    /// but can later be customized by the user. This is not applied
    /// unless m_enableUserColor is true.
    hkColor::Argb m_userColor;

    /// If true, the user color is applied and the server color is ignored.
    hkBool m_enableUserColor;

    /// Determines the color of the object during "randomized" visualization.
    /// This is super-ceded by the user color if m_enableUserColor is true.
    hkColor::Argb m_randomColor;

    /// Get the current server color.
    /// This can change at any moment based on server commands.
    HK_INLINE hkColor::Argb getServerColor() const { return m_serverColor; }

    /// If m_enableUserColor is true, returns the user color; else the server color.
    HK_INLINE hkColor::Argb getCurrentColor() const { return m_enableUserColor ? m_userColor : m_serverColor; }
    

private:

    /// Get the effective color used for internal setting (affected by visibility)
    HK_INLINE hkColor::Argb getEffectiveColor() const
    {
        hkColor::Argb effectiveColor = getCurrentColor();
        bool wasNone = ( effectiveColor == hkColor::NONE );
        effectiveColor &= ( ( m_visibility == NEVER_VISIBLE ) ? 0x00ffffff : 0xffffffff );
        bool isNone = ( effectiveColor == hkColor::NONE );
        effectiveColor -= ( isNone && !wasNone ); // make sure our visibility didn't make us equal to hkColor::NONE
        return effectiveColor;
    }

    HK_INLINE bool isEffectivelyVisibile( const hkgVdbRenderFlags& globalFlags )
    {
        if ( globalFlags.noneIsSet( hkgVdbRenderFlags::HIDE_ALL ) && ( m_visibility != NEVER_VISIBLE ) ||
            globalFlags.anyIsSet( hkgVdbRenderFlags::HIDE_ALL ) && ( m_visibility == ALWAYS_VISIBLE ) )
        {
            
            
            
            
            return true;
        }
        else
        {
            return false;
        }
    }

    /// Current server color
    hkColor::Argb m_serverColor;

    friend class hkgDisplayObjectExInterface;
};

//////////////////////////////////////////////////////////////////////////
/// Controls the geometry in the display world.
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkgVdbGeometryControl : public hkVdbDefaultErrorReporter
{
public:
    HK_DECLARE_CLASS( hkgVdbGeometryControl, New );
    hkgVdbGeometryControl( hkgWindow& window );
    ~hkgVdbGeometryControl();

    //
    // Registration
    //

    hkResult registerSelf( hkgVdbPlugin& plugin, hkVdbDisplayHandler& handler, hkVdbClient& client );
    hkResult unregisterSelf( hkgVdbPlugin& plugin, hkVdbDisplayHandler& handler, hkVdbClient& client );

    //
    // Rendering
    //

    /// Render all geometry to the provided viewport.
    void render( hkgViewport& viewport ) const;

    //
    // Member Access
    //

    /// Raw access to the underlying display world so it can be shared with other entities.
    HK_INLINE hkgDisplayWorld& accessDisplayWorld() { return *m_displayWorld; }

    //
    // Render Control
    //

    /// Sets the current flags being used to render the geometry.
    hkResult setRenderFlags( hkgVdbRenderFlags flags );

    /// Gets the current flags being used to render the geometry.
    HK_INLINE hkgVdbRenderFlags getRenderFlags() const { return m_renderFlags; }

    /// Usually, the server will provide a color for geometry, but if not, this color will be used.
    static hkColor::Argb DEFAULT_COLOR;

    //
    // Geometry Control
    //

    /// Set the display options for the geometry associated with id.
    hkResult setGeometryDisplayOptions( hkUint64 id, const hkgVdbGeometryDisplayOptions& options );

    /// Get the display options for the geometry associated with id.
    const hkgVdbGeometryDisplayOptions* getGeometryDisplayOptions( hkUint64 id ) const;

    /// A func that is used to filter geometry for saving.
    typedef hkBool32( *GeometryFilterFunc )( hkUint64 id, const hkgDisplayObject& object, void* handle );

    /// A GeometryFilterFunc which will save geometry based on selection.
    /// If this func is used, the "filterHandle" is expected to be a pointer to SaveSelectionOnlyOptions.
    static GeometryFilterFunc SaveSelectionOnly;
    enum class SaveSelectionOnlyOptions { SaveSelectedAndHighlighted = 0, SaveSelected, SaveHighlighted };

    /// A GeometryFilterFunc which will save a single geometry.
    /// If this func is used, the "filterHandle" is expected to be a pointer to SaveSingleGeometryOptions (aka an hkgDisplayObject).
    static GeometryFilterFunc SaveSingleGeometry;
    typedef hkgDisplayObject SaveSingleGeometryOptions;

    /// Save geometry using an optional geometry filter and handle for the filter callback.
    hkResult saveGeometry( const char* filepath, GeometryFilterFunc filter = HK_NULL, void* filterHandle = HK_NULL );

    //
    // Query
    //

    /// Find the geometry id at the specified viewport point (or zero if not found).
    /// Optionally returns world space and local space hit points.
    hkUint64 findGeometryAt( hkgViewport& viewport, hkgVdbPoint p, hkVector4* worldSpaceHitPointOut = HK_NULL, hkVector4* localSpaceHitPointOut = HK_NULL ) const;

    /// Gets the geometry from the specified id.
    hkgDisplayObject* getGeometryFromId( hkUint64 id ) const;

    //
    // Signals
    //

    HK_DECLARE_SIGNAL( GeometryAddedSignal, hkSignal1< hkgDisplayObject& > );
    /// Fires when a geometry is added to the display world.
    GeometryAddedSignal m_geometryAdded;

    HK_DECLARE_SIGNAL( GeometryRemovedSignal, hkSignal1< hkgDisplayObject& > );
    /// Fires when a geometry is removed from the display world.
    GeometryRemovedSignal m_geometryRemoved;

    HK_DECLARE_SIGNAL( GeometryHiddenSignal, hkSignal1< hkgDisplayObject& > );
    /// Fires when a geometry is hidden.
    GeometryHiddenSignal m_geometryHidden;

    HK_DECLARE_SIGNAL( GeometryShownSignal, hkSignal1< hkgDisplayObject& > );
    /// Fires when a geometry is shown.
    GeometryShownSignal m_geometryShown;

    //
    // Internal use
    //

    void onDisconnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection );
    void onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection );
    void onDisplayOptionsSetSignal( const hkDebugDisplayHandler::Options& options, hkVdbSignalResult& result );
    void onGeometryCmdReceivedSignal( const hkVdbDisplayHandler::GeometryCmd& geomCmd, hkVdbSignalResult& result );
    void onPlaybackInfoReceivedSignal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result );
    void onGeometrySelectionChangedSignal( const hkgVdbSelectionWidget::SelectionSets& set, const hkgVdbSelectionWidget::SelectionChangedInfo& info );
    void onWaitForCompletionSignal();

protected:

    typedef hkVdbMap<hkUint64, hkgDisplayObject*>::Iterator IdMapIt;
    typedef hkVdbMap<const hkgDisplayObject*, hkUint64>::Iterator ObjectMapIt;

    hkResult processGeometryCmdQueue();
    hkResult processGeometryCmd( const hkVdbDisplayHandler::GeometryCmd& geomCmd );
    hkResult processAddGeomCmd( const hkVdbDisplayHandler::AddGeometryCmd& addCmd );
    hkResult processInstanceGeomCmd( const hkVdbDisplayHandler::InstanceGeometryCmd& instanceCmd );
    hkResult processUpdateGeomCmd( const hkVdbDisplayHandler::UpdateGeometryCmd& updateCmd );
    hkResult processRemoveGeomCmd( const hkVdbDisplayHandler::RemoveGeometryCmd& removeCmd );
    hkResult processDisposeGeomCmd( const hkVdbDisplayHandler::DisposeGeometryCmd& disposeCmd );
    void addAndTrackDisplayObject(
        hkgDisplayObject& object,
        const hkMatrix4& transform,
        hkUint64 id,
        int tag );
    void removeDisplayObjects();
    void removeDisplayObject( IdMapIt iter );
    void addDisplayObjectToWorld( hkgDisplayObject& object );
    void removeDisplayObjectFromWorld( hkgDisplayObject& object );

    struct SelectionMode
    {
        enum Enum
        {
            NONE = 0,
            HIGHLIGHTED = 1,
            SELECTED = 2,
        };
    };
    void setSelection( const hkgVdbSelectionWidget::SelectionSet& sets, SelectionMode::Enum selection );

    hkRefPtr<hkgWindow> m_window;
    hkgVdbRenderFlags m_renderFlags;
    hkRefPtr<hkgDisplayWorld> m_displayWorld;
    hkRefPtr<hkgVdbSelectionWidget> m_selectionWidget;

    hkArray<hkVdbDisplayHandler::AnyGeometryCmd> m_cmdQueue;
    hkVdbMap<hkUint64, hkgDisplayObject*> m_idToObjectMap;

    hkRefPtr<hkVdbDisplayHandler> m_displayHandler;

    friend class hkgDisplayObjectExInterface;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
