// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPlugin.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Control/hkgVdbGeometryControl.h>

#include <Common/Base/Types/hkSignalSlots.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>

#include <Common/Visualize/Shape/hkDisplayGeometry.h>
#include <Common/Visualize/Shape/hkDisplayWireframe.h>

#include <Common/SceneData/hkSceneData.h>
#include <Common/SceneData/Scene/hkxScene.h>

#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Common/DisplayObject/hkgWireframeDisplayObject.h>
#include <Graphics/Common/Geometry/VertexSet/hkgVertexSet.h>
#include <Graphics/Common/Shader/hkgShaderContext.h>
#include <Graphics/Bridge/DisplayHandler/hkgGeometryConverter.h>

#include <Graphics/Dx11/hkGraphicsDX11.h> // precompiled header
#include <Graphics/Dx11/Shared/DisplayObject/hkgInstancedDisplayObjectDX11.h>

#include <VisualDebugger/VdbDisplay/Hkg/System/hkgVdbPluginControl.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPluginApi.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/hkgVdbWidgetManager.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/3d/hkgVdbSelectionWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Utils/hkgVdbObjectConversionUtil.h>

#ifndef HK_VDB_USE_HASH_MAP
#include <Common/Base/Container/PointerMap/hkMap.h>
#include <Common/Base/Container/PointerMap/hkMap.hxx>
template class HK_EXPORT_COMMON hkMapBase<hkUint64, hkgDisplayObject*>;
template class HK_EXPORT_COMMON hkMap<hkUint64, hkgDisplayObject*>;
template class HK_EXPORT_COMMON hkMapBase<const hkgDisplayObject*, hkUint64>;
template class HK_EXPORT_COMMON hkMap<const hkgDisplayObject*, hkUint64>;
#endif

#define DEBUG_LOG_IDENTIFIER "vdb.hkg.System.Control.Geometry"
#include <Common/Base/System/Log/hkLog.hxx>

class hkgDisplayObjectExInterface
{
public:
    HK_DECLARE_CLASS( hkgDisplayObjectExInterface, New );
    hkgDisplayObjectExInterface(
        hkgVdbGeometryControl& owner,
        hkDisplayGeometryFlags flags,
        hkColor::Argb serverColor,
        hkColor::Argb randomColor ) :
        m_owner( owner ),
        m_self( HK_NULL ),
        m_currentColor( hkColor::NONE ),
        m_renderFlags( owner.getRenderFlags() ),
        m_serverFlags( flags )
    {
        m_options.m_serverColor = serverColor;
        m_options.m_randomColor = randomColor;
    }

    HK_INLINE void postInit( hkgDisplayObject& self )
    {
        m_self = &self;
        // Ignore some asserts/warnings that are fine in the initialization case
        hkDisableError disable22441025( 0x22441025 );
        hkDisableError disable22441026( 0x22441026 );
        updateRenderProperties();
    }

    HK_INLINE void setServerColor( hkColor::Argb newColor )
    {
        if ( newColor == hkColor::NONE ) return;
        m_options.m_serverColor = newColor;
        updateRenderProperties();
    }

    HK_INLINE void setGlobalRenderFlags( const hkgVdbRenderFlags& flags )
    {
        bool wasHidden = m_renderFlags.anyIsSet( hkgVdbRenderFlags::HIDE_ALL );
        {
            m_renderFlags = flags;
        }
        bool isHidden = m_renderFlags.anyIsSet( hkgVdbRenderFlags::HIDE_ALL );
        if ( wasHidden != isHidden )
        {
            updateRenderProperties();
        }
    }

    HK_INLINE void setOptions( const hkgVdbGeometryDisplayOptions& options )
    {
        m_options = options;
        updateRenderProperties();
    }

    HK_INLINE void setServerFlagBits( hkDisplayGeometryFlags flagBits )
    {
        HK_ASSERT_DEBUG( 0x22441025, m_owner.m_idToObjectMap.hasKey( m_self->m_id ), "Object must be added first" );

        if ( m_renderFlags.noneIsSet( hkgVdbRenderFlags::HIDE_ALL ) )
        {
            setServerFlagBitsInternal( flagBits );
        }
        m_serverFlags.orWith( flagBits );
    }

    HK_INLINE void clearServerFlagBits( hkDisplayGeometryFlags flagBits )
    {
        HK_ASSERT_DEBUG( 0x22441026, m_owner.m_idToObjectMap.hasKey( m_self->m_id ), "Object must be added first" );

        if ( m_renderFlags.noneIsSet( hkgVdbRenderFlags::HIDE_ALL ) )
        {
            clearServerFlagBitsInternal( flagBits );
        }
        m_serverFlags.andNotWith( flagBits );
    }

    HK_INLINE void updateStatusFromFlags()
    {
        // Ignore some asserts/warnings that are fine in the update case
        hkDisableError disable22441027( 0x22441027 );
        hkDisableError disable22441028( 0x22441028 );
        hkDisplayGeometryFlags setFlags = m_serverFlags;
        setServerFlagBitsInternal( setFlags );
        hkDisplayGeometryFlags clearFlags = hkDisplayGeometryFlags::DEFAULT;
        clearFlags.andNotWith( setFlags );
        clearServerFlagBitsInternal( clearFlags );
    }

    HK_INLINE const hkDisplayGeometryFlags getServerFlags() const
    {
        return m_serverFlags;
    }

    HK_INLINE const hkgVdbGeometryDisplayOptions& getOptions() const
    {
        return m_options;
    }

    HK_INLINE void markRemoved()
    {
        
        
        
        
        
        
        
        
        
        m_serverFlags.clear( hkDisplayGeometryFlags::VISIBLE );
    }

protected:

    HK_INLINE void setServerFlagBitsInternal( hkDisplayGeometryFlags flagBits )
    {
        if ( flagBits.anyIsSet( hkDisplayGeometryFlags::PICKABLE ) )
        {
            m_self->setStatusFlags( m_self->getStatusFlags() & ~HKG_DISPLAY_OBJECT_NOT_PICKABLE );
        }
        if ( flagBits.anyIsSet( hkDisplayGeometryFlags::VISIBLE ) )
        {
#if 0 
            if ( object.m_geomsVisibilityMask.getSize() == 0 )
            {
                const int numGeoms = hkMath::max2( object.m_solidGeoms.getSize(), object.m_alphaGeoms.getSize() );
                object.m_geomsVisibilityMask.setSizeAndFill( 0, numGeoms, 1 );
            }
            object.m_geomsVisibilityMask.assignAll<1>();
#else
            m_owner.addDisplayObjectToWorld( *m_self );
#endif
        }
        if ( ( flagBits.anyIsSet( hkDisplayGeometryFlags::MUTABLE_TRANSFORM ) ) ||
            ( flagBits.anyIsSet( hkDisplayGeometryFlags::MUTABLE_VERTICES ) ) )
        {
#ifdef HK_DEBUG
            HK_ASSERT( 0x22441027, false, "hkgVdbPlugin does not currently support dynamically setting the geometry flags" );
#else
            HK_WARN_ONCE( 0x22441027, "hkgVdbPlugin does not currently support dynamically setting the geometry flags ( " << flagBits.get() << " )" );
#endif
        }
    }

    HK_INLINE void clearServerFlagBitsInternal( hkDisplayGeometryFlags flagBits )
    {
        if ( flagBits.anyIsSet( hkDisplayGeometryFlags::PICKABLE ) )
        {
            m_self->setStatusFlags( m_self->getStatusFlags() | HKG_DISPLAY_OBJECT_NOT_PICKABLE );
        }
        if ( flagBits.anyIsSet( hkDisplayGeometryFlags::VISIBLE ) )
        {
#if 0 //todo.jg.4vdb faster but won't work for render graphed stuff
            if ( object.m_geomsVisibilityMask.getSize() == 0 )
            {
                const int numGeoms = hkMath::max2( object.m_solidGeoms.getSize(), object.m_alphaGeoms.getSize() );
                object.m_geomsVisibilityMask.setSizeAndFill( 0, numGeoms, 1 );
            }
            object.m_geomsVisibilityMask.assignAll<0>();
#else
            m_owner.removeDisplayObjectFromWorld( *m_self );
#endif
        }
        if ( ( flagBits.anyIsSet( hkDisplayGeometryFlags::MUTABLE_TRANSFORM ) ) ||
            ( flagBits.anyIsSet( hkDisplayGeometryFlags::MUTABLE_VERTICES ) ) )
        {
#ifdef HK_DEBUG
            HK_ASSERT( 0x22441028, false, "hkgVdbPlugin does not currently support dynamically clearing the geometry flags" );
#else
            HK_WARN_ONCE( 0x22441028, "hkgVdbPlugin does not currently support dynamically clearing the geometry flags ( " << flagBits.get() << " )" );
#endif
        }
    }

    HK_INLINE void updateRenderProperties()
    {
        if ( m_options.isEffectivelyVisibile( m_renderFlags ) )
        {
            updateStatusFromFlags();
        }
        else
        {
            m_self->setStatusFlags( m_self->getStatusFlags() | HKG_DISPLAY_OBJECT_NOT_PICKABLE );
        }
        updateColor();
    }

    void updateColor()
    {
        // Get our effective color based on visibility and user color settings.
        hkColor::Argb newColor = m_options.getEffectiveColor();

        // Early out if we have a no-op situation
        if ( newColor == hkColor::NONE )
        {
            return;
        }

        // We use the user color for our random color if it's enabled
        
        
        
        
        
        m_self->setRandomColor( 0xff000000 | ( m_options.m_enableUserColor ? newColor : m_options.m_randomColor ) );

        // hkgDisplayObject::getColor can be slow
        if ( m_currentColor == hkColor::NONE )
        {
            m_self->getUserColor( m_currentColor );
        }

#ifdef HK_DEBUG_SLOW
        hkColor::Argb currentColor;
        const hkBool32 hasColor = ( m_self->getUserColor( currentColor ).isSuccess() );
        HK_ASSERT_DEBUG( 0x22441085, !hasColor || ( currentColor == m_currentColor ), "Color has changed out from under the framework" );
#endif

        if ( m_currentColor != newColor )
        {
            
            const hkBool32 sharedGeometry =
                m_self->getNumGeometry() > 0 &&
                ( m_self->getGeometry( 0 )->getReferenceCount() > 1 );

            if ( sharedGeometry )
            {
                // Make a deep copy so we can change color without affecting other instances
                hkgDisplayObject* objectCopy = hkgDisplayObject::copy(
                    m_self,
                    HKG_DISPLAY_OBJECT_SHARE_FACESETS | HKG_DISPLAY_OBJECT_SHARE_MATERIALS,
                    HK_NULL,
                    hkgDisplayObject::create );

                // Steal the geometries from the copy so we keep the same pointer.
                for ( int i = 0; i < objectCopy->getNumGeometry(); i++ )
                {
                    m_self->setGeometry( i, objectCopy->getGeometry( i ) );
                }
                HK_ASSERT( 0x22440801, m_self->getNumGeometry() == objectCopy->getNumGeometry(), "Copy failed in some way." );

                objectCopy->removeReference();
            }

            m_self->setUserColor( newColor );
            m_currentColor = newColor;
        }

        // Used in conjunction with OPERATION_OVERRIDE_VISIBILITY in render to ensure
        // that hideable objects get hidden by that mode, while non-hideable objects
        // appear as usual.
        hkColor::Argb debugColor =
            ( m_options.m_visibility == hkgVdbGeometryDisplayOptions::HIDEABLE ) ?
            hkSetBits( newColor, 1 ) :
            hkClearBits( newColor, 1 );
        m_self->setDebugColor( 0xff000000 | debugColor );
    }

    void renderWithDisplayOptions( hkgDisplayContext* context, hkArray<hkgGeometry*> const& geom, const hkBitField* visibilityMask /*= HK_NULL*/ ) const
    {
        // Make sure our user-set color is always used, this is a no-op if it's already current.
        const_cast<hkgDisplayObjectExInterface*>( this )->updateColor();

        // Use overriding material if randomized colors is on and we don't have a user-set color.
        hkgMaterial* overridingMaterial = HK_NULL;
        if (context->getDebugMaterialMode() != hkgDebugMaterialFlags::COLOR_USER)
        {
            overridingMaterial = const_cast<hkgMaterial*>(context->getMaterialOverride());
            context->getShaderContext()->setPackedDebugMaterial(m_self->getPackedDebugMaterial(context->getDebugMaterialMode()));
        }

        int ng = geom.getSize();
        for ( int i = 0; i < ng; ++i )
        {
            if ( visibilityMask && ( i < visibilityMask->getSize() ) && !visibilityMask->get( i ) )
            {
                continue;
            }
            geom[i]->render( context, overridingMaterial );
        }
    }

protected:

    hkgVdbGeometryControl& m_owner;
    hkgDisplayObject* m_self;
    hkgVdbGeometryDisplayOptions m_options;
    hkColor::Argb m_currentColor;
    hkgVdbRenderFlags m_renderFlags;
    hkDisplayGeometryFlags m_serverFlags;
};

class hkgDisplayObjectEx : public hkgDisplayObject, public hkgDisplayObjectExInterface
{
public:
    HK_DECLARE_CLASS( hkgDisplayObjectEx, New );
    hkgDisplayObjectEx(
        hkgVdbGeometryControl& owner,
        hkDisplayGeometryFlags flags,
        hkColor::Argb serverColor,
        hkColor::Argb randomColor ) :
        hkgDisplayObjectExInterface( owner, flags, serverColor, randomColor )
    {}

protected:
    virtual void renderGeomArray( hkgDisplayContext* context, hkArray<hkgGeometry*> const& geom, const hkBitField* visibilityMask /*= HK_NULL*/ ) const HK_OVERRIDE
    {
        renderWithDisplayOptions( context, geom, visibilityMask );
    }
};

class hkgInstancedDisplayObjectEx : public hkgInstancedDisplayObjectDX11, public hkgDisplayObjectExInterface
{
public:
    HK_DECLARE_CLASS( hkgDisplayObjectEx, New );
    hkgInstancedDisplayObjectEx(
        hkgDisplayContextDX11* displayContext,
        hkgVdbGeometryControl& owner,
        hkDisplayGeometryFlags flags,
        hkColor::Argb serverColor,
        hkColor::Argb randomColor ) :
        hkgInstancedDisplayObjectDX11( displayContext ),
        hkgDisplayObjectExInterface( owner, flags, serverColor, randomColor )
    {
        
        
        
        
        setRenderMode( HKG_INSTANCED_DISPLAY_CPU );
    }

protected:
    virtual void renderGeomArray( hkgDisplayContext* context, hkArray<hkgGeometry*> const& geom, const hkBitField* visibilityMask /*= HK_NULL*/ ) const HK_OVERRIDE
    {
        renderWithDisplayOptions( context, geom, visibilityMask );
    }
};

class hkgWireframeDisplayObjectEx : public hkgWireframeDisplayObject, public hkgDisplayObjectExInterface
{
public:
    HK_DECLARE_CLASS( hkgWireframeDisplayObjectEx, New );
    hkgWireframeDisplayObjectEx(
        hkgVdbGeometryControl& owner,
        hkDisplayGeometryFlags flags,
        hkColor::Argb serverColor,
        hkColor::Argb randomColor ) :
        hkgDisplayObjectExInterface( owner, flags, serverColor, randomColor )
    {}

protected:
    virtual void renderGeomArray( hkgDisplayContext* context, hkArray<hkgGeometry*> const& geom, const hkBitField* visibilityMask /*= HK_NULL*/ ) const HK_OVERRIDE
    {
        renderWithDisplayOptions( context, geom, visibilityMask );
    }
};

namespace
{
    HK_INLINE hkgDisplayObjectExInterface* getEx( hkgDisplayObject* object )
    {
        if ( object )
        {
            if ( object->getStatusFlags() & HKG_DISPLAY_OBJECT_INSTANCED )
            {
                hkgInstancedDisplayObjectEx* objectEx = static_cast<hkgInstancedDisplayObjectEx*>( object );
                return static_cast<hkgDisplayObjectExInterface*>( objectEx );
            }
            else if ( object->getStatusFlags() & HKG_DISPLAY_OBJECT_WIREFRAME )
            {
                hkgWireframeDisplayObjectEx* objectEx = static_cast< hkgWireframeDisplayObjectEx* >( object );
                return static_cast< hkgDisplayObjectExInterface* >( objectEx );
            }
            else
            {
                hkgDisplayObjectEx* objectEx = static_cast< hkgDisplayObjectEx* >( object );
                return static_cast< hkgDisplayObjectExInterface* >( objectEx );
            }
        }
        return HK_NULL;
    }

    hkgDisplayObject* buildDisplayObjectFromDisplayGeometries(
        hkgDisplayContext& displayContext,
        hkgVdbGeometryControl& owner,
        const hkArrayBase<hkDisplayGeometry*>& geometries,
        hkDisplayGeometryFlags flags,
        hkColor::Argb serverColor,
        hkColor::Argb randomColor )
    {
        // Create hkg object
        hkgDisplayObject* object;
        if ( flags & hkDisplayGeometryFlags::PARTICLE_TEMPLATE )
        {
            hkgDisplayContextDX11& displayContextDx11 = static_cast<hkgDisplayContextDX11&>(displayContext);
            hkgInstancedDisplayObjectEx* instanced = new hkgInstancedDisplayObjectEx(
                &displayContextDx11,
                owner,
                flags,
                serverColor,
                randomColor );
            instanced->setMaxNumObjects( 1 );
            instanced->setNumObjects( 1 );
            object = instanced;
        }
        else
        {
            object = new hkgDisplayObjectEx(
                owner,
                flags,
                serverColor,
                randomColor );
        }
        hkgGeometryConverter::applyDisplayGeometryFlags( object, flags );

        if ( hkgGeometry* graphicsDisplayGeometry = hkgGeometryConverter::createCombinedGraphicsDisplay(
                geometries,
                &displayContext,
                flags,
                
                HK_NULL,
                1.0f,
                false,
                &serverColor ) )
        {
            object->addGeometry( graphicsDisplayGeometry );
            graphicsDisplayGeometry->removeReference();
            return object;
        }
        else
        {
            object->removeReference();
            return HK_NULL;
        }
    }

    hkgDisplayObject* buildWireframeObjectFromDisplayGeometries(
        hkgDisplayContext& displayContext,
        hkgVdbGeometryControl& owner,
        const hkArrayBase<hkDisplayGeometry*>& geometries,
        hkDisplayGeometryFlags flags,
        hkColor::Argb serverColor,
        hkColor::Argb randomColor )
    {
        hkgWireframeDisplayObjectEx* wfdo = new hkgWireframeDisplayObjectEx(
            owner,
            flags,
            serverColor,
            randomColor );
        hkgGeometryConverter::applyDisplayGeometryFlags( wfdo, flags );
        for ( int i = 0; i < geometries.getSize(); ++i )
        {
            if ( geometries[i]->getType() == HK_DISPLAY_WIREFRAME )
            {
                const hkDisplayWireframe* displayWireframe = static_cast< hkDisplayWireframe* >( geometries[i] );
                wfdo->appendDisplayLines( displayWireframe->m_lines );
            }
        }
        return wfdo;
    }

    hkgDisplayObject* buildInstanceDisplayObjectFromSource(
        hkgDisplayContext& displayContext,
        hkgVdbGeometryControl& owner,
        hkgDisplayObject& sourceObject,
        hkColor::Argb serverColor,
        hkColor::Argb randomColor )
    {
        hkgDisplayObject* object = new hkgDisplayObjectEx(
            owner,
            getEx( &sourceObject )->getServerFlags(),
            serverColor,
            randomColor );
        // Heterogeneous batching will cause an explosion of memory since the vertices will be
        // copied for *all* these objects sharing geometry, disable for now until we can get memory down.
        
        
        
        
        object->setStatusMutabilityFlags( HKG_DISPLAY_OBJECT_DYNAMIC );
        for ( int i = 0; i < sourceObject.getNumGeometry(); i++ )
        {
            object->addGeometry( sourceObject.getGeometry( i ) );
        }
        object->bake( &displayContext );
        return object;
    }

    hkgDisplayObject* buildInstanceWireframeObjectFromSource(
        hkgDisplayContext& displayContext,
        hkgVdbGeometryControl& owner,
        hkgDisplayObject& sourceObject,
        hkColor::Argb serverColor,
        hkColor::Argb randomColor )
    {
        // Wireframe bodies must be added specially.
        hkgWireframeDisplayObject& wfdo = static_cast< hkgWireframeDisplayObject& >( sourceObject );
        hkgWireframeDisplayObjectEx* newWfdo = new hkgWireframeDisplayObjectEx(
            owner,
            getEx( &sourceObject )->getServerFlags(),
            serverColor,
            randomColor );
        // Heterogeneous batching will cause an explosion of memory since the vertices will be
        // copied for *all* these objects sharing geometry, disable for now until we can get memory down.
        newWfdo->setStatusMutabilityFlags( HKG_DISPLAY_OBJECT_DYNAMIC );
        newWfdo->appendDisplayLines( wfdo.getDisplayLines() );
        return newWfdo;
    }

    hkBool32 filterBySelection( hkUint64 id, const hkgDisplayObject& object, void* handle )
    {
        hkTuple<const hkVdbMap<hkUint64, hkRefPtr<hkgDisplayObject> >&, const hkVdbMap<hkUint64, hkRefPtr<hkgDisplayObject> >&, hkgVdbGeometryControl::SaveSelectionOnlyOptions>* tuple =
        reinterpret_cast< hkTuple<const hkVdbMap<hkUint64, hkRefPtr<hkgDisplayObject> >&, const hkVdbMap<hkUint64, hkRefPtr<hkgDisplayObject> >&, hkgVdbGeometryControl::SaveSelectionOnlyOptions>* >( handle );

        // Check highlighted
        if ( ( tuple->m_2 == hkgVdbGeometryControl::SaveSelectionOnlyOptions::SaveSelectedAndHighlighted ) ||
            ( tuple->m_2 == hkgVdbGeometryControl::SaveSelectionOnlyOptions::SaveHighlighted ) )
        {
            if ( tuple->m_0.contains( id ) )
            {
                return true;
            }
        }

        // Check selected
        if ( ( tuple->m_2 == hkgVdbGeometryControl::SaveSelectionOnlyOptions::SaveSelectedAndHighlighted ) ||
            ( tuple->m_2 == hkgVdbGeometryControl::SaveSelectionOnlyOptions::SaveSelected ) )
        {
            if ( tuple->m_1.contains( id ) )
            {
                return true;
            }
        }

        return false;
    }
    hkBool32 saveSingleFuncMarker( hkUint64 id, const hkgDisplayObject& object, void* handle ) { return false; }
};

hkgVdbGeometryControl::GeometryFilterFunc hkgVdbGeometryControl::SaveSelectionOnly = HK_NULL;
hkgVdbGeometryControl::GeometryFilterFunc hkgVdbGeometryControl::SaveSingleGeometry = HK_NULL;

hkgVdbGeometryControl::hkgVdbGeometryControl( hkgWindow& window ) :
    hkVdbDefaultErrorReporter( &s_debugLog ),
    m_window( &window ),
    m_renderFlags( hkgVdbRenderFlags::DEFAULT ),
    m_displayWorld( hkRefNew<hkgDisplayWorld>( hkgDisplayWorld::create( window.getContext() ) ) )
{
    clearError();
    hkgVdbGeometryControl::SaveSelectionOnly = &filterBySelection;
    hkgVdbGeometryControl::SaveSingleGeometry = &saveSingleFuncMarker;
}

hkgVdbGeometryControl::~hkgVdbGeometryControl()
{}

hkResult hkgVdbGeometryControl::registerSelf( hkgVdbPlugin& plugin, hkVdbDisplayHandler& handler, hkVdbClient& client )
{
    HK_SUBSCRIBE_TO_SIGNAL( client.m_disconnected, this, hkgVdbGeometryControl );
    HK_SUBSCRIBE_TO_SIGNAL( client.m_connected, this, hkgVdbGeometryControl );
    HK_SUBSCRIBE_TO_SIGNAL( handler.m_displayOptionsSet, this, hkgVdbGeometryControl );
    HK_SUBSCRIBE_TO_SIGNAL( handler.m_geometryCmdReceived, this, hkgVdbGeometryControl );
    HK_SUBSCRIBE_TO_SIGNAL( handler.m_waitForCompletion, this, hkgVdbGeometryControl );
    HK_SUBSCRIBE_TO_SIGNAL( client.getCmdHandler<hkVdbPlaybackHandler>()->m_playbackInfoReceived, this, hkgVdbGeometryControl );
    m_selectionWidget = plugin.getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>();
    HK_SUBSCRIBE_TO_SIGNAL( m_selectionWidget->m_geometrySelectionChanged, this, hkgVdbGeometryControl );
    m_displayHandler = &handler;
    return HK_SUCCESS;
}

hkResult hkgVdbGeometryControl::unregisterSelf( hkgVdbPlugin& plugin, hkVdbDisplayHandler& handler, hkVdbClient& client )
{
    client.m_disconnected.unsubscribeAll( this );
    client.m_connected.unsubscribeAll( this );
    handler.m_displayOptionsSet.unsubscribeAll( this );
    handler.m_geometryCmdReceived.unsubscribeAll( this );
    handler.m_waitForCompletion.unsubscribeAll( this );
    client.getCmdHandler<hkVdbPlaybackHandler>()->m_playbackInfoReceived.unsubscribeAll( this );
    m_selectionWidget->m_geometrySelectionChanged.unsubscribeAll( this );
    m_selectionWidget = HK_NULL;
    m_displayHandler = HK_NULL;
    removeDisplayObjects();
    return HK_SUCCESS;
}

void hkgVdbGeometryControl::render( hkgViewport& viewport ) const
{
    hkgDisplayContext& context = *m_window->getContext();
    context.lock();

    if ( m_idToObjectMap.getSize() )
    {
        // Apply render flags
        {
            // Determine debug material flags
            hkgDebugMaterialFlags flags = 0;
            {
                // If we are hidden, we use override visibility operation in conjunction with the hkgDisplayObjectExInterface's use of debug color
                // to hide any objects which should be affected by the global hide.
                flags.orWith( bool( m_renderFlags.anyIsSet( hkgVdbRenderFlags::HIDE_ALL ) ) * ( hkgDebugMaterialFlags::COLOR_DEBUG | hkgDebugMaterialFlags::OPERATION_OVERRIDE_VISIBILITY ) );

                // If we are randomized, then use random color with heavy saturation.
                flags.orWith( bool( !flags && m_renderFlags.anyIsSet( hkgVdbRenderFlags::RANDOMIZED_COLORS ) ) * ( hkgDebugMaterialFlags::COLOR_RANDOM | hkgDebugMaterialFlags::MODIFIER_SATURATE ) );

                // Default to user color if aren't doing anything special.
                // Note: this is actually a no-op no matter what because COLOR_USER == 0, but it may change down the road.
                flags.orWith( bool( !flags ) * hkgDebugMaterialFlags::COLOR_USER );

                // If we have any highlight/selection (and we've enabled selection coloring), we should modify the color with minimizing effect.
                // Selected items get their own debug material flags to help them pop to the front, see onGeometrySelectionChangedSignal.
                flags.orWith(
                    bool(
                        ( m_selectionWidget->getHighlightedGeometries().getSize() || m_selectionWidget->getSelectedGeometries().getSize()
                            || m_selectionWidget->getHighlightedDisplays().getSize() || m_selectionWidget->getSelectedDisplays().getSize() )
                        && m_renderFlags.anyIsSet( hkgVdbRenderFlags::VISUALIZE_GEOMETRY_SELECTION ) )
                    * ( hkgDebugMaterialFlags::MODIFIER_DESATURATE | hkgDebugMaterialFlags::MODIFIER_DARKEN ) );
            }

            // Setup the context
            context.setDebugMaterialMode( flags );
            context.setWireframeState( m_renderFlags.anyIsSet( hkgVdbRenderFlags::WIREFRAME ) );
            context.setCullfaceMode( m_renderFlags.anyIsSet( hkgVdbRenderFlags::CULLMODE_CCW ) ? HKG_CULLFACE_CCW : HKG_CULLFACE_CW );
            context.setCullFaceState( m_renderFlags.anyIsSet( hkgVdbRenderFlags::BACKFACE_CULLING ) );
        }

        if ( m_renderFlags.anyIsSet( hkgVdbRenderFlags::OUTLINE_FACES ) )
        {
            // Get current state values
            bool wasDepthRead = ( context.getEnabledState() & HKG_ENABLED_ZREAD );
            bool wasDepthWrite = ( context.getEnabledState() & HKG_ENABLED_ZWRITE );
            bool wasLighting = ( context.getEnabledState() & HKG_ENABLED_LIGHTING );
            bool wasWireframe = ( context.getEnabledState() & HKG_ENABLED_WIREFRAME );

            // Enable Z read & Z write
            context.setDepthReadState( true );
            context.setDepthWriteState( true );

            // Render solid polys but with a +Z bias
            context.setDepthBias( +12.0f );
            m_displayWorld->render( &context, true, false );

            // Wireframe mode
            context.setLightingState( false );
            context.setWireframeState( true );

            // Enable Z read only and draw pixels nearer than the current z buff value
            context.setDepthReadState( true );
            context.setDepthWriteState( false );

            // Render wireframe polys but with a -Z bias
            context.setDepthBias( -12.0f );
            m_displayWorld->render( &context, true, false, true );

            // Reset
            context.setDepthBias( 0.0f );
            context.setDepthReadState( wasDepthRead );
            context.setDepthWriteState( wasDepthWrite );
            context.setLightingState( wasLighting );
            context.setWireframeState( wasWireframe );
        }
        else
        {
            m_displayWorld->render( &context, true, false );
        }
    }

    context.unlock();
}

hkResult hkgVdbGeometryControl::setRenderFlags( hkgVdbRenderFlags flags )
{
    hkgDisplayContext& context = *m_window->getContext();
    context.lock();
    {
        
        m_renderFlags = flags;
        for ( IdMapIt iter = m_idToObjectMap.getIterator();
            m_idToObjectMap.isValid( iter );
            iter = m_idToObjectMap.getNext( iter ) )
        {
            if ( hkgDisplayObjectExInterface* ex = getEx( m_idToObjectMap.getValue( iter ) ) )
            {
                ex->setGlobalRenderFlags( flags );
            }
        }
    }
    context.unlock();
    return HK_SUCCESS;
}

hkColor::Argb hkgVdbGeometryControl::DEFAULT_COLOR = hkColor::WHITESMOKE;

hkResult hkgVdbGeometryControl::setGeometryDisplayOptions( hkUint64 id, const hkgVdbGeometryDisplayOptions& options/*, int viewportIdx */ )
{
    hkResult result = HK_FAILURE;
    hkgDisplayContext& context = *m_window->getContext();
    context.lock();
    {
        if ( hkgDisplayObject* object = m_idToObjectMap.getWithDefault( id, HK_NULL ) )
        {
            getEx( object )->setOptions( /*viewportIdx, */options );
            result = HK_SUCCESS;
        }
    }
    context.unlock();
    return result;
}

const hkgVdbGeometryDisplayOptions* hkgVdbGeometryControl::getGeometryDisplayOptions( hkUint64 id /*, int viewportIdx */ ) const
{
    const hkgVdbGeometryDisplayOptions* result = HK_NULL;
    hkgDisplayContext& context = *m_window->getContext();
    context.lock();
    {
        if ( hkgDisplayObject* object = m_idToObjectMap.getWithDefault( id, HK_NULL ) )
        {
            result = &getEx( object )->getOptions();
        }
    }
    context.unlock();
    return result;
}

hkResult hkgVdbGeometryControl::saveGeometry( const char* filepath, GeometryFilterFunc filter, void* filterHandle )
{
    hkRefPtr<hkStreamWriter> writer = hkFileSystem::getInstance().openWriter( filepath );
    HK_VDB_VERIFY_CONDITION_MSG(
        writer,
        0xadb00047,
        hkVdbError::PLUGIN_WRITE_ERROR,
        "Could not open file \"" << filepath << "\" for write" );

    hkRefPtr<hkxScene> scene;
    if ( filter == SaveSingleGeometry )
    {
        
        // The below doesn't work, possibly because of double-inheritance and hkgDisplayGeometryEx
        // if ( hkgDisplayObject* dispObject = hkDynCast( filterHandle ) )
        if ( filterHandle )
        {
            hkgDisplayObject* dispObject = ( hkgDisplayObject* ) filterHandle;
            hkgDisplayContext& context = *m_window->getContext();
            context.lock();
            hkPointerMap< int, hkxMaterial*> materialMap;
            hkRefPtr<hkxMesh> mesh = hkgVdbObjectConversionUtil::createHkxMesh( dispObject, materialMap );
            hkMatrix4 meshT; meshT.set4x4ColumnMajor( dispObject->getTransform() );
            context.unlock();
            scene = hkgVdbObjectConversionUtil::createHkxScene(
                hkArrayViewT::fromSingleObject( mesh ),
                hkArrayViewT::fromSingleObject( meshT ) );
        }
        else
        {
            HK_ASSERT( 0x22441393, false, "No geometry provided for SaveSingleGeometry func" );
        }
    }
    else
    {
        // Not using auto here so that we get compile error when this changes (and therefore need to change SaveSelectionOnly function)
        hkTuple<const hkVdbMap<hkUint64, hkRefPtr<hkgDisplayObject> >&, const hkVdbMap<hkUint64, hkRefPtr<hkgDisplayObject> >&, SaveSelectionOnlyOptions>
            selectionTuple(
                m_selectionWidget->getHighlightedGeometries(),
                m_selectionWidget->getSelectedGeometries(),
                // Default is both
                SaveSelectionOnlyOptions::SaveSelectedAndHighlighted );
        if ( filter == SaveSelectionOnly )
        {
            if ( filterHandle ) selectionTuple.m_2 = *reinterpret_cast< SaveSelectionOnlyOptions* >( filterHandle );
            filterHandle = &selectionTuple;
        }

        hkgDisplayContext& context = *m_window->getContext();
        context.lock();
        hkLocalArray<hkRefPtr<hkxMesh>> meshes( m_displayWorld->getNumDisplayObjects() );
        hkPointerMap< int, hkxMaterial*> materialMap;
        hkLocalArray<hkMatrix4> meshesTs( m_displayWorld->getNumDisplayObjects() );
        {
            for ( int i = 0; i < m_displayWorld->getNumDisplayObjects(); i++ )
            {
                if ( hkgDisplayObject* dispObject = m_displayWorld->getDisplayObject( i ) )
                {
                    hkUint64 id = dispObject->m_id;
                    if ( !filter || filter( id, *dispObject, filterHandle ) )
                    {
                        meshes.expandOne() = hkgVdbObjectConversionUtil::createHkxMesh( dispObject, materialMap );
                        meshesTs.expandOne().set4x4ColumnMajor( dispObject->getTransform() );
                    }
                }
            }
        }
        context.unlock();
        scene = hkgVdbObjectConversionUtil::createHkxScene( meshes, meshesTs );
    }

    if ( scene )
    {
        return hkgVdbObjectConversionUtil::writeHkxScene( *writer, scene );
    }
    else
    {
        return HK_FAILURE;
    }
}

hkUint64 hkgVdbGeometryControl::findGeometryAt( hkgViewport& viewport, hkgVdbPoint p, hkVector4* worldSpaceHitPointOut, hkVector4* localSpaceHitPointOut ) const
{
    hkUint64 result = 0;
    hkgDisplayContext& context = *m_window->getContext();
    context.lock();
    {
        hkgViewportPickData dataOut;
        if ( viewport.pick( p.m_x, p.m_y, m_displayWorld, dataOut ) )
        {
            HK_ALIGN16( float hkgvec[3] );
            if ( worldSpaceHitPointOut )
            {
                hkgVec3Copy( hkgvec, dataOut.getClosestObjectWorldPos() );
                worldSpaceHitPointOut->load<3>( hkgvec );
            }
            if ( localSpaceHitPointOut )
            {
                hkgVec3Copy( hkgvec, dataOut.getClosestGeometryLocalPos() );
                localSpaceHitPointOut->load<3>( hkgvec );
            }
            const int objectIdx = dataOut.getClosestObjectIndex();
            result = m_displayWorld->getDisplayObject( objectIdx )->m_id;
        }
    }
    context.unlock();
    return result;
}

hkgDisplayObject* hkgVdbGeometryControl::getGeometryFromId( hkUint64 id ) const
{
    hkgDisplayContext& context = *m_window->getContext();
    context.lock();
    hkgDisplayObject* displayObject = m_idToObjectMap.getWithDefault( id, HK_NULL );
    context.unlock();
    return displayObject;
}

void hkgVdbGeometryControl::onDisconnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    // Clear our cmd queue if it has lingering data
    processGeometryCmdQueue();
}

void hkgVdbGeometryControl::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    // Clear our cmd queue so any cmds from connection will get purged
    processGeometryCmdQueue();
}

void hkgVdbGeometryControl::onDisplayOptionsSetSignal( const hkDebugDisplayHandler::Options& options, hkVdbSignalResult& result )
{
    
    
}

void hkgVdbGeometryControl::onGeometryCmdReceivedSignal( const hkVdbDisplayHandler::GeometryCmd& geomCmd, hkVdbSignalResult& result )
{
    
    if ( const hkVdbDisplayHandler::AddGeometryCmd* addCmd = geomCmd.asAddCmd() )
    {
        m_cmdQueue.expandOne() = *addCmd;
    }
    else if ( const hkVdbDisplayHandler::InstanceGeometryCmd* instanceCmd = geomCmd.asInstanceCmd() )
    {
        m_cmdQueue.expandOne() = *instanceCmd;
    }
    else if ( const hkVdbDisplayHandler::UpdateGeometryCmd* updateCmd = geomCmd.asUpdateCmd() )
    {
        m_cmdQueue.expandOne() = *updateCmd;
    }
    else if ( const hkVdbDisplayHandler::RemoveGeometryCmd* removeCmd = geomCmd.asRemoveCmd() )
    {
        m_cmdQueue.expandOne() = *removeCmd;
    }
    else if ( const hkVdbDisplayHandler::DisposeGeometryCmd* disposeCmd = geomCmd.asDisposeCmd() )
    {
        m_cmdQueue.expandOne() = *disposeCmd;
    }
    else
    {
        HK_VDB_SIGNAL_ERROR_MSG( 0xadb00002, hkVdbError::PLUGIN_ERROR, "Unknown geometry cmd" );
        result.signalError( *this );
    }
}

void hkgVdbGeometryControl::onPlaybackInfoReceivedSignal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result )
{
    if ( info.flagWasSet( hkVdbPlaybackFlags::FRAME_ENDED ) )
    {
        if ( processGeometryCmdQueue().isFailure() ) result.signalError( *this );
    }
}

void hkgVdbGeometryControl::onGeometrySelectionChangedSignal( const hkgVdbSelectionWidget::SelectionSets& sets, const hkgVdbSelectionWidget::SelectionChangedInfo& info )
{
    // Order here matters if there was something was removed from selected AND highlighted sets at the same time.
    setSelection( info.m_removedFromSelected, sets.m_highlighted.getSize() ? SelectionMode::HIGHLIGHTED : SelectionMode::NONE );
    setSelection( info.m_removedFromHighlighted, SelectionMode::NONE );
    setSelection( info.m_addedToHighlighted, SelectionMode::HIGHLIGHTED );
    setSelection( info.m_addedToSelected, SelectionMode::SELECTED );
}

void hkgVdbGeometryControl::onWaitForCompletionSignal()
{
    
}

hkResult hkgVdbGeometryControl::processGeometryCmdQueue()
{
    bool succeeded = true;
    hkgDisplayContext& context = *m_window->getContext();
    context.lock();
    {
        for ( int i = 0; i < m_cmdQueue.getSize(); i++ )
        {
            hkVdbDisplayHandler::GeometryCmd& geomCmd = m_cmdQueue[i];
            succeeded &= ( processGeometryCmd( geomCmd ).isSuccess() );
        }
        m_cmdQueue.clear();
    }
    context.unlock();
    return succeeded ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkgVdbGeometryControl::processGeometryCmd( const hkVdbDisplayHandler::GeometryCmd& geomCmd )
{
    if ( const hkVdbDisplayHandler::AddGeometryCmd* addCmd = geomCmd.asAddCmd() )
    {
        return processAddGeomCmd( *addCmd );
    }
    else if ( const hkVdbDisplayHandler::InstanceGeometryCmd* instanceCmd = geomCmd.asInstanceCmd() )
    {
        return processInstanceGeomCmd( *instanceCmd );
    }
    else if ( const hkVdbDisplayHandler::UpdateGeometryCmd* updateCmd = geomCmd.asUpdateCmd() )
    {
        return processUpdateGeomCmd( *updateCmd );
    }
    else if ( const hkVdbDisplayHandler::RemoveGeometryCmd* removeCmd = geomCmd.asRemoveCmd() )
    {
        return processRemoveGeomCmd( *removeCmd );
    }
    else if ( const hkVdbDisplayHandler::DisposeGeometryCmd* disposeCmd = geomCmd.asDisposeCmd() )
    {
        return processDisposeGeomCmd( *disposeCmd );
    }
    else
    {
        HK_ASSERT( 0x22441124, false, "Unknown delayed cmd" );
        HK_VDB_SIGNAL_ERROR_MSG( 0xadb00028, hkVdbError::PLUGIN_ERROR, "Unknown geometry cmd" );
        return HK_FAILURE;
    }
}

hkResult hkgVdbGeometryControl::processAddGeomCmd( const hkVdbDisplayHandler::AddGeometryCmd& addCmd )
{
    hkBool32 hasNonWireframe = false;
    for ( int i = 0; i < addCmd.m_geometry.getSize(); i++ )
    {
        if ( addCmd.m_geometry[i]->getType() != HK_DISPLAY_WIREFRAME )
        {
            hasNonWireframe = true;
            break;
        }
    }

    hkColor::Argb color;
    hkDisplayGeometryFlags flags;
    if ( const hkVdbDisplayHandler::AddGeometryCmd::Ex* addCmdEx = addCmd.getExtended() )
    {
        color = addCmdEx->m_color;
        if ( color == hkColor::NONE ) color = DEFAULT_COLOR;
        flags = addCmdEx->m_flags;
    }
    else
    {
        color = DEFAULT_COLOR;
        flags = hkDisplayGeometryFlags::DEFAULT;
    }

    // Generate a random color based on object ID
    hkColor::Argb randomColor;
    {
        const hkUint32 seed = hkUint32( addCmd.m_id ) * 1000;
        hkPseudoRandomGenerator rand( seed );
        randomColor = hkColor::getRandomColor( rand );
    }

    hkgDisplayObject* object;
    if ( hasNonWireframe )
    {
        object = buildDisplayObjectFromDisplayGeometries(
            *m_window->getContext(),
            *this,
            addCmd.m_geometry,
            flags,
            color,
            randomColor );
    }
    else
    {
        object = buildWireframeObjectFromDisplayGeometries(
            *m_window->getContext(),
            *this,
            addCmd.m_geometry,
            flags,
            color,
            randomColor );
    }

    if ( object )
    {
        addAndTrackDisplayObject( *object, addCmd.m_transform, addCmd.m_id, addCmd.m_tag );
        getEx( object )->postInit( *object );
    }

    return HK_SUCCESS;
}

hkResult hkgVdbGeometryControl::processInstanceGeomCmd( const hkVdbDisplayHandler::InstanceGeometryCmd& instanceCmd )
{
    if ( hkgDisplayObject* sourceObject = m_idToObjectMap.getWithDefault( instanceCmd.m_sourceGeometryId, HK_NULL ) )
    {
        if ( ( sourceObject->getStatusFlags() & HKG_DISPLAY_OBJECT_INSTANCED ) != 0 )
        {
            HK_VDB_SIGNAL_ERROR_MSG( 0xadb00102, hkVdbError::PLUGIN_ERROR, "Cannot explicitly add instances to instanced objects, use UPDATE_PARTICLE_TRANSFORMS instead" );
            return HK_FAILURE;
        }

        // Need to note the current server color
        hkColor::Argb serverColor;
        sourceObject->getUserColor( serverColor );

        // Generate a random color based on object ID
        hkColor::Argb randomColor;
        {
            const hkUint32 seed = hkUint32( instanceCmd.m_id ) * 1000;
            hkPseudoRandomGenerator rand( seed );
            randomColor = hkColor::getRandomColor( rand );
        }

        hkgDisplayObject* object;
        if ( sourceObject->getStatusFlags() & HKG_DISPLAY_OBJECT_WIREFRAME )
        {
            object = buildInstanceWireframeObjectFromSource(
                *m_window->getContext(),
                *this,
                *sourceObject,
                serverColor,
                randomColor );
        }
        else
        {
            object = buildInstanceDisplayObjectFromSource(
                *m_window->getContext(),
                *this,
                *sourceObject,
                serverColor,
                randomColor );
        }

        if ( object )
        {
            addAndTrackDisplayObject(
                *object,
                instanceCmd.m_transform,
                instanceCmd.m_id,
                instanceCmd.m_tag );
            getEx( object )->postInit( *object );
        }

        return HK_SUCCESS;
    }

    HK_VDB_SIGNAL_ERROR_MSG( 0xadb00029, hkVdbError::PLUGIN_ERROR, "Unknown geometry instanced id" );
    return HK_FAILURE;
}

hkResult hkgVdbGeometryControl::processUpdateGeomCmd( const hkVdbDisplayHandler::UpdateGeometryCmd& updateCmd )
{
    if ( hkgDisplayObject* object = m_idToObjectMap.getWithDefault( updateCmd.m_id, HK_NULL ) )
    {
        // Do flags/alpha first since they can affect visibility and some hkg functions are optimized
        // when hkgDisplayObject is currently in the world (eg. setting color can use immutable/shared material)
        if ( const hkDisplayGeometryFlags* flagBitsToSet = updateCmd.getSetFlagBits() )
        {
            getEx( object )->setServerFlagBits( *flagBitsToSet );
        }
        if ( const hkDisplayGeometryFlags* flagBitsToClear = updateCmd.getClearFlagBits() )
        {
            getEx( object )->clearServerFlagBits( *flagBitsToClear );
        }
        if ( const hkReal* alpha = updateCmd.getAlpha() )
        {
            hkColor::Argb currentColor;
            if ( object->getUserColor( currentColor ).isSuccess() )
            {
                if ( hkColor::getAlphaAsFloat( currentColor ) != *alpha )
                {
                    hkColor::Argb newColor = hkColor::replaceAlpha( ( unsigned char ) ( *alpha * 255 ), currentColor );
                    getEx( object )->setServerColor( newColor );
                }
            }
            else
            {
                HK_ASSERT( 0x22441006, false, "Could not get object color" );
            }
        }

        // Everything else...
        if ( const hkColor::Argb* color = updateCmd.getColor() )
        {
            getEx( object )->setServerColor(
                ( *color != hkColor::NONE ) ? *color : DEFAULT_COLOR );
        }
        if ( const hkMatrix4* transform = updateCmd.getTransform() )
        {
            HK_ALIGN16( float cm16[16] );
            transform->get4x4ColumnMajor( cm16 );
            object->setTransform( cm16 );
        }
        if ( const hkArray<hkVector4>* verts = updateCmd.getVerts() )
        {
            hkgGeometry* geom = object->getGeometry( 0 );
            hkgMaterialFaceSet* materialSet = geom->getMaterialFaceSet( 0 );
            hkgFaceSet* faceSet = materialSet->getFaceSet( 0 );
            hkgVertexSet* vertexSet = faceSet->getVertexSet();

            if ( vertexSet->m_vertexMapping )
            {
                vertexSet->lock( HKG_LOCK_READWRITE );

                const hkArray< hkArray<int> >& map = *vertexSet->m_vertexMapping;
                for ( int v = 0; v < verts->getSize(); v++ )
                {
                    for ( int m = 0; m < map[v].getSize(); m++ )
                    {
                        vertexSet->setVertexComponentData( HKG_VERTEX_COMPONENT_POS, map[v][m], verts->begin() + v );
                    }
                }

                vertexSet->unlock();

                object->recomputeNormals();
                object->computeAABB();
            }
        }
        if ( const hkArray<hkTransform>* particleTransforms = updateCmd.getParticleTransforms() )
        {
            if ( (object->getStatusFlags() & HKG_DISPLAY_OBJECT_INSTANCED) == 0 )
            {
                HK_VDB_SIGNAL_ERROR_MSG( 0xadb00100, hkVdbError::PLUGIN_ERROR, "Cannot update display objects with particle transforms that weren't specified as particles" );
                return HK_FAILURE;
            }

            hkgInstancedDisplayObject* instanced = static_cast<hkgInstancedDisplayObject*>( object );
            hkUint32 numTransforms = particleTransforms->getSize();
            instanced->setNumObjects( numTransforms );

            // Update transforms
            for ( hkUint32 i = 0; i < numTransforms; i++ )
            {
                instanced->setTransformAligned( (*particleTransforms)[i], i );
                instanced->setInstanceColor( *(hkColor::Argb*)&(*particleTransforms)[i].getTranslation()(3), i );
            }
        }
        if ( const int* particleUpdateId = updateCmd.getParticleUpdateId() )
        {
            if ( (object->getStatusFlags() & HKG_DISPLAY_OBJECT_INSTANCED) == 0 )
            {
                HK_VDB_SIGNAL_ERROR_MSG( 0xadb00103, hkVdbError::PLUGIN_ERROR, "Cannot set particle update id for objects that weren't specified as particles" );
                return HK_FAILURE;
            }

            hkgInstancedDisplayObject* instanced = static_cast<hkgInstancedDisplayObject*>(object);
            instanced->m_userPtr = (void*)(hkUint64)*particleUpdateId;
        }
        return HK_SUCCESS;
    }

    HK_VDB_SIGNAL_ERROR_MSG( 0xadb00030, hkVdbError::PLUGIN_ERROR, "Unknown geometry updated id" );
    return HK_FAILURE;
}

hkResult hkgVdbGeometryControl::processRemoveGeomCmd( const hkVdbDisplayHandler::RemoveGeometryCmd& removeCmd )
{
    if ( hkgDisplayObject* object = m_idToObjectMap.getWithDefault( removeCmd.m_id, HK_NULL ) )
    {
        // We only remove from world at this point, because geometry is still in circular buffer
        // (see DisposeGeometryCmd handling).
        getEx( object )->markRemoved();
        removeDisplayObjectFromWorld( *object );
        return HK_SUCCESS;
    }

    HK_VDB_SIGNAL_ERROR_MSG( 0xadb00031, hkVdbError::PLUGIN_ERROR, "Unknown geometry removed id" );
    return HK_FAILURE;
}

hkResult hkgVdbGeometryControl::processDisposeGeomCmd( const hkVdbDisplayHandler::DisposeGeometryCmd& disposeCmd )
{
    IdMapIt iter = m_idToObjectMap.findKey( disposeCmd.m_id );
    if ( m_idToObjectMap.isValid( iter ) )
    {
        // We are done with the geometry, completely remove it.
        removeDisplayObject( iter );
        m_idToObjectMap.remove( iter );
        return HK_SUCCESS;
    }

    HK_VDB_SIGNAL_ERROR_MSG( 0xadb00032, hkVdbError::PLUGIN_ERROR, "Unknown geometry disposed id" );
    return HK_FAILURE;
}

void hkgVdbGeometryControl::addAndTrackDisplayObject(
    hkgDisplayObject& object,
    const hkMatrix4& transform,
    hkUint64 id,
    int tag )
{
    HK_ASSERT( 0x22440999, id != 0, "Id of zero is reserved as an invalid id" );

    // initialize the transform of the display object
    {
        HK_ALIGN16( float cm16[16] );
        transform.get4x4ColumnMajor( cm16 );
        object.setTransform( cm16 );
    }

    // Setup the mapping between id and object
    {
        IdMapIt iter = m_idToObjectMap.findOrInsertKey( id, &object );
        if ( m_idToObjectMap.getValue( iter ) != &object )
        {
            HK_WARN( 0xd9881eac, "Duplicate IDs added. Overwriting existing geometry." );
            removeDisplayObject( iter );
            m_idToObjectMap.setValue( iter, &object );
        }
        object.m_id = id;
    }

    addDisplayObjectToWorld( object );

    m_geometryAdded.fire( object );
}

void hkgVdbGeometryControl::removeDisplayObjects()
{
    // Check that everything there aren't extra objects in the display world.
    // This should be the case if we aren't sharing the world.
#if 0
    HK_ASSERT( 0x22441076, m_displayWorld->getNumDisplayObjects() <= m_idToObjectMap.getSize(), "Unexpected display world objects" );
#endif

    
    
    for ( IdMapIt iter = m_idToObjectMap.getIterator();
        m_idToObjectMap.isValid( iter );
        iter = m_idToObjectMap.getNext( iter ) )
    {
        removeDisplayObject( iter );
    }

    // Check that everything is gone from the world.
    // This should be the case if we aren't sharing the world.
#if 0
    HK_ASSERT( 0x22441077, m_displayWorld->getNumDisplayObjects() == 0, "We have display objects" );
#endif

    m_idToObjectMap.clear();
}

void hkgVdbGeometryControl::removeDisplayObject( IdMapIt iter )
{
    hkgDisplayObject* object = m_idToObjectMap.getValue( iter );
    removeDisplayObjectFromWorld( *object );
    m_geometryRemoved.fire( *object );
    object->removeReference();
}

void hkgVdbGeometryControl::addDisplayObjectToWorld( hkgDisplayObject& object )
{
    if ( !m_displayWorld->containsDisplayObject( &object ) )
    {
        m_displayWorld->addDisplayObject( &object );
        object.removeReference();
        m_geometryShown.fire( object );
    }
}

void hkgVdbGeometryControl::removeDisplayObjectFromWorld( hkgDisplayObject& object )
{
    m_displayWorld->removeDisplayObject( &object );
    m_geometryHidden.fire( object );
}

void hkgVdbGeometryControl::setSelection( const hkgVdbSelectionWidget::SelectionSet& set, SelectionMode::Enum selection )
{
    // Always let the system determine base color
    hkgDebugMaterialFlags flags = hkgDebugMaterialFlags::COLOR_SYSTEM;
    // If we have a highlight/selection, we want to override the operations/modifiers
    flags.orWith( ( selection != SelectionMode::NONE ) * ( hkgDebugMaterialFlags::OVERRIDE_SYSTEM_OPERATION | hkgDebugMaterialFlags::OVERRIDE_SYSTEM_MODIFIER ) );
    // If highlighted, then modify the base color with a DESATURATE.
    flags.orWith( ( selection == SelectionMode::HIGHLIGHTED ) * hkgDebugMaterialFlags::MODIFIER_DESATURATE );
    // Else if selected, we will just be normally colored by the system.
    for ( hkgVdbSelectionWidget::SelectionSet::Iterator iter = set.getIterator();
        set.isValid( iter );
        iter = set.getNext( iter ) )
    {
        if ( hkgDisplayObject* obj = set.getValue( iter ) )
        {
            obj->setDebugMaterialFlags( flags );
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
