// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbDisplayHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/3d/hkgVdbSelectionWidget.h>

class hkgWindow;
class hkgFont;
class hkgViewport;
class hkVdbClient;
struct hkgDisplayVertex;
struct hkgDisplaySphere;
struct hkgDisplayText;
struct hkgDisplay3dText;
class hkgParticleDisplayObject;
class hkgVdbPlugin;
class hkVdbTextHandler;

//////////////////////////////////////////////////////////////////////////
/// Flags which control how we draw immediate-mode-display items.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkgVdbDrawFlags : public hkFlagsEx<hkUint8>
{
    enum Bits
    {
        /// POIs will be drawn.
        DRAW_POINTS_OF_INTEREST = ( 1 << 0 ),

        /// Draw orange/yellow box corner tick marks around highlights.
        DRAW_HIGHLIGHTED_BOX = ( 1 << 1 ),

        /// Draw white box corner tick marks around selection.
        DRAW_SELECTED_BOX = ( 1 << 2 ),

        /// The system will apply a visualization to the current display selection/highlights.
        VISUALIZE_DISPLAY_SELECTION = ( 1 << 3 ),

        /// Default draw flags.
        DEFAULT = ( DRAW_POINTS_OF_INTEREST | DRAW_HIGHLIGHTED_BOX | DRAW_SELECTED_BOX | VISUALIZE_DISPLAY_SELECTION )
    };

    HK_DECLARE_FLAGS_EX_CLASS( hkgVdbDrawFlags, hkgVdbDrawFlags::Bits );
};

//////////////////////////////////////////////////////////////////////////
/// Controls the immediate-mode-display in the display world.
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkgVdbDisplayControl : public hkVdbDefaultErrorReporter
{
public:
    HK_DECLARE_CLASS( hkgVdbDisplayControl, New );
    hkgVdbDisplayControl( hkgWindow& window );
    ~hkgVdbDisplayControl();

    //
    // Registration
    //

    hkResult registerSelf( hkgVdbPlugin& plugin, hkVdbDisplayHandler& handler, hkVdbClient& client );
    hkResult unregisterSelf( hkgVdbPlugin& plugin, hkVdbDisplayHandler& handler, hkVdbClient& client );

    //
    // Rendering
    //

    /// Render all immediate-mode-display to the provided viewport.
    void render( hkgViewport& viewport ) const;

    /// Swaps the write buffer (filled in by vdb client signals) with the read buffer (drawn by render).
    void swapBuffers();

    /// Flushes the write buffer to the read buffer (no swapping).
    void flushWriteBuffer();

    //
    // Render Control
    //

    /// Sets the current flags being used to draw immediate-mode-display.
    hkResult setDrawFlags( hkgVdbDrawFlags flags );

    /// Gets the current flags being used to draw immediate-mode-display.
    HK_INLINE hkgVdbDrawFlags getDrawFlags() const { return m_drawFlags; }

    //
    // Internal use
    //

    void onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection );
    void onPlaybackInfoReceivedSignal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result );
    void onFlushDisplaySignal( bool clear );
    void onDisplayCmdReceivedSignal( const hkVdbDisplayHandler::DisplayCmd& displayCmd, hkVdbSignalResult& result );
    void onGeometrySelectionChangedSignal( const hkgVdbSelectionWidget::SelectionSets& set, const hkgVdbSelectionWidget::SelectionChangedInfo& info );
    void onDisplaySelectionChangedSignal( const hkgVdbSelectionWidget::SelectionSets& set, const hkgVdbSelectionWidget::SelectionChangedInfo& info );
    void onWaitForCompletionSignal();

protected:

    struct DisplayBuffer
    {
        HK_DECLARE_CLASS( DisplayBuffer, New );

        hkgDisplayVertex* expandPoints( int numPoints )
        {
            return m_points.expandBy( numPoints );
        }

        hkgDisplaySphere* expandThickPoints( int numPoints )
        {
            return m_thickPoints.expandBy( numPoints );
        }

        hkgDisplayVertex* expandLines( int numLines )
        {
            return m_lines.expandBy( numLines * 2 );
        }

        hkgDisplayVertex* expandTriangles( int numTris )
        {
            return m_triangles.expandBy( numTris * 3 );
        }

        hkgDisplay3dText* expand3dText( int num )
        {
            return m_text3d.expandBy( num );
        }

        hkgDisplayText* expandText( int num )
        {
            return m_text.expandBy( num );
        }

        void append( const DisplayBuffer& buffer );

        hkArray<hkgDisplayVertex> m_points;
        hkArray<hkgDisplaySphere> m_thickPoints;
        hkArray<hkgDisplayVertex> m_lines;
        hkArray<hkgDisplayVertex> m_triangles;
        hkArray<hkgDisplayText> m_text;
        hkArray<hkgDisplay3dText> m_text3d;

        void reset()
        {
            m_points.clear();
            m_thickPoints.clear();
            m_lines.clear();
            m_triangles.clear();
            m_text.clear();
            m_text3d.clear();
        }
    };

    void renderBuffer( hkgViewport& viewport, DisplayBuffer& buffer ) const;
    void renderPointGroup( hkgDisplayContext &context, const hkArray<hkgDisplayVertex>& points ) const;
    void renderThickPointGroup( hkgDisplayContext& context, const hkArray<hkgDisplaySphere>& points ) const;
    void renderLineGroup( hkgDisplayContext& context, const hkArray<hkgDisplayVertex>& points ) const;
    void renderTriangleGroup( hkgDisplayContext& context, const hkArray<hkgDisplayVertex>& points ) const;
    void renderText3dGroup( hkgDisplayContext& context, hkgViewport& viewport, hkArray<hkgDisplay3dText>& text ) const;
    void processDisplayCmd(
        const hkVdbDisplayHandler::DisplayCmd& displayCmd,
        hkVdbSignalResult& result,
        DisplayBuffer& buffer );
    void updateSelectionDisplayBuffers(
        const hkgVdbSelectionWidget::SelectionChangedInfo* changeInfo = HK_NULL,
        hkBool32 changeInfoForGeometry = false );

    hkRefPtr<hkgWindow> m_window;
    hkgVdbDrawFlags m_drawFlags;
    hkRefPtr<hkgFont> m_font;
    hkRefPtr<hkgParticleDisplayObject> m_thickPointDisplayObject;
    hkRefPtr<hkgVdbSelectionWidget> m_selectionWidget;

    hkRefPtr<hkVdbTextHandler> m_textHandler;

    // These fixed buffers are at the end of our buffers array.
    enum FIXED_BUFFER_INDICES
    {
        // Note: 0 and 1 skipped since they are not fixed.
        //       They are our read/write buffers and get swapped by swapBuffers.
        FLUSHED_WRITE_BUFFER_INDEX = 2,
        PAUSE_BUFFER_INDEX,
        SELECTION_BUFFER_INDEX,
        POI_BUFFER_INDEX,
        NUM_BUFFERS
    };
    mutable DisplayBuffer m_buffers[NUM_BUFFERS];
    int m_writeBufferIndex;
    int m_renderBufferIndex;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
