// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPlugin.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Control/hkgVdbDisplayControl.h>

#include <Common/Base/Types/hkSignalSlots.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Visualize/hkVisualDebuggerCmd.h>

#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/Font/hkgFont.h>
#include <Graphics/Common/DisplayObject/hkgParticleDisplayObject.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbTextHandler.h>

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPluginApi.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/hkgVdbWidgetManager.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/Utils/hkgVdb3dBoxDrawer.h>

#define DEBUG_LOG_IDENTIFIER "vdb.hkg.System.Control.Display"
#include <Common/Base/System/Log/hkLog.hxx>

namespace
{
    struct Sort3dText
    {
        Sort3dText( hkgCamera* cam ) : m_camera( cam ) {}

        inline bool operator() ( const hkgDisplay3dText& a, const hkgDisplay3dText& b )
        {
            const float* nearPlane = m_camera->getFrustumPlane(4);
            return hkgVec3Dot( nearPlane, b.m_pos ) < hkgVec3Dot( nearPlane, a.m_pos );
        }

        hkgCamera* m_camera;
    };

    static void drawSet(
        hkgDisplayContext& context,
        const hkgVdbSelectionWidget::SelectionSet& set,
        hkColor::Argb color )
    {
        hkTransform selectionSetTransform;
        hkAabb selectionSetAabb; selectionSetAabb.setEmpty();

        if ( set.getSize() == 1 )
        {
            // Note: HKG is strange in that the AABB is actually not updated to be a true AABB...
            // rather it's the local AABB translated by the object transform.
            // In the single selection case, get the local aabb and allow the draw to rotate
            // it so it's not a true AABB

            auto iter = set.getIterator();
            if ( set.isValid( iter ) )
            {
                if ( hkgDisplayObject* object = set.getValue( iter ) )
                {
                    if ( !object->hasAABB() )
                    {
                        object->computeAABB();
                    }

                    if ( object->hasAABB() )
                    {
                        object->getAABBLocal( selectionSetAabb.m_min, selectionSetAabb.m_max );
                        selectionSetTransform.set4x4ColumnMajor( object->getTransform() );
                    }
                }
            }
        }
        else
        {
            set.computeAabb( selectionSetAabb );
            selectionSetTransform.setIdentity();
        }

        if ( !selectionSetAabb.isEmpty() && selectionSetAabb.isValid() )
        {
            hkgVdb3dBoxDrawerOptions options;
            {
                options.m_style = hkgVdb3dBoxDrawerOptions::CORNERS;
                options.m_color = color;
            }
            hkgVdb3dBoxDrawer::draw3dBox(
                context,
                selectionSetTransform,
                selectionSetAabb,
                options );
        }
    }

    static HK_INLINE hkUint64 packIdAndTag( hkUint64 id, int tag )
    {
        // Note: We can safely use HK_VDB_RESERVED_PERSISTENT_ID_BITS
        // because we don't preserve imm display cache entries across frames.
        // So these are effectively unused bytes for imm displays.
        using namespace hkPack;
        return Pack<hkUint64>(
            hkPackInt<64 - HK_VDB_RESERVED_PERSISTENT_ID_BITS>( id ),
            hkPackInt<HK_VDB_RESERVED_PERSISTENT_ID_BITS>( tag ) );
    }

    static HK_INLINE hkTuple<hkUint64, int> unpackIdAndTag( hkUint64& data )
    {
        using namespace hkPack;
        hkPackInt<64 - HK_VDB_RESERVED_PERSISTENT_ID_BITS> id;
        hkPackInt<HK_VDB_RESERVED_PERSISTENT_ID_BITS> tag;
        Unpack( data, id, tag );
        return hkTupleT::make( hkUint64( id ), int( tag ) );
    }
};

hkgVdbDisplayControl::hkgVdbDisplayControl( hkgWindow& window ) :
    hkVdbDefaultErrorReporter( &s_debugLog ),
    m_window( &window ),
    m_drawFlags( hkgVdbDrawFlags::DEFAULT ),
    m_writeBufferIndex( 0 ),
    m_renderBufferIndex( -1 )
{
    clearError();

    window.getContext()->lock();

    // Create particle display object for thick points
    {
        m_thickPointDisplayObject = hkRefNew<hkgParticleDisplayObject>( hkgParticleDisplayObject::create( window.getContext() ) );

        // Material
        hkgMaterial* mat = hkgMaterial::create();
        {
            m_thickPointDisplayObject->setRenderMode( HKG_PARTICLE_DISPLAY_POINTSPRITE );

            hkgShader* vertShader = window.findShaderByName( "thickPointVShader" );
            hkgShader* pixelShader = window.findShaderByName( "thickPointPShader" );
            hkgShader* geomShader = window.findShaderByName( "thickPointGShader" );

            hkgShaderEffectCollection* effect = hkgShaderEffectCollection::create();

            if ( vertShader && pixelShader )
            {
                effect->addShaderEffect( vertShader, pixelShader, geomShader );
            }

            mat->setShaderCollection( effect );
            effect->removeReference();
        }

        m_thickPointDisplayObject->setMaterial( mat );
        mat->removeReference();

        m_thickPointDisplayObject->setMaxNumParticles( 10000, true );
        m_thickPointDisplayObject->setBlendMode( HKG_PARTICLE_BLEND_MODULATE );
    }

    // Create font for 3d text rendering
    {
        m_font = hkRefNew<hkgFont>( hkgFont::create() );
        m_font->setCharHeight( 16 );
        m_font->setCharWidth( 8 );
        m_font->loadFromBuiltin( window.getContext(), false );
    }

    window.getContext()->unlock();
}

hkgVdbDisplayControl::~hkgVdbDisplayControl()
{}

hkResult hkgVdbDisplayControl::registerSelf( hkgVdbPlugin& plugin, hkVdbDisplayHandler& handler, hkVdbClient& client )
{
    HK_SUBSCRIBE_TO_SIGNAL( client.m_connected, this, hkgVdbDisplayControl );
    HK_SUBSCRIBE_TO_SIGNAL( handler.m_displayCmdReceived, this, hkgVdbDisplayControl );
    HK_SUBSCRIBE_TO_SIGNAL( handler.m_waitForCompletion, this, hkgVdbDisplayControl );
    HK_SUBSCRIBE_TO_SIGNAL( handler.m_flushDisplay, this, hkgVdbDisplayControl );
    HK_SUBSCRIBE_TO_SIGNAL( client.getCmdHandler<hkVdbPlaybackHandler>()->m_playbackInfoReceived, this, hkgVdbPluginControl );
    m_selectionWidget = plugin.getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>();
    HK_SUBSCRIBE_TO_SIGNAL( m_selectionWidget->m_geometrySelectionChanged, this, hkgVdbDisplayControl );
    HK_SUBSCRIBE_TO_SIGNAL( m_selectionWidget->m_displaySelectionChanged, this, hkgVdbDisplayControl );
    m_textHandler = client.getCmdHandler<hkVdbTextHandler>();
    return HK_SUCCESS;
}

hkResult hkgVdbDisplayControl::unregisterSelf( hkgVdbPlugin& plugin, hkVdbDisplayHandler& handler, hkVdbClient& client )
{
    client.m_connected.unsubscribeAll( this );
    handler.m_displayCmdReceived.unsubscribeAll( this );
    handler.m_waitForCompletion.unsubscribeAll( this );
    handler.m_flushDisplay.unsubscribeAll( this );
    client.getCmdHandler<hkVdbPlaybackHandler>()->m_playbackInfoReceived.unsubscribeAll( this );
    m_selectionWidget->m_geometrySelectionChanged.unsubscribeAll( this );
    m_selectionWidget->m_displaySelectionChanged.unsubscribeAll( this );
    m_selectionWidget = HK_NULL;
    m_textHandler = HK_NULL;
    return HK_SUCCESS;
}

void hkgVdbDisplayControl::render( hkgViewport& viewport ) const
{
    hkgDisplayContext& context = *m_window->getContext();
    context.lock();

    
    m_buffers[POI_BUFFER_INDEX].reset();
    auto const &pois = m_selectionWidget->getPointsOfInterest();
    if ( m_drawFlags.anyIsSet( hkgVdbDrawFlags::DRAW_POINTS_OF_INTEREST ) && pois.getSize() )
    {
        context.matchState(
            HKG_ENABLED_ALPHABLEND | HKG_ENABLED_ZREAD | HKG_ENABLED_ZWRITE,
            context.getCullfaceMode(),
            HKG_BLEND_MODULATE_OPAQUE_DEST,
            HKG_ALPHA_SAMPLE_COVERAGE );

        hkVdbSignalResult result;
        hkVdbDisplayHandler::Display1PtCmd poiCmd;
        poiCmd.m_cmdType = hkVdbCmdType::DISPLAY_POINT_EX;
        for ( auto iter = pois.getIterator();
            pois.isValid( iter );
            iter = pois.getNext( iter ) )
        {
            auto const &poi = pois.getValue( iter );
            {
                poiCmd.m_id = 0;
                poiCmd.m_tag = 0;
                poiCmd.m_color = poi.m_color;
                poiCmd.m_pt1 = poi.m_worldSpacePos;

                // Const-cast not ideal, but we created this and are just using it to share code-paths.
                hkVdbDisplayHandler::Display1PtCmd::Ex* ex =
                    const_cast<hkVdbDisplayHandler::Display1PtCmd::Ex*>( poiCmd.getExtended() );
                {
                    ex->m_style = poi.m_style;

                    // Scale POI based on camera from/to vec (if using star, dot is has SP billboard size effect anyway)
                    hkReal scale = poi.m_scale;
                    const hkgCamera* c = viewport.getCamera();
                    if ( c && ( poi.m_style == hk1PointDisplayStyle::STAR ) )
                    {
                        HK_ALIGN16( float lenVec[3] );
                        HK_ALIGN16( float toVec[3] );
                        poi.m_worldSpacePos.store<3>( toVec );
                        hkgVec3Sub( lenVec, toVec, c->getFromPtr() );
                        float d = hkgVec3Length( lenVec );
                        // Reference is roughly 1.0f at 50m looks good.
                        // Of course, not perfect given fov, etc.
                        scale = ( d / 50.f );
                    }
                    ex->m_scale = scale;
                }
            }

            // Const-cast not ideal, but this can technically set error state.
            const_cast< hkgVdbDisplayControl* >(this)->processDisplayCmd(
                poiCmd,
                result,
                m_buffers[POI_BUFFER_INDEX] );
            HK_ASSERT( 0x22441415, result.isSuccess(), "Error state changed unexpectedly" );
        }
    }

    if ( m_renderBufferIndex != -1 )
    {
        const int effRenderIndex =
            ( m_drawFlags.anyIsSet( hkgVdbDrawFlags::VISUALIZE_DISPLAY_SELECTION ) &&
                m_selectionWidget &&
            ( m_selectionWidget->getSelectedGeometries().getSize()
                || m_selectionWidget->getHighlightedDisplays().getSize()
                || m_selectionWidget->getSelectedDisplays().getSize() ) ) ?
                SELECTION_BUFFER_INDEX :
                m_renderBufferIndex;

        DisplayBuffer* displayBuffers[] = {
            &m_buffers[effRenderIndex],
            &m_buffers[FLUSHED_WRITE_BUFFER_INDEX],
            &m_buffers[POI_BUFFER_INDEX]
        };

        for ( int i = 0; i < HK_COUNT_OF( displayBuffers ); i++ )
        {
            DisplayBuffer& displayBuffer = *displayBuffers[i];
            renderBuffer( viewport, displayBuffer );
        }
    }

    // Draw extras
    
    if ( m_drawFlags.anyIsSet( hkgVdbDrawFlags::DRAW_HIGHLIGHTED_BOX ) )
    {
        // Havok Yellow (used for highlighted nodes in vdb)
        drawSet( context, m_selectionWidget->getHighlightedGeometries(), 0xFFFFB300 );
    }
    if ( m_drawFlags.anyIsSet( hkgVdbDrawFlags::DRAW_SELECTED_BOX ) )
    {
        // Vdb Blue is use for selected nodes background, but the text is white and that helps it pop more
        drawSet( context, m_selectionWidget->getSelectedGeometries(), hkColor::WHITESMOKE );
    }

    context.unlock();
}

void hkgVdbDisplayControl::renderBuffer( hkgViewport& viewport, DisplayBuffer& buffer ) const
{
    hkgDisplayContext& context = *m_window->getContext();

    context.matchState(
        HKG_ENABLED_ALPHABLEND | HKG_ENABLED_ZREAD | HKG_ENABLED_ZWRITE,
        context.getCullfaceMode(),
        HKG_BLEND_MODULATE_OPAQUE_DEST,
        HKG_ALPHA_SAMPLE_COVERAGE );

    // Render points
    if ( buffer.m_points.getSize() )
    {
        renderPointGroup( context, buffer.m_points );
    }

    // Render thick points
    if ( buffer.m_thickPoints.getSize() )
    {
        renderThickPointGroup( context, buffer.m_thickPoints );
    }

    // Render lines
    if ( buffer.m_lines.getSize() )
    {
        renderLineGroup( context, buffer.m_lines );
    }

    // Render triangles
    if ( buffer.m_triangles.getSize() )
    {
        renderTriangleGroup( context, buffer.m_triangles );
    }

    // Render 3d text
    if ( buffer.m_text3d.getSize() )
    {
        renderText3dGroup( context, viewport, buffer.m_text3d );
    }

    // lastly, draw 2d text in overlay window
    
    
    
    {
    }
}

void hkgVdbDisplayControl::renderPointGroup( hkgDisplayContext &context, const hkArray<hkgDisplayVertex>& points ) const
{
    context.beginGroup( HKG_IMM_POINTS );
    for ( int i = 0; i < points.getSize(); i++ )
    {
        const hkgDisplayVertex& point = points[i];
        context.setCurrentColorPacked( point.m_color );
        context.setCurrentPosition( point.m_pos );
    }
    context.endGroup();
}

void hkgVdbDisplayControl::renderThickPointGroup( hkgDisplayContext& context, const hkArray<hkgDisplaySphere>& points ) const
{
    // Clear shader/material state to avoid error 0x004B3241 (attempting to render point sprites without a geometry shader).
    context.setCurrentShaderEffect(HK_NULL, HK_NULL);
    context.setCurrentMaterial(HK_NULL, HK_NULL);

    int numPoints = points.getSize();
    int batchSize = m_thickPointDisplayObject->getMaxNumParticles();
    int numBatches = numPoints / batchSize + !!( numPoints % batchSize );
    for ( int batch = 0; batch < numBatches; batch++ )
    {
        int thisBatch = hkMath::min2( numPoints - batch * batchSize, batchSize );
        m_thickPointDisplayObject->setNumParticles( thisBatch );
        hkVector4 pos;
        for ( int i = 0; i < thisBatch; i++ )
        {
            const hkgDisplaySphere& p = points[i + batch * batchSize];
            pos.load<3>( p.m_pos );
            pos.setComponent<3>( p.m_radius );
            m_thickPointDisplayObject->setPosition( pos, i );
            m_thickPointDisplayObject->setUserData( hkColorf( p.m_color ), i );
        }

        m_thickPointDisplayObject->render( &context );
    }
}

void hkgVdbDisplayControl::renderLineGroup( hkgDisplayContext& context, const hkArray<hkgDisplayVertex>& linePoints ) const
{
    context.executeLineGroup( linePoints );
}

void hkgVdbDisplayControl::renderTriangleGroup( hkgDisplayContext &context, const hkArray<hkgDisplayVertex>& trianglePoints ) const
{
    
    

    // Set state for opaque triangles
    context.setBlendState( false );
    context.setDepthWriteState( true );

    int firstTransparentIndex = -1;
    context.beginGroup( HKG_IMM_TRIANGLE_LIST );
    {
        // Render opaque triangles
        for ( int i = 0; i < trianglePoints.getSize(); i += 3 )
        {
            const hkgDisplayVertex* pts = &trianglePoints[i];

            if ( hkColor::getAlphaAsFloat( pts[0].m_color ) >= 1.f )
            {
                context.setCurrentColorPacked( pts[0].m_color );
                context.setCurrentPosition( pts[0].m_pos );
                context.setCurrentPosition( pts[1].m_pos );
                context.setCurrentPosition( pts[2].m_pos );
            }
            else if ( firstTransparentIndex == -1 )
            {
                firstTransparentIndex = i;
            }
        }
    }
    context.endGroup();

    // Set state for transparent triangles
    context.setBlendState( true );
    context.setDepthWriteState( false );
    context.beginGroup( HKG_IMM_TRIANGLE_LIST );
    {
        // Render transparent triangles
        for ( int i = firstTransparentIndex; ( i >= 0 ) && ( i < trianglePoints.getSize() ); i += 3 )
        {
            const hkgDisplayVertex* pts = &trianglePoints[i];

            if ( hkColor::getAlphaAsFloat( pts[0].m_color ) < 1.f )
            {
                context.setCurrentColorPacked( pts[0].m_color );
                context.setCurrentPosition( pts[0].m_pos );
                context.setCurrentPosition( pts[1].m_pos );
                context.setCurrentPosition( pts[2].m_pos );
            }
        }
    }
    context.endGroup();
}

void hkgVdbDisplayControl::renderText3dGroup( hkgDisplayContext& context, hkgViewport& viewport, hkArray<hkgDisplay3dText>& text ) const
{
    hkgFont::setDrawState( &context );
    context.setBlendMode( HKG_BLEND_MODULATE_OPAQUE_DEST );

    // Want depth for 3D text so that it fits in scene properly
    HKG_ENABLED_STATE state = context.getEnabledState();
    if ( !( state & HKG_ENABLED_ZWRITE ) )
        context.setDepthWriteState( true );
    if ( !( state & HKG_ENABLED_ZREAD ) )
        context.setDepthReadState( true );

    // sort the 3d text array wrt to camera
    // as 3d text frequently overlaps, it is better to
    // draw in depth order to improve visual effect
    hkgCamera* cam = viewport.getCamera();
    hkAlgorithm::explicitStackQuickSort<hkgDisplay3dText, Sort3dText>( text.begin(), text.getSize(), Sort3dText( cam ) );

    m_font->render( &context, text, &viewport );
}

void hkgVdbDisplayControl::swapBuffers()
{
    m_window->getContext()->lock();
    m_writeBufferIndex = ( ( m_writeBufferIndex + 1 ) % 2 );
    m_renderBufferIndex = ( ( m_writeBufferIndex + 1 ) % 2 );
    m_buffers[m_writeBufferIndex].reset();
    m_buffers[FLUSHED_WRITE_BUFFER_INDEX].reset();
    updateSelectionDisplayBuffers();
    m_window->getContext()->unlock();
}

void hkgVdbDisplayControl::flushWriteBuffer()
{
    m_window->getContext()->lock();
    if ( m_renderBufferIndex != -1 )
    {
        m_buffers[FLUSHED_WRITE_BUFFER_INDEX].reset();
        m_buffers[FLUSHED_WRITE_BUFFER_INDEX].append( m_buffers[m_writeBufferIndex] );
    }
    m_window->getContext()->unlock();
}

hkResult hkgVdbDisplayControl::setDrawFlags( hkgVdbDrawFlags flags )
{
    m_window->getContext()->lock();
    {
        hkBool32 wasVisualizingSelection = m_drawFlags.anyIsSet( hkgVdbDrawFlags::VISUALIZE_DISPLAY_SELECTION );
        m_drawFlags = flags;
        if ( !wasVisualizingSelection &&
            m_drawFlags.anyIsSet( hkgVdbDrawFlags::VISUALIZE_DISPLAY_SELECTION ) )
        {
            // If this just turned on, we need to update our selection buffers
            updateSelectionDisplayBuffers();
        }
    }
    m_window->getContext()->unlock();
    return HK_SUCCESS;
}

void hkgVdbDisplayControl::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        m_window->getContext()->lock();
        m_writeBufferIndex = 0;
        m_renderBufferIndex = -1;
        m_window->getContext()->unlock();
    }
}

void hkgVdbDisplayControl::onPlaybackInfoReceivedSignal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result )
{
    // If we've changed frames during playback, we need to present our filled buffer.
    if ( info.flagWasSet( hkVdbPlaybackFlags::FRAME_ENDED ) )
    {
        // Note: lock inside function call.
        swapBuffers();
    }

    // Handle save/restore of current working buffer during pause/unpause.
    // This is to preserve current frame's work if it wasn't completed before the pause.
    if ( ( info.m_previousState != hkVdbPlaybackState::PAUSED ) && ( info.m_targetState == hkVdbPlaybackState::PAUSED ) )
    {
        // Save
        m_window->getContext()->lock();
        m_buffers[PAUSE_BUFFER_INDEX].reset();
        m_buffers[PAUSE_BUFFER_INDEX].append( m_buffers[m_writeBufferIndex] );
        m_window->getContext()->unlock();
    }
    else if ( ( info.m_previousState == hkVdbPlaybackState::PAUSED ) && ( info.m_targetState != hkVdbPlaybackState::PAUSED ) )
    {
        // Restore
        m_window->getContext()->lock();
        m_buffers[m_writeBufferIndex].reset();
        m_buffers[m_writeBufferIndex].append( m_buffers[PAUSE_BUFFER_INDEX] );
        m_window->getContext()->unlock();
    }
}

void hkgVdbDisplayControl::onFlushDisplaySignal( bool clear )
{
    // Note: locks inside function calls.
    if ( clear )
    {
        swapBuffers();
    }
    else
    {
        flushWriteBuffer();
    }
}

void hkgVdbDisplayControl::onDisplayCmdReceivedSignal( const hkVdbDisplayHandler::DisplayCmd& displayCmd, hkVdbSignalResult& result )
{
    DisplayBuffer& writeBuffer = m_buffers[m_writeBufferIndex];
    processDisplayCmd( displayCmd, result, writeBuffer );
}

void hkgVdbDisplayControl::onGeometrySelectionChangedSignal( const hkgVdbSelectionWidget::SelectionSets& sets, const hkgVdbSelectionWidget::SelectionChangedInfo& info )
{
    m_window->getContext()->lock();
    // If we have a display selection, then this is taken care of by onDisplaySelectionChangedSignal
    
    
    
    if ( !m_selectionWidget->getHighlightedDisplays().getSize()
        && !m_selectionWidget->getSelectedDisplays().getSize() )
    {
        updateSelectionDisplayBuffers();
    }
    m_window->getContext()->unlock();
}

void hkgVdbDisplayControl::onDisplaySelectionChangedSignal( const hkgVdbSelectionWidget::SelectionSets& sets, const hkgVdbSelectionWidget::SelectionChangedInfo& info )
{
    m_window->getContext()->lock();
    {
        updateSelectionDisplayBuffers();
    }
    m_window->getContext()->unlock();
}

void hkgVdbDisplayControl::onWaitForCompletionSignal()
{
    
    
}

void hkgVdbDisplayControl::processDisplayCmd(
    const hkVdbDisplayHandler::DisplayCmd& displayCmd,
    hkVdbSignalResult& result,
    DisplayBuffer& buffer )
{
    auto writeLine = [&buffer]( hkUint64 id, hkVector4Parameter a, hkVector4Parameter b, hkColor::Argb color, int tag )
    {
        hkgDisplayVertex* pts = buffer.expandLines( 1 );
        a.store<3>( pts[0].m_pos );
        b.store<3>( pts[1].m_pos );
        pts[0].m_color = pts[1].m_color = color;
        pts[0].m_userData = pts[1].m_userData = packIdAndTag( id, tag );
    };

    if ( const hkVdbDisplayHandler::DisplayGeometryCmd* displayGeomCmd = displayCmd.asGeometryCmd() )
    {
        hkTransform rootTransform;
        displayGeomCmd->m_transform.get( rootTransform );

        hkArray<hkVector4> lineArray;
        lineArray.reserve( 128 );
        for ( int i = 0; i < displayGeomCmd->m_geometry.getSize(); ++i )
        {
            const hkDisplayGeometry* displayGeom = displayGeomCmd->m_geometry[i];

            hkTransform composedTransform;
            composedTransform.setMul( rootTransform, displayGeom->getTransform() );

            lineArray.clear();
            displayGeom->getWireframeGeometry( lineArray );

            hkgDisplayVertex* linePts = buffer.expandLines( lineArray.getSize() / 2 );
            for ( int j = lineArray.getSize() - 1; j >= 0; j -= 2 )
            {
                hkVector4 start; start.setTransformedPos( composedTransform, lineArray[j] );
                hkVector4 end; end.setTransformedPos( composedTransform, lineArray[j - 1] );

                start.store<3>( linePts[j].m_pos );
                end.store<3>( linePts[j - 1].m_pos );
                linePts[j].m_color = linePts[j - 1].m_color = displayGeomCmd->m_color;
                linePts[j].m_userData = linePts[j - 1].m_userData = packIdAndTag( displayGeomCmd->m_id, displayGeomCmd->m_tag );
            }
        }
    }
    else if ( const hkVdbDisplayHandler::Display1PtCmd* display1PtCmd = displayCmd.as1PtCmd() )
    {
        if ( const hkVdbDisplayHandler::Display1PtCmd::Ex* ex = display1PtCmd->getExtended() )
        {
            switch ( ex->m_style )
            {
                case hk1PointDisplayStyle::DOT:
                {
                    hkgDisplaySphere* point = buffer.expandThickPoints( 1 );
                    display1PtCmd->m_pt1.store<3>( point->m_pos );
                    point->m_radius = ex->m_scale;
                    point->m_color = display1PtCmd->m_color;
                    point->m_userData = packIdAndTag( display1PtCmd->m_id, display1PtCmd->m_tag );
                    break;
                }
                case hk1PointDisplayStyle::STAR:
                {
                    hkDebugDisplayHandlerUtils::displayStarUsing2PointBatchesFunc( writeLine, display1PtCmd->m_id, display1PtCmd->m_pt1, display1PtCmd->m_color, ex->m_scale, display1PtCmd->m_tag );
                    break;
                }
                default:
                {
                    HK_VDB_SIGNAL_ERROR_MSG( 0xadb00045, hkVdbError::PLUGIN_ERROR, "Unknown display1Pt cmd" );
                    result.signalError( *this );
                    break;
                }
            }
        }
        else
        {
            hkgDisplayVertex* point = buffer.expandPoints( 1 );
            display1PtCmd->m_pt1.store<3>( point->m_pos );
            point->m_color = display1PtCmd->m_color;
            point->m_userData = packIdAndTag( display1PtCmd->m_id, display1PtCmd->m_tag );
        }
    }
    else if ( const hkVdbDisplayHandler::Display2PtCmd* display2PtCmd = displayCmd.as2PtCmd() )
    {
        if ( const hkVdbDisplayHandler::Display2PtCmd::Ex* ex = display2PtCmd->getExtended() )
        {
            switch ( ex->m_style )
            {
                case hk2PointDisplayStyle::LINE_SEGMENT:
                {
                    writeLine( display2PtCmd->m_id, display2PtCmd->m_pt1, display2PtCmd->m_pt2, display2PtCmd->m_color, display2PtCmd->m_tag );
                    break;
                }
                case hk2PointDisplayStyle::ARROW:
                {
                    hkDebugDisplayHandlerUtils::displayArrowUsing2PointBatchesFunc( writeLine, display2PtCmd->m_id, display2PtCmd->m_pt1, display2PtCmd->m_pt2, display2PtCmd->m_color, display2PtCmd->m_tag );
                    break;
                }
                default:
                {
                    HK_VDB_SIGNAL_ERROR_MSG( 0xadb00045, hkVdbError::PLUGIN_ERROR, "Unknown display1Pt cmd" );
                    result.signalError( *this );
                    break;
                }
            }
        }
        else
        {
            writeLine( display2PtCmd->m_id, display2PtCmd->m_pt1, display2PtCmd->m_pt2, display2PtCmd->m_color, display2PtCmd->m_tag );
        }
    }
    else if ( const hkVdbDisplayHandler::Display3PtCmd* display3PtCmd = displayCmd.as3PtCmd() )
    {
        hkgDisplayVertex* pts = buffer.expandTriangles( 1 );
        display3PtCmd->m_pt1.store<3>( pts[0].m_pos );
        display3PtCmd->m_pt2.store<3>( pts[1].m_pos );
        display3PtCmd->m_pt3.store<3>( pts[2].m_pos );
        pts[0].m_color = pts[1].m_color = pts[2].m_color = display3PtCmd->m_color;
        pts[0].m_userData = pts[1].m_userData = pts[2].m_userData = packIdAndTag( display3PtCmd->m_id, display3PtCmd->m_tag );
    }
    else if ( const hkVdbDisplayHandler::DisplayTextCmd* displayTextCmd = displayCmd.asTextCmd() )
    {
        if ( 1 )
        {
            hkVdbTextHandler::TextInfo info;
            {
                info.m_id = displayTextCmd->m_id;
                info.m_tag = displayTextCmd->m_tag;
                info.m_level = hkLog::Level::Info;
                info.m_text = displayTextCmd->m_text;
            }
            m_textHandler->m_textInfoReceived.fire( info, result );
        }
        else
        {
            
            
            hkgDisplayText* text = buffer.expandText( 1 );
            text->m_color = displayTextCmd->m_color;
            text->m_text = displayTextCmd->m_text;
            text->m_userData = packIdAndTag( displayTextCmd->m_id, displayTextCmd->m_tag );
        }
    }
    else if ( const hkVdbDisplayHandler::Display3dTextCmd* display3dTextCmd = displayCmd.as3dTextCmd() )
    {
        hkgDisplay3dText* text3d = buffer.expand3dText( 1 );
        text3d->m_color = display3dTextCmd->m_color;
        display3dTextCmd->m_position.store<3>( text3d->m_pos );
        text3d->m_text = display3dTextCmd->m_text;
        text3d->m_userData = packIdAndTag( display3dTextCmd->m_id, display3dTextCmd->m_tag );
    }
    else
    {
        HK_VDB_SIGNAL_ERROR_MSG( 0xadb00001, hkVdbError::PLUGIN_ERROR, "Unknown display cmd" );
        result.signalError( *this );
    }
}

void hkgVdbDisplayControl::updateSelectionDisplayBuffers(
    const hkgVdbSelectionWidget::SelectionChangedInfo* changeInfo,
    hkBool32 changeInfoForGeometry )
{
    if ( m_drawFlags.noneIsSet( hkgVdbDrawFlags::VISUALIZE_DISPLAY_SELECTION ) )
    {
        return;
    }

    DisplayBuffer& selectionBuffer = m_buffers[SELECTION_BUFFER_INDEX];
    if ( changeInfo )
    {
        
        //if ( changeInfo->m_addedToHighlighted.getSize() || changeInfo->m_removedFromHighlighted.getSize() )
        //{
        //  // heap sort? buffer entries? or is full rebuild going to be faster usually...
        //  hkAlgorithm::heapSort<hkgDisplay3dText, Sort3dText>( text.begin(), text.getSize(), Sort3dText( cam ) );
        //}
        //if ( changeInfo->m_addedToSelected.getSize() || changeInfo->m_removedFromSelected.getSize() )
        //{
        //}
        HK_ASSERT_NOT_IMPLEMENTED( 0x22441465 );
    }
    else
    {
        selectionBuffer.reset();
        hkBool32 hasGeometrySelection = ( m_selectionWidget->getSelectedGeometries().getSize() );
        hkBool32 hasDisplayHighlightOrSelection = ( m_selectionWidget->getHighlightedDisplays().getSize() || m_selectionWidget->getSelectedDisplays().getSize() );
        if ( hasGeometrySelection || hasDisplayHighlightOrSelection )
        {
            DisplayBuffer& srcBuffer = m_buffers[m_renderBufferIndex];
            const hkgVdbSelectionWidget::SelectionSet& highlighted = m_selectionWidget->getHighlightedDisplays();
            const hkgVdbSelectionWidget::SelectionSet& selected = m_selectionWidget->getSelectedDisplays();

            // We iterate over the entire buffer and include the imm display feature if any of the following are true:
            // a) it's tag is zero - this is a special tag for debug display and we don't want to hide these
            // b) we currently have a selection/highlight and it's part of the selection/highlight - we aren't in this
            //      loop at all if we have no selection/highlight, but we could have one and not the other.

            
            for ( int j = 0; j < srcBuffer.m_points.getSize(); ++j )
            {
                hkTuple<hkUint64, int> meta = unpackIdAndTag( srcBuffer.m_points[j].m_userData );
                if ( !meta.m_1 ||
                    selected.contains( meta.m_0 ) )
                {
                    *selectionBuffer.expandPoints( 1 ) = srcBuffer.m_points[j];
                }
                else if ( highlighted.contains( meta.m_0 ) )
                {
                    hkgDisplayVertex* point = selectionBuffer.expandPoints( 1 );
                    *point = srcBuffer.m_points[j];
                    point->m_color = hkColor::darken( point->m_color );
                }
            }

            for ( int j = 0; j < srcBuffer.m_thickPoints.getSize(); ++j )
            {
                hkTuple<hkUint64, int> meta = unpackIdAndTag( srcBuffer.m_thickPoints[j].m_userData );
                if ( !meta.m_1 ||
                    selected.contains( meta.m_0 ) )
                {
                    *selectionBuffer.expandThickPoints( 1 ) = srcBuffer.m_thickPoints[j];
                }
                else if ( highlighted.contains( meta.m_0 ) )
                {
                    hkgDisplayVertex* thickPoint = selectionBuffer.expandThickPoints( 1 );
                    *thickPoint = srcBuffer.m_thickPoints[j];
                    thickPoint->m_color = hkColor::darken( thickPoint->m_color );
                }
            }

            for ( int j = 0; ( j + 1 ) < srcBuffer.m_lines.getSize(); j += 2 )
            {
                hkTuple<hkUint64, int> meta = unpackIdAndTag( srcBuffer.m_lines[j].m_userData );
                if ( !meta.m_1 ||
                    selected.contains( meta.m_0 ) )
                {
                    hkgDisplayVertex* line = selectionBuffer.expandLines( 1 );
                    line[0] = srcBuffer.m_lines[j];
                    line[1] = srcBuffer.m_lines[j+1];
                }
                else if ( highlighted.contains( meta.m_0 ) )
                {
                    hkgDisplayVertex* line = selectionBuffer.expandLines( 1 );
                    line[0] = srcBuffer.m_lines[j];
                    line[1] = srcBuffer.m_lines[j+1];
                    line[0].m_color = hkColor::darken( line[0].m_color );
                    line[1].m_color = hkColor::darken( line[1].m_color );
                }
            }

            for ( int j = 0; ( j + 2 ) < srcBuffer.m_triangles.getSize(); j += 3 )
            {
                hkTuple<hkUint64, int> meta = unpackIdAndTag( srcBuffer.m_triangles[j].m_userData );
                if ( !meta.m_1 ||
                    selected.contains( meta.m_0 ) )
                {
                    hkgDisplayVertex* tri = selectionBuffer.expandTriangles( 1 );
                    tri[0] = srcBuffer.m_triangles[j];
                    tri[1] = srcBuffer.m_triangles[j+1];
                    tri[2] = srcBuffer.m_triangles[j+2];
                }
                else if ( highlighted.contains( meta.m_0 ) )
                {
                    hkgDisplayVertex* tri = selectionBuffer.expandTriangles( 1 );
                    tri[0] = srcBuffer.m_triangles[j];
                    tri[1] = srcBuffer.m_triangles[j+1];
                    tri[2] = srcBuffer.m_triangles[j+2];
                    tri[0].m_color = hkColor::darken( tri[0].m_color );
                    tri[1].m_color = hkColor::darken( tri[1].m_color );
                    tri[2].m_color = hkColor::darken( tri[2].m_color );
                }
            }

            for ( int j = 0; j < srcBuffer.m_text.getSize(); ++j )
            {
                hkTuple<hkUint64, int> meta = unpackIdAndTag( srcBuffer.m_points[j].m_userData );
                if ( !meta.m_1 ||
                    selected.contains( meta.m_0 ) )
                {
                    *selectionBuffer.expandText( 1 ) = srcBuffer.m_text[j];
                }
                else if ( highlighted.contains( meta.m_0 ) )
                {
                    hkgDisplayText* text = selectionBuffer.expandText( 1 );
                    *text = srcBuffer.m_text[j];
                    text->m_color = hkColor::darken( text->m_color );
                }
            }

            for ( int j = 0; j < srcBuffer.m_text3d.getSize(); ++j )
            {
                hkTuple<hkUint64, int> meta = unpackIdAndTag( srcBuffer.m_text3d[j].m_userData );
                if ( !meta.m_1 ||
                    selected.contains( meta.m_0 ) )
                {
                    *selectionBuffer.expand3dText( 1 ) = srcBuffer.m_text3d[j];
                }
                else if ( highlighted.contains( meta.m_0 ) )
                {
                    hkgDisplayText* text3d = selectionBuffer.expand3dText( 1 );
                    *text3d = srcBuffer.m_text3d[j];
                    text3d->m_color = hkColor::darken( text3d->m_color );
                }
            }
        }
    }
}

void hkgVdbDisplayControl::DisplayBuffer::append( const DisplayBuffer& buffer )
{
    m_points.append( buffer.m_points );
    m_thickPoints.append( buffer.m_thickPoints );
    m_lines.append( buffer.m_lines );
    m_triangles.append( buffer.m_triangles );
    m_text.append( buffer.m_text );
    m_text3d.append( buffer.m_text3d );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
