// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Container/Hash/hkHashMap.h>

#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbDisplayHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>

#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/3d/hkgVdbSelectionWidget.h>

class hkgWindow;
class hkVdbClient;
class hkgDisplayContext;
class hkgViewport;
class hkgCamera;
class hkgVdbPlugin;
class hkgVdbSelectionWidget;

//////////////////////////////////////////////////////////////////////////
/// Flags which control how a camera behaves.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkgVdbCameraFlags : public hkFlagsEx<hkUint8>
{
    enum Bits
    {
        /// Change the camera's "to" position to be
        /// on the hkgVdbSelectionWidget::PrimaryPointOfInterest.
        /// Note: This does not apply when a server camera is being used.
        LOOK_AT_PRIMARY_POINT_OF_INTEREST = ( 1 << 0 ),

        /// Change the camera's "to" position to be
        /// on the hkgVdbSelectionWidget::PrimaryPointOfInterest's
        /// geometry center (if it's a local POI).
        /// This is a no-op if the PrimaryPointOfInterest is not local.
        /// Note: This does not apply when a server camera is being used.
        LOOK_AT_PRIMARY_GEOMETRY_CENTER = ( 1 << 1 ),

        /// Change the camera's "from/to" values to maintain
        /// relative position to the hkgVdbSelectionWidget::PrimaryPointOfInterest.
        /// Note: This does not apply when a server camera is being used.
        KEEP_RELATIVE_TO_PRIMARY_POINT_OF_INTEREST = ( 1 << 2 ),

        /// Change the camera's "from/to" values to maintain
        /// relative position to the hkgVdbSelectionWidget::PrimaryPointOfInterest's
        /// geometry center (if it's a local POI).
        /// Note: This does not apply when a server camera is being used.
        KEEP_RELATIVE_TO_PRIMARY_GEOMETRY_CENTER = ( 1 << 3 ),

        /// Default flags.
        DEFAULT = 0
    };

    HK_DECLARE_FLAGS_EX_CLASS( hkgVdbCameraFlags, hkgVdbCameraFlags::Bits );
};

//////////////////////////////////////////////////////////////////////////
/// Describes a vdb camera.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkgVdbCameraInfo
{
    HK_DECLARE_CLASS( hkgVdbCameraInfo, New );

    /// True, it this was defined by the server.
    hkBool32 m_server;

    /// The camera properties.
    /// (Only basic properties as determined by hkVdbDisplayHandler::UpdateCameraCmd are copied).
    hkRefPtr<hkgCamera> m_camera;

    /// The camera flags.
    hkgVdbCameraFlags m_flags;

    /// The PointOfInterest id to look at and/or keep relative to (see flags).
    hkUint64 m_poiId;

    /// If kept relative to POI, this is the offset to POI.
    /// Note: this is set during the first render pass after it is needed.
    hkVector4 m_keepRelativeOffset;

protected:

    /// The last position the system moved us to.
    hkVector4 m_lastFrom;
    hkVector4 m_lastTo;

    friend class hkgVdbCameraControl;
};

//////////////////////////////////////////////////////////////////////////
/// Controls the camera in the display world.
/// We do not offer functionality here that is present in hkgViewport
/// or hkgCamera directly (such as control schemes).
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkgVdbCameraControl : public hkVdbDefaultErrorReporter
{
public:
    HK_DECLARE_CLASS( hkgVdbCameraControl, New );
    hkgVdbCameraControl( hkgWindow& window, const hkgVdbGeometryControl& geometryControl );
    ~hkgVdbCameraControl();

    typedef hkHashMap<hkStringPtr, hkgVdbCameraInfo> CameraInfos;

    //
    // Registration
    //

    hkResult registerSelf( hkgVdbPlugin& plugin, hkVdbDisplayHandler& handler, hkVdbClient& client );
    hkResult unregisterSelf( hkgVdbPlugin& plugin, hkVdbDisplayHandler& handler, hkVdbClient& client );

    //
    // Rendering
    //

    /// Sets up the viewport camera for rendering.
    void render( hkgViewport& viewport, const char* renderCameraName = HK_NULL ) const;

    //
    // Control
    //

    /// Set flags applied to whatever camera is assigned to the viewport at the time of render.
    hkResult setGlobalCameraFlags( hkgVdbCameraFlags flags );

    /// Get flags applied to whatever camera is assigned to the viewport at the time of render.
    HK_INLINE hkgVdbCameraFlags getGlobalCameraFlags() const { return m_globalCameraInfo.m_flags; }

    /// Add/update a camera with the given properties and/or flags.
    HK_INLINE hkResult updateCamera(
        const char* cameraName,
        const hkgCamera* cameraProperties,
        const hkgVdbCameraFlags* cameraFlags,
        const hkUint64* poiId = &hkgVdbSelectionWidget::PrimaryPointOfInterest );

    /// Remove cameras with the specified name.
    bool removeCameras(
        hkArrayView<const char*> cameraNames = hkVdbValidity<hkArrayView<const char*>>::invalid() );

    //
    // Query
    //

    /// Get the camera infos.
    /// Note: these can be updated by server, so should be protected by MT lock while accessing.
    HK_INLINE const CameraInfos& getCameraInfos() const;

    /// Get a camera info by camera name.
    /// Note: these can be updated by server, so should be protected by MT lock while accessing.
    HK_INLINE const hkgVdbCameraInfo* getCameraInfo( const char* cameraName ) const;

    //
    // Signals
    //

    HK_DECLARE_SIGNAL( CameraUpdatedSignal, hkSignal1< const hkgVdbCameraInfo& > );
    /// Fired when a camera is updated (or added for the first time).
    CameraUpdatedSignal m_cameraUpdated;

    HK_DECLARE_SIGNAL( CameraRemovedSignal, hkSignal1< const hkgVdbCameraInfo& > );
    /// Fired when a camera is removed.
    CameraRemovedSignal m_cameraRemoved;

    //
    // Internal use
    //

    void onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection );
    void onDisconnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection );
    void onWritingToSinkSignal( hkVdbConnection& connection, const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, const hkVdbCmd*& cmdToWriteOut );
    void onCameraCmdReceivedSignal( const hkVdbDisplayHandler::CameraCmd& cameraCmd, hkVdbSignalResult& result );
    void onPlaybackInfoReceivedSignal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result );

protected:

    hkgVdbCameraInfo* updateCameraInternal(
        const char* cameraName,
        hkBool32 isFromServer,
        const hkgCamera* cameraProperties,
        const hkgVdbCameraFlags* cameraFlags,
        const hkUint64* poiId );

    void stepCameraInfo( hkgVdbCameraInfo& cameraInfo ) const;

    const hkgVdbGeometryControl& m_geometryControl;
    hkRefPtr<hkgWindow> m_window;
    hkRefPtr<hkgVdbSelectionWidget> m_selectionWidget;

    mutable hkStringPtr m_renderCameraName;
    mutable hkRefPtr<hkgCamera> m_renderCamera;
    mutable hkRefPtr<hkgCamera> m_sinkedCamera;

    mutable hkgVdbCameraInfo m_globalCameraInfo;
    mutable CameraInfos m_nameToCameraInfo;
};

#include <VisualDebugger/VdbDisplay/Hkg/System/Control/hkgVdbCameraControl.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
