// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPlugin.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Control/hkgVdbCameraControl.h>

#include <Common/Base/Types/hkSignalSlots.h>
#include <Common/Visualize/hkServerDebugDisplayHandler.h>

#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/Camera/hkgCamera.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbInternalHandler.h>

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPluginApi.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/hkgVdbWidgetManager.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Control/hkgVdbGeometryControl.h>

#define DEBUG_LOG_IDENTIFIER "vdb.hkg.System.Control.Camera"
#include <Common/Base/System/Log/hkLog.hxx>

namespace
{
    const hkUint8 LOOK_AT_FLAGS =
        hkUint8( hkgVdbCameraFlags::LOOK_AT_PRIMARY_POINT_OF_INTEREST | hkgVdbCameraFlags::LOOK_AT_PRIMARY_GEOMETRY_CENTER );
    const hkUint8 KEEP_RELATIVE_FLAGS =
        hkUint8( hkgVdbCameraFlags::KEEP_RELATIVE_TO_PRIMARY_POINT_OF_INTEREST | hkgVdbCameraFlags::KEEP_RELATIVE_TO_PRIMARY_GEOMETRY_CENTER );

    void copyCameraMinimal( hkgCamera& dst, const hkgCamera& src )
    {
        dst.setFrom( src.getFromPtr() );
        dst.setTo( src.getToPtr() );
        dst.setUp( src.getUpPtr() );
        dst.setNear( src.getNear() );
        dst.setFar( src.getFar() );
        dst.setFOV( src.getFOV() );
    }

    void initCameraInfo( hkgVdbCameraInfo& info, const char* cameraName )
    {
        info.m_server = false;
        info.m_camera = hkRefNew<hkgCamera>( hkgCamera::create() );
        info.m_camera->setCameraName( cameraName );
        info.m_flags = hkgVdbCameraFlags::DEFAULT;
        info.m_poiId = hkgVdbSelectionWidget::PrimaryPointOfInterest;
        info.m_keepRelativeOffset.setXYZ_W( hkVector4::getZero(), -1 );
    }

    void updateCameraProperties( hkgVdbCameraInfo& info, hkVector4Parameter lastFrom, hkVector4Parameter lastTo, const hkgCamera* cameraProperties )
    {
        if ( cameraProperties )
        {
            // Copy
            copyCameraMinimal( *info.m_camera, *cameraProperties );

            // Reset offset if need be
            if ( info.m_flags.anyIsSet( KEEP_RELATIVE_FLAGS ) )
            {
                HK_ALIGN16( float hkgvec[3] );
                info.m_camera->getFrom( hkgvec );
                hkVector4 currentFrom; currentFrom.load<3>( hkgvec );
                if ( !currentFrom.allEqual<3>( lastFrom, hkSimdReal::getConstant<HK_QUADREAL_EPS>() ) )
                {
                    // Reset our offset next render so that we are up-to-date.
                    info.m_keepRelativeOffset.setW( -1 );
                }
            }
        }
    }

    void updateCameraFlags( hkgVdbCameraInfo& info, const hkgVdbCameraFlags* cameraFlags )
    {
        if ( cameraFlags )
        {
            info.m_flags = *cameraFlags;
            if ( cameraFlags->noneIsSet( KEEP_RELATIVE_FLAGS ) )
            {
                // Reset our offset since we've turned off this flag.
                info.m_keepRelativeOffset.setW( -1 );
            }
        }
    }
}

hkgVdbCameraControl::hkgVdbCameraControl( hkgWindow& window, const hkgVdbGeometryControl& geometryControl ) :
    hkVdbDefaultErrorReporter( &s_debugLog ),
    m_geometryControl( geometryControl ),
    m_window( &window )
{
    clearError();
    initCameraInfo( m_globalCameraInfo, HK_NULL );
}

hkgVdbCameraControl::~hkgVdbCameraControl()
{}

hkResult hkgVdbCameraControl::registerSelf( hkgVdbPlugin& plugin, hkVdbDisplayHandler& handler, hkVdbClient& client )
{
    HK_SUBSCRIBE_TO_SIGNAL( client.m_connected, this, hkgVdbCameraControl );
    HK_SUBSCRIBE_TO_SIGNAL( client.m_disconnected, this, hkgVdbCameraControl );
    HK_SUBSCRIBE_TO_SIGNAL( client.m_writingToSink, this, hkgVdbCameraControl );
    HK_SUBSCRIBE_TO_SIGNAL( handler.m_cameraCmdReceived, this, hkgVdbCameraControl );
    HK_SUBSCRIBE_TO_SIGNAL( client.getCmdHandler<hkVdbPlaybackHandler>()->m_playbackInfoReceived, this, hkgVdbCameraControl );
    m_selectionWidget = plugin.getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>();
    return HK_SUCCESS;
}

hkResult hkgVdbCameraControl::unregisterSelf( hkgVdbPlugin& plugin, hkVdbDisplayHandler& handler, hkVdbClient& client )
{
    client.m_connected.unsubscribeAll( this );
    client.m_disconnected.unsubscribeAll( this );
    client.m_writingToSink.unsubscribeAll( this );
    handler.m_cameraCmdReceived.unsubscribeAll( this );
    client.getCmdHandler<hkVdbPlaybackHandler>()->m_playbackInfoReceived.unsubscribeAll( this );
    return HK_SUCCESS;
}

void hkgVdbCameraControl::render( hkgViewport& viewport, const char* renderCameraName ) const
{
    hkgDisplayContext& context = *m_window->getContext();
    context.lock();

    // Update our client-camera infos
    for ( auto iter = m_nameToCameraInfo.getIterator();
        m_nameToCameraInfo.isValid( iter );
        iter = m_nameToCameraInfo.getNext( iter ) )
    {
        auto && cameraInfo = m_nameToCameraInfo.getValue( iter );
        stepCameraInfo( cameraInfo );
    }

    // Update our global camera info (if needed)
    if ( m_globalCameraInfo.m_flags )
    {
        updateCameraProperties( m_globalCameraInfo, m_globalCameraInfo.m_lastFrom, m_globalCameraInfo.m_lastTo, viewport.getCamera() );
        stepCameraInfo( m_globalCameraInfo );
        if ( hkgCamera* viewportCamera = viewport.getCamera() )
        {
            copyCameraMinimal( *viewportCamera, *m_globalCameraInfo.m_camera );
        }
    }

    // Create/destroy our render camera if needed.
    {
        m_renderCameraName = renderCameraName;
        if ( m_renderCameraName )
        {
            if ( !m_renderCamera || ( hkString::strCmp( m_renderCamera->getCameraName(), m_renderCameraName ) != 0 ) )
            {
                if ( const hkgVdbCameraInfo* renderCameraInfo = getCameraInfo( m_renderCameraName ) )
                {
                    m_renderCamera = hkRefNew<hkgCamera>( hkgCamera::create() );
                    copyCameraMinimal( *m_renderCamera, *renderCameraInfo->m_camera );
                    m_renderCamera->setCameraName( renderCameraInfo->m_camera->getCameraName() );
                }
                else
                {
                    m_renderCamera = HK_NULL;
                }
            }
        }
        else
        {
            m_renderCamera = HK_NULL;
        }
    }

    // Apply render camera and update projection/modelview matrices
    {
        hkRefPtr<hkgCamera> currentCamera = viewport.getCamera();
        if ( m_renderCamera )
        {
            // We use the setCamera function to automatically adjust viewport-dependent properties
            // like aspect ratio for the server camera.
            viewport.setCamera( m_renderCamera );

            // If we have a currentCamera, copy renderCamera details over to that camera.
            // That way, when the caller stops supplying the renderCameraName
            // we'll be right where we left off from the render camera.
            // This could be user-configurable down the road.
            if ( currentCamera )
            {
                copyCameraMinimal( *currentCamera, *m_renderCamera );
                viewport.setCamera( currentCamera );
            }
        }
        else if ( currentCamera )
        {
            currentCamera->setAspect( float( viewport.getWidth() ) / viewport.getHeight() );
            currentCamera->computeProjection();
            currentCamera->computeModelView();
        }
    }

    // Render and sink camera
    if ( hkgCamera* camera = viewport.getCamera() )
    {
        viewport.setAsCurrent( &context );
        if ( m_sinkedCamera ) copyCameraMinimal( *m_sinkedCamera, *camera );
    }

    context.unlock();
}

hkResult hkgVdbCameraControl::setGlobalCameraFlags( hkgVdbCameraFlags flags )
{
    hkgDisplayContext& context = *m_window->getContext();
    context.lock();
    {
        updateCameraFlags( m_globalCameraInfo, &flags );
    }
    context.unlock();
    return HK_SUCCESS;
}

bool hkgVdbCameraControl::removeCameras(
    hkArrayView<const char*> cameraNames )
{
    bool removed = false;
    if ( cameraNames.getSize() )
    {
        if ( hkVdbValidity<hkArrayView<const char*>>::isValid( cameraNames ) )
        {
            for ( int i = 0; i < cameraNames.getSize(); i++ )
            {
                const char* cameraName = cameraNames[i];
                auto iter = m_nameToCameraInfo.findKey( cameraName );
                if ( m_nameToCameraInfo.isValid( iter ) )
                {
                    hkgVdbCameraInfo infoCopy = m_nameToCameraInfo.getValue( iter );
                    m_nameToCameraInfo.remove( iter );
                    m_cameraRemoved.fire( infoCopy );
                    removed = true;
                }
            }
        }
        else
        {
            hkLocalArray<hkgVdbCameraInfo> removedInfos( m_nameToCameraInfo.getSize() );
            removed = m_nameToCameraInfo.getSize();
            for ( auto iter = m_nameToCameraInfo.getIterator();
                m_nameToCameraInfo.isValid( iter );
                iter = m_nameToCameraInfo.getNext( iter ) )
            {
                removedInfos.emplaceBack( m_nameToCameraInfo.getValue( iter ) );
            }
            m_nameToCameraInfo.clear();
            for ( int i = 0; i < removedInfos.getSize(); i++ )
            {
                m_cameraRemoved.fire( removedInfos[i] );
            }
        }
    }
    return removed;
}

void hkgVdbCameraControl::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    hkgDisplayContext& context = *m_window->getContext();
    context.lock();

    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        m_nameToCameraInfo.clear();
        m_sinkedCamera = hkRefNew<hkgCamera>( hkgCamera::create() );
    }
    else if ( use == hkVdbConnectionUse::SINK )
    {
        m_sinkedCamera = hkRefNew<hkgCamera>( hkgCamera::create() );
    }

    context.unlock();
}

void hkgVdbCameraControl::onDisconnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    hkgDisplayContext& context = *m_window->getContext();
    context.lock();
    if ( use == hkVdbConnectionUse::SINK )
    {
        m_sinkedCamera = HK_NULL;
    }
    context.unlock();
}

void hkgVdbCameraControl::onWritingToSinkSignal( hkVdbConnection& connection, const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, const hkVdbCmd*& cmdToWriteOut )
{
    if ( cmd.getType() == hkVdbCmdType::STEP )
    {
        hkgDisplayContext& context = *m_window->getContext();
        context.lock();
        // Add our current rendering camera into this step so it'll be in the recording
        {
            HK_ASSERT( 0x22441129, m_sinkedCamera, "Sinked camera was not created" );

            if ( hkStreamWriter* sinkWriter = connection.getWriter() )
            {
                // Write the cmd to the buffer
                hkArray<char> cmdBuffer;
                {
                    // Need to sync to the render camera if we have one to avoid one-off frame problems
                    if ( m_renderCamera )
                    {
                        if ( const hkgVdbCameraInfo* renderCameraInfo = getCameraInfo( m_renderCamera->getCameraName() ) )
                        {
                            copyCameraMinimal( *m_sinkedCamera, *renderCameraInfo->m_camera );
                        }
                    }

                    hkVdbLocalOStream os( cmdBuffer );
                    hkCriticalSection cs( 1000 );
                    hkServerDebugDisplayHandler sdh( &os, &cs );

                    hkVector4 vfrom;
                    m_sinkedCamera->getFrom( vfrom );
                    hkVector4 vto;
                    m_sinkedCamera->getTo( vto );
                    hkVector4 vup;
                    m_sinkedCamera->getUp( vup );

                    sdh.updateCamera(
                        "Recorded Camera",
                        vfrom,
                        vto,
                        vup,
                        m_sinkedCamera->getNear(),
                        m_sinkedCamera->getFar(),
                        m_sinkedCamera->getFOV() );
                }

                // Write the buffer as a dispatch cmd (so protocol can be
                // used during deserialization despite current server protocol).
                if ( cmdBuffer.getSize() )
                {
                    hkVdbInternalHandler::writeDispatchCmd(
                        *sinkWriter,
                        *reinterpret_cast< hkVdbCmd* >( cmdBuffer.begin() ),
                        protocol );
                }
            }
        }
        context.unlock();
    }
}

void hkgVdbCameraControl::onCameraCmdReceivedSignal( const hkVdbDisplayHandler::CameraCmd& cameraCmd, hkVdbSignalResult& result )
{
    hkgDisplayContext& context = *m_window->getContext();
    context.lock();

    if ( const hkVdbDisplayHandler::UpdateCameraCmd* updateCmd = cameraCmd.asUpdateCmd() )
    {
        // Create camera if needed
        const hkgVdbCameraInfo* serverCameraInfo = getCameraInfo( updateCmd->m_name );
        if ( !serverCameraInfo )
        {
            hkgVdbCameraFlags noFlags = 0;
            serverCameraInfo = updateCameraInternal(
                updateCmd->m_name,
                true,
                HK_NULL,
                &noFlags,
                HK_NULL );
        }

        // Copy minimal state of the camera
        if ( serverCameraInfo )
        {
            hkgCamera* camera = serverCameraInfo->m_camera;
            camera->setFrom( updateCmd->m_from );
            camera->setTo( updateCmd->m_to );
            camera->setUp( updateCmd->m_up );
            camera->setNear( updateCmd->m_nearPlane );
            camera->setFar( updateCmd->m_farPlane );
            camera->setFOV( updateCmd->m_fov );
            m_cameraUpdated.fire( *serverCameraInfo );
        }
        else
        {
            result.signalError( *this );
        }
    }
    else if ( const hkVdbDisplayHandler::RemoveCameraCmd* removeCmd = cameraCmd.asRemoveCmd() )
    {
        
        const char* cameraName = removeCmd->m_name;
        removeCameras( hkArrayViewT::fromSingleObject( cameraName ) );
    }
    else
    {
        HK_VDB_SIGNAL_ERROR_MSG( 0xadb00000, hkVdbError::PLUGIN_ERROR, "Unknown camera cmd" );
        result.signalError( *this );
    }

    context.unlock();
}

void hkgVdbCameraControl::onPlaybackInfoReceivedSignal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result )
{
    if ( info.flagWasSet( hkVdbPlaybackFlags::FRAME_ENDED ) )
    {
        hkgDisplayContext& context = *m_window->getContext();
        context.lock();
        {
            if ( m_renderCamera )
            {
                if ( const hkgVdbCameraInfo* renderCameraInfo = getCameraInfo( m_renderCamera->getCameraName() ) )
                {
                    copyCameraMinimal( *m_renderCamera, *renderCameraInfo->m_camera );
                }
            }
        }
        context.unlock();
    }
}

hkgVdbCameraInfo* hkgVdbCameraControl::updateCameraInternal(
    const char* cameraName,
    hkBool32 isFromServer,
    const hkgCamera* cameraProperties,
    const hkgVdbCameraFlags* cameraFlags,
    const hkUint64* poiId )
{
    hkgDisplayContext& context = *m_window->getContext();
    context.lock();
    auto iter = m_nameToCameraInfo.findOrInsertKey( cameraName, hkgVdbCameraInfo() );
    auto && value = m_nameToCameraInfo.getValue( iter );
    {
        // Do initialization if needed for optional args
        if ( !value.m_camera )
        {
            initCameraInfo( value, cameraName );
        }

        // Update
        {
            value.m_server = isFromServer;
            updateCameraProperties( value, value.m_lastFrom, value.m_lastTo, cameraProperties );
            updateCameraFlags( value, cameraFlags );
            if ( poiId )
            {
                value.m_poiId = *poiId;
            }
        }
    }
    context.unlock();

    return &value;
}

void hkgVdbCameraControl::stepCameraInfo( hkgVdbCameraInfo& cameraInfo ) const
{
    // Server camera's aren't defined locally
    if ( cameraInfo.m_server )
    {
        return;
    }

    hkBool32 lookAtPOI = cameraInfo.m_flags.anyIsSet( hkgVdbCameraFlags::LOOK_AT_PRIMARY_POINT_OF_INTEREST );
    hkBool32 lookAtCenter = cameraInfo.m_flags.anyIsSet( hkgVdbCameraFlags::LOOK_AT_PRIMARY_GEOMETRY_CENTER );
    hkBool32 keepRelativePOI = cameraInfo.m_flags.anyIsSet( hkgVdbCameraFlags::KEEP_RELATIVE_TO_PRIMARY_POINT_OF_INTEREST );
    hkBool32 keepRelativeCenter = cameraInfo.m_flags.anyIsSet( hkgVdbCameraFlags::KEEP_RELATIVE_TO_PRIMARY_GEOMETRY_CENTER );
    if ( lookAtPOI || lookAtCenter || keepRelativePOI || keepRelativeCenter )
    {
        if ( const hkgVdbSelectionWidget::PointOfInterest* poi =
            m_selectionWidget->getPointOfInterest( hkgVdbSelectionWidget::PrimaryPointOfInterest ) )
        {
            // These options only work for local pois.
            lookAtCenter = ( lookAtCenter && poi->m_geometryId );
            keepRelativeCenter = ( keepRelativeCenter && poi->m_geometryId );

            // Get target pos (might be center of local poi).
            hkVector4 targetPos = poi->m_worldSpacePos;
            if ( lookAtCenter || keepRelativeCenter )
            {
                if ( const hkgDisplayObject* displayObject = m_geometryControl.getGeometryFromId( poi->m_geometryId ) )
                {
                    hkBool32 validAabb = false;
                    if ( !displayObject->hasAABB() )
                    {
                        const_cast< hkgDisplayObject* >( displayObject )->computeAABB();
                    }
                    if ( displayObject->hasAABB() )
                    {
                        HK_ALIGN16( float vec3[3] );
                        hkgVec3Transform( vec3, displayObject->getAABBCentPtr(), displayObject->getTransform() );
                        targetPos.load<3>( vec3 );
                        validAabb = true;
                    }

                    // Just use transform WS as best guess?
                    // Could be way off depending on vertices, etc. of geometries.
                    if ( !validAabb )
                    {
                        HK_ALIGN16( float mat4[16] );
                        hkgMat4Copy( mat4, displayObject->getTransform() );
                        hkTransform t; t.set4x4ColumnMajor( mat4 );
                        targetPos = t.getTranslation();
                    }
                }
                else
                {
                    // Not a valid local POI, disable these options
                    lookAtCenter = false;
                    keepRelativeCenter = false;
                }
            }

            // Apply keep relative
            if ( keepRelativePOI || keepRelativeCenter )
            {
                HK_ALIGN16( float hkgvec[3] );
                cameraInfo.m_camera->getFrom( hkgvec );
                hkVector4 currentFrom; currentFrom.load<3>( hkgvec );
                if ( cameraInfo.m_keepRelativeOffset.getW() == -1 )
                {
                    cameraInfo.m_keepRelativeOffset.setXYZ0( currentFrom - targetPos );
                    cameraInfo.m_lastFrom = currentFrom;
                    cameraInfo.m_camera->getTo( hkgvec );
                    cameraInfo.m_lastTo.load<3>( hkgvec );
                }
                else
                {
                    cameraInfo.m_camera->getTo( hkgvec );
                    hkVector4 currentTo; currentTo.load<3>( hkgvec );
                    hkVector4 newFrom;
                    hkVector4 newTo;
                    newFrom = ( targetPos + cameraInfo.m_keepRelativeOffset );
                    newTo = ( currentTo + ( newFrom - currentFrom ) );
                    cameraInfo.m_camera->setFrom( newFrom );
                    cameraInfo.m_camera->setTo( newTo );
                    cameraInfo.m_lastFrom = newFrom;
                    cameraInfo.m_lastTo = newTo;
                }
            }

            // Apply look at
            if ( lookAtPOI || lookAtCenter )
            {
                cameraInfo.m_camera->setTo( targetPos );
                cameraInfo.m_lastTo = targetPos;
            }
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
