// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#ifndef HAVOK_VDB_DESERIALIZER
#define HAVOK_VDB_DESERIALIZER

#include <Common/Base/System/Stopwatch/hkStopwatch.h>
#include <Common/Visualize/hkVisualDebuggerCmdType.h>

// Handlers for the bits
class hkVdbPluginHandler;
class hkVdbDestructionHandler;
class hkVdbCameraHandler;
class hkVdbDataObjectHandler;
class hkVdbPerfStatsHandler;
class hkVdbConsoleHandler;
class hkVdbDisplayHandler;
class hkVdbMemorySnapshotHandler;
class hkVdbCache;
class hkVdbGeometryCache;
struct hkVdbCmd;
class hkVdbIStream;
namespace hkSerialize
{
    class TagfileReadFormat;
    class Load;
    class Save;
}

//
class hkVdbIStream;
class hkStreamReader;
class hkStreamWriter;
class hkMonitorCaptureStreamUtility;

// the Deserializer is given chunks in the form of a streambuf and decodes them to
// commands
class hkVdbDeserializer
{
public:

    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DISPLAY, hkVdbDeserializer );

    enum Status
    {
        HK_OK = 0,
        HK_FAIL,
        HK_VERSION_CLIENT_OLDER,
        HK_VERSION_CLIENT_NEWER,
        HK_VERSION_CONFLICT,
        HK_WARNING, // The deserializer is OK but there is a warning which means some data might be missing
    };

    enum Mode
    {
        HK_NOT_CONNECTED = 0,
        HK_LIVE_PLAYBACK,
        HK_MOVIE_CACHE,
        HK_MOVIE_PLAYBACK
    };

    hkVdbDeserializer(
        hkVdbLiveConnection* connection,
        hkVdbPluginHandler* pluginHandler,
        hkVdbDestructionHandler* destructionHandler,
        hkVdbDisplayHandler* displayHandler,
        hkVdbPerfStatsHandler* statisticsHandler,
        hkVdbPerfStatsHandler* memStatsHandler,
        hkVdbCameraHandler* cameraHandler,
        hkVdbConsoleHandler* consoleHandler,
        hkVdbMemorySnapshotHandler* memorySnapshotHandler,
        hkVdbDataObjectHandler* objectHandler,
        hkVdbGeometryCache* geometryCache );

    ~hkVdbDeserializer();

    inline Mode getMode() const { return m_mode; }
    inline void setMode( Mode mode )
    {
        m_mode = mode;
        if ( mode == HK_MOVIE_PLAYBACK ) { resetFilteredViewers(); }
    }

    inline void setContinueOnUnknownClass( hkBool newVal ) { m_continueOnUnknownClass = newVal; }
    inline hkBool getContinueOnUnknownClass() const { return m_continueOnUnknownClass; }

    void setStepCallback( hkVdbStepCallback c, void* userHandle ) { m_stepCallback = c; m_stepHandle = userHandle; }
    void setVersionCallback( hkVdbVersionCallback c, void* userHandle ) { m_versionCallback = c; m_versionHandle = userHandle; }
    void setReflectViewerStateCallback( hkVdbReflectViewerStateCallback c, void* userHandle ) { m_reflectViewerStateCallback = c; m_reflectViewerStateHandle = userHandle; }

    Status processChunk( hkVdbCmd* chunk );

    // In milliseconds
    inline float getCurrentChunkTime() const { return m_currentTime; }
    inline int getCurrentChunkStepCount() const { return m_currentStepCount; }
    inline int getCurrentFrameSize() const { return m_currentFrameByteSize; }

    inline float getServerFps() { return m_serverFps; }

    // call this when you know that sim has been reset, before deserializing more frames.
    inline void resetCounters() { setCurrentChunkTime( 0 ); setCurrentChunkStepCount( 0 ); }
    inline void setCurrentChunkTime( float t ) { m_currentTime = t; }
    inline void setCurrentChunkStepCount( int c ) { m_currentStepCount = c; }

    inline const char* getRemotePlatformString() const { return m_remotePlatformString.cString(); };

    inline bool gotRemoteVersion() { return m_remoteProtocolVersion != -1; }
    inline int getRemoteProtocolVersion() { return m_remoteProtocolVersion; }
    inline int getRemoteProtocolMinimumCompatible() { return m_remoteProtocolMinimumCompatible; }

    bool gotServerLayout();

    void filterViewer( int id, bool filter );
    bool isViewerFiltered( int id );

    void movieSelectViewer( int id, bool selected );
    bool isMovieViewerSelected( int id );

    void resetFilteredViewers();

protected:

    bool viewerIsPresent( int tag );
    void setServerPointerSize( hkUchar pointerSize );
    void setServerIsLittleEndian( hkBool isLittleEndian );

    void processStep( hkVdbIStream& reader );
    void processServerInfo( hkVdbIStream& reader );
    void processAddGeometry( hkVdbCmdType::Enum type, hkVdbIStream& reader );
    void processInstanceGeometry( hkVdbIStream& reader );
    void processUpdateGeometryTransform( hkVdbCmdType::Enum type, hkVdbIStream& reader );
    void processUpdateGeometryVertices( hkVdbIStream& reader );
    void processSetGeometryColor( hkVdbCmdType::Enum type, hkVdbIStream& reader );
    void processUpdateGeometryFlags( hkVdbCmdType::Enum type, hkVdbIStream& reader );
    void processRemoveGeometry( hkVdbIStream& reader );
    void processDisplayGeometry( hkVdbCmdType::Enum type, hkVdbIStream& reader );
    void processDisplayPoint( hkVdbCmdType::Enum type, hkVdbIStream& reader );
    void processDisplay2Points( hkVdbCmdType::Enum type, hkVdbIStream& reader );
    void processDisplay3Points( hkVdbCmdType::Enum type, hkVdbIStream& reader );
    void processDisplayText( hkVdbCmdType::Enum type, hkVdbIStream& reader );
    void processDisplay3dText( hkVdbIStream& reader );
    void processUpdateCamera( hkVdbIStream& reader );
    void processRegisterProcess( hkVdbIStream& reader );
    void processSelectProcess( hkVdbIStream& reader );
    void processPerfStats( hkVdbCmdType::Enum type, hkVdbIStream& reader );
    void processStatsMaps( hkVdbCmdType::Enum type, hkVdbIStream& reader );
    void processLiveObject( hkVdbIStream& reader );
    void processInternal( hkVdbCmd& cmd, hkVdbCmdType::Enum type, hkVdbIStream& reader );

protected:

    Mode m_mode;

    hkVdbLiveConnection* m_connection;
    hkVdbDisplayHandler* m_displayHandler;
    hkVdbPluginHandler* m_pluginHandler; 
    hkVdbDestructionHandler* m_destructionHandler;
    hkVdbPerfStatsHandler* m_perfStatsHandler;
    hkVdbPerfStatsHandler* m_memStatsHandler;
    hkVdbCameraHandler* m_cameraHandler;
    hkVdbConsoleHandler* m_consoleHandler;
    hkVdbDataObjectHandler* m_objectHandler;
    hkVdbMemorySnapshotHandler* m_memorySnapshotHandler;
    hkVdbGeometryCache* m_geometryCache;

    hkVdbStepCallback m_stepCallback;
    void* m_stepHandle;

    hkVdbVersionCallback m_versionCallback;
    void* m_versionHandle;

    hkVdbReflectViewerStateCallback m_reflectViewerStateCallback;
    void* m_reflectViewerStateHandle;

    float m_currentTime;
    int m_currentStepCount;
    int m_currentFrameByteSize;

    // Track the rate of steps coming from the server
    int m_serverFpsFrames;
    float m_serverFps;
    hkStopwatch m_serverFpsTimer;

    // Store of the remote version information if it has been received
    // (it will be -1 of not)
    int m_remoteProtocolVersion;
    int m_remoteProtocolMinimumCompatible;

    // storage for remote magic platform string
    hkStringPtr m_remotePlatformString;

    // storage for remote compiler string
    hkStringPtr m_remoteCompilerString;

    hkArray<int> m_movieSelectedViewers;
    hkArray<int> m_filteredViewers;

    // These are all of the objects read between HK_BEGIN_OBJECT and HK_END_OBJECT.
    // They are discarded after HK_END_OBJECT.
    hkArray<hkUint64> m_tempObjects;

    // Whether we are between HK_BEGIN_OBJECT and HK_END_OBJECT.
    hkBool m_objectBegun;

    // Whether or not to silently ignore an unknown class when processing an HK_LIVE_OBJECT command
    // Defaults to false, but is set to true in the VDB frontend
    hkBool m_continueOnUnknownClass;

public:
    void sendObject( hkStreamWriter* writer, hkReferencedObject* object );
    void clearSerializers();

protected:
    hkSerialize::Load* getObjectDeserializer( int tag );
    hkSerialize::Save* getObjectSerializer();
    // Map of deserializers. One per process.
    hkHashMap<int, hkSerialize::Load*> m_deserializers;
    // Util to write objects back to connected server
    hkSerialize::Save* m_objectSerializer;
    HK_DEBUG_ONLY_MEMBER( const hkVdbCmd*, m_lastProcessedCmd );
};

#endif // HAVOK_VDB_DESERIALIZER

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
