// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#ifndef HAVOK_VDB_COMMAND_CACHE_H
#define HAVOK_VDB_COMMAND_CACHE_H

#include <Common/Base/Types/hkSignalSlots.h>
#include <Common/Base/Container/Hash/hkHashMap.h>
#include <Common/Base/Container/Set/hkSet.h>
#include <Common/Base/Container/BitField/hkBitField.h>

struct hkVdbCmd;
class hkVdbLiveConnection;
class hkVdbDeserializer;
class hkVdbDisplayHandler;
class hkVdbPerfStatsHandler;
namespace hkSerialize { class TagfileReadFormat; }





/// This class holds a frame of vdb command data.
struct hkVdbFrame
{
    hkVdbFrame();

    //
    // Iteration
    //

    typedef const hkVdbCmd* Iterator;
    Iterator getIterator() const;
    Iterator getNext( Iterator i ) const;
    Iterator getPrevious( Iterator i ) const;
    const hkVdbCmd* getValue( Iterator i ) const;
    hkBool32 isValid( Iterator i ) const;

    //
    // Query
    //

    int getFrameNumber() const { return m_frameNumber; }
    int getNumChunks() const { return m_numChunks; }
    int getTotalBytes() const { return m_chunkStorage.getSize() + m_customStorage.getSize(); }
    float getStartTime() const { return m_startTime; }
    float getDuration() const { return m_duration; }

    //
    // Custom Data
    // This storage is determined by calling hkVdbCache::setCustomFrameBytes()
    //

    hkInt8* getCustomBytes( hkUint32 id, int& numBytesOut ) const;
    template<typename Type> Type* getCustomObjects( hkUint32 id, int numObjects = 1 ) const
    {
        const int expectedBytes = sizeof( Type ) * numObjects;
        int numBytes;
        if ( hkInt8* customBytes = getCustomBytes( id, numBytes ) )
        {
            if ( expectedBytes <= numBytes )
            {
                return reinterpret_cast<Type*>( customBytes );
            }
        }
        return HK_NULL;
    }

private:

    /// Peek at the current chunk we are reading.
    const hkVdbCmd* peek() const { return m_currentChunk.peek(); }

    /// Read as much from the stream as is available and advance to the next chunk
    /// if we complete our current chunk.
    const hkVdbCmd* readAndAdvance( hkVdbLiveConnection* connection );

    /// Reset the frame so it can be reused.
    /// No memory is deallocated.
    void reset();

    /// This class represents the chunk of vdb command data that isn't fully read yet.
    struct hkVdbPartialChunk
    {
        hkVdbPartialChunk( int* numChunks = HK_NULL, hkArray<hkInt8>* backingStore = HK_NULL );

        /// Peek at the current, incomplete, vdb chunk.
        const hkVdbCmd* peek() const;

        /// We are done with the current chunk, advance in the backing store to a new vdb chunk.
        /// Return the chunk we just advanced over (not the new one).
        const hkVdbCmd* advance();

        /// Get the size of the vdb chunk header.
        int getHeaderBytes() const { return 4; }

        /// Get the size of the vdb chunk body (only valid after the header has been read).
        int getBodyBytes() const;

        /// Get the location in the current vdb chunk to write more data.
        hkInt8* getWriteLoc();

        /// Set the num of bytes reserved for this vdb chunk.
        void reserve( int numBytes );

        /// Set the number of bytes we've written to for this vdb chunk.
        void setCompleted( int numBytes );

        /// Get the number of bytes we've written to for this vdb chunk.
        int getNumCompleted() const;

        /// Clear our completed bytes.
        void clear();

    private:

        int m_partialChunkStartIndex;
        int* m_numChunks;
        hkArray<hkInt8>* m_backingStore;

        friend struct hkVdbFrame;
    };

    /// This class wraps a subset of bytes that live in an external hkArray of hkInt8s to be used for custom data.
    class CustomStorage : public hkArray<hkInt8>
    {
    public:
        CustomStorage( hkInt8* storage = HK_NULL, int size = 0 ) : hkArray<hkInt8>( storage, size, size ) {}
        CustomStorage( const CustomStorage& cs ) { operator=( cs ); }
        HK_INLINE CustomStorage& operator= ( const CustomStorage& cs )
        {
            CustomStorage* mutableThis = const_cast< CustomStorage* >( this );
            hkString::memCpy( mutableThis, &cs, sizeof( CustomStorage ) );
            return *mutableThis;
        }
    };

    int m_frameNumber;
    int m_numChunks;
    float m_startTime;
    float m_duration;

    

    hkArray<hkInt8> m_chunkStorage;
    hkVdbPartialChunk m_currentChunk;
    hkArray<hkInt8> m_customStorage;
    mutable hkHashMap<hkUint32, CustomStorage> m_customStorageById;

    friend class hkVdbCache;
};

/// This is a circular cache (or buffer) of vdb frames.

class hkVdbCache
{
public:

    /// Create a new cache that will be buffer enough vdb frames to cover "duration" milliseconds.
    
    
    
    
    hkVdbCache( float duration = 10000.0f );

    /// Reset the cache to it's initial created state.
    /// If duration is not provided, the original duration is used.
    void reset( float duration = 0.0f );

    /// Save the contents of the cache to the stream.
    /// Note: If the cache has been filled, then the commands won't be sufficient to reproduce the current state of
    /// the Vdb (see hkVdbPersistentStateCache).
    void save( hkStreamWriter* streamWriter );

    //
    // Advancing
    //

    /// Read chunk data from the live stream.
    /// If the chunk is completed during the call, it is returned.
    /// Otherwise, HK_NULL is returned.
    const hkVdbCmd* readChunk( hkVdbLiveConnection* liveConnection );

    /// Notify the vdb cache of a chunk read by some other means.
    /// Note: This does not copy the chunk into the caches buffer it just
    /// advances the internals. This is for advanced use.
    const void notifyChunk( const hkVdbCmd* completedChunk );

    //
    // Iteration
    // These only iterate over completed frames and will, therefore, not include
    // the current frame that is being read. To get that frame see Query interfaces.
    //

    typedef int Iterator;
    Iterator getIterator( int frame = -1 ) const;
    Iterator getNext( Iterator i ) const;
    Iterator getPrevious( Iterator i ) const;
    const hkVdbFrame* getValue( Iterator i ) const;
    hkBool32 isValid( Iterator i ) const;

    //
    // Query
    //

    int getNumFrames() const;
    int getStartingFrameNumber() const { return m_startingFrameNumber; }
    float getDuration() const { return m_duration; }

    /// Peek at the current frame we are working on.
    /// This is not available through the Iterator interface.
    const hkVdbFrame* peek() const;

    /// Get the oldest frame in the buffer.
    const hkVdbFrame* begin() const;

    /// Get the newest completed frame in the buffer.
    const hkVdbFrame* back() const;

    /// Get a specific frame from the buffer.
    /// This can be the current frame, which is not available through the Iterator interface.
    const hkVdbFrame* getFrame( int frame ) const;

    //
    // Custom Data
    //

    /// Enum of different initialization operations which can be performed on custom data
    /// when it's reserved for a new vdb frame.
    enum InitializationOp
    {
        /// Don't do anything to the reserved memory.
        UNINITIALIZED = 0,
        /// Zero out the reserved memory.
        ZERO = 1,
        /// Copy the memory from the previous frame (if available).
        COPY_FROM_PREVIOUS = 2
    };

    /// This function sets the number of custom bytes that will be allocated for a particular id
    /// on any new vdb frames.  It has no impact on the current vdb frame.
    
    
    void setCustomFrameBytes( hkUint32 id, int numBytesToReserve, InitializationOp flags );

    //
    // Signals
    //

    /// This signal is fired before the cache saves its contents to a stream.
    HK_DECLARE_SIGNAL( StartSaveSignal, hkSignal2< hkVdbCache*, hkStreamWriter* > );
    StartSaveSignal m_startSave;

    /// This signal is fired after the cache saves its contents to a stream.
    HK_DECLARE_SIGNAL( EndSaveSignal, hkSignal2< hkVdbCache*, hkStreamWriter* > );
    EndSaveSignal m_endSave;

    /// This signal occurs after a frame is completed (before we advance to the next frame).
    HK_DECLARE_SIGNAL( FrameCompletedSignal, hkSignal2< hkVdbCache*, const hkVdbFrame* > );
    FrameCompletedSignal m_frameCompleted;

    /// This signal occurs after we've advanced to a new frame.
    HK_DECLARE_SIGNAL( FrameAdvancedSignal, hkSignal2< hkVdbCache*, const hkVdbFrame* > );
    FrameAdvancedSignal m_frameAdvanced;

    /// This signal occurs just before a frame is evicted (so it can be accessed).
    HK_DECLARE_SIGNAL( FrameEvictedSignal, hkSignal2< hkVdbCache*, const hkVdbFrame* > );
    FrameEvictedSignal m_frameEvicted;

private:

    /// Information about reserving new memory.
    struct MemoryReservationInfo
    {
        int m_bytesToReserve;
        InitializationOp m_flags;
    };

    /// Advance the cached to the next frame.
    /// "timeStep" should be the time the last step took to complete.
    hkVdbFrame* advance( float timeStep );

    /// Get the current frame we are working on.
    hkVdbFrame* current();

    /// Get the current frame's number.
    /// Note: current()->getFrameNumber() only works when we've completed the frame.
    int getCurrentFrameNumber() const { return ( m_startingFrameNumber + getNumFrames() ); }

    float m_duration;
    int m_startingFrameIdx;
    int m_startingFrameNumber;
    int m_currentFrameIdx;
    hkArray<hkVdbFrame> m_frameStorage;
    hkHashMap<hkUint32, MemoryReservationInfo> m_customReserveSizes;
};

/// A cinfo describing cached geometry.
struct hkVdbCachedGeometryCinfo
{
    HK_DECLARE_CLASS( hkVdbCachedGeometryCinfo, New );

    hkVdbCachedGeometryCinfo(
        int tag = -1,
        hkUint64 streamId = 0,
        int frameAdded = -1,
        const hkMatrix4& initialTransform = hkMatrix4::getIdentity(),
        
        hkColor::Argb initialColor = hkColor::rgbFromFloats( 0.4f, 0.4f, 0.4f, 1.0f ) ) :
        m_tag( tag ),
        m_streamId( streamId ),
        m_frameAdded( frameAdded ),
        m_initialTransform( initialTransform ),
        m_initialColor( initialColor )
    {}

    /// The tag for the viewer which created this geometry.
    int m_tag;

    /// The original id from the live command stream.
    hkUint64 m_streamId;

    /// The frame this geometry was added.
    int m_frameAdded;

    /// The initial transform for the geometry.
    hkMatrix4 m_initialTransform;

    /// The initial color for the geometry.
    hkColor::Argb m_initialColor;
};

/// A class representing a cached geometry.
class hkVdbCachedGeometry : protected hkVdbCachedGeometryCinfo
{
public:
    HK_DECLARE_CLASS( hkVdbCachedGeometry, New );

    
    

    //
    // Query
    //

    /// Return a unique id for this geometry which won't conflict with a geometry having the same streamId.
    /// Any number of removed geometries can have the same streamId as other geometries.
    hkUint64 getUid();

    /// Return the original streamId for the geometry.
    hkUint64 getStreamId() { return m_streamId; }

    /// Get the transform of this geometry at a particular frame.
    const hkMatrix4& getTransform( int frame ) const;

    /// Get the color of this geometry at a particular frame.
    hkColor::Argb getColor( int frame ) const;

    /// Get the vertices of this geometry at a particular frame (if any).
    const hkArray<hkVector4>* getVertices( int frame ) const;

    /// Get the skinning transforms of this geometry at a particular frame (if any).
    const hkArray<hkMatrix4>* getSkinningTransforms( int frame ) const;

    //
    // Registration/Updating
    //

    /// Register that the transform was updated on a particular frame.
    void registerTransformUpdate( int frame, const hkMatrix4& transform );

    /// Register that the color was updated on a particular frame.
    void registerColorUpdate( int frame, hkColor::Argb color );

    /// Register that the vertices were updated on a particular frame.
    void registerVerticesUpdate( int frame, hkArray<hkVector4>& newVertsForSwap );

    /// Register that the skinning transforms were updated on a particular frame.
    void registerSkinningTransformsUpdate( int frame, const hkMatrix4& transform, hkArray<hkMatrix4>& newSkinningTransformsForSwap );

    /// Register that the geometry was removed on a particular frame.
    void registerGeometryRemoved( int frame );

protected:

    /// Flags indicating various states of the geometry.
    enum StateFlags
    {
        /// The geom has been added to the display handler
        GEOM_ADDED = 1 << 0,
        /// The geom is currently visible
        GEOM_VISIBLE = 1 << 1,
        /// The geom is currently mapped to the streamId in the geometry cache map.
        GEOM_OWNS_STREAM_ID = 1 << 2,
        /// Has vertices updates
        GEOM_HAS_VERTICES_UPDATES = 1 << 3,
        /// Has skinning transforms updates
        GEOM_HAS_SKINNING_TRANSFORMS_UPDATES = 1 << 4,

        /// Both added and visible
        GEOM_ADDED_VISIBLE_MASK = ( GEOM_ADDED | GEOM_VISIBLE )
    };

    hkVdbCachedGeometry(
        hkVdbCachedGeometryCinfo& cinfo,
        hkVdbCache* vdbCache,
        int geometryCachedIndex );
    ~hkVdbCachedGeometry();

    /// Returns if this geometry is added on the particular frame in question.
    /// That is, if it *was* added before the frame and hasn't been removed yet or was removed *after* the frame.
    hkBool32 isAdded( int frame ) const;

    /// Create sparse data for the cached geometry.
    /// This is data which we don't place in the vdb frame's custom byte buffer (to be allocated every frame).
    template<typename T>
    T* createSparseData()
    {
        
        T* sparseData = new T();
        m_geometriesSparseData.insert( sparseData, sparseData );
        return sparseData;
    }

    /// Delete previously created sparse data.
    template<typename T>
    void deleteSparseData( T* sparseData )
    {
        
        if ( sparseData )
        {
            hkHashMap<const void*, hkReflect::Var>::Iterator iter = m_geometriesSparseData.findKey( reinterpret_cast< const void* >( sparseData ) );
            HK_ASSERT_NO_MSG( 0x22440847, m_geometriesSparseData.isValid( iter ) );
            if ( m_geometriesSparseData.isValid( iter ) )
            {
                m_geometriesSparseData.getValue( iter ).destroy();
                m_geometriesSparseData.remove( iter );
            }
        }
    }

    /// Get our custom frame bytes for the specified id
    hkInt8* getCustomFrameBytes( hkUint32 id, int frame, int elementSize );

    /// Get our custom frame bytes for the specified id
    template<typename T> T* getCustomFrameBytes( hkUint32 id, int frame )
    { return reinterpret_cast< T* >( getCustomFrameBytes( id, frame, sizeof( T ) ) ); }

    hkVdbCache* m_vdbCache;

    int m_geometryCachedIndex;
    int m_frameRemoved;

    hkFlags<StateFlags, hkInt8> m_stateFlags;

    hkArray<hkMatrix4>* m_initialSkinningTransforms;
    hkArray<hkVector4>* m_initialVertices;

    
    
    hkHashMap<const void*, hkReflect::Var> m_geometriesSparseData;

    friend class hkVdbGeometryCache;
};

/// A cache of geometries.
/// Some of the geometry data lives in the vdb cache back-end, this class (and hkVdbCachedGeometry) simplifies
/// the retrieving and storage of extra information to be able to quickly zip around the cache.
class hkVdbGeometryCache
{
public:

    /// Create a new geometry cache with the provided vdb cache back-end.
    hkVdbGeometryCache( hkVdbCache* vdbCache );
    ~hkVdbGeometryCache();

    /// Get the total number of geometries in the cache.
    int getNumGeometries() { return m_geometries.getSize(); }

    /// Create a new cached geometry.
    hkVdbCachedGeometry* createGeometry( hkVdbCachedGeometryCinfo& cinfo );

    /// Access a previously created "live" geometry by streamId.
    /// This is a hash-table lookup.
    
    
    hkVdbCachedGeometry* accessGeometry( hkUint64 streamId );

    /// Search for a particular geometry by its uid.
    hkVdbCachedGeometry* findGeometryByUid( hkUint64 uid );

    /// Clear the geometry cache.
    void clear();

    /// Using the provided deserializer and displayHandler,
    /// replay the data in the cache from startFrame -> endFrame.
    /// This interface assumes that frames will not be skipped in subsequent calls, starting at 0.
    
    
    
    void replay(
        int startFrame,
        int endFrame,
        hkVdbDeserializer* deserializer,
        hkVdbDisplayHandler* displayHandler );

    /// Hide all geometry in the cache using the provided displayHandler.
    /// Optionally remove references on the hidden objects.
    
    void hideAllGeometry( hkVdbDisplayHandler* displayHandler, bool removeReferences );

    //
    // Signal callbacks
    //

    void onFrameCompletedSignal( hkVdbCache* cache, const hkVdbFrame* frame );
    void onFrameAdvancedSignal( hkVdbCache* cache, const hkVdbFrame* frame );
    void onFrameEvictedSignal( hkVdbCache* cache, const hkVdbFrame* frame );

protected:

    void showGeometry( hkVdbCachedGeometry &geom, hkVdbDisplayHandler* displayHandler );
    void hideGeometry( hkVdbCachedGeometry &geom, hkVdbDisplayHandler* displayHandler );

    
    static const hkUint32 HK_VDB_TRANSFORM_CACHE_ID = 1;
    static const hkUint32 HK_VDB_COLOR_CACHE_ID = 2;
    static const hkUint32 HK_VDB_VERTEX_CACHE_ID = 3;
    static const hkUint32 HK_VDB_SKINNING_TRANSFORM_CACHE_ID = 4;

    hkVdbCache* m_vdbCache;
    hkInt8* m_currentFrameTransformsCache;
    hkInt8* m_currentFrameColorCache;
    hkArray<hkVdbCachedGeometry*> m_geometries;
    hkArray<hkVdbCachedGeometry*> m_newlyCreatedGeometries;
    hkBitField m_geometryIndicesPool;
    hkHashMap<hkUint64, hkVdbCachedGeometry*> m_geometriesByStreamId;
    int m_numGeometriesLastFrame;

    friend class hkVdbCachedGeometry;
};

#endif // HAVOK_VDB_COMMAND_CACHE_H

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
