// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#ifndef TYPE_OUT_H
#define TYPE_OUT_H

PRAGMA_WARNING_PUSH
#include <clang/Basic/SourceLocation.h>
PRAGMA_WARNING_POP

#include "utils/TextBuilder.h"
#include "utils/StlUtils.h"

namespace clang
{
    class NamedDecl;
    class EnumDecl;
    class TypedefDecl;
    class CXXRecordDecl;
    class FieldDecl;
}

class Database;

/// Contains the information to produce a reflected type. Can be filled with data items and optionals.
/// Data items will be printed as variables in an _Auto struct (details depend on the type of Decl).
class TypeOut
{
    public:

        TypeOut(const Database& db, const clang::NamedDecl* decl);

        /// Specify special behaviors for producing the definition of a data item.
        enum DataFlags
        {
            FLAG_NONE           = 0,

            /// The item is declared externally and must be only defined.
            FLAG_EXTERN_DECL    = 0x1,

            /// The item must be initialized by constructor call rather than static initialization.
            FLAG_CONSTRUCTOR    = 0x2,

            /// The type of the item is another item of this TypeOut and must be decorated with the appropriate qualifier.
            FLAG_QUALIFY_TYPE   = 0x4,

            /// The OPT will pull in code and should be conditionally guarded
            FLAG_IS_CODE        = 0x8,

            /// Weaker version of FLAG_IS_CODE, the opt is needed to construct types so sometimes we still want it
            FLAG_NEEDED_FOR_CONSTRUCTION = 0x16
        };

        /// Add a data item. The item will be declared inside the _Auto struct and defined outside.
        /// Definition will be printed in the form: <type> ...::_Auto::[<section>::]<name> = { <defn> }.
        /// The sectionParentString is placed after the section name so the section struct can optionally subclass other types.
        TypeOut& addData(const std::string& name, const std::string& type, const std::string& defn, const std::string& section = "", int flags = FLAG_NONE, const std::string& sectionParentString = "");

        /// Add a data item without external definition. decl will be just pasted in the _Auto struct.
        TypeOut& addData(const std::string& decl);

        /// Add an opt.
        TypeOut& addOpt(clang::SourceLocation srcLoc, const std::string& opt, const std::string& name, int flags = FLAG_NONE);

        /// Add a data item, then add its address as an opt.
        TypeOut& addOpt(clang::SourceLocation srcLoc, const std::string& opt, const std::string& name,
            const std::string& type, const std::string& defn, const std::string& section = "", int flags = FLAG_NONE);

        /// Add a data item, then add its typecast address as an opt.
        TypeOut& addOptCast(clang::SourceLocation srcLoc, const std::string& opt, const std::string& name,
            const std::string& realtype, const std::string& casttype, const std::string& defn,
            const std::string& section = "", int flags = FLAG_NONE);

        bool hasOpt(const std::string& opt) const;

        bool hasOpts() const { return m_items.size() > 0; }

        TypeOut& setFormat(clang::SourceLocation srcLoc, const std::string& format)
        {
            return addOpt(srcLoc, "Opt::FORMAT", format);
        }
        TypeOut& setSubType(clang::SourceLocation srcLoc, const std::string& subType)
        {
            return addOpt(srcLoc, "Opt::SUBTYPE", subType);
        }

        TypeOut& addFlag(clang::SourceLocation srcLoc, const std::string& flag);

        TypeOut& setParent(const std::string& p)
        {
            m_parent = p;
            return *this;
        }

        void addSizeAlign(const clang::SourceLocation& loc, const std::string& size, const std::string& align, const std::string& reqAlign="0")
        {
            const std::string& val = formatString("HK_REFLECT_TYPE_OPTIONAL_SIZE_ALIGN(%s,%s,%s)", size.c_str(), align.c_str(), reqAlign.c_str() );
            addOpt(loc, "Opt::SIZE_ALIGN", val);
        }

        TextBuilder::Section& getRawCode() { return m_rawCode;  }

        /// Raw code section that can be included multiple times
        TextBuilder::Section& getRawCodeTypeDeclarations() { return m_rawCodeTypeDeclarations; }

        /// Returns true if the type has template parameters (pure template or partial specialization).
        bool hasTemplateParams() const { return !m_tplHeader.empty(); }

        /// Get the qualifier to be used to reference data items from outside the _Auto struct
        /// (e.g. "MyClass::_Auto::")
        std::string getDataQualifier() const { return m_dataQualifier; }

        /// Returns the template header of the type.
        std::string getTemplateHeader() const { return m_tplHeader; }

        /// Appends all the data definitions and the reflected type to the program listing.
        /// Specializes ReflectionOf and adds a type registration entry if necessary.
        void appendTo(TextBuilder& cxxBody, TextBuilder& inlBody) const;

        /// Appends the data definition.
        void appendDataTo(TextBuilder::Section& section) const;

        /// Appends the reflected type content.
        void appendTypeContentTo(TextBuilder::Section& section) const;

    protected:

        struct DataItem
        {
            std::string type;
            std::string name;
            std::string defn;
            std::string section;
            std::string sectionParentString;
            int flags;
        };

        /// Optional value.
        struct OptItem
        {
            enum CodeFilter
            {
                ALWAYS_INCLUDE = 0,
                INCLUDE_FOR_CONSTRUCTION, // Not for data only but needed if we want to construct the type
                INCLUDE_FOR_FULL_REFLECTION, // Only needed for the full reflection
            };
            std::string opt;
            std::string name;
            CodeFilter isCodeOpt;
        };

        void appendItemsTo(TextBuilder::Section& declarations, TextBuilder::Section& definitions) const;
        void appendAutoTo(TextBuilder::Section& section, const std::string& declarations) const;
        void appendOptsTo(TextBuilder::Section& section) const;

        std::string getTypeRegEntrySym() const;

        // not implemented
        TypeOut& operator=(const TypeOut& rhs);

        const Database& m_db;
        const clang::NamedDecl* m_decl;

        /// Template header, prepended to every declaration. Null if the decl has no params.
        std::string m_tplHeader;

        /// Qualifier used for definitions and to reference data items from outside (e.g. "MyClass::_Auto::").
        std::string m_dataQualifier;

        /// Name of the _Auto struct.
        std::string m_dataStructName;

        bool m_declareReflection;
        bool m_externalReflection;

        std::vector<OptItem> m_items;
        std::set<std::string> m_flags;
        std::string m_parent;

        std::vector<DataItem> m_dataItems;

        TextBuilder::VerbatimSection m_rawCode;
        TextBuilder::VerbatimSection m_rawCodeTypeDeclarations;
};

class TrackerTypeOut : public TypeOut
{
    public:
        TrackerTypeOut(const Database& db, const clang::NamedDecl* decl);

        void appendTo(TextBuilder& cxxBody, TextBuilder& inlBody) const;
};

#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
