// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
#include "utils/TypeOut.h"
#include "utils/Database.h"
#include "utils/ElementUtils.h"
#include "utils/TextBuilder.h"

namespace
{
    std::string getIsReflectedCondition(const Database& db, const clang::NamedDecl* decl, bool externalReflection)
    {
        const clang::TemplateParameterList* params = getTemplateParams(decl);
        assert(params);

        std::string res;
        Optional<Attribute> ignoredAttr = db.getAttributeValue< Optional<Attribute> >(decl, "hk::IgnoreTemplateParams");
        {
            llvm::raw_string_ostream os(res);
            if (externalReflection)
            {
                os << "true";
            }
            else
            {
                // Check that this instantiation has a valid nested reflected check (catches non-reflected explicit
                // specializations of reflected templates).
                os << "hkReflect::Detail::NestedReflectedCheck< " << db.getScopedMangledName(decl) << " >::result";
            }
            if (!ignoredAttr || (unsigned)ignoredAttr->getNumPosValues() < params->size())
            {
                for (auto param : *params)
                {
                    if (clang::isa<clang::TemplateTypeParmDecl>(param) &&
                        (!ignoredAttr || std::find(ignoredAttr->posBegin(), ignoredAttr->posEnd(), param->getNameAsString()) == ignoredAttr->posEnd()))
                    {
                        
                        os << " &&\n    hkReflect::Detail::IsReflectedBase< " << param->getNameAsString() << " >::result";
                    }
                }
            }
        }
        return res;
    }

    void printHeader(const clang::NamedDecl* decl, const std::string& declName, TextBuilder& structsListing, TextBuilder& programListing)
    {
        programListing["append"].append("").append(std::string(80, '/'))
            .append("// %s: %s\n//", decl->getDeclKindName(), declName.c_str());
        if (&programListing != &structsListing)
        {
            structsListing["append"].append("").append(std::string(80, '/'))
                .append("// %s: %s (_Auto definition)\n//", decl->getDeclKindName(), declName.c_str());
        }
    }

    std::string addTypename(const std::string& str)
    {
        if (!stringStartsWith(str, "typename "))
        {
            return "typename " + str;
        }
        return str;
    }
}

TypeOut::TypeOut(const Database& db, const clang::NamedDecl* decl)
    : m_db(db)
    , m_decl(decl)
    , m_parent("0")
{
    m_declareReflection = true;
    auto declaringRecord = clang::dyn_cast<clang::CXXRecordDecl>(decl);
    if (declaringRecord)
    {
        // if this is a property declaration, the declaring record is actually its context
        if (const clang::CXXRecordDecl* outer = clang::dyn_cast<clang::CXXRecordDecl>(decl->getDeclContext()))
        {
            if (isPropertyDecl(decl))
            {
                declaringRecord = outer;
                m_declareReflection = false;
            }
        }
    }
    else
    {
        declaringRecord = clang::dyn_cast<clang::CXXRecordDecl>(decl->getDeclContext());
        m_declareReflection = clang::isa<clang::TypeDecl>(decl);
    }

    m_externalReflection = declaringRecord && m_db.getAttributeValue<bool>(declaringRecord, "hk::ExternalReflection", false);

    // get template strings
    m_tplHeader = getTemplateParametersStringRecursive(m_decl);

    // todo.rt this forbids access to private record members from the reflected type of a field
    if (!m_externalReflection && m_decl == declaringRecord)
    {
        // If the decl is a reflected record, data will live in MyType::_Auto<void, Unused>.
        std::string cxxName = m_db.getScopedMangledNameNoPrefix(m_db.astContext.getTypeDeclType(declaringRecord));

        // The _Auto name is used for the declaration so it must not contain template disambiguators.
        // todo.rt scopedName must contain the outer classes but not the namespaces
        std::string scopedName = m_db.getScopedMangledName(declaringRecord, Database::NAME_DECLARATION);
        m_dataStructName = formatString("%s::_Auto", scopedName.c_str());

        /// The data qualifier will use the regular name instead.
        m_dataQualifier = formatString("%s::_Auto::", cxxName.c_str());

    }
    else
    {
        // Otherwise (top-level enum/typedef, reflected type declared in non-reflected record,
        // record with external reflection) data will live in a global struct (MyType_Auto).
        std::vector<std::string> tplParams;
        if (declaringRecord)
        {
            getTemplateParametersString(declaringRecord, tplParams, false);
        }
        const std::string& paramsString = tplParams.empty() ? "" : formatString("< %s >", joinStrings(tplParams, ", ").c_str());

        std::string ns = getNamespaceString(decl);
        const std::string& nameWithoutNs = decl->getQualifiedNameAsString().substr(ns.size(), std::string::npos);

        m_dataStructName = formatString("%s_Auto", pyNameFromCxxName(nameWithoutNs).c_str());
        m_dataQualifier = formatString("%s%s%s::", ns.c_str(), m_dataStructName.c_str(), paramsString.c_str());
    }
}

TypeOut& TypeOut::addOpt(clang::SourceLocation srcLoc, const std::string& opt, const std::string& name, int flags)
{
    OptItem::CodeFilter optFlags = OptItem::CodeFilter::ALWAYS_INCLUDE;
    if((flags & FLAG_NEEDED_FOR_CONSTRUCTION) != 0)
    {
        optFlags = OptItem::CodeFilter::INCLUDE_FOR_CONSTRUCTION;
    }
    else if((flags & FLAG_IS_CODE) != 0)
    {
        optFlags = OptItem::CodeFilter::INCLUDE_FOR_FULL_REFLECTION;
    }

    OptItem newItem = { opt.find("hkReflect::") == 0 ? opt : "hkReflect::" + opt, name, optFlags};

    const int newOptionValue = m_db.getOptionValue(srcLoc, opt);

    std::vector<OptItem>::iterator it = m_items.begin();
    for(; it != m_items.end(); ++it)
    {
        if(newOptionValue <= m_db.getOptionValue(srcLoc, it->opt))
            break;
    }

    if(it != m_items.end() && newOptionValue == m_db.getOptionValue(srcLoc, it->opt))
    {
        // update the existing item
        *it = newItem;
    }
    else
    {
        m_items.insert(it, newItem);
    }

    return *this;
}

TypeOut& TypeOut::addOpt(clang::SourceLocation srcLoc,
                          const std::string& opt,
                          const std::string& name,
                          const std::string& type,
                          const std::string& defn,
                          const std::string& section,
                          int flags)
{
    return addOptCast(srcLoc, opt, name, type, "", defn, section, flags);
}

TypeOut& TypeOut::addOptCast(clang::SourceLocation srcLoc,
    const std::string& opt,
    const std::string& name,
    const std::string& realtype,
    const std::string& casttype,
    const std::string& defn,
    const std::string& section,
    int flags)
{
    assert(!realtype.empty());

    // adds the data item
    addData(name, realtype, defn, section, flags);

    // adds an opt containing the address of the data item
    std::string context;
    if ((flags & FLAG_EXTERN_DECL) == 0)
    {
        context = getDataQualifier();
        if (section.size())
        {
            context += section + "::";
        }
    }
    const std::string& qualifiedName = formatString("%s(&%s%s)", casttype.c_str(), context.c_str(), name.c_str());
    addOpt(srcLoc, opt, qualifiedName, flags);

    return *this;
}

TypeOut& TypeOut::addData(const std::string& decl)
{
    return addData(decl, "", "");
}

TypeOut& TypeOut::addData(const std::string& name, const std::string& type, const std::string& defn, const std::string& section, int flags, const std::string& sectionParentString)
{
    DataItem newData = {type, name, defn, section, sectionParentString, flags};
    m_dataItems.push_back(newData);
    return *this;
}

bool TypeOut::hasOpt(const std::string& opt) const
{
    std::string optFull = opt.find("hkReflect::") == 0 ? opt : "hkReflect::" + opt;
    for(unsigned i = 0; i < m_items.size(); ++i)
    {
        if(m_items[i].opt == optFull)
        {
            return true;
        }
    }
    return false;
}

TypeOut& TypeOut::addFlag(clang::SourceLocation srcLoc, const std::string& flag)
{
    m_flags.insert(flag);
    const std::string& flagsStr = m_flags.empty() ? "0" : joinStrings(m_flags, " | ");
    addOpt(srcLoc, "Opt::FLAGS", flagsStr);
    return *this;
}

namespace
{
    void appendPropertyImpl(TextBuilder::Section& section, const clang::Decl* decl, const std::string& reflectedTypeName)
    {
        // specialize
        if( auto dc = clang::dyn_cast<clang::DeclContext>(decl) )
        {
            typedef clang::DeclContext::specific_decl_iterator<clang::TypedefDecl> iter;
            for( iter it(dc->decls_begin()), end(dc->decls_end()); it!=end; ++it )
            {
                if( it->getName() == "PropertyImpl" )
                {
                    section.append("typedef typename %s::PropertyImpl PropertyImpl;", reflectedTypeName.c_str());
                    break;
                }
            }
        }
    }
}

void TypeOut::appendTo(TextBuilder& cxxBody, TextBuilder& inlBody) const
{
    assert(m_declareReflection);
    assert(clang::isa<clang::TypeDecl>(m_decl));

    const std::string& declName = m_db.getScopedMangledName(m_decl, Database::NAME_DECLARATION);
    const std::string& reflectedTypeName = m_db.getScopedMangledName(m_decl, Database::NAME_REFLECTION);

    // The ReflectionOf/Auto definitions must be visible in the header if this is a template definition/specialization.
    TextBuilder& structsListing = isTemplateRecursive(m_decl) ? inlBody : cxxBody;

    // The rest of the definitions should go in the inl for pure templates
    // and partial specializations, in the cxx othwerwise.
    TextBuilder& programListing = hasTemplateParams() ? inlBody : cxxBody;

    if (!hasTemplateParams()) // only for .cxx
    {
        programListing["append"].append("#ifdef HK_DETAIL_REFLECT_DEFINITIONS");
    }
    printHeader(m_decl, declName, structsListing, programListing);

    std::string reflectionHeader;

    // if the type declaration is dependent, this is a type nested in a template, so generate a placeholder type
    if (m_db.isDependentTypeDecl(clang::dyn_cast<clang::TypeDecl>(m_decl)))
    {
        
        assert(false);
#if 0
        std::vector<const clang::NamedDecl*> scopes;
        for (auto curr = clang::dyn_cast<clang::NamedDecl>(m_decl); curr; curr = clang::dyn_cast<clang::NamedDecl>(curr->getDeclContext()))
        {
            scopes.push_back(curr);
        }

        std::string placeholder;
        llvm::raw_string_ostream o(placeholder);
        std::vector<std::string> headerParams;
        for (std::vector<const clang::NamedDecl*>::const_reverse_iterator it = scopes.rbegin(); it != scopes.rend(); it++)
        {
            // generate the type name
            if (it != scopes.rbegin())
            {
                o << "_";
            }
            o << (*it)->getNameAsString();

            // generate the parameter list
            getTemplateParametersString(m_db, *it, headerParams, true);
        }
        o.flush();

        reflectionHeader = formatString("template< %s >\n", joinStrings(headerParams, ", ").c_str());

        programListing["extern"].append("%sstruct %s { %s val; };\n", reflectionHeader.c_str(), placeholder.c_str(), m_db.getScopedMangledName(m_decl).c_str());
#endif
    }
    else
    {
        reflectionHeader = m_tplHeader;
    }

    // Append raw code first.
    programListing["append"].append(m_rawCode.toString());

    const std::string& registeredType = formatString("HK_REFLECT_GET_NON_CONST_TYPE( %s )", reflectedTypeName.c_str());

    if (isTemplateRecursive(m_decl))
    {
        std::string paramsAreReflected;

        // if the type is/explicitly specializes a template, ReflectionOf needs to be explicitly specialized
        const char* header = reflectionHeader.empty() ? "template<>\n" : reflectionHeader.c_str();

        if (hasTemplateParams() && !m_externalReflection)
        {
            structsListing["append"].append("\n%shkReflect::Detail::TypeData %s::typeData = { 0 };\n", header, reflectedTypeName.c_str());
        }

        structsListing["append"].append("namespace hkReflect\n{");
        if (hasTemplateParams())
        {
            // If this is not a fully specialized template, whether it is reflected depends on whether its arguments
            // are reflected.
            paramsAreReflected = getIsReflectedCondition(m_db, m_decl, m_externalReflection);

            // Declare the reflection data inside ReflectionOf<T> rather than inside the type.
            // This way it will not be compiled for instantiations which are not referenced by reflection (even if they are dll-exported).
            structsListing["append"].append("%sstruct ReflectionOf< %s >\n{", header, reflectedTypeName.c_str());
            structsListing["append"].append("typedef typename hkTrait::If<")
                .append("    %s,", paramsAreReflected.c_str())
                .append("ReflectionOf< %s >, void >::Type Holder;", reflectedTypeName.c_str());

            structsListing["append"].append("static Detail::TypeRegNode typeRegNode;")
                .append("static Detail::TypeData typeData;");

            appendPropertyImpl(structsListing["append"], m_decl, reflectedTypeName);
            structsListing["append"].append("}; //ReflectionOf\n");

            
            // Point hkReflect::Typedef::Resolve<T> to the original type corresponding to T
            // (where all typedef tags in template arguments are resolved).
            // HK_REFLECT_AS_TYPEDEF types must use the main Resolve template instead, which redirects to the
            // typedef-ed type.
            if ( !m_db.getAttribute( m_decl, "hk::ReflectAsTypedef" ) )
            {
                structsListing["append"].append( "namespace Typedef\n{" )
                    .append( "%sstruct Resolve< %s > { typedef %s Type; };", header, reflectedTypeName.c_str(),
                        m_db.getScopedMangledName( m_decl, Database::NAME_RESOLVE_TAGS ).c_str() )
                    .append( "}" );
            }
        }
        else
        {
            // Full specializations. Declare the reflection data inside the type so that it is dll-exported with the type.
            structsListing["append"].append("%sstruct ReflectionOf< %s >\n{", header, reflectedTypeName.c_str())
                .append("typedef %s Holder;", reflectedTypeName.c_str());
            appendPropertyImpl(structsListing["append"], m_decl, reflectedTypeName);
            structsListing["append"].append("}; //ReflectionOf\n");
        }

        structsListing["append"].append("} // hkReflect\n");
    }

    // Generate code for data declarations and definitions.
    TextBuilder::VerbatimSection declarations;
    TextBuilder::VerbatimSection definitions;
    appendItemsTo(declarations, definitions);

    // Append Auto.
    appendAutoTo(structsListing["auto"], declarations.toString());
    // Then the registration entry (references the TypeData address, is referenced by the TypeData content).
    if (m_db.getAttributeValue<bool>(m_decl, "hk::RegisterType", true))
    {
        // We always register templates in the inl
        if (m_db.getAttributeValue<bool>(m_decl, "hk::ManualTypeRegistration", false) && !isTemplateRecursive(m_decl))
        {
            programListing["append"].append("// Not registering %s because of hk::ManualTypeRegistration", registeredType.c_str());
        }
        else
        {
            programListing["append"].append("%shkReflect::Detail::TypeRegNode %s( %s );\n",
            reflectionHeader.c_str(), getTypeRegEntrySym().c_str(), registeredType.c_str());
        }
    }
    else
    {
        programListing["append"].append("// Not registering %s because of hk::RegisterType(false)", registeredType.c_str());
    }

    // Append the TypeData.
    // Could always use hkReflect::ReflectionOf< %s >::Holder but that causes a VS ICE with templates.
    const std::string& structName = (hasTemplateParams() || !clang::isa<clang::CXXRecordDecl>(m_decl)) ?
        formatString("hkReflect::ReflectionOf< %s >::typeData", reflectedTypeName.c_str()) :
        formatString("hkReflect::ReflectionOf< %s >::Holder::typeData", reflectedTypeName.c_str());
    programListing["append"].append("%shkReflect::Detail::TypeData %s =",
        hasTemplateParams() ? reflectionHeader.c_str() : "", structName.c_str());
    appendTypeContentTo(programListing["append"]);

    // Finally the definitions (may reference the TypeData and the registration entry).
    programListing["append"].append(definitions.toString());
    if (!hasTemplateParams())
    {
        programListing["append"].append(hasTemplateParams() ? "" : "#endif // HK_DETAIL_REFLECT_DEFINITIONS\n");
    }

    if (!m_rawCodeTypeDeclarations.isEmpty())
    {
        programListing["append"].append(m_rawCodeTypeDeclarations.toString());
    }
}

void TypeOut::appendDataTo(TextBuilder::Section& section) const
{
    TextBuilder::VerbatimSection declarations;
    TextBuilder::VerbatimSection definitions;

    appendItemsTo(declarations, definitions);

    if (!declarations.isEmpty())
    {
        appendAutoTo(section, declarations.toString());
        section.append(definitions.toString());
        section.append("");
    }
}

void TypeOut::appendItemsTo(TextBuilder::Section& declarations, TextBuilder::Section& definitions) const
{
    // declare and define the additional data
    if (!m_dataItems.empty())
    {
        for (std::vector<DataItem>::const_iterator dataIt = m_dataItems.begin();
            dataIt != m_dataItems.end(); ++dataIt)
        {
            std::string name = dataIt->name;
            std::string type = dataIt->type;
            bool externDecl = (dataIt->flags & FLAG_EXTERN_DECL) != 0;
            bool useCtor = (dataIt->flags & FLAG_CONSTRUCTOR) != 0;
            bool qualifyType = (dataIt->flags & FLAG_QUALIFY_TYPE) != 0;

            // workaround for transforming "hkReal[3] x" -> "hkReal x[3]" BUT
            // we want AggregateOf<hkReal[3]>::Type to remain unchanged.
            const std::vector<std::string>& parts = splitStringWithScoping(type, '[');
            if (parts.size() == 2) // hkReal[2] -> [ "hkReal", "2]" ]
            {
                type = parts[0];
                name = name + "[" + parts[1];
            }

            if (dataIt->defn.empty())
            {
                // declaration only
                assert(!externDecl);
                assert(dataIt->section.empty());
                declarations.append("%s", name.c_str());
            }
            else
            {
                // declaration + external definition
                bool inSection = !dataIt->section.empty();
                if (!externDecl)
                {
                    // put the declaration in a separate struct if a section is specified
                    const std::string& header = inSection ? formatString("struct %s %s{ ", dataIt->section.c_str(), dataIt->sectionParentString.c_str()) : std::string("");
                    const std::string& trailer = inSection ? "};" : "";
                    declarations.append("%sstatic %s %s;%s",
                        header.c_str(), type.c_str(), name.c_str(), trailer.c_str());
                }
                const std::string& defName = formatString("%s%s%s%s",
                    externDecl ? "" : m_dataQualifier.c_str(), 
                    dataIt->section.c_str(), 
                    inSection ? "::" : "",
                    name.c_str());
                const std::string& defValue = formatString(useCtor ? "( %s )" : " = %s", dataIt->defn.c_str());
                definitions.append("%s%s%s %s%s;\n",
                    m_tplHeader.c_str(),
                    qualifyType ? ((hasTemplateParams() ? "typename " : "") + m_dataQualifier).c_str() : "",
                    type.c_str(),
                    defName.c_str(),
                    defValue.c_str());
            }
        }
    }
}

void TypeOut::appendAutoTo(TextBuilder::Section& section, const std::string& declarations) const
{
    if (!declarations.empty())
    {
        std::string namespaceStart, namespaceEnd;
        std::tie(namespaceStart,namespaceEnd) = getNamespaceBlock(m_decl);

        // print the _Auto struct
        section.append(formatString("%s%sstruct %s", namespaceStart.c_str(), m_tplHeader.c_str(), m_dataStructName.c_str()));
        section.append("{");
        section.append(declarations);
        section.append("};\n");
        section.append(namespaceEnd);
    }
}



void TypeOut::appendOptsTo(TextBuilder::Section& section) const
{
    const std::string& optionalBits = joinStrings(m_items, &OptItem::opt, " | ");

    section.append("HK_FILTER_REFLECT_OPT(%s),//optionals\n(hkUlong)%s,//parent",
        optionalBits.empty() ? "hkReflect::Opt::NONE" : optionalBits.c_str(), m_parent.c_str());

    for(std::size_t i = 0; i < m_items.size(); ++i)
    {
        OptItem::CodeFilter isCodeOpt = m_items[i].isCodeOpt;

        if((m_items[i].opt == "hkReflect::Opt::DEF_CONSTRUCTOR"))
        {
            isCodeOpt = OptItem::CodeFilter::INCLUDE_FOR_FULL_REFLECTION;
        }

        if (isCodeOpt == OptItem::CodeFilter::INCLUDE_FOR_FULL_REFLECTION)
        {
            section.append("HK_EXCLUDE_FROM_DATA_REFLECTION((HK_REFLECT_TYPE_OPTIONAL(%s, %s),))", m_items[i].opt.c_str(), m_items[i].name.c_str());
        }
        else if (isCodeOpt == OptItem::CodeFilter::INCLUDE_FOR_CONSTRUCTION)
        {
            section.append("HK_INCLUDE_IN_CONSTRUCTABLE_REFLECTION((HK_REFLECT_TYPE_OPTIONAL(%s, %s),))", m_items[i].opt.c_str(), m_items[i].name.c_str());
        }
        else
        {
            section.append("HK_REFLECT_TYPE_OPTIONAL(%s, %s),", m_items[i].opt.c_str(), m_items[i].name.c_str());
        }
    }
}

void TypeOut::appendTypeContentTo(TextBuilder::Section& section) const
{
    section.append("{");
    appendOptsTo(section);
    // If the class is a template the registration node must be referenced by
    // something, otherwise it will be stripped away.
    const std::string& typeReg = getTypeRegEntrySym();
    if (!typeReg.empty() && hasTemplateParams())
    {
        section.append("(hkUlong)&%s,", typeReg.c_str());
    }

    section.append("};");
}

std::string TypeOut::getTypeRegEntrySym() const
{
    if(!m_declareReflection)
    {
        return "";
    }

    return hasTemplateParams() ?
        formatString("hkReflect::ReflectionOf< %s >::typeRegNode", m_db.getScopedMangledName(m_decl, Database::NAME_REFLECTION).c_str()) :
        pyNameFromCxxName(m_db.getScopedMangledName(m_decl)) + "_typeRegNode";
}

TrackerTypeOut::TrackerTypeOut(const Database& db, const clang::NamedDecl* decl)
    : TypeOut(db, decl)
{
    assert(clang::isa<clang::CXXRecordDecl>(m_decl));

    // Declare data in _TrackerAuto rather than _Auto.
    
    size_t i1 = m_dataStructName.rfind("_Auto");
    size_t i2 = m_dataQualifier.rfind("_Auto");
    assert(i1 != std::string::npos);
    assert(i2 != std::string::npos);
    m_dataStructName.insert(i1 + 1, "Tracker");
    m_dataQualifier.insert(i2 + 1, "Tracker");
}

void TrackerTypeOut::appendTo(TextBuilder& cxxBody, TextBuilder& inlBody) const
{
    const std::string& reflectedTypeName = m_db.getScopedMangledName(m_decl);

    if (m_externalReflection)
    {
        hkLog.warning(m_decl, "Type %s is marked as hk::ExternalReflection and cannot be memory-tracked", reflectedTypeName.c_str());
        return;
    }

    // The ReflectionOf/Auto definitions must be visible in the header if this is a template definition/specialization.
    TextBuilder& structsListing = isTemplateRecursive(m_decl) ? inlBody : cxxBody;

    // The rest of the definitions should go in the inl for pure templates
    // and partial specializations, in the cxx othwerwise.
    TextBuilder& programListing = hasTemplateParams() ? inlBody : cxxBody;

    printHeader(m_decl, m_db.getScopedMangledName(m_decl, Database::NAME_DECLARATION), structsListing, programListing);

    // Generate code for data declarations and definitions.
    TextBuilder::VerbatimSection declarations;
    TextBuilder::VerbatimSection definitions;

    std::string regNode;
    std::string typeData;
    if (hasTemplateParams())
    {
        // Define trackerRegNode in _TrackerAuto.
        declarations.append("static hkReflect::Detail::TrackerRegNode trackerRegNode;");
        regNode = formatString("%s::_TrackerAuto::trackerRegNode", reflectedTypeName.c_str());
        // Use _TrackerHelper so that the type is visible and can be used directly
        typeData = formatString("%s::_TrackerHelper::typeData", reflectedTypeName.c_str());
    }
    else
    {
        // Define trackerRegNode as a global symbol (so that it can be referenced in the classlist).
        regNode = formatString("%s_trackerRegNode", pyNameFromCxxName(reflectedTypeName).c_str());
        // No _TrackerHelper, put type in _TrackerAuto.
        declarations.append("struct Helper { static hkReflect::Detail::TypeData typeData; };");
        typeData = formatString("%s::_TrackerAuto::Helper::typeData", reflectedTypeName.c_str());
    }

    appendItemsTo(declarations, definitions);

    // Append Auto.
    appendAutoTo(structsListing["append"], declarations.toString());

    // Append the TrackerRegNode.
    bool usesOldMacro = findMemberDecl<clang::TypedefDecl>(clang::dyn_cast<clang::CXXRecordDecl>(m_decl), "hkUsesOldMacro") != nullptr;
    programListing["append"]
        .append("%shkReflect::Detail::TrackerRegNode %s( %s, HK_TRACKER_GET_HANDLE_UNCHECKED( %s%s ), true );",
        m_tplHeader.c_str(), regNode.c_str(), typeData.c_str(),  reflectedTypeName.c_str(), usesOldMacro ? "::_OldMacroHack" : "");

    if (hasTemplateParams())
    {
        // Append the definition of trackerValue, which will force-link the TrackerRegNode.
        programListing["append"].append("%sconst hkUint16 %s::_trackerValue = HK_TRACKER_CAST_TO_HANDLE(&%s);", m_tplHeader.c_str(), reflectedTypeName.c_str(), regNode.c_str());
    }

    programListing["append"].append("");

    // Append the TypeData contents.
    programListing["append"].append("%shkReflect::Detail::TypeData %s =",
        hasTemplateParams() ? m_tplHeader.c_str() : "", typeData.c_str());
    programListing["append"].append("{");
    appendOptsTo(programListing["append"]);
    programListing["append"].append("};");

    // Finally the definitions (may reference the TypeData and the registration entry).
    programListing["append"].append(definitions.toString());
    programListing["append"].append("");
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
