// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
PRAGMA_WARNING_PUSH
#include <llvm/Support/FileSystem.h>
#include <llvm/Support/Path.h>
PRAGMA_WARNING_POP

#include "Database.h"
#include "TkbmsRegistry.h"
#include "StlUtils.h"
#include "RuntimeError.h"
#include "Filesystem.h"


TkbmsRegistry::Entry::Entry(const std::string& tkbms)
    : fullTkbms(tkbms), requireAllProducts(true)
{
    if(!tkbms.empty())
    {
        std::istringstream istr(tkbms);

        std::string line;

        // skip the first line (header)
        std::getline(istr, line);

        while (std::getline(istr, line))
        {
            std::string::size_type valueStart = line.find(':');
            if (valueStart != std::string::npos)
            {
                // line contains a value
                valueStart += 1;
                while(line[valueStart] == ' ' || line[valueStart] == '\t')
                    ++valueStart;

                std::string::size_type valueEnd = line.size() - 1;
                while(line[valueEnd] == ' ' || line[valueEnd] == '\t' || line[valueEnd] == '\r' || line[valueEnd] == '\n')
                    --valueEnd;

                const std::string& value = line.substr(valueStart, valueEnd - valueStart + 1);

                if(line.find("PLATFORM") != std::string::npos)
                {
                    platform = splitString(value, ' ');
                }
                else if(line.find("PRODUCT") != std::string::npos)
                {
                    product = splitString(value, '+');
                    if(product.size() == 1)
                    {
                        product = splitString(value, ' ');
                        requireAllProducts = false;
                    }
                }
                else if(line.find("VISIBILITY") != std::string::npos)
                {
                    visibility = value;
                }
                platform.erase(std::remove_if(platform.begin(), platform.end(), [](const std::string& s) { return s.empty(); }), platform.end());
                product.erase(std::remove_if(product.begin(), product.end(), [](const std::string& s) { return s.empty(); }), product.end());
            }
        }
    }
}

bool TkbmsRegistry::Entry::isValid() const
{
    return !fullTkbms.empty() && !platform.empty() && !product.empty() && !visibility.empty();
}

bool TkbmsRegistry::Entry::isProbablyTKBMS() const
{
    return fullTkbms.find("TKBMS") != std::string::npos;
}

bool TkbmsRegistry::Entry::has(KeyType key, const std::string& value)
{
    switch(key)
    {
        case KEY_PRODUCT:
            return std::find(product.begin(), product.end(), value) != product.end();

        case KEY_PLATFORM:
            return std::find(platform.begin(), platform.end(), value) != platform.end();

        case KEY_VISIBILITY:
            return visibility == value;
    }

    throw FatalError("Internal Error: Invalid TKBMS key.");
}
bool TkbmsRegistry::Entry::matches(KeyType key, const std::set<std::string>& values)
{
    switch(key)
    {
        case KEY_PRODUCT:
        {
            for(std::vector<std::string>::const_iterator it = product.begin(); it != product.end(); ++it)
            {
                std::set<std::string>::const_iterator v = values.find(*it);
                if(requireAllProducts && v == values.end())
                    return false;
                else if(!requireAllProducts && v != values.end())
                    return true;
            }

            if(requireAllProducts)
                return true;
            else
                return false;
        }
        case KEY_PLATFORM:
        {
            // Check if the file is explicitly included in any of the platforms.
            for(std::vector<std::string>::const_iterator it = platform.begin(); it != platform.end(); ++it)
            {
                if(values.find(*it) != values.end())
                    return true;
            }

            // Check if the file is explicitly excluded by all the platforms.
            bool allExcluded = true;
            for(auto it = values.begin(); it != values.end(); ++it)
            {
                if (std::find(platform.begin(), platform.end(), "!" + *it) == platform.end())
                {
                    allExcluded = false;
                }
            }
            if (allExcluded)
            {
                return false;
            }

            // ALL does not imply REFLECT so if we are looking for REFLECT and we didn't find it explicitly
            // then consider we didn't find even if ALL is specified.
            // If we are looking for REFLECT it means "--explicit-reflect" has been set and we are looking
            // for headers explicitly reflected.
            if(values.find("REFLECT") != values.end())
            {
                return false;
            }

            // Check if the file is "ALL" by default.
            return std::find(platform.begin(), platform.end(), "ALL") != platform.end();
        }
        case KEY_VISIBILITY:
        {
            return values.find(visibility) != values.end();
        }
    }
    throw FatalError("Invalid key: %d", key);
}


namespace
{
    const char* keyNames[] = { "PLATFORM", "PRODUCT", "VISIBILITY" };

    template<typename T>
    std::string toSetString( const T& container, char sep = ',' )
    {
        std::ostringstream os;
        os << "{";
        typename T::const_iterator it = container.begin();
        os << it->c_str();
        while ( ++it != container.end() )
        {
            os << sep << it->c_str();
        }
        os << "}";
        return os.str();
    }

}

std::string TkbmsRegistry::Entry::printableValue(KeyType key)const
{
    if (key == KEY_VISIBILITY)
    {
        return visibility;
    }
    else if ( key == KEY_PLATFORM )
    {
        return toSetString(platform);
    }
    else
    {
        return toSetString(product, requireAllProducts ? '+' : ',' );
    }
}

TkbmsRegistry::TkbmsRegistry(const std::string& projectProduct)
    : tkbmsEntries(&Filesystem::sortFileNames)
    , projectProduct(projectProduct)
{

}

void TkbmsRegistry::scanFilesystem(const std::string& project, const std::vector<std::string>& excludeDirs)
{
    llvm::error_code ec;
    const std::string dot_h = ".h";
    const std::string dot_hpp = ".hpp";
    for (llvm::sys::fs::recursive_directory_iterator i(project, ec), end; i != end; i.increment(ec))
    {
        if (ec != 0)
        {
            hkLog.error("%s: skipped due to \"%s\"", i->path().c_str(), ec.message().c_str());
            i.no_push();
        }

        std::string path = i->path();

        // Test the file name FIRST before we stat the file. This allows us to avoid
        // the stat completely for files we were never going to include anyway based in the filename
        const std::string& fullFileName = Filesystem::fixSeparatorsAndRemoveLeadingDot(path);
        if(stringEndsWith(fullFileName, dot_h) || stringEndsWith(fullFileName, dot_hpp))
        {
            // Catch the case where a folder matches the file format (e.g.: Source\Common\Oops.h\...).
            llvm::sys::fs::file_status stat;
            if(i->status(stat) && stat.type() == llvm::sys::fs::file_type::directory_file)
                continue;

            scanSingleFile(fullFileName);
        }
        else
        {
            for(std::vector<std::string>::const_iterator cit = excludeDirs.begin(); cit != excludeDirs.end(); ++cit)
            {
                if(stringEndsWith(fullFileName, *cit))
                {
                    hkLog.debug("%s: Directory explicitly excluded.", fullFileName.c_str());
                    i.no_push();
                    break;
                }
            }
        }
    }
}

bool TkbmsRegistry::scanSingleFile(const std::string& fullFileName)
{
    bool excludeTag = false;
    std::string absoluteFileName = Filesystem::absoluteCanonicalPath(fullFileName);
    const std::string& fullTkbms = readTkbmsFrom(absoluteFileName, excludeTag);
    Entry e(fullTkbms);
    if(e.isValid())
    {
        hkLog.debug("%s: found with TKBMS.", absoluteFileName.c_str());

        if (excludeTag && !e.has(KEY_PLATFORM, "!REFLECT"))
        {
            e.platform.push_back("!REFLECT");
        }
        tkbmsEntries[absoluteFileName] = e;
        return true;
    }
    else if(e.isProbablyTKBMS())
    {
        hkLog.error("%s: Looks like a TKBMS but failed to parse.", fullFileName.c_str());
    }
    return false;
}

void TkbmsRegistry::removeEntries(KeyType key, const std::string& value)
{
    std::vector<EntryMap::iterator> toDelete;

    for(EntryMap::iterator it = tkbmsEntries.begin(); it != tkbmsEntries.end(); ++it)
    {
        if(it->second.has(key, value))
        {
            hkLog.debug("%s: Excluded by TKBMS because %s in %s.", it->first.c_str(), value.c_str(), keyNames[key]);
            toDelete.push_back(it);
        }
    }

    for(std::vector<EntryMap::iterator>::const_iterator cit = toDelete.begin(); cit != toDelete.end(); ++cit)
        tkbmsEntries.erase(*cit);
}

void TkbmsRegistry::keepEntries(KeyType key, const std::set<std::string>& values)
{
    std::vector<EntryMap::iterator> toDelete;

    // Pre-build a string version of value for debug messages.

    for(EntryMap::iterator it = tkbmsEntries.begin(); it != tkbmsEntries.end(); ++it)
    {
        if(!it->second.matches(key, values))
        {
            hkLog.debug("%s: Excluded by TKBMS because %s=%s not contained in %s.", it->first.c_str(), keyNames[key], it->second.printableValue(key).c_str(), toSetString(values).c_str() );
            toDelete.push_back(it);
        }
    }

    for(std::vector<EntryMap::iterator>::const_iterator cit = toDelete.begin(); cit != toDelete.end(); ++cit)
        tkbmsEntries.erase(*cit);
}

const TkbmsRegistry::Entry& TkbmsRegistry::getTkbmsEntry(const std::string& file)
{
    std::string absoluteFileName = Filesystem::absoluteCanonicalPath(file);

    EntryMap::const_iterator cit = tkbmsEntries.find(absoluteFileName);

    if(cit == tkbmsEntries.end())
        throw FatalError("Internal error: Didn't find TKBMS for file: " + file);

    return cit->second;
}

bool TkbmsRegistry::tryGetTkbmsEntry(const std::string& file, TkbmsRegistry::Entry& entryOut)
{
    EntryMap::const_iterator cit = tkbmsEntries.find(file);

    if(cit == tkbmsEntries.end())
    {
        return false;
    }

    entryOut = cit->second;
    return true;
}

std::string TkbmsRegistry::readTkbmsFrom(const std::string& file)
{
    bool dummy;
    return readTkbmsFrom(file, dummy);
}

std::string TkbmsRegistry::readTkbmsFrom(const std::string& file, bool& hasOldExcludeTag)
{
    std::ifstream f;

    f.open(file.c_str());
    if(!f.is_open())
        throw FatalError("Couldn't open file: " + file);

    std::string tkbms;
    std::string line;

    // Read the text between the "TKBMS v1.0" guards
    hasOldExcludeTag = false;

    // find the first TKBMS v1.0
    while(std::getline(f, line))
    {
        if(line.find("TKBMS v1.0") != std::string::npos)
        {
            tkbms = line;
            tkbms += '\n';
            break;
        }
    }
    // add everything up to the end marker
    while(std::getline(f, line))
    {
        tkbms += line;
        tkbms += '\n';
        if(line.find("TKBMS v1.0") != std::string::npos)
        {
            break;
        }
    }
    // check the next 10 lines for an exclude
    for(int limit = 10; std::getline(f, line) && limit; --limit)
    {
        if(line.find("HK_REFLECTION_PARSER_EXCLUDE_FILE") != std::string::npos)
        {
            hasOldExcludeTag = true;
        }
    }

    return tkbms;
}

std::string TkbmsRegistry::getProjectProduct() const
{
    return projectProduct.size() ? projectProduct : "COMMON";
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
