// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
PRAGMA_WARNING_PUSH
#include <llvm/Support/FileSystem.h>
PRAGMA_WARNING_POP

#include "TextBuilder.h"
#include "Logger.h"
#include "StlUtils.h"
#include "RuntimeError.h"
#include "Filesystem.h"

TextBuilder::Section::~Section()
{
}

TextBuilder::TextBuilder()
{
    m_sections.reserve(16);
}

TextBuilder::Section& TextBuilder::operator[](const std::string& name)
{
    std::shared_ptr<Section> section = findSection(name);

    if(!section)
        throw FatalError("Internal error: Invalid section: " + name);

    return *section;
}

TextBuilder::Section& TextBuilder::addSection(const std::string& name, Section* section)
{
    if (findSection(name))
        throw FatalError("Internal error: Adding section '" + name + "' twice");
    m_sections.push_back(std::make_pair(name, std::shared_ptr<Section>(section)));
    return *section;
}

std::shared_ptr<TextBuilder::Section> TextBuilder::findSection(const std::string& name)
{
    for(std::vector< std::pair< std::string, std::shared_ptr<Section> > >::const_iterator cit = m_sections.begin();
        cit != m_sections.end();
        ++cit)
        if(cit->first == name)
            return cit->second;

    return std::shared_ptr<Section>();
}

std::string TextBuilder::format()
{
    std::string out;

    for(std::vector< std::pair< std::string, std::shared_ptr<Section> > >::const_iterator cit = m_sections.begin();
        cit != m_sections.end();
        ++cit)
    {
        if(!cit->second->isEmpty())
        {
            cit->second->format(out);
            out += "\n\n";
        }
    }

    return out;
}

bool TextBuilder::isEmpty() const
{
    for(std::vector< std::pair< std::string, std::shared_ptr<Section> > >::const_iterator cit = m_sections.begin();
        cit != m_sections.end();
        ++cit)
        if(!cit->second->isEmpty())
            return false;
    return true;
}

std::string TextBuilder::Section::toString() const
{
    std::string temp;
    format(temp);
    if (!temp.empty() && temp.back() == '\n')
        temp = temp.substr(0, temp.length()-1);
    return temp;
}

TextBuilder::Section& TextBuilder::VerbatimSection::append(const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);

    append(formatStringImpl(fmt, args));

    va_end(args);

    return *this;
}

TextBuilder::Section& TextBuilder::VerbatimSection::append(const std::string& str)
{
    m_data << str << "\n";

    return *this;
}

TextBuilder::Section& TextBuilder::CodeSection::append(const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);

    append(formatStringImpl(fmt, args));

    va_end(args);

    return *this;
}

TextBuilder::Section& TextBuilder::CodeSection::append(const std::string& str)
{
    // Split str into lines and append a trailing \n.
    m_lines.push_back(std::string());
    for(std::size_t cursor = 0; cursor < str.size(); ++cursor)
    {
        std::string& current = m_lines.back();
        while(cursor < str.size())
        {
            const char ch = str[cursor];
            current += ch;
            if(str[cursor] == '\n')
                break;
            ++cursor;
        }

        m_lines.push_back(std::string());
    }
    m_lines.back() += '\n';


    return *this;
}

void TextBuilder::CodeSection::format(std::string& out) const
{
    int indent = 0;

    for(std::vector<std::string>::const_iterator cit = m_lines.begin(); cit != m_lines.end(); ++cit)
    {
        const std::string& line = *cit;

        // Count the number of scopes closed/opened.
        int diff = 0;
        char quoteBegin = 0;
        for (unsigned i = 0; i < line.length(); ++i)
        {
            switch(line[i])
            {
                case '(':
                case '[':
                case '{':
                    if (!quoteBegin) ++diff;
                    break;
                case ')':
                case ']':
                case '}':
                    if (!quoteBegin) --diff;
                    break;
                case '"':
                case '\'':
                    if (quoteBegin == 0) quoteBegin = line[i];
                    else if (quoteBegin == line[i]) quoteBegin = 0;
                    break;
                case'\\':
                    ++i;
                    break;
            }
        }

        if (indent + diff < 0)
        {
            const std::string& noNewline = line.substr(0, line.length() - 1);
            hkLog.error("Mismatching brackets on line \"%s\"", noNewline.c_str());
            // Reset the indent so that the file is printed anyway.
            indent = -diff;
        }

        const bool comment = (indent+diff) != 0 && line.find("///") != std::string::npos;

        const int closingBrace = diff < 0 ? diff : 0;
        const int totalIndent = indent + comment + closingBrace;
        assert(totalIndent >= 0);
        out.insert(out.end(), totalIndent*4, ' ');
        out += line;

        indent += diff;
    }
}

TextBuilder::Section& TextBuilder::DeclarationSection::append(const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);

    append(formatStringImpl(fmt, args));

    va_end(args);

    return *this;
}

TextBuilder::Section& TextBuilder::DeclarationSection::append(const std::string& str)
{
    if(m_decls.insert(str).second)
    {
        m_data << str << "\n";
    }

    return *this;
}

TextBuilder::Section& TextBuilder::IncludeSection::append(const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);

    append(formatStringImpl(fmt, args));

    va_end(args);

    return *this;
}

TextBuilder::Section& TextBuilder::IncludeSection::append(const std::string& str)
{
    // If the string is a preprocessor directive, paste it as it is.
    if (str.size() && str[0] == '#')
    {
        m_data << str << "\n";
    }
    else if(m_decls.insert(str).second)
    {
        std::string pathToInclude = str;
        for(std::vector<std::string>::const_iterator cit = m_includePaths.begin(); cit != m_includePaths.end(); ++cit)
        {
            const std::string& subPath = str.substr(0, cit->size());
            if(llvm::sys::fs::exists(subPath) && Filesystem::compareFileNames(*cit, subPath))
            {
                pathToInclude = str.substr(cit->size() + 1, std::string::npos);
                std::replace(pathToInclude.begin(), pathToInclude.end(), '\\', '/');
                break;
            }
        }

        if(m_statementType == USE_ANGLE_BRACKETS)
        {
            m_data << "#include <" << pathToInclude << ">\n";
        }
        else
        {
            m_data << "#include \"" << pathToInclude << "\"\n";
        }
    }

    return *this;
}

void TextBuilder::IncludeSection::format(std::string& out) const
{
    if(!m_ifGuard.empty())
        out += "\n#if " + m_ifGuard + "\n";

    DeclarationSection::format(out);

    if(!m_ifGuard.empty())
        out += "\n#endif //" + m_ifGuard + "\n";
}

TextBuilder::Section& TextBuilder::GuardedSection::append(const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);

    const std::string& newFmt = std::string("# define ") + m_prefix + "%s\n" + fmt + "\n#endif\n";
    const std::string& decl = formatStringImpl(newFmt.c_str(), args);

    if(m_decls.insert(decl).second)
    {
        const char* guard = va_arg(args, const char*);
        m_data << "#ifndef " << m_prefix << guard << "\n" << decl;
    }

    va_end(args);

    return *this;
}

TextBuilder::Section& TextBuilder::GuardedSection::append(const std::string&)
{
    throw FatalError("Internal Error: Shouldn't call GuardedSection::append(const std::string&).");
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
