// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
#include "StlUtils.h"
#include "utils/Logger.h"
#include "utils/RuntimeError.h"

#if defined(va_copy)
#define hk_va_copy(DST, SRC) va_copy(DST, SRC)
#elif defined(__va_copy)
#define hk_va_copy(DST, SRC) __va_copy(DST, SRC)
#else
#define hk_va_copy(DST, SRC) ((DST) = (SRC))
#endif

#ifdef _WIN32
    inline char * local_strdup(const char* a) { return _strdup(a); }
#else
    inline char * local_strdup(const char* a) { return strdup(a); }
#endif

#ifdef HK_COMPILER_RAZZLE
#define vsnprintf(...) _vsnprintf(__VA_ARGS__)
#endif

std::string formatString(const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);

    const std::string& res = formatStringImpl(fmt, args);

    va_end(args);

    return res;
}

std::string formatStringImpl(const char* fmt, va_list args)
{
    std::vector<char> backupBuffer;
    char smallBuffer[4096] = { 0 };

    char* buffer = smallBuffer;
    std::size_t bufferSize = sizeof(smallBuffer);

    for(;;)
    {
        va_list copy;
        hk_va_copy(copy, args);
        const int nbBytes = vsnprintf(buffer, bufferSize, fmt, copy);
        va_end(copy);

        if(nbBytes < 0 ) // error, try a larger size
        {
            backupBuffer.resize( bufferSize * 2 );
        }
        else if (std::size_t(nbBytes) >= bufferSize) // return value is size needed
        {
            backupBuffer.resize( nbBytes + 1 );
        }
        else
        {
            break;
        }
        buffer = &backupBuffer[0];
        bufferSize = backupBuffer.size();
    }

    if(buffer[0] == 0 && fmt[0] != 0)
    {
        throw FatalError("Internal error while formatting string with format='%s'", fmt);
    }

    if(buffer != smallBuffer)
    {
        hkLog.debug("formatString: buffer size = %u", bufferSize);
    }

    return std::string(buffer);
}

std::vector<std::string> splitString(const std::string& value, char splitChar, int maxSplit)
{
    std::vector<std::string> res;

    std::string::const_iterator strIt = value.begin();
    while(strIt != value.end())
    {
        std::string::const_iterator nextPlus = maxSplit-- != 0
            ? std::find(strIt, value.end(), splitChar)
            : value.end();
        res.push_back(std::string(strIt, nextPlus));
        strIt = nextPlus;
        if(strIt != value.end())
            ++strIt;
    }

    return res;
}

namespace
{
    enum ScopeTypes
    {
        NONE,
        PARENTHESIS,
        CURLY_BRACES,
        SQUARE_BRACES,
        DOUBLE_QUOTES,
        ANGLE_BRACKETS,

        NUM_SCOPE_TYPES,
    };

    ScopeTypes isOpeningScopeDelimiter(const char c)
    {
        if(c == '(')
            return PARENTHESIS;
        else if(c == '{')
            return CURLY_BRACES;
        else if(c == '[')
            return SQUARE_BRACES;
        else if(c == '"')
            return DOUBLE_QUOTES;
        else if(c == '<')
            return ANGLE_BRACKETS;
        else
            return NONE;
    }

    ScopeTypes isClosingScopeDelimiter(const char c, unsigned char count[NUM_SCOPE_TYPES])
    {
        if(c == ')')
            return PARENTHESIS;
        else if(c == '}')
            return CURLY_BRACES;
        else if(c == ']')
            return SQUARE_BRACES;
        else if(c == '"' && count[DOUBLE_QUOTES] > 0)
            return DOUBLE_QUOTES;
        else if (c == '>')
            return ANGLE_BRACKETS;
        else
            return NONE;
    }

    union ScopeCount
    {
        unsigned char count[NUM_SCOPE_TYPES];
        int64_t isNonZero;
    };
}

std::vector<std::string> splitStringWithScoping(const std::string& value, char splitChar)
{
    ScopeCount scopes = { 0 };

    std::vector<std::string> res;

    // Used to handle the case where the splitChar is a scope delimiter
    const ScopeTypes splitScopeType = ScopeTypes(int(isOpeningScopeDelimiter(splitChar)) | isClosingScopeDelimiter(splitChar, scopes.count));

    std::string::const_iterator strIt = value.begin();
    while(strIt != value.end())
    {
        std::string token;

        while(strIt != value.end())
        {
            if(ScopeTypes closingScopeType = isClosingScopeDelimiter(*strIt, scopes.count))
            {
                if(closingScopeType != splitScopeType)
                    --scopes.count[closingScopeType];
            }
            else if(ScopeTypes openingScopeType = isOpeningScopeDelimiter(*strIt))
            {
                if(openingScopeType != splitScopeType)
                    ++scopes.count[openingScopeType];
            }

            if(!scopes.isNonZero && *strIt == splitChar)
                break;

            token += *strIt++;
        }

        res.push_back(token);
        if(strIt != value.end())
            ++strIt;
    }

    return res;
}

bool stringStartsWith(const std::string& str, const std::string& prefix)
{
    if(str.size() >= prefix.size())
        return std::equal(prefix.begin(), prefix.end(), str.begin());
    return false;
}

bool stringEndsWith(const std::string& str, const std::string& end)
{
    if(end.size() > str.size())
        return false;

    return std::equal(str.rbegin(), str.rbegin() + end.size(), end.rbegin());
}

ArgcArgv::ArgcArgv()
    : argv(0), argc(0)
{

}

ArgcArgv::ArgcArgv(const std::string& str)
{
    const std::vector<std::string>& args = splitStringWithScoping(str, ' ');
    *this = ArgcArgv(args);
}

ArgcArgv::ArgcArgv(const std::vector<std::string>& args)
{
    argc = (int)args.size();

    if(argc > 0)
    {
        argv = new char*[argc];

        for(int i = 0; i < argc; ++i)
        {
            argv[i] = local_strdup(args[i].c_str());
        }
    }
}

ArgcArgv::ArgcArgv(ArgcArgv&& rv)
{
    *this = std::move(rv);
}

ArgcArgv::~ArgcArgv()
{
    for(int i = 0; i < argc; ++i)
        delete[] argv[i];
    delete argv;
}

ArgcArgv& ArgcArgv::operator=(ArgcArgv&& rv)
{
    if(this != &rv)
    {
        argc = rv.argc;
        argv = rv.argv;

        rv.argc = 0;
        rv.argv = 0;
    }

    return *this;
}

std::string toLower(const std::string& str)
{
    std::string copy(str);

    std::transform(copy.begin(), copy.end(), copy.begin(), &::tolower);

    return copy;
}

std::string prefixScope(const std::string& prefix, const std::string& name)
{
    if( prefix.size() )
    {
        return prefix + "::" + name;
    }
    return name;
}

void replaceAllInplace( std::string& s, const std::string& search, const std::string& replace )
{
    for( size_t pos = 0; ; pos += replace.length() )
    {
        // Locate the substring to replace
        pos = s.find( search, pos );
        if( pos == std::string::npos ) break;
        // Replace by erasing and inserting
        s.replace( pos, search.length(), replace );
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
