// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
#include "Logger.h"
#include "Database.h"
#include "utils/Filesystem.h"

Logger hkLog;

#if defined(va_copy)
#define hk_va_copy(DST, SRC) va_copy(DST, SRC)
#elif defined(__va_copy)
#define hk_va_copy(DST, SRC) __va_copy(DST, SRC)
#else
#define hk_va_copy(DST, SRC) ((DST) = (SRC))
#endif

void Logger::log(LevelType level, const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    logImpl(level, fmt, args);
    va_end(args);
}

void Logger::error(const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    logImpl(LOG_ERROR, fmt, args);
    va_end(args);
}

void Logger::warning(const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    logImpl(LOG_WARNING, fmt, args);
    va_end(args);
}

void Logger::info(const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    logImpl(LOG_INFO, fmt, args);
    va_end(args);
}

void Logger::debug(const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    logImpl(LOG_DEBUG, fmt, args);
    va_end(args);
}

void Logger::error(const clang::Decl* decl, const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    logImpl(LOG_ERROR, decl->getASTContext().getSourceManager(), decl->getLocation(), fmt, args);
    va_end(args);
}

void Logger::warning(const clang::Decl* decl, const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    logImpl(LOG_WARNING, decl->getASTContext().getSourceManager(), decl->getLocation(), fmt, args);
    va_end(args);
}

void Logger::info(const clang::Decl* decl, const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    logImpl(LOG_INFO, decl->getASTContext().getSourceManager(), decl->getLocation(), fmt, args);
    va_end(args);
}

void Logger::debug(const clang::Decl* decl, const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    logImpl(LOG_DEBUG, decl->getASTContext().getSourceManager(), decl->getLocation(), fmt, args);
    va_end(args);
}

void Logger::error(const clang::SourceManager& mgr, clang::SourceLocation loc, const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    logImpl(LOG_ERROR, mgr, loc, fmt, args);
    va_end(args);
}

void Logger::warning(const clang::SourceManager& mgr, clang::SourceLocation loc, const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    logImpl(LOG_WARNING, mgr, loc, fmt, args);
    va_end(args);
}

void Logger::info(const clang::SourceManager& mgr, clang::SourceLocation loc, const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    logImpl(LOG_INFO, mgr, loc, fmt, args);
    va_end(args);
}

void Logger::debug(const clang::SourceManager& mgr, clang::SourceLocation loc, const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    logImpl(LOG_DEBUG, mgr, loc, fmt, args);
    va_end(args);
}

void Logger::errorWithoutDatabase(unsigned int line, const char* fileName, const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    logImpl(LOG_ERROR, line, fileName, fmt, args);
    va_end(args);
}

void Logger::warningWithoutDatabase(unsigned int line, const char* fileName, const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    logImpl(LOG_WARNING, line, fileName, fmt, args);
    va_end(args);
}

void Logger::logImpl(LevelType level, const char* fmt, va_list arg)
{
    logImpl(level, 0, NULL, fmt, arg);
}

std::string Logger::emitLineDirective(unsigned int line, const char* fileName)
{
    std::string fileString;
    Filesystem::makePathRelativeToIncludes(fileName, fileString);
    std::replace( fileString.begin(), fileString.end(), '\\', '/' );
    return formatString("#line %d \"%s\"\n", line, fileString.c_str());
}

void Logger::logImpl(LevelType level, const clang::SourceManager& mgr, const clang::SourceLocation loc, const char* fmt, va_list arg)
{
    unsigned int line;
    std::string fileNameAbs;
    getLocation(mgr, loc, line, fileNameAbs);
    std::string fileName;
    Filesystem::makePathRelativeToCurrentDirectory(fileNameAbs, fileName);

    if (fileName.empty())
    {
        logImpl(level, 0, NULL, fmt, arg);
    }
    else
    {
        logImpl(level, line, fileName.c_str(), fmt, arg);
    }
}

void Logger::logImpl(LevelType level, unsigned int line, const char* fileName, const char* fmt, va_list arg)
{
    va_list copyErrorWarning;
    va_list copyMessage;
    hk_va_copy(copyErrorWarning, arg);
    hk_va_copy(copyMessage, arg);

// A build tool SHALL NOT embed absolute path names in any generated file.
// https://osgwiki.com/wiki/Environment_Independence_for_OSG_build_tools
#ifdef HK_COMPILER_RAZZLE
    std::string fileString;
    if (fileName)
    {
        std::string filePath;
        Filesystem::getFileNameAndPath(fileName, fileString, filePath);
        fileName = fileString.c_str();
    }
#endif

    if(level <= m_level)
    {
        setTextColor(level);
        if(fileName)
            fprintf(stderr, "%s(%d):", fileName, line);
        fprintf(stderr, "%s PREBUILD: ", stringFromLevel(level));
        vfprintf(stderr, fmt, arg);
        fprintf(stderr, "\n");
        setTextColor(LOG_NONE);
    }

    // Only add errors to the output file, because the project might consider warnings as errors.
    if(level == LOG_ERROR)
    {
        if(fileName)
        {
            m_loggedErrors += emitLineDirective(line, fileName);
        }
        m_loggedErrors += "#error ";
        m_loggedErrors += formatStringImpl(fmt, copyErrorWarning);
        m_loggedErrors += "\n";
    }
    else if (level == LOG_WARNING)
    {
        if(fileName)
        {
            m_loggedWarnings += emitLineDirective(line, fileName);
        }
        m_loggedWarnings += "#error Warning: ";

        m_loggedWarnings += formatStringImpl(fmt, copyErrorWarning);
        m_loggedWarnings += "\n";
    }

    // Log everything
    {
        m_loggedMessages += stringFromLevel(level);
        m_loggedMessages += ": ";
        if(fileName)
        {
            m_loggedMessages += formatString("%s:%d:", fileName, line);
        }
        m_loggedMessages += formatStringImpl(fmt, copyMessage);
        m_loggedMessages += "\n";
    }

    va_end(copyMessage);
    va_end(copyErrorWarning);
}

#ifdef WIN32
#include <windows.h>
#include <wincon.h>
#undef ERROR
namespace
{
    const WORD levelToWindowsColor[] =
    {
        FOREGROUND_RED | FOREGROUND_INTENSITY,                    // ERROR
        FOREGROUND_RED | FOREGROUND_GREEN | FOREGROUND_INTENSITY, // WARNING
        FOREGROUND_GREEN,                                         // INFO
        FOREGROUND_RED | FOREGROUND_BLUE,                         // DEBUG
        FOREGROUND_BLUE | FOREGROUND_GREEN | FOREGROUND_RED       // NONE
    };

    WORD getConsoleTextAttribute(HANDLE hConsoleOutput)
    {
        CONSOLE_SCREEN_BUFFER_INFO csbi;
        if (GetConsoleScreenBufferInfo(hConsoleOutput, &csbi))
        {
            return csbi.wAttributes;
        }
        else
        {
            // Assume white text on black background if we failed to get info
            return levelToWindowsColor[Logger::LOG_NONE];
        }
    }
}

void Logger::setTextColor(LevelType level)
{
    HANDLE  hConsole = GetStdHandle(STD_ERROR_HANDLE);
    static const WORD originalColor = getConsoleTextAttribute(hConsole);
    if (level == LOG_NONE)
    {
        // Restore text attributes to original value
        SetConsoleTextAttribute(hConsole, originalColor);
    }
    else
    {
        SetConsoleTextAttribute(hConsole, levelToWindowsColor[level]);
    }
}
#else

namespace
{
    const char* levelToAnsiColor[]
    {
        "\x1b[31m", // ERROR
        "\x1b[33m", // WARNING
        "\x1b[32m", // INFO
        "\x1b[35m", // DEBUG
        "\x1b[0m"   // NONE
    };
}

void Logger::setTextColor(LevelType level)
{
    fprintf(stderr, "%s", levelToAnsiColor[level]);
}
#endif

const char* Logger::stringFromLevel(LevelType level)
{
    switch(level)
    {
    case LOG_ERROR: return "error";
    case LOG_WARNING: return "warning";
    case LOG_INFO: return "info";
    case LOG_DEBUG: return "debug";
    case LOG_NONE: return "NONE ";
    default: return "<unknown>";
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
