// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#ifndef FILESYSTEM_H
#define FILESYSTEM_H

PRAGMA_WARNING_PUSH
# include <vector>
# include <string>
# include <iostream>
# include <memory>
# include <llvm/ADT/SmallVector.h>
PRAGMA_WARNING_POP

#include "TextBuilder.h"

class Filesystem
{
    public:

        typedef std::ios_base::openmode OpenMode;
        static const OpenMode RO = std::ios_base::in;
        static const OpenMode RW = std::ios_base::out | std::ios_base::trunc;

        static const char pathSeparator;
        static const char foreignPathSeparator;

        struct FileHandle
        {
            ~FileHandle() { if(stream.use_count() == 2) fs->applyChanges(*this); }

            bool operator==(const FileHandle& fh) const { return path == fh.path; }
            template<typename T>
            FileHandle& operator<<(const T& t) { (*stream) << t; return *this; }

            Filesystem* fs;
            OpenMode mode;
            std::string path;
            std::shared_ptr<std::iostream> stream;
            bool forceWrite;
        };

        typedef std::shared_ptr<std::iostream> (*OpenFunctionType)(const std::string& path, OpenMode mode);

        static std::shared_ptr<std::iostream> simpleOpen(const std::string& path, OpenMode mode);
        static std::shared_ptr<std::iostream> perforceOpen(const std::string& path, OpenMode mode);

        /// Replace \ with / and remove ./ at the start of the path
        static std::string fixSeparatorsAndRemoveLeadingDot(const std::string& path);

        /// Return a canonical-form path.
        static std::string getCanonicalizedPath(const char* path, std::size_t pathSize);

        /// Get the absolute canonical path, this is what we pass to clang and always use at run-time.
        /// It cannot be output to a file, as it will contain the path from the machine it was run on
        static std::string absoluteCanonicalPath(const std::string& path);

        // Decompose a filePath into a fileName and path. Assumes filePath is canonical.
        static void getFileNameAndPath(const std::string& filePath, std::string& fileName, std::string& path);

        /// Returns true if pathA and pathB represent the same path taking into account case etc. Both paths have to be
        /// either relative or absolute.
        static bool compareFileNames(const std::string& pathA, const std::string& pathB);

        /// Returns true if pathA < pathB taking into account case etc.
        static bool sortFileNames(const std::string& pathA, const std::string& pathB);

        static bool makePathRelativeToIncludes(const std::string& path, std::string& pathOut);
        static bool makePathRelativeToCurrentDirectory(const std::string& path, std::string& pathOut);
        Filesystem(OpenFunctionType openFun = &simpleOpen);

        FileHandle open(const std::string& path, OpenMode mode);
        FileHandle openRO(const std::string& path);
        FileHandle openRW(const std::string& path, bool forceWrite = false);

        void setDryRun(bool d) { m_dryRun = d; }
        static llvm::error_code makeAbsolute(llvm::SmallVectorImpl< char >& path);

        void cleanOutputPath(const std::string& outputPath);

        static llvm::error_code ensureContainingDirectoryExists(const std::string& pathAndFileName);

    private:

        void applyChanges(FileHandle& handle);

        OpenFunctionType m_openFun;
        std::vector<FileHandle> m_operations;
        std::vector<std::string> m_filesWritten;
        bool m_dryRun;
        static bool s_isTerminating;
};

#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
