// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
//#include <process.h>
PRAGMA_WARNING_PUSH
#include <llvm/Support/FileSystem.h>
#include <llvm/Support/Path.h>
#include <llvm/Support/raw_ostream.h>
PRAGMA_WARNING_POP

#include "Filesystem.h"
#include "Logger.h"
#include "RuntimeError.h"

#ifdef _WIN32
#include <windows.h>
#endif
std::vector<std::string>* getGlobalIncludePaths();

// This does most of the work in this file
namespace
{
    void canonicalizePath(const char* path, std::size_t pathSize, llvm::SmallVectorImpl<char>& result)
    {
        std::string pathStr(path, pathSize);
#if 0 //def _WIN32
        if (strncmp(pathStr.c_str(), "\\\\?\\", 4) != 0)
        {
            std::string newPathStr("\\\\?\\");
            newPathStr.append(pathStr);
            pathStr.swap(newPathStr);
        }
#endif

        std::replace(pathStr.begin(), pathStr.end(), Filesystem::foreignPathSeparator, Filesystem::pathSeparator);

        std::vector<std::string> pathComponents = splitString(pathStr, Filesystem::pathSeparator);

        std::vector<std::string>::iterator it = pathComponents.begin();
        while (it != pathComponents.end())
        {
            if (*it == "..")
            {
                // Whenever we find a .. delete it and the element before.
                // Can't normalize if the previous element is . or .. so leave it unchanged.
                if (it != pathComponents.begin() && (*(it-1) != ".") && (*(it-1) != ".."))
                {
                    it = pathComponents.erase(it - 1, it + 1);
                }
                else
                {
                    ++it;
                }
            }
            else if (*it == ".")
            {
                it = pathComponents.erase(it);
            }
            else
            {
                ++it;
            }
        }


#ifdef _WIN32
        const std::string& newPath = joinStrings(pathComponents, "\\");
#else
        const std::string& newPath = joinStrings(pathComponents, "/");
#endif
        result.append(newPath.begin(), newPath.end());
    }
}

llvm::error_code Filesystem::ensureContainingDirectoryExists(const std::string& pathAndFileName)
{
    std::string filePath;
    std::string fileName;
    std::string pathAndFileNameFS(pathAndFileName);
    std::replace(pathAndFileNameFS.begin(), pathAndFileNameFS.end(), Filesystem::foreignPathSeparator, Filesystem::pathSeparator);
    Filesystem::getFileNameAndPath(pathAndFileNameFS, fileName, filePath);
    return llvm::sys::fs::create_directories(filePath);
}


// for read while another hkPreBuild process has it open for write. In any other case, the fs
// error is probably fatal and we are just wasting a lot of time retrying.
std::shared_ptr<std::iostream> Filesystem::simpleOpen(const std::string& pathIn, OpenMode mode)
{
    llvm::SmallVector<char, 1024> path;
    canonicalizePath(pathIn.data(), pathIn.length(), path);
    path.push_back(0);

    std::shared_ptr<std::fstream> out(new std::fstream);
    const int numAttempsToMake = 40;
    int numAttempts = numAttempsToMake;
    while(!out->is_open() && (--numAttempts >= 0))
    {
        out->open(path.data(), mode);
        if(!out->is_open())
        {
            // Most of the time, the directory exists
            // So we only do this if it fails
            ensureContainingDirectoryExists(std::string(path.begin(), path.end()));
            out->open(path.data(), mode);
        }
#ifdef _WIN32
        // Either the open is impossible, or it is already open for read and we are trying to open for write.
        // This is not a great situation and ideally it would be made impossible by the build order, but for now
        // just retry a few times
        if(!out->is_open())
        {
            hkLog.debug("Unable to open %s, %d attempts remaining", path.data(), numAttempts);
            Sleep(250);
        }
#endif
    }
    if(!out->is_open())
    {
        hkLog.error("Unable to open %s after %d retries", path.data(), numAttempsToMake);
        if(!s_isTerminating)
        {
            s_isTerminating = true;
            throw FatalError("Couldn't open file: %s", path.data());
        }
    }
    return out;
}

std::shared_ptr<std::iostream> Filesystem::perforceOpen(const std::string& pathIn, OpenMode mode)
{
    if(mode == RO)
        return simpleOpen(pathIn, mode);
    else
    {
        llvm::SmallVector<char, 1024> path;
        canonicalizePath(pathIn.data(), pathIn.length(), path);
        path.push_back(0);

        std::shared_ptr<std::fstream> out(new std::fstream);
        out->open(path.data(), mode);
        if(!out->is_open())
        {
            const std::string& cmd = "p4 edit " + std::string(path.begin(), path.end());
            int status = system(cmd.c_str());
            if( status != 0 )
                hkLog.warning("Failed to execute '%s'", cmd.c_str());

            out->open(path.data(), mode);
            if(!out->is_open())
            {
                if(!s_isTerminating)
                {
                    s_isTerminating = true;
                    throw FatalError("Impossible to open %s for writing.", path.data());
                }
            }
        }
        return out;
    }
}

Filesystem::Filesystem(OpenFunctionType openFun)
    : m_openFun(openFun), m_dryRun(false)
{
}

Filesystem::FileHandle Filesystem::open(const std::string& path, OpenMode mode)
{
    if(mode == RO)
        return openRO(path);
    else
        return openRW(path);
}

Filesystem::FileHandle Filesystem::openRO(const std::string& pathIn)
{
    llvm::SmallVector<char, 1024> path;
    canonicalizePath(pathIn.data(), pathIn.length(), path);
    path.push_back(0);

    Filesystem::FileHandle op = { this, RO, path.data(), (*m_openFun)(path.data(), RO), false };
    m_operations.push_back(op);
    return op;
}

Filesystem::FileHandle Filesystem::openRW(const std::string& pathIn, bool forceWrite)
{
    llvm::SmallVector<char, 1024> path;
    canonicalizePath(pathIn.data(), pathIn.length(), path);
    path.push_back(0);

    hkLog.info("Writing %s", path.data());
    m_filesWritten.push_back(path.data());
    Filesystem::FileHandle op = { this, RW, path.data(), std::make_shared<std::stringstream>(), forceWrite };
    m_operations.push_back(op);
    return op;
}

void Filesystem::applyChanges(FileHandle& fh)
{
    // TODO: Domino build should always write all files!
    // (The way domino cleans output means this works currently but we shouldn't rely on it)
    if(!m_dryRun && fh.mode == RW)
    {
        bool writeToFile = true;

        fh.stream->seekg(0, std::ios_base::end);
        const std::streamoff newFileSize = fh.stream->tellg();
        fh.stream->seekg(0);
        char* newFileContent = new char[static_cast<unsigned int>(newFileSize) + 1];
        fh.stream->read(newFileContent, newFileSize);

        if(!fh.forceWrite)
        {
            std::ifstream existingFile(fh.path);
            if(existingFile.is_open())
            {
                existingFile.seekg(0, std::ios_base::end);
                const std::streamoff existingFileSize = existingFile.tellg();
                existingFile.seekg(0);

                // If the existing file is smaller to the new one no need to compare it we know it has to be overwritten.
                // Note that the reported size might be bigger but the content still be the same.
                if(existingFileSize >= newFileSize)
                {
                    char* existingFileContent = new char[static_cast<unsigned int>(existingFileSize) + 1];
                    existingFile.read(existingFileContent, existingFileSize);
                    const std::streamsize bytesRead = existingFile.gcount();
                    existingFile.close();

                    if(bytesRead == newFileSize)
                        writeToFile = memcmp(existingFileContent, newFileContent, static_cast<std::size_t>(newFileSize)) != 0;

                    delete[] existingFileContent;
                }
            }
        }

        if(writeToFile)
        {
            (*m_openFun)(fh.path, RW)->write(newFileContent, newFileSize);
        }

        delete[] newFileContent;
    }
    fh.stream.reset();
    m_operations.erase(std::find(m_operations.begin(), m_operations.end(), fh));
}

std::string Filesystem::fixSeparatorsAndRemoveLeadingDot(const std::string& path)
{
    std::string res(path);
    std::replace(res.begin(), res.end(), Filesystem::foreignPathSeparator, Filesystem::pathSeparator);
    // Remove ./ at the start
    while((res.length() > 1) && (res[0] == '.') && (res[1] == Filesystem::pathSeparator))
    {
        res.erase(res.begin(), res.begin() + 2);
    }

    return res;
}

std::string Filesystem::absoluteCanonicalPath(const std::string& path)
{
    // After make_absolute we might stil have a path looking like /foo/bar/baz/../../Demo/Demos
    // so we have to canonicalize it.

    // We would like to ask clang to do the work but the implementation of canonicalize is missing (as of llvm 3.2).
    // See: http://llvm.org/bugs/show_bug.cgi?id=9842
    // llvm::sys::fs::canonicalize(absolutePath.data(), projectPath);


    std::string includePath = Filesystem::fixSeparatorsAndRemoveLeadingDot(path);
    llvm::SmallVector<char, 1024> absolutePath(includePath.begin(), includePath.end());
    Filesystem::makeAbsolute(absolutePath);
    llvm::SmallVector<char, 1024> includePathCanon;
    canonicalizePath(absolutePath.data(), absolutePath.size(), includePathCanon);
    return std::string(includePathCanon.begin(), includePathCanon.end());
}

std::string Filesystem::getCanonicalizedPath(const char* path, std::size_t pathSize)
{
    llvm::SmallVector<char, 1024> canonicalFilePath;
    canonicalizePath(path, pathSize, canonicalFilePath);
    std::string ret;
    ret.assign(canonicalFilePath.data(), canonicalFilePath.size());
    return ret;
}

bool Filesystem::makePathRelativeToIncludes(const std::string& path, std::string& pathOut)
{
    if (llvm::sys::path::is_relative(path))
    {
        pathOut = path;
        return true;
    }
    if(std::vector<std::string>* includePaths = getGlobalIncludePaths())
    {
        for (std::vector<std::string>::const_iterator cit = includePaths->begin(); cit != includePaths->end(); ++cit)
        {
            const std::string& subPath = path.substr(0, cit->size());
            if (llvm::sys::fs::exists(subPath) && Filesystem::compareFileNames(*cit, subPath))
            {
                pathOut = path.substr(cit->size() + 1, std::string::npos);
                std::replace(pathOut.begin(), pathOut.end(), '\\', '/');
                return true;
            }
        }
    }
    llvm::errs() << "Failed to relativize " << path << "\n";
    if (getGlobalIncludePaths())
    {
        int pathCount = 0;
        for (auto& globalPath : *getGlobalIncludePaths())
        {
            llvm::errs() << "PATH " << pathCount++ << ": " << globalPath << "\n";
        }
    }
    return false;
}

bool Filesystem::makePathRelativeToCurrentDirectory(const std::string& path, std::string& pathOut)
{
    pathOut = path;
    if (llvm::sys::path::is_relative(path))
    {
        return true;
    }
    llvm::SmallVector<char, 512> curp;
    if (llvm::sys::fs::current_path(curp))
    {
        return false;
    }
    const std::string subPath = path.substr(0, curp.size());
    if (llvm::sys::fs::exists(subPath) && Filesystem::compareFileNames(std::string(curp.begin(),curp.end()), subPath))
    {
        pathOut = path.substr(curp.size() + 1, std::string::npos);
        std::replace(pathOut.begin(), pathOut.end(), '\\', '/');
        return true;
    }
    return false;
}

void Filesystem::getFileNameAndPath(const std::string& filePath, std::string& fileName, std::string& path)
{
    size_t lastForeslash = filePath.find_last_of(Filesystem::pathSeparator);
    if ( lastForeslash != std::string::npos )
    {
        fileName.assign(filePath.begin() + lastForeslash + 1, filePath.end());
        path.assign(filePath.begin(), filePath.begin() + lastForeslash + 1);
    }
    else
    {
        fileName = filePath;
    }
}

bool Filesystem::compareFileNames(const std::string& pathA, const std::string& pathB)
{
    // We would like to use PathV2 here instead but we can't since
    // llvm::sys::fs::equivalent will always return true if the two
    // arguments are (different or not) folders and not files.
    //bool equivalentPaths = false;
    //if(llvm::sys::fs::equivalent(*cit, subPath, equivalentPaths) && equivalentPaths)
    //   return equivalentPaths;

#ifdef WIN32 // case insensitive systems
    const std::string& fixedPathA = toLower(pathA);
    const std::string& fixedPathB = toLower(pathB);
#else
    const std::string& fixedPathA = pathA;
    const std::string& fixedPathB = pathB;
#endif

    return fixedPathA == fixedPathB;
}

bool Filesystem::sortFileNames(const std::string& pathA, const std::string& pathB)
{
    return toLower(pathA) < toLower(pathB);
}

llvm::error_code Filesystem::makeAbsolute(llvm::SmallVectorImpl< char >& path)
{
    llvm::error_code res = llvm::sys::fs::make_absolute(path);
    // It would seem that this is no longer the case
    #if 0 && !defined(_WIN32)
        //fix error in apple implementation (absolute path has no '/' at the beginning)
        path.insert( path.begin(), '/' );
    #endif
    return res;
}

namespace
{
    bool considerFileNameForCleaning(const std::string& fullFileName)
    {
        std::string filePath;
        std::string fileName;

        Filesystem::getFileNameAndPath(fullFileName, fileName, filePath);
        const std::string dot_cxx = ".cxx";
        const std::string dot_hxx = ".hxx";
        const std::string dot_inl = ".inl";

        if(stringEndsWith(fileName, dot_cxx) || stringEndsWith(fileName, dot_inl) || stringEndsWith(fileName, dot_hxx))
        {
            return true;
        }
        else
        {
            return false;
        }
    }
}

void Filesystem::cleanOutputPath(const std::string& outputPath)
{
    llvm::error_code ec;

    std::set<std::string> filesInOutputPath;
    for (llvm::sys::fs::recursive_directory_iterator i(outputPath, ec), end; i != end; i.increment(ec))
    {
        // Catch the case where a folder matches the file format (e.g.: Source\Common\Oops.h\...).
        llvm::sys::fs::file_status stat;
        if(i->status(stat) && stat.type() == llvm::sys::fs::file_type::directory_file)
            continue;

        const std::string& fullFileName = Filesystem::fixSeparatorsAndRemoveLeadingDot(i->path());
        if(considerFileNameForCleaning(fullFileName))
        {
            // If not, we aren't interested
            filesInOutputPath.insert(fullFileName);
        }
    }

    // Remove everything that we wrote
    for(const std::string& file : m_filesWritten)
    {
        for ( auto outputFile = filesInOutputPath.begin(); outputFile != filesInOutputPath.end(); ++outputFile )
        {
            if ( llvm::StringRef( *outputFile ).compare_lower( file ) == 0 )
            {
                filesInOutputPath.erase( outputFile );
                break;
            }
        }
    }

    // Everything left we want to remove
    for(const std::string& file : filesInOutputPath)
    {
        bool success = false;
        for(int i=0; (i < 100) && !success; i++)
        {
            std::string newPath(file);
            std::stringstream ss; ss << "." << i;
            newPath += ss.str();
            llvm::sys::fs::file_status fileStatus;
            if(llvm::sys::fs::status(newPath, fileStatus))
            {
                if(!llvm::sys::fs::exists(fileStatus))
                {
                    if(!llvm::sys::fs::rename(file, newPath)) // Returns false when it worked
                    {
                        success = true;
                    }
                }
            }
        }
    }
}


bool Filesystem::s_isTerminating = false;

#ifdef _WIN32
const char Filesystem::pathSeparator = '\\';
const char Filesystem::foreignPathSeparator = '/';
#else
const char Filesystem::pathSeparator = '/';
const char Filesystem::foreignPathSeparator = '\\';
#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
