// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
#include "ElementUtils.h"
#include "StlUtils.h"
#include "Database.h"
#include "RuntimeError.h"
#include "utils/Logger.h"
#include "utils/HavokUtils.h"

static const clang::CXXMethodDecl* findDefaultConstructor(const clang::CXXRecordDecl* recDecl)
{
    for (clang::CXXRecordDecl::ctor_iterator it = recDecl->ctor_begin(); it != recDecl->ctor_end(); ++it)
    {
        if ((*it)->isDefaultConstructor())
        {
            return *it;
        }
    }
    return NULL;
}

static const clang::CXXMethodDecl* findCopyConstructor(const clang::CXXRecordDecl* recDecl)
{
    for (clang::CXXRecordDecl::ctor_iterator it = recDecl->ctor_begin(); it != recDecl->ctor_end(); ++it)
    {
        if ((*it)->isCopyConstructor())
        {
            return *it;
        }
    }
    return NULL;
}

static const clang::CXXMethodDecl* findCopyAssignment(const clang::CXXRecordDecl* recDecl)
{
    for (clang::CXXRecordDecl::method_iterator it = recDecl->method_begin(); it != recDecl->method_end(); ++it)
    {
        if ((*it)->isCopyAssignmentOperator())
        {
            return *it;
        }
    }
    return NULL;
}

static const clang::CXXMethodDecl* findDestructor(const clang::CXXRecordDecl* recDecl)
{
    return recDecl->getDestructor();
}

static const SpecialMethodsData specialMethodsData[] =
{
    {
        "Opt::DEF_CONSTRUCTOR",
        &clang::CXXRecordDecl::hasTrivialDefaultConstructor,
        &findDefaultConstructor
    },
    {
        "Opt::COPY_CONSTRUCTOR",
        &clang::CXXRecordDecl::hasTrivialCopyConstructor,
        &findCopyConstructor
    },
    {
        "Opt::DESTRUCTOR",
        &clang::CXXRecordDecl::hasTrivialDestructor,
        &findDestructor
    },
    {
        "Opt::COPY_ASSIGNMENT",
        &clang::CXXRecordDecl::hasTrivialCopyAssignment,
        &findCopyAssignment
    },
};


const SpecialMethodsData& getSpecialMethodsData(SpecialMethod method)
{
    return specialMethodsData[method];
}

const clang::TemplateParameterList* getTemplateParams(const clang::Decl* decl)
{
    if (auto specDecl = clang::dyn_cast<clang::ClassTemplatePartialSpecializationDecl>(decl))
    {
        return specDecl->getTemplateParameters();
    }
    else if (!clang::isa<clang::ClassTemplateSpecializationDecl>(decl))
    {
        if (auto tplDecl = getTemplateDecl(decl)) 
        {
            return tplDecl->getTemplateParameters();
        }
    }
    return nullptr;
}

bool hasTemplateParamsRecursive(const clang::Decl* decl)
{
    while (decl)
    {
        if (getTemplateParams(decl))
        {
            return true;
        }
        decl = clang::dyn_cast<clang::TypeDecl>(decl->getDeclContext());
    }
    return false;
}

std::string getTemplateParametersString(const clang::Decl* decl, std::vector<std::string>& tplParams, bool addKeyword)
{
    const clang::TemplateParameterList* params = getTemplateParams(decl);

    if (!params)
    {
        return "";
    }

    std::string ret;
    llvm::raw_string_ostream out(ret);
    const clang::PrintingPolicy& policy = decl->getASTContext().getPrintingPolicy();

    out << "template <";

    for (clang::TemplateParameterList::const_iterator cit = params->begin(); cit != params->end(); ++cit)
    {
        tplParams.push_back("");
        llvm::raw_string_ostream tplParamsOut(tplParams.back());

        if (cit != params->begin())
            out << ", ";

        if (const clang::TemplateTypeParmDecl* typeParam = clang::dyn_cast<clang::TemplateTypeParmDecl>(*cit))
        {
            const char* keyword = typeParam->wasDeclaredWithTypename() ? "typename" : "class";
            out << keyword << " " << *typeParam;
            if(addKeyword)
                tplParamsOut << keyword << " ";
            tplParamsOut << typeParam->getName();
        }
        else if (const clang::NonTypeTemplateParmDecl* nonTypeParam = clang::dyn_cast<clang::NonTypeTemplateParmDecl>(*cit))
        {
            out << nonTypeParam->getType().getAsString(policy);

            if(addKeyword)
                tplParamsOut << nonTypeParam->getType().getAsString();

            if (clang::IdentifierInfo* name = nonTypeParam->getIdentifier())
            {
                out << ' ' << name->getName();
                tplParamsOut << ' ' << name->getName();
            }
        }
        else if (clang::dyn_cast<clang::TemplateTemplateParmDecl>(*cit))
        {
            throw RuntimeError((*cit), "Template template parameter are not supported by reflection");
        }
    }

    out << ">\n";
    out.flush();

    return ret;
}

std::string getTemplateParametersString(const clang::Decl* tplDecl)
{
    std::vector<std::string> params;
    return getTemplateParametersString(tplDecl, params, true);
}

bool isTemplateRecursive(const clang::Decl* decl)
{
    while (decl)
    {
        if (getTemplateDecl(decl))
        {
            return true;
        }
        decl = clang::dyn_cast<clang::TypeDecl>(decl->getDeclContext());
    }
    return false;
}

std::string getTemplateParametersStringRecursive(const clang::Decl* decl)
{
    std::string res;
    if (const clang::CXXRecordDecl* declaring = clang::dyn_cast<clang::CXXRecordDecl>(decl->getDeclContext()))
    {
        res = getTemplateParametersStringRecursive(declaring);
    }
    return res + getTemplateParametersString(decl);
}

/// Returns a boolean stating if the given record object needs a bypass constructor
/// trampoline function to be produced in the TypeInfo object by the reflection system.
bool needsBypassConstructor(const Database& db, const clang::CXXRecordDecl* recDecl, std::string& reason)
{
    /*
    ** The check to see if a class needs a bypass constructor is a recursive one:
    ** 1 check if the class is polymorphic, in that case it always needs the reflect
    **   constructor as it is used to setup the vtables properly when loading an
    **   object.
    ** 2 check if the class declares the bypass constructor explicitly, in this case
    **   the user may have special operations to be performed on reflected loading.
    ** 3 check if any of the parents or fields of the class require the bypass constructor
    **   call, in this case we need the bypass constructor to be called for this as well.
    */
    checkIsDefinition(recDecl);

    // If this type isn't reflected or serializable it doesn't need a bypass ctor
    if (!db.isReflected(recDecl)
        || !db.getAttributeValue(recDecl, "hk::Serialize", true)
        || !db.getAttributeValue(recDecl, "hk::GenerateBypassCtor", true))
        return false;

    for (auto bit = recDecl->bases_begin(); bit != recDecl->bases_end(); ++bit)
    {
        if (hasBypassAndPublicDefaultConstructor(bit->getType()))
        {
            reason = db.getScopedMangledName(bit->getType());
            return true; // a parent needs to be bypass constructed
        }
    }

    for(clang::CXXRecordDecl::field_iterator cit = recDecl->field_begin(); cit != recDecl->field_end(); ++cit)
    {
        if(hasBypassAndPublicDefaultConstructor(cit->getType()))
        {
            reason = cit->getNameAsString();
            return true; // a field needs to be reflect constructed
        }
    }

    return false;
}

bool hasBypassConstructor(const clang::CXXRecordDecl* recDecl)
{
    for(clang::CXXRecordDecl::ctor_iterator cit = recDecl->ctor_begin(); cit != recDecl->ctor_end(); ++cit)
    {
        const clang::CXXConstructorDecl* ctor = *cit;
        if(isBypassConstructor(ctor))
        {
            return true;
        }
    }

    return false;
}

bool hasBypassAndPublicDefaultConstructor(clang::QualType type)
{
    if (auto recDecl = getCXXRecordFrom(type))
    {
        bool hasBypass = false, hasPublicDefault = false;
        for (clang::CXXRecordDecl::ctor_iterator cit = recDecl->ctor_begin(); cit != recDecl->ctor_end(); ++cit)
        {
            const clang::CXXConstructorDecl* ctor = *cit;
            if (isBypassConstructor(ctor))
            {
                if (hasPublicDefault)
                {
                    return true;
                }
                hasBypass = true;
            }
            else if (ctor->isDefaultConstructor() && ctor->getAccess() == clang::AS_public)
            {
                if (hasBypass)
                {
                    return true;
                }
                hasPublicDefault = true;
            }
        }
    }
    return false;
}

bool isBypassConstructor(const clang::CXXConstructorDecl* ctorDecl)
{
    if(ctorDecl->getMinRequiredArguments() == 1)
    {
        const clang::ParmVarDecl* p = ctorDecl->getParamDecl(0);
        // Don't do a simple equality check (==) here because there might be a "class" keyword
        // in front of the type depending on the printing policy.
        if(p->getType().getAsString().find("hkReflect::BypassCtorFlag") != std::string::npos)
        {
            return true;
        }
    }
    return false;
}

/// returns a boolean stating if the record is an interface record (Java-style interface).
/// A record will be an interface record if it does not define any data member.
bool isInterfaceRecord(const clang::CXXRecordDecl* decl)
{
    if(!decl->field_empty())
        return false;

    for(clang::CXXRecordDecl::base_class_const_iterator cit = decl->bases_begin(); cit != decl->bases_end(); ++cit)
        if(!isInterfaceRecord(cit->getType().getTypePtr()->getAsCXXRecordDecl()))
            return false;

    return true;
}

clang::QualType resolveTypedefType(clang::QualType type)
{
    while(auto typedefType = clang::dyn_cast<clang::TypedefType>(type))
    {
        unsigned qual = type.getLocalFastQualifiers();
        type = typedefType->getDecl()->getUnderlyingType();
        type.addFastQualifiers(qual);
    }
    return type;
}

clang::QualType resolveElaboratedType(clang::QualType type)
{
    while(auto elabType = clang::dyn_cast<clang::ElaboratedType>(type))
    {
        unsigned qual = type.getLocalFastQualifiers();
        type = elabType->getNamedType();
        type.addFastQualifiers(qual);
    }
    return type;
}

clang::QualType resolveType(clang::QualType type)
{
    for (;;)
    {
        if (auto typedefType = clang::dyn_cast<clang::TypedefType>(type))
        {
            unsigned qual = type.getLocalFastQualifiers();
            type = typedefType->getDecl()->getUnderlyingType();
            type.addFastQualifiers(qual);
        }
        else if (auto elabType = clang::dyn_cast<clang::ElaboratedType>(type))
        {
            unsigned qual = type.getLocalFastQualifiers();
            type = elabType->getNamedType();
            type.addFastQualifiers(qual);
        }
        else if ( auto substType = clang::dyn_cast<clang::SubstTemplateTypeParmType>(type) )
        {
            unsigned qual = type.getLocalFastQualifiers();
            type = substType->getReplacementType();
            type.addFastQualifiers( qual );
        }
        else
        {
            return type;
        }
    }
}

std::string getStaticStringType(const std::string& str)
{
    std::string typeName = "hkReflect::StaticString< ";
    typeName += "'" + str.substr(0, 4) + "'";
    for(std::size_t i = 4; i < str.size(); i += 4)
    {
        typeName += ", '";
        typeName += str.substr(i, 4);
        typeName += "'";
    }
    typeName += " >";
    return typeName;
}

std::string pyNameFromCxxName(const std::string& cxxName)
{
    std::string pyName;
    pyName.reserve(cxxName.size());
    for(std::string::const_iterator cit = cxxName.begin(); cit != cxxName.end(); ++cit)
    {
        switch(*cit)
        {
            case ':':
            {
                pyName.push_back('_');
                if (*(cit+1) == ':')
                {
                    ++cit;
                }
                break;
            }
            case '*':
            {
                pyName.append("Ptr");
                break;
            }
            case '&':
            {
                pyName.append("Ref");
                break;
            }
            case '[':
            {
                pyName.append("Arr");
                break;
            }
            case '<':
            case '>':
            case ',':
            case ' ':
            case ']':
            case '\'':
            {
                //skip
                break;
            }
            default:
            {
                pyName.push_back(*cit);
            }
        }
    }
    return pyName;
}

const clang::CXXRecordDecl* getCXXRecordFrom(const clang::Type* type)
{
    const clang::CXXRecordDecl* rec = type->getAsCXXRecordDecl();
    if(rec && rec->getDefinition())
    {
        return rec->getDefinition();
    }
    else
    {
        // Try to get the definition somewhere else.
        const clang::CXXRecordDecl* fallback = nullptr;
        if(auto spcType = clang::dyn_cast<clang::TemplateSpecializationType>(type))
        {
            // todo.nt4m This specialization is dependent on some template parameters of the current decl,
            // so it is not possible to associate a specific record decl to it. We use the record decl
            // of its template, but this might be unsafe (e.g. if some specializations have different attributes
            // from their template).
            if(const clang::ClassTemplateDecl* tplDecl = clang::dyn_cast<clang::ClassTemplateDecl>(spcType->getTemplateName().getAsTemplateDecl()))
            {
                fallback = getCXXRecordFromTemplate(tplDecl);
            }
        }
        else if (auto dType = clang::dyn_cast<clang::DependentNameType>(type))
        {
            const clang::NestedNameSpecifier* scope = dType->getQualifier();
            const clang::IdentifierInfo* id = dType->getIdentifier();
            fallback = clang::dyn_cast_or_null<clang::CXXRecordDecl>(getDependentTypeDefinition(scope, id));
        }
        else if (auto dtType = clang::dyn_cast<clang::DependentTemplateSpecializationType>(type))
        {
            const clang::NestedNameSpecifier* scope = dtType->getQualifier();
            const clang::IdentifierInfo* id = dtType->getIdentifier();
            fallback = clang::dyn_cast_or_null<clang::CXXRecordDecl>(getDependentTypeDefinition(scope, id));
        }
        else if (auto injType = clang::dyn_cast<clang::InjectedClassNameType>(type))
        {
            fallback = getCXXRecordFromTemplate(injType->getDecl()->getDescribedClassTemplate());
        }
        if (fallback && fallback->getDefinition())
        {
            return fallback->getDefinition();
        }
    }

    if (rec)
    {
        hkLog.debug(rec, "Found forward-declared type %s", rec->getNameAsString().c_str());
    }
    return rec;
}

void checkIsDefinition(const clang::CXXRecordDecl* decl)
{
    if (decl == nullptr || !decl->isThisDeclarationADefinition())
    {
        throw RuntimeError(decl, "Could not find definition for '%s'", decl->getQualifiedNameAsString().c_str());
    }
}

const clang::CXXMethodDecl* hasAfterReflectNew( const clang::CXXRecordDecl* recDecl )
{
    for(clang::CXXRecordDecl::method_iterator it = recDecl->method_begin(); it != recDecl->method_end(); ++it)
    {
        clang::CXXMethodDecl* m = *it;
        if( m->getNameAsString() == "afterReflectNew" )
        {
            if( m->isVirtual() ) { hkLog.error(m, "afterReflectNew must not be virtual"); }
            if( m->isStatic() ) { hkLog.error(m, "afterReflectNew must not be static"); }
            return m;
        }
    }
    return nullptr;
}

bool hasFields(const clang::CXXRecordDecl* decl)
{
    if(!decl->field_empty())
        return true;

    for(clang::CXXRecordDecl::base_class_const_iterator it = decl->bases_begin(); it != decl->bases_end(); ++it)
    {
        if(const clang::CXXRecordDecl* baseDecl = it->getType()->getAsCXXRecordDecl())
        {
            if(hasFields(baseDecl))
                return true;
        }
        // todo.nt if the base isn't a CXXRecordDecl it's a template parameter.
    }

    return false;
}

std::pair<std::string, std::string> getNamespaceBlock(const clang::Decl* decl)
{
    std::pair<std::string, std::string> res;

    for(const clang::DeclContext* declContext = decl->getDeclContext(); declContext; declContext = declContext->getParent())
    {
        if(const clang::NamespaceDecl* nsDecl = clang::dyn_cast<clang::NamespaceDecl>(declContext))
        {
            res.first.insert(0, "namespace " + nsDecl->getNameAsString() + "\n{\n");
            res.second += "}\n";
        }
    }

    return res;
}

std::string getNamespaceString(const clang::Decl* decl)
{
    for(const clang::DeclContext* declContext = decl->getDeclContext(); declContext; declContext = declContext->getParent())
    {
        if(const clang::NamespaceDecl* nsDecl = clang::dyn_cast<clang::NamespaceDecl>(declContext))
        {
            return nsDecl->getQualifiedNameAsString() + "::";
        }
    }
    return "";
}

bool definitionIsA(const clang::CXXRecordDecl* parentDefinition, const clang::CXXRecordDecl* definitionBeingTested)
{
    bool ret = false;

    const clang::CXXRecordDecl* parDef = definitionBeingTested;
    while(parDef)
    {
        if(parDef == parentDefinition)
        {
            ret = true;
            break;
        }

        clang::CXXRecordDecl::base_class_const_iterator it = parDef->bases_begin();
        clang::CXXRecordDecl::base_class_const_iterator end = parDef->bases_end();
        parDef = NULL;
        if(it != end)
        {
            parDef = it->getType()->getAsCXXRecordDecl();
        }

        if(parDef)
        {
            if(!parDef->isThisDeclarationADefinition())
            {
                parDef = parDef->getDefinition();
            }
            if(!parDef->isThisDeclarationADefinition()) // Parent record should always be defined.
            {
                throw RuntimeError(parDef, "Couldn't find a definition for %s\n", parDef->getNameAsString().c_str());
            }
        }
    }

    return ret;
}

MethodType getMethodType(const Database& db, clang::QualType type, SpecialMethod specialMethod, clang::AccessSpecifier& asOut, bool& mightBeAbstract);

MethodType getMethodType(
    const Database& db,
    const clang::CXXRecordDecl* recDecl,
    SpecialMethod specialMethod,
    clang::AccessSpecifier& asOut,
    bool& mightBeAbstract)
{
    asOut = clang::AS_public;
    mightBeAbstract = recDecl->isAbstract();

    // Check for explicit attributes.
    if (db.getAttributeValue(recDecl, "hk::Pod", false))
    {
        return METHOD_TRIVIAL;
    }
    if (db.getAttribute(recDecl, "hk::FixupReflectedMethods"))
    {
        return METHOD_UNKNOWN;
    }

    // Early out for missing default constructor.
    if (specialMethod == DEF_CONSTRUCTOR && !recDecl->hasDefaultConstructor())
    {
        return METHOD_UNAVAILABLE;
    }

    const SpecialMethodsData& smData = getSpecialMethodsData(specialMethod);

    // Early out if the special method is trivial.
    // Clang will consider all non-fully-specialized template fields/parents as trivial, so we cannot
    // use this shortcut with template decls
    if ((recDecl->*smData.declHasTrivialMethod)() && !hasTemplateParamsRecursive(recDecl))
    {
        return METHOD_TRIVIAL;
    }

    // Get the special method.
    const clang::CXXMethodDecl* method = smData.findMethodFunc(recDecl);

    if (method && !method->isImplicit())
    {
        asOut = method->getAccess();
        return METHOD_EXPLICIT;
    }
    else
    {
        // If the type has vtables reflection must use the native default ctor (no way to get the vtables properly
        // initialized otherwise). Templates are not correctly recognized as polymorphic so we always use explicit for them.
        if (specialMethod == DEF_CONSTRUCTOR && (recDecl->isPolymorphic() || hasTemplateParamsRecursive(recDecl)))
        {
            return METHOD_EXPLICIT;
        }

        // Implicit method, must check the type manually.
        // The method type will be: trivial, if the method is trivial in all parents and fields; implicit, if the method is
        // non-trivial in at least one parent/field and trivial in all the others; or unavailable, if the method is
        // unavailable in at least one parent/field.
        MethodType res = METHOD_TRIVIAL;
        for (clang::CXXRecordDecl::base_class_const_iterator it = recDecl->bases_begin(); it != recDecl->bases_end(); ++it)
        {
            clang::AccessSpecifier parentAs;
            bool parentMightBeAbstract;
            MethodType parentType = getMethodType(db, it->getType(), specialMethod, parentAs, parentMightBeAbstract);

            if (parentType == METHOD_UNAVAILABLE)
            {
                return METHOD_UNAVAILABLE;
            }
            else if (parentMightBeAbstract && (specialMethod == DEF_CONSTRUCTOR || specialMethod == COPY_CONSTRUCTOR))
            {
                // Constructors of abstract types can only be called natively.
                res = METHOD_EXPLICIT;
            }
            else if (parentType == METHOD_EXPLICIT)
            {
                switch (parentAs)
                {
                    case clang::AS_private: return METHOD_UNAVAILABLE; 
                    case clang::AS_protected: res = METHOD_EXPLICIT; break; 
                    default: if (res < METHOD_IMPLICIT) res = METHOD_IMPLICIT; break;
                }
            }
            else if (res < parentType)
            {
                // METHOD_UNKNOWN or METHOD_IMPLICIT
                res = parentType;
            }
        }
        for (clang::CXXRecordDecl::field_iterator it = recDecl->field_begin(); it != recDecl->field_end(); ++it)
        {
            // types with const fields cannot be copy-assigned
            if (specialMethod == COPY_ASSIGNMENT && it->getType().isConstQualified())
            {
                return METHOD_UNAVAILABLE;
            }

            clang::AccessSpecifier fieldAs;
            bool fieldMightBeAbstract;
            MethodType fieldType = getMethodType(db, it->getType(), specialMethod, fieldAs, fieldMightBeAbstract);
            if (fieldType == METHOD_UNAVAILABLE || fieldAs != clang::AS_public)
            {
                return METHOD_UNAVAILABLE;
            }
            else if (fieldType == METHOD_UNKNOWN)
            {
                if (res < METHOD_UNKNOWN) res = METHOD_UNKNOWN;
            }
            else if (fieldType != METHOD_TRIVIAL)
            {
                if (res < METHOD_IMPLICIT) res = METHOD_IMPLICIT;
            }
        }
        return res;
    }
}

MethodType getMethodType(
    const Database& db,
    clang::QualType type,
    SpecialMethod specialMethod,
    clang::AccessSpecifier& asOut,
    bool& mightBeAbstract)
{
    asOut = clang::AS_public;
    mightBeAbstract = false;

    if (const clang::CXXRecordDecl* recDecl = getCXXRecordFrom(type))
    {
        checkIsDefinition(recDecl);
        return getMethodType(db, recDecl, specialMethod, asOut, mightBeAbstract);
    }
    else if (const clang::ArrayType* arr = clang::dyn_cast<clang::ArrayType>(type))
    {
        return getMethodType(db, arr->getElementType(), specialMethod, asOut, mightBeAbstract);
    }
    else if (const clang::ReferenceType* ref = clang::dyn_cast<clang::ReferenceType>(type))
    {
        return specialMethod == DEF_CONSTRUCTOR ? METHOD_UNAVAILABLE : METHOD_TRIVIAL;
    }
    else if (clang::isa<clang::TemplateTypeParmType>(type) ||
        clang::isa<clang::DependentNameType>(type) ||
        clang::isa<clang::DependentTemplateSpecializationType>(type))
    {
        // no information about the type
        mightBeAbstract = true;
        return METHOD_UNKNOWN;
    }
    else
    {
        return METHOD_TRIVIAL;
    }
}

MethodType getMethodType(const Database& db, const clang::CXXRecordDecl* recDecl, SpecialMethod specialMethod)
{
    clang::AccessSpecifier access;
    bool dummy;
    MethodType res = getMethodType(db, recDecl, specialMethod, access, dummy);
    return access == clang::AS_public ? res : METHOD_UNAVAILABLE;
}

clang::QualType getFieldTypeFrom(const Database& db, const clang::Decl* decl)
{
    if (const clang::FieldDecl* fieldDecl = clang::dyn_cast<clang::FieldDecl>(decl))
        return fieldDecl->getType();
    else if (auto propDecl = isPropertyDecl(decl))
    {
        // Return the result type of the get method.
        const clang::CXXMethodDecl* getmeth;
        std::tie(std::ignore, getmeth, std::ignore) = getPropertyData(db, propDecl);
        return getmeth->getResultType();
    }
    return clang::QualType();
}

clang::QualType getTypeFrom(const Database& db, const clang::Decl* decl)
{
    auto type = getFieldTypeFrom(db, decl);
    if (!type.isNull())
    {
        return type;
    }

    if (auto typeDecl = clang::dyn_cast<clang::TypeDecl>(decl))
    {
        return db.astContext.getTypeDeclType(typeDecl);
    }
    throw FatalError(decl, "Cannot get the type of a %s decl", decl->getDeclKindName());
}

const clang::TypeDecl* getDeclFrom(clang::QualType type)
{
    if (auto tdefType = clang::dyn_cast<clang::TypedefType>(type))
    {
        return tdefType->getDecl();
    }
    else if (auto enumType = clang::dyn_cast<clang::EnumType>(type))
    {
        return enumType->getDecl();
    }
    else
    {
        return getCXXRecordFrom(type);
    }
}

const clang::TypeDecl* getTypeDeclFrom(const Database& db, const clang::Decl* decl)
{
    auto type = getFieldTypeFrom(db, decl);
    if (!type.isNull())
    {
        return getDeclFrom(type);
    }
    return clang::dyn_cast<clang::TypeDecl>(decl);
}

const clang::CXXRecordDecl* getCXXRecordFromTemplate(const clang::ClassTemplateDecl* tplDecl)
{
    const clang::CXXRecordDecl* rec = tplDecl->getTemplatedDecl();
    if (rec->getDefinition())
    {
        return rec->getDefinition();
    }

    // If this is a template nested in a specialization, its definition is contained inside the main
    // template of the specialization.
    if (auto outerDecl = clang::dyn_cast<clang::ClassTemplateSpecializationDecl>(tplDecl->getDeclContext()))
    {
        const clang::CXXRecordDecl* outerTpl = getCXXRecordFromTemplate(outerDecl->getSpecializedTemplate());
        if (outerTpl && outerTpl->isThisDeclarationADefinition())
        {
            if (const clang::ClassTemplateDecl* origDecl = findMemberDecl<clang::ClassTemplateDecl>(outerTpl, tplDecl->getNameAsString()))
            {
                const clang::CXXRecordDecl* fallback = origDecl->getTemplatedDecl();
                if (fallback && fallback->getDefinition())
                {
                    return fallback->getDefinition();
                }
            }
        }
    }

    hkLog.debug(rec, "Found forward-declared type %s", rec->getNameAsString().c_str());
    return rec;
}

const clang::TypeDecl* getDependentTypeDefinition(const clang::NestedNameSpecifier* scope, const clang::IdentifierInfo* id)
{
    clang::QualType declaring;
    return getDependentTypeDefinition(scope, id, declaring);
}

const clang::TypeDecl* getDependentTypeDefinition(const clang::NestedNameSpecifier* scope, const clang::IdentifierInfo* id, clang::QualType& scopeAsType)
{
    if (scope->getKind() == clang::NestedNameSpecifier::TypeSpec ||
        scope->getKind() == clang::NestedNameSpecifier::TypeSpecWithTemplate)
    {
        scopeAsType = resolveType(clang::QualType(scope->getAsType(), 0));
        const clang::CXXRecordDecl* recDecl = getCXXRecordFrom(scopeAsType);

        if(recDecl && recDecl->isThisDeclarationADefinition())
        {
            // search a decl with the same name in the scope
            if (auto member = findMemberDecl<clang::NamedDecl>(recDecl, id->getName()))
            {
                if (auto tplDecl = clang::dyn_cast<clang::ClassTemplateDecl>(member))
                {
                    return getCXXRecordFromTemplate(tplDecl);
                }
                return clang::dyn_cast<clang::TypeDecl>(member);
            }
        }
    }
    scopeAsType = clang::QualType();
    return nullptr;
}

const clang::CXXRecordDecl* isPropertyDecl(const clang::Decl* decl)
{
    if (auto rec = clang::dyn_cast<clang::CXXRecordDecl>(decl))
    {
        if (clang::isa<clang::CXXRecordDecl>(decl->getDeclContext()) &&
            stringStartsWith(rec->getNameAsString(), "PropertyPlaceholder")) 
        {
            return rec;
        }
    }
    return nullptr;
}

std::string getPropertyDeclName(const clang::CXXRecordDecl* decl)
{
    // todo.rt use this and remove name from the attribute
    return fieldNameWithoutPrefix("PropertyPlaceholder", decl->getNameAsString());
}

PropertyData getPropertyData(const Database& db, const clang::CXXRecordDecl* propertyDecl)
{
    auto outerDecl = clang::dyn_cast<clang::CXXRecordDecl>(propertyDecl->getDeclContext());
    assert(outerDecl);

    const std::string& cxxName = db.getScopedMangledName(outerDecl);

    Optional<Attribute> optAttr = db.getAttributeValue< Optional<Attribute> >(propertyDecl, "hk::PropertyInternal");
    if (!optAttr)
    {
        throw RuntimeError(propertyDecl, "Property placeholder '%s::%s' does not have a property attribute",
            cxxName.c_str(), propertyDecl->getNameAsString().c_str());
    }

    const Attribute& propertyAttr = *optAttr;
    std::string name = propertyAttr.get("name"); // name is mandatory
    std::string getName = propertyAttr.get("getter"); // getter is mandatory
    std::string setName = propertyAttr.get("setter");

    const clang::CXXMethodDecl* getmeth = 0;
    const clang::CXXMethodDecl* setmeth = 0;
    for (clang::CXXRecordDecl::method_iterator it = outerDecl->method_begin();
        it != outerDecl->method_end() && (getmeth == 0 || setmeth == 0);
        ++it)
    {
        const std::string& methName = (*it)->getNameAsString();
        if (methName == getName)
            getmeth = *it;
        else if (methName == setName)
            setmeth = *it;
    }

    if (getmeth == 0)
        throw RuntimeError(propertyDecl, "%s::%s: nonexistent getter method in property.", cxxName.c_str(), getName.c_str());

    if (!getmeth->isConst())
        throw RuntimeError(getmeth, "%s::%s must be const to be used as a getter in property.", cxxName.c_str(), getName.c_str());

    if (!setName.empty())
    {
        if (setmeth == 0)
            throw RuntimeError(propertyDecl, "%s::%s: nonexistent setter method in property.", cxxName.c_str(), setName.c_str());

        if (setmeth->isConst())
            throw RuntimeError(setmeth, "%s::%s can't be const to be used as a setter in property.", cxxName.c_str(), setName.c_str());

        if (setmeth->getNumParams() != 1)
            throw RuntimeError(setmeth, "%s::%s must have exactly one argument.", cxxName.c_str(), setName.c_str());
    }

    return std::make_tuple(name, getmeth, setmeth);
}

const clang::TypedefDecl* getHkDeclareClassDecl( const clang::CXXRecordDecl* recDecl )
{
    typedef clang::DeclContext::specific_decl_iterator<clang::TypedefDecl> TypedefIterator;
    for ( TypedefIterator it = TypedefIterator( recDecl->decls_begin() ); it != TypedefIterator( recDecl->decls_end() ); ++it )
    {
        const clang::TypedefDecl* typedefDecl = *it;
        if ( typedefDecl->getName() == "_HkDeclareClassName" )
        {
            return typedefDecl;
        }
    }
    return NULL;
}

std::string getAttrName(const clang::AnnotateAttr* annotation, const clang::SourceManager& sm)
{
    const std::string& fullAttr = annotation->getAnnotation();
    if (fullAttr.find(Database::attributePrefix) == std::string::npos)
    {
        throw RuntimeError(sm, annotation->getLocation(), "Trying to get the value of a non-Havok attribute.");
    }
    hkAttributeParser parser(hkStringView(fullAttr.data() + Database::attributePrefixLength,
        int(fullAttr.length() - Database::attributePrefixLength)));
    if (!parser.advance())
    {
        throw RuntimeError(sm, annotation->getLocation(), "Invalid attribute string: '%s'", fullAttr.c_str());
    }

    hkStringBuf name;
    if (!parser.currentAttrName(name).isSuccess())
        throw RuntimeError(sm, annotation->getLocation(), name.cString());
    return name;
}

std::string getAttrNameOrEmpty(const clang::AnnotateAttr* annotation, const clang::SourceManager& sm)
{
    try
    {
        return getAttrName(annotation, sm);
    }
    catch (const RuntimeError&)
    {
        return std::string();
    }
}

namespace
{
    typedef std::vector<const clang::CXXMethodDecl*> Methods;
    void getPureMethods(const clang::CXXRecordDecl* recDecl, Methods& methods)
    {
        // Get inherited pure methods.
        for (auto baseIt = recDecl->bases_begin(); baseIt != recDecl->bases_end(); ++baseIt)
        {
            if (auto recParent = getCXXRecordFrom(baseIt->getType()))
            {
                if (!getTemplateDecl(recParent) && !recParent->isAbstract())
                {
                    // Early out.
                    continue;
                }

                getPureMethods(recParent, methods);
            }
            else
            {
                // todo.rt database?
                hkLog.info("Cannot detect pure virtual methods in class '%s', one of its parents is a dependent type. Assuming it has none.",
                    recDecl->getQualifiedNameAsString().c_str());
            }
        }

        for (auto methIt = recDecl->method_begin(); methIt != recDecl->method_end(); ++methIt)
        {
            if (methIt->isPure())
            {
                methods.push_back(*methIt);
            }
            for (auto overriddenIt = methIt->begin_overridden_methods(); overriddenIt != methIt->end_overridden_methods(); ++overriddenIt)
            {
                auto found = std::find(methods.begin(), methods.end(), *overriddenIt);
                if (found != methods.end())
                {
                    methods.erase(found);
                }
            }
        }
    }
}

bool isAbstract(const clang::CXXRecordDecl* recDecl)
{
    if (!getTemplateDecl(recDecl))
    {
        return recDecl->isAbstract();
    }

    // CXXRecordDecl->isAbstract() does not work for templates, must check it manually.
    Methods pureMethods;
    getPureMethods(recDecl, pureMethods);
    return pureMethods.size() != 0;
}

bool isPubliclyAccessible(const clang::Decl* decl)
{
    auto outer = decl;
    do
    {
        if (outer->getAccess() == clang::AS_protected || outer->getAccess() == clang::AS_private)
        {
            return false;
        }
        outer = clang::dyn_cast<clang::CXXRecordDecl>(outer->getDeclContext());
    } while (outer);
    return true;
}

std::string getAggregateTypeFor(const std::string& cxxName, bool isTemplate)
{
    return formatString("%shkReflect::Detail::AggregateOf< %s%s%s >::Type",
        isTemplate ? "typename " : "",
        isTemplate ? "HK_REFLECT_RESOLVE( " : "",
        cxxName.c_str(),
        isTemplate ? " )" : "");
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
