// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#ifndef DATABASE_H
#define DATABASE_H

PRAGMA_WARNING_PUSH
# include <vector>
# include <string>
# include <map>
# include <sstream>

# include <clang/AST/AST.h>
# include <clang/AST/Attr.h>
# include <clang/Sema/Sema.h>
# include <llvm/Support/CommandLine.h>
PRAGMA_WARNING_POP

#include "Attribute.h"
#include "Logger.h"

class Filesystem;
class TkbmsRegistry;
class BuildFileDeclMapVisitor;
namespace clang { class Sema; }

class Database
{
    public:

        friend class BuildFileDeclMapVisitor;

        typedef std::vector<clang::TypeDecl*> DeclListType;

        struct FileInfo
        {
            DeclListType decls;
            std::set<std::string> extraIncludes;
        };

        typedef std::map<std::string, FileInfo> FileDeclMapType;

        // A map of #included filename to the pair (path in #include, source file).
        struct DirectInclude
        {
            /// The input file containing the include
            std::string m_sourceFile;
            /// If m_resolved, then the full path to the included file. Otherwise, the path stated in the include.
            std::string m_path;
            /// The path leads to an actual file found by the preprocessor.
            bool m_resolved;
        };

        /// From filename to DirectInclude.
        typedef std::multimap<std::string, DirectInclude> DirectIncludes;

        Database(Filesystem& fs, const std::vector<std::string>& inputFiles, clang::ASTContext& astContext, clang::Sema& s, const std::vector<std::string>& includePaths,
                 const std::vector<std::string>& invocAttributes, const std::vector<std::string>& forceIncludes, TkbmsRegistry& tkbms, DirectIncludes& dIncludes, const std::string& outputDir,
                 const std::string& preamble, const std::string& postamble, const std::string& preambleCpp, const std::string& postambleCpp, bool outputTkbms);

        void selfCheck() const;
        clang::TranslationUnitDecl* getRoot() const { return astContext.getTranslationUnitDecl(); }
        clang::SourceManager& getSourceManager() const { return astContext.getSourceManager(); }

        bool isDeclaredInInputs(const clang::Decl* decl) const;
        const std::string& getOutputDir() const;
        const std::string& getProjectNameRoot() const { return projectNameRoot; }
        std::string tkbms(const std::string& product = "COMMON", const std::string& platform = "ALL", const std::string& visibility = "PUBLIC") const;

        bool isReflected(const clang::Decl* decl) const;
        bool isReflected(const clang::Type* type) const;
        bool isReflected(const clang::QualType type) const;

        bool isOpaqueTypeAllowed(const clang::FieldDecl* field) const;
        bool isSerializable(const clang::Decl* field) const;

        bool hasTrackerHandle(const clang::TypeDecl* decl) const;
        bool trackContent(const clang::Decl* decl) const;

        enum NameType
        {
            NAME_NORMAL,
            NAME_REFLECTION,    
            NAME_DECLARATION,   
            NAME_RESOLVE_TAGS,  
        };

        std::string getScopedMangledName(const clang::NamedDecl* decl, NameType mode = NAME_NORMAL) const;
        std::string getScopedMangledName(clang::QualType type, NameType mode = NAME_NORMAL) const;

        /// Does not add the "typename" and "const" keywords even if the type needs it.
        std::string getScopedMangledNameNoPrefix(clang::QualType type, NameType mode = NAME_NORMAL) const;

        std::string getScopeName(const clang::NamedDecl* decl, NameType mode = NAME_NORMAL) const;

        /// Get an identifier for a method inside the declaring class (mangles name, arguments, return type and constness).
        std::string getMangledMethodName(const clang::CXXMethodDecl* method) const;

        std::string getAttrName(const clang::AnnotateAttr* annotation) const;
        std::string getAttrNameOrEmpty(const clang::AnnotateAttr* annotation) const;

        const clang::AnnotateAttr* getAttribute(const clang::Decl* decl, const std::string& attrName) const;
        const clang::AnnotateAttr* getAttributeInHierarchy(const clang::Decl* decl, const std::string& attrName) const;

        std::vector<const clang::AnnotateAttr*> getAllAttributes(const clang::Decl* decl) const;
        std::vector<const clang::AnnotateAttr*> getAllAttributes(const clang::Decl* decl, const std::string& attrName) const;

        const AttributeDecl* getRuntimeAttributeDecl(const std::string& name) const;
        const AttributeDecl* getStaticAttributeDecl(const std::string& name) const;
        const AttributeDecl* getAttributeDecl(const std::string& name) const;

        const clang::CXXRecordDecl* getMessageTypeDecl(const std::string& name) const;

        /// Finds the record declaration corresponding to the definition of the record whose declared name
        /// matches the passed in string. Returns NULL if no definition is found.
        /// The declared name represents an unqualified name. If multiple records have the same unqualified name
        /// then the first one will be returned.
        const clang::NamedDecl* findDecl(const std::string& name) const;
        const clang::CXXRecordDecl* findRecordDefinition(const std::string& name) const;

        template<typename ValueType>
            ValueType getAttributeValue(const clang::AnnotateAttr* attr) const;

        template<typename ValueType>
            ValueType getAttributeValue(const clang::Decl* decl, const std::string& attrName, ValueType notFoundValue) const;

        template<typename ValueType>
            ValueType getAttributeValue(const clang::Decl* decl, const std::string& attrName) const { return getAttributeValue(decl, attrName, ValueType()); }

        template<typename ValueType>
            ValueType getAttributeValueInHierarchy(const clang::Decl* decl, const std::string& attrName, ValueType notFoundValue) const;

        template<typename ValueType>
            ValueType getAttributeValueInHierarchy(const clang::Decl* decl, const std::string& attrName) const { return getAttributeValueInHierarchy(decl, attrName, ValueType()); }

        template<typename ValueType>
            std::vector<ValueType> getAllAttributesValues(const clang::Decl* decl, const std::string& attrName) const;

        std::vector<Attribute> getAllRuntimeAttributes(const clang::Decl* decl) const;
        std::vector<Attribute> getAllStaticAttributes(const clang::Decl* decl, const char* attrName=nullptr) const;

        void addAttribute(clang::Decl* decl, const std::string& hkAttrString,
            clang::SourceRange attrRange = clang::SourceRange());

        void replaceAttribute(clang::Decl* decl, const clang::AnnotateAttr* orig, const std::string& hkAttrString,
            clang::SourceRange attrRange = clang::SourceRange());

        static std::string getSourceFileName(const clang::Decl* decl);

        int getOptionValue(clang::SourceLocation srcLoc, const std::string& option) const;

        static bool isOldStyleAttribute(const std::string& name);

        bool isDependentTypeDecl(const clang::TypeDecl* decl) const;

        Attribute getReflectDetails(const clang::Decl* decl) const;

        std::string getReflectedTypeAddress(const clang::QualType& type, bool tracker = false) const;
        std::string getReflectedTypeAddress(const clang::NamedDecl* decl, bool tracker = false) const;
        std::string getReflectedTypeAddress(const std::string& typeName, bool tracker = false) const;

        bool shouldBeExpanded(clang::QualType type) const;
        bool hasOwnVtable(const clang::CXXRecordDecl* recDecl) const;

        std::string getPreamble(const std::string& outputName, const std::string& headerName = "") const;
        std::string getPostamble(const std::string& outputName) const;

        Filesystem& fs;
        std::vector<std::string> inputFiles;
        clang::ASTContext& astContext;
        clang::Sema& sema;
        FileDeclMapType inputFilesDecls;
        const std::vector<std::string>& includePaths;
        const std::vector<std::string> forceIncludes;
        std::string forceIncludesCode;
        TkbmsRegistry& tkbmsRegistry;

        std::string preamble;
        std::string postamble;
        std::string preambleCpp;
        std::string postambleCpp;

        /// Although this is populated during preprocessing, the database carries a reference to it for convenience.
        DirectIncludes& directIncludes;

        static const char* const attributePrefix;
        static const int attributePrefixLength;
        static const char* allowedOldAttributes[];
    private:

        bool isMethodReflectedByAttribute(const clang::CXXMethodDecl* methDecl, const Attribute& reflectMethodsAttr) const;
        std::string getMangledName(const clang::NamedDecl* decl, NameType mode) const;
        std::pair<std::string, const clang::TypeDecl*> getScopedMangledNameInternal(clang::QualType type, NameType mode) const;

        std::string projectNameRoot;
        std::map<std::string, std::string> invocationAttributes;

        std::map<std::string, AttributeDecl> runtimeAttributeDecl;
        std::map<std::string, AttributeDecl> staticAttributeDecl;

        std::map<std::string, const clang::CXXRecordDecl*> messageTypeDecl;
        std::map<std::string, int> optionEnumValues;

        Database(const Database&);
        Database& operator=(const Database&);

        std::string m_outputDir;
        bool m_outputTkbms;
};

#include "Database.inl"

#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
