// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
PRAGMA_WARNING_PUSH
# include <algorithm>
# include <fstream>

# include <clang/Basic/SourceManager.h>
# include <clang/Basic/FileManager.h>
# include <clang/AST/RecursiveASTVisitor.h>
# include <clang/Sema/Lookup.h>
# include <clang/Sema/Scope.h>
# include <clang/AST/RecordLayout.h>
# include <llvm/Support/ManagedStatic.h>
PRAGMA_WARNING_POP

#include "Database.h"
#include "Logger.h"
#include "utils/ElementUtils.h"
#include "utils/StlUtils.h"
#include "utils/RuntimeError.h"
#include "utils/Optional.h"
#include "utils/Filesystem.h"
#include "utils/HavokUtils.h"
#include "utils/TkbmsRegistry.h"

class BuildFileDeclMapVisitor : public clang::RecursiveASTVisitor<BuildFileDeclMapVisitor>
{
public:
    explicit BuildFileDeclMapVisitor(Database& db, const clang::ASTContext& astCtx) : m_db(db), m_astCtx(astCtx) {}

    bool VisitTypeDecl(clang::TypeDecl* decl)
    {
        // Exclude implicit specializations and template parameters
        if (auto specDecl = clang::dyn_cast<clang::ClassTemplateSpecializationDecl>(decl))
        {
            if (!specDecl->isExplicitSpecialization())
            {
                return true;
            }
        }
        else if (clang::isa<clang::TemplateTypeParmDecl>(decl))
        {
            return true;
        }

        // Exclude record/enum non-definitions
        if (auto tagDecl = clang::dyn_cast<clang::TagDecl>(decl))
        {
            if (!tagDecl->isThisDeclarationADefinition())
            {
                return true;
            }
        }

        // Add the decl to the input map.
        const std::string& declFile = Database::getSourceFileName(decl);
        if (!declFile.empty())
        {
            if(std::binary_search(m_db.inputFiles.begin(), m_db.inputFiles.end(), declFile, Filesystem::sortFileNames))
                m_db.inputFilesDecls[declFile].decls.push_back(decl);
        }

        // Check the decl attributes.
        if(decl->hasAttrs())
        {
            const clang::AttrVec& attrVec = decl->getAttrs();
            typedef clang::specific_attr_iterator<clang::AnnotateAttr> SpecificIterator;
            for( SpecificIterator iterator = clang::specific_attr_begin<clang::AnnotateAttr>(attrVec), end_iterator = clang::specific_attr_end<clang::AnnotateAttr>(attrVec);
                iterator != end_iterator;
                ++iterator )
            {
                if(iterator->getAnnotation().startswith(Database::attributePrefix)) // Check that the attribute starts with hk_attr (Havok attribute).
                {
                    // Get attribute name.
                    std::string attrName;
                    try
                    {
                        attrName = m_db.getAttrName(*iterator);
                    }
                    catch (const RuntimeError& e)
                    {
                        hkLog.error("%s", e.what());
                        continue;
                    }

                    // Assumes that decls for hk::RuntimeAttribute, hk::StaticAttribute, hk::MessageType are visited
                    // before any type decorated with one of them.
                    if(attrName == "hk::RuntimeAttribute")
                    {
                        if(const clang::CXXRecordDecl* recDecl = clang::dyn_cast<clang::CXXRecordDecl>(decl))
                        {
                            if(getTemplateDecl(recDecl))
                            {
                                hkLog.error(decl, "Attribute '%s' is a template, this is not supported", m_db.getScopedMangledName(decl).c_str());
                                continue;
                            }

                            m_db.runtimeAttributeDecl.insert(std::make_pair(recDecl->getQualifiedNameAsString(), AttributeDecl(m_db, recDecl)));
                        }
                        else
                            hkLog.error(decl, "Unexpected decl got hk::RuntimeAttribute: %s (%s)", decl->getDeclKindName(), decl->getNameAsString().c_str());
                        return true;
                    }
                    else if(attrName == "hk::StaticAttribute")
                    {
                        if(const clang::CXXRecordDecl* typeDecl = clang::dyn_cast<clang::CXXRecordDecl>(decl))
                        {
                            if(getTemplateDecl(typeDecl))
                            {
                                hkLog.error(decl, "Attribute '%s' is a template, this is not supported", m_db.getScopedMangledName(decl).c_str());
                                continue;
                            }
                            m_db.staticAttributeDecl.insert(std::make_pair(typeDecl->getQualifiedNameAsString(), AttributeDecl(m_db, typeDecl)));
                        }
                        else
                            hkLog.error(decl, "Unexpected decl got hk::StaticAttribute: %s (%s)", decl->getDeclKindName(), decl->getNameAsString().c_str());
                        return true;
                    }
                    else if (attrName == "hk::MessageType")
                    {
                        if(const clang::CXXRecordDecl* typeDecl = clang::dyn_cast<clang::CXXRecordDecl>(decl))
                        {
                            m_db.messageTypeDecl[m_db.getScopedMangledName(decl)] = typeDecl;
                        }
                        else
                            hkLog.error(decl, "Message type '%s' should be a record declaration.", decl->getNameAsString().c_str());
                        return true;
                    }
                }
            }
        }

        // store the values of hkReflect::Optional
        if(m_db.optionEnumValues.empty())
        {
            if(const clang::EnumDecl* enumDecl = clang::dyn_cast<clang::EnumDecl>(decl))
            {
                if(enumDecl->isCompleteDefinition() && enumDecl->getQualifiedNameAsString() == "hkReflect::Opt::Values")
                {
                    for(clang::EnumDecl::enumerator_iterator it = enumDecl->enumerator_begin(); it != enumDecl->enumerator_end(); ++it)
                    {
                        m_db.optionEnumValues[it->getNameAsString()] = int(it->getInitExpr()->EvaluateKnownConstInt(m_db.astContext).getZExtValue());
                    }
                }
            }
        }

        return true;
    }

private:

    BuildFileDeclMapVisitor(const BuildFileDeclMapVisitor&);
    BuildFileDeclMapVisitor& operator=(const BuildFileDeclMapVisitor&);

    Database& m_db;
    const clang::ASTContext& m_astCtx;
};

class AttributeCheckVisitor : public clang::RecursiveASTVisitor<AttributeCheckVisitor>
{
    public:

        AttributeCheckVisitor(const Database& db) : m_db(db) {}

        typedef std::pair< const clang::Decl *, const clang::AnnotateAttr * > AttrElementType;
        static bool sortByLocation(const AttrElementType& a, const AttrElementType& b)
        {
            if ( a.second && b.second )
            {
                return a.second->getLocation() < b.second->getLocation();
            }
            else
            {
                return a.first->getLocation() < b.first->getLocation();
            }
        }
        void run()
        {
            for (auto fileIt : m_db.inputFilesDecls)
            {
                for (auto decl : fileIt.second.decls)
                {
                    TraverseDecl(decl);
                }
            }

            // Warn for attributes on non-reflected classes (one warning per class).
            std::vector< std::pair< const clang::Decl *, const clang::AnnotateAttr * > > invalidAttrs(m_invalidAttrs.begin(), m_invalidAttrs.end());
            std::sort(invalidAttrs.begin(), invalidAttrs.end(), sortByLocation);
            for (auto declAttr : invalidAttrs)
            {
                if (declAttr.second)
                {
                    auto named = clang::dyn_cast<clang::NamedDecl>(declAttr.first);

                    if (named && (named->getQualifiedNameAsString() == "hkReflect::ReflectionOf"))
                    {
                        // Attaching attributes to reflected typedefs will generate this. Ignore it
                        continue;
                    }
                    hkLog.warning(m_db.astContext.getSourceManager(), declAttr.second->getLocation(),
                        "Attributes found on non-reflected field/type in '%s', they will be ignored",
                        named ? named->getQualifiedNameAsString().c_str() : "unnamed");
                }
            }
        }

        bool VisitNamedDecl(clang::NamedDecl* decl)
        {
            if (decl->hasAttr<clang::AnnotateAttr>() && m_invalidAttrs.find(decl) == m_invalidAttrs.end())
            {
                bool reflected = m_db.isReflected(decl);
                bool isPropertyPlaceholder = isPropertyDecl(decl) != nullptr;
                clang::AttrVec& attrs = decl->getAttrs();
                for (auto it = clang::specific_attr_begin<clang::AnnotateAttr>(attrs),
                    end = clang::specific_attr_end<clang::AnnotateAttr>(attrs);
                    it != end; ++it)
                {
                    std::string attrName;
                    try
                    {
                        attrName = m_db.getAttrName(*it);
                    }
                    catch (const RuntimeError&)
                    {
                        // Not a Havok attribute.
                        continue;
                    }

                    // todo.ntm Silently ignore some old-style attributes (hk.*, dotnet.*, ...). Should be removed eventually.
                    // Run the clang parser with -v=4 to trigger debug prints for those attributes.
                    if(Database::isOldStyleAttribute(attrName))
                    {
                        hkLog.debug(m_db.astContext.getSourceManager(), it->getLocation(),
                            formatString("Found python-style attribute '%s'", attrName.c_str()));
                        continue;
                    }
                    else if (!m_db.getAttributeDecl(attrName))
                    {
                        hkLog.error(m_db.astContext.getSourceManager(), it->getLocation(),
                            "'%s' is not a valid attribute type\nIt needs to be reflected AND have a RuntimeAttribute "
                            "or StaticAttribute attribute", attrName.c_str());
                        m_invalidAttrs[decl] = nullptr;
                        continue;
                    }

                    // todo.nt6 this should be more general, checking that all the attributes are applied to a decl
                    // they are compatible with
                    const char* allowedOnNonReflected[] =
                    {
                        "hk::Reflect",
                        "hk::MemoryTracker",
                        "hk::Type",
                        "hk::Pod",
                        "hk::Feature",
                        "hk::AlignReal",
                        "hk::ReflectAsTypedef",
                        "hke::RuntimeDataSection",
                    };
                    const char** anrEnd = allowedOnNonReflected + sizeof(allowedOnNonReflected) / sizeof(char*);

                    if (!reflected && !isPropertyPlaceholder && std::find(allowedOnNonReflected, anrEnd, attrName) == anrEnd)
                    {
                        const clang::Decl* keyDecl = decl;
                        if (isPropertyPlaceholder || clang::isa<clang::FieldDecl>(decl) || clang::isa<clang::CXXMethodDecl>(decl))
                        {
                            // Assign any error to the containing record.
                            keyDecl = clang::dyn_cast<clang::Decl>(decl->getDeclContext());
                            assert(keyDecl);
                        }
                        bool isStaticAttribute = false;
                        if (auto named = clang::dyn_cast<clang::NamedDecl>(keyDecl))
                        {
                            isStaticAttribute = m_db.getStaticAttributeDecl(named->getQualifiedNameAsString()) != HK_NULL;
                        }
                        const char* allowedOnStaticAttribute[] =
                        {
                            "hk::StaticAttribute",
                            "hk::Optional",
                            "hk::Default",
                        };
                        const char** asaEnd = allowedOnStaticAttribute + sizeof(allowedOnStaticAttribute) / sizeof(char*);
                        if (!isStaticAttribute || std::find(allowedOnStaticAttribute, asaEnd, attrName) == asaEnd)
                        {
                            m_invalidAttrs.insert(std::pair<const clang::Decl*, const clang::AnnotateAttr*>(keyDecl, *it));
                        }
                    }
                }
            }
            return true;
        }

    private:
        void operator=(const AttributeCheckVisitor&);

        const Database& m_db;
        std::map<const clang::Decl*, const clang::AnnotateAttr*> m_invalidAttrs;
};

namespace
{
    // to be used on [Dependent]TemplateSpecializationType
    template<typename T>
    llvm::ArrayRef<clang::TemplateArgument> getArgs(const T* type)
    {
        return llvm::ArrayRef<clang::TemplateArgument>(type->getArgs(), type->getNumArgs());
    }


    void printSpecializationArguments(const Database& db,
                                      llvm::ArrayRef<clang::TemplateArgument> args,
                                      llvm::raw_string_ostream& ostr,
                                      const clang::TemplateParameterList* params,
                                      Database::NameType nt)
    {
        bool addComma = false;
        for(unsigned i = 0; i < args.size(); ++i)
        {
            const clang::TemplateArgument& arg = args[i];
            if(addComma)
                ostr << ", ";
            addComma = true;

            if(arg.getKind() == clang::TemplateArgument::Type)
            {
                const std::string& paramName = db.getScopedMangledName(arg.getAsType(), nt);
                if (paramName.empty() && arg.getAsType()->isTemplateTypeParmType())
                {
                    // the type might contain no information if it is not specialized;
                    // get the name from the class template
                    if(params)
                    {
                        if (nt == Database::NAME_RESOLVE_TAGS)
                        {
                            ostr << "HK_REFLECT_RESOLVE(" << db.getScopedMangledName(params->getParam(i)) << ")";

                        }
                        else
                        {
                            ostr << db.getScopedMangledName(params->getParam(i));
                        }
                    }
                    else
                    {
                        // Should never happen, but give a compile error in case it does
                        ostr << "PREBUILD_UNKNOWN_TEMPLATE_PARAMETER";
                    }
                }
                else
                {
                    ostr << paramName;
                }
            }
            else
            {
                arg.print(db.astContext.getPrintingPolicy(), ostr);
            }
        }

        // handle default arguments
        if (params)
        {
            for (size_t i = args.size(); i < params->size(); ++i)
            {
                const clang::NamedDecl* param = params->getParam((int)i);
                if(addComma)
                    ostr << ", ";
                addComma = true;

                if (auto typeParam = clang::dyn_cast<clang::TemplateTypeParmDecl>(param))
                {
                    if (typeParam->hasDefaultArgument())
                    {
                        ostr << db.getScopedMangledName(typeParam->getDefaultArgument(), nt);
                    }
                }
                else if (auto nonTypeParam = clang::dyn_cast<clang::NonTypeTemplateParmDecl>(param))
                {
                    if (nonTypeParam->hasDefaultArgument())
                    {
                        nonTypeParam->getDefaultArgument()->printPretty(ostr, nullptr, clang::PrintingPolicy(db.astContext.getLangOpts()));
                    }
                }
                else if (clang::isa<clang::TemplateTemplateParmDecl>(param))
                {
                    hkLog.error(param, "Template template parameter are not supported by reflection");
                }
            }
        }
    }

    std::string printSpecializationArguments(const Database& db, llvm::ArrayRef<clang::TemplateArgument> args, const clang::TemplateParameterList* params, Database::NameType nt)
    {
        std::string res;
        llvm::raw_string_ostream o(res);
        printSpecializationArguments(db, args, o, params, nt);
        o.flush();
        return res;
    }

    std::string getNestedTypePlaceholder(const Database& db, const clang::Type* type)
    {
        // resolve the type, but stop at the first reflected typedef
        for (;;)
        {
            if (auto typedefType = clang::dyn_cast<clang::TypedefType>(type))
            {
                if (db.getAttributeValue<bool>(typedefType->getDecl(), "hk::Reflect", false))
                {
                    break;
                }
                type = typedefType->getDecl()->getUnderlyingType().getTypePtr();
            }
            else if (auto elabType = clang::dyn_cast<clang::ElaboratedType>(type))
            {
                type = elabType->getNamedType().getTypePtr();
            }
            else
            {
                break;
            }
        }

        typedef std::pair<std::string, std::string> ScopeLvl; 

        // collect the scopes of the type
        std::vector<ScopeLvl> scopes;
        {
            // Get:
            // - the type definition (needed for name and template parameters)
            // - the template arguments of the type, if it is a specialization
            // - the scope of the type, if this is a nested type
            const clang::TypeDecl* typeDefinition = nullptr;
            clang::QualType scopeAsType;
            llvm::ArrayRef<clang::TemplateArgument> tplArgs;

            if (auto tdefType = clang::dyn_cast<clang::TypedefType>(type))
            {
                typeDefinition = tdefType->getDecl();
            }
            else if (auto enumType = clang::dyn_cast<clang::EnumType>(type))
            {
                typeDefinition = enumType->getDecl();
            }
            else if (auto dType = clang::dyn_cast<clang::DependentNameType>(type))
            {
                typeDefinition = getDependentTypeDefinition(dType->getQualifier(), dType->getIdentifier(), scopeAsType);
            }
            else if (auto dtType = clang::dyn_cast<clang::DependentTemplateSpecializationType>(type))
            {
                typeDefinition = getDependentTypeDefinition(dtType->getQualifier(), dtType->getIdentifier(), scopeAsType);
                tplArgs = getArgs(dtType);
            }
            else if (const clang::CXXRecordDecl* recDefinition = getCXXRecordFrom(type))
            {
                if (recDefinition->isThisDeclarationADefinition())
                {
                    typeDefinition = recDefinition;
                    if (auto sType = clang::dyn_cast<clang::TemplateSpecializationType>(type))
                    {
                        tplArgs = getArgs(sType);
                    }
                    else if (auto iType = clang::dyn_cast<clang::InjectedClassNameType>(type))
                    {
                        const clang::TemplateSpecializationType* tsType = iType->getInjectedTST();
                        tplArgs = getArgs(tsType);
                    }
                    else if (auto specDecl = clang::dyn_cast<clang::ClassTemplateSpecializationDecl>(recDefinition))
                    {
                        tplArgs = specDecl->getTemplateArgs().asArray();
                    }

                    // if the type is nested in a specialization we cannot get its scope from its definition, because
                    // its definition will not be in the same specialization (it will be rather be in the main template),
                    // so we get its scope from the declaration
                    if (type->getAsCXXRecordDecl())
                    {
                        if (auto outerDecl = clang::dyn_cast<clang::TypeDecl>(type->getAsCXXRecordDecl()->getDeclContext()))
                        {
                            scopeAsType = db.astContext.getTypeDeclType(outerDecl);
                        }
                    }
                }
            }

            if (!typeDefinition)
            {
                return "";
            }

            // get the declaration for the scope
            const clang::NamedDecl* scopeAsDecl = scopeAsType.isNull() ?
                clang::dyn_cast<clang::CXXRecordDecl>(typeDefinition->getDeclContext()) :
                getCXXRecordFrom(scopeAsType);

            // return an empty string if the type is not nested in a template
            if (!scopeAsDecl || !isTemplateRecursive(scopeAsDecl))
            {
                return "";
            }

            // push the first level
            std::string tplArgsString;
            if (!tplArgs.empty())
            {
                tplArgsString = printSpecializationArguments(db, tplArgs, getTemplateDecl(typeDefinition)->getTemplateParameters(), Database::NAME_REFLECTION);
            }
            scopes.push_back(ScopeLvl(typeDefinition->getNameAsString(), tplArgsString));

            // push the scopes which may not have a declaration (incomplete/dependent template specializations)
            while (!scopeAsType.isNull())
            {
                if (auto scopeSpec = clang::dyn_cast<clang::DependentTemplateSpecializationType>(scopeAsType))
                {
                    auto classTemplate = clang::dyn_cast<clang::ClassTemplateDecl>(getTemplateDecl(scopeAsDecl));
                    scopes.push_back(ScopeLvl(scopeSpec->getIdentifier()->getName(),
                        printSpecializationArguments(db, getArgs(scopeSpec),
                        classTemplate->getTemplateParameters(), Database::NAME_REFLECTION)));

                    const clang::TypeDecl* typeDecl = getDependentTypeDefinition(scopeSpec->getQualifier(), scopeSpec->getIdentifier(), scopeAsType);
                    assert(typeDecl);
                    scopeAsDecl = clang::dyn_cast<clang::NamedDecl>(typeDecl->getDeclContext());
                }
                else
                {
                    if (auto templateScopeSpec = clang::dyn_cast<clang::TemplateSpecializationType>(scopeAsType))
                    {
                        auto classTemplate = clang::dyn_cast<clang::ClassTemplateDecl>(templateScopeSpec->getTemplateName().getAsTemplateDecl());
                        scopes.push_back(ScopeLvl(classTemplate->getNameAsString(),
                            printSpecializationArguments(db, getArgs(templateScopeSpec),
                            classTemplate->getTemplateParameters(), Database::NAME_REFLECTION)));

                        scopeAsDecl = clang::dyn_cast<clang::NamedDecl>(
                            templateScopeSpec->getAsCXXRecordDecl() ?
                            templateScopeSpec->getAsCXXRecordDecl()->getDeclContext() :
                            scopeAsDecl->getDeclContext());
                    }
                    scopeAsType = clang::QualType();
                }
            }

            // push the remaining scopes
            for (auto curr = scopeAsDecl; curr; curr = clang::dyn_cast<clang::NamedDecl>(curr->getDeclContext()))
            {
                const clang::ClassTemplateDecl* classTemplate = getTemplateDecl(curr);
                if (const clang::ClassTemplateSpecializationDecl* spec = clang::dyn_cast<clang::ClassTemplateSpecializationDecl>(curr))
                {
                    const clang::TemplateArgumentList& argsList = spec->getTemplateArgs();
                    scopes.push_back(ScopeLvl(curr->getNameAsString(),
                        printSpecializationArguments(db, argsList.asArray(), classTemplate->getTemplateParameters(), Database::NAME_REFLECTION)));
                }
                else if (classTemplate)
                {
                    std::vector<std::string> tplParams;
                    getTemplateParametersString(classTemplate, tplParams, false);
                    scopes.push_back(ScopeLvl(classTemplate->getNameAsString(), joinStrings(tplParams, ", ")));
                }
                else
                {
                    scopes.push_back(ScopeLvl(curr->getNameAsString(), ""));
                }
            }
        }

        // print the name
        std::string typeName;
        std::string tplArgs;
        {
            llvm::raw_string_ostream typeNameStr(typeName);
            llvm::raw_string_ostream tplArgsStr(tplArgs);
            tplArgsStr << "< ";
            bool addComma = false;
            for (std::vector< std::pair<std::string, std::string> >::const_reverse_iterator it = scopes.rbegin(); it != scopes.rend(); it++)
            {
                typeNameStr << it->first;
                if (it + 1 != scopes.rend())
                {
                    typeNameStr << "_";
                }
                if (!it->second.empty())
                {
                    if (addComma)
                    {
                        tplArgsStr << ", ";
                    }
                    addComma = true;
                    tplArgsStr << it->second;
                }
            }
            tplArgsStr << " >";
        }

        return typeName + tplArgs;
    }

    std::string concatWithColon(const std::string& scope, const std::string& name)
    {
        if (scope.empty())
        {
            return name;
        }
        if (name.empty())
        {
            return scope;
        }
        return scope + "::" + name;
    }
}

Database::Database(Filesystem& f, const std::vector<std::string>& inputs, clang::ASTContext& astCtx, clang::Sema& s,
    const std::vector<std::string>& includes, const std::vector<std::string>& invocAttributes,
    const std::vector<std::string>& forceInc, TkbmsRegistry& tkbms, Database::DirectIncludes& dIncludes,
    const std::string& outputDir, const std::string& preamble, const std::string& postamble,
    const std::string& preambleCpp, const std::string& postambleCpp, bool outputTkbms)
    : fs(f)
    , inputFiles(inputs.begin(), inputs.end())
    , astContext(astCtx)
    , sema(s)
    , includePaths(includes)
    , forceIncludes(forceInc.begin(), forceInc.end())
    , tkbmsRegistry(tkbms)
    , preamble(preamble)
    , postamble(postamble)
    , preambleCpp(preambleCpp)
    , postambleCpp(postambleCpp)
    , directIncludes(dIncludes)
    , m_outputDir(outputDir)
    , m_outputTkbms(outputTkbms)
{

    for(std::vector<std::string>::iterator it = inputFiles.begin(); it != inputFiles.end(); ++it)
        *it = Filesystem::absoluteCanonicalPath(*it);
    std::sort(inputFiles.begin(), inputFiles.end(), Filesystem::sortFileNames);

    BuildFileDeclMapVisitor(*this, astCtx).TraverseDecl(getRoot());

    for( llvm::cl::list<std::string>::const_iterator iter = invocAttributes.begin(); iter != invocAttributes.end(); ++iter )
    {
        const std::string::size_type index = iter->find_first_of('=');
        std::string macro, value;
        if(index != std::string::npos)
        {
            macro = iter->substr(0, index);
            value = iter->substr(index+1, std::string::npos);
        }
        else
        {
            macro = (*iter);
        }

        invocationAttributes[macro] = value;
    }

    auto projNameIt = invocationAttributes.find("ProjectName");
    if (projNameIt != invocationAttributes.end())
    {
        projectNameRoot = projNameIt->second;
    }
    else
    {
        // outDir looks like C:/.../Source/XXX/YYY/_Auto and we want to extract YYY
        const std::string& outDir = getOutputDir();
        const std::string& outDirWithoutAuto = outDir.substr(0, outDir.find_last_of(Filesystem::pathSeparator));
        projectNameRoot = outDirWithoutAuto.substr(outDirWithoutAuto.find_last_of(Filesystem::pathSeparator) + 1, std::string::npos);
    }

    // apply HavokPrefix if provided
    auto hkPrefixIt = invocationAttributes.find("HavokPrefix");
    if (hkPrefixIt != invocationAttributes.end())
    {
        projectNameRoot = hkPrefixIt->second + projectNameRoot;
    }

    for(std::vector<std::string>::const_iterator cit = forceIncludes.begin(); cit != forceIncludes.end(); ++cit)
        forceIncludesCode += "#include <" + *cit + ">\n";
}

void Database::selfCheck() const
{
    AttributeCheckVisitor(*this).run();
}


bool Database::isDeclaredInInputs(const clang::Decl* decl) const
{
    const clang::SourceManager& srcManager = getSourceManager();
    clang::PresumedLoc loc = srcManager.getPresumedLoc(decl->getLocation());
    if (loc.isValid())
    {
        const std::string& file = Filesystem::absoluteCanonicalPath(loc.getFilename());
        return std::binary_search(inputFiles.begin(), inputFiles.end(), file, &Filesystem::sortFileNames);
    }
    return false;
}

const std::string& Database::getOutputDir() const
{
    return m_outputDir;
}


bool Database::isMethodReflectedByAttribute(const clang::CXXMethodDecl* methDecl, const Attribute& reflectMethodsAttr) const
{
    // "reflected" must be present (checked by the parser)
    if(reflectMethodsAttr["reflected"]->as<bool>())
    {
        // If the "prefix" is set check the method name
        if(const Optional<AttributeValue>& methPrefix = reflectMethodsAttr["prefix"])
            return stringStartsWith(methDecl->getNameAsString(), methPrefix->as<std::string>());
        else
            return true;
    }

    return false;
}

bool Database::isReflected(const clang::Decl* decl) const
{
    if(const Optional<bool>& reflAttr = getAttributeValue< Optional<bool> >(decl, "hk::Reflect"))
    {
        return *reflAttr;
    }
    else if(const clang::TagDecl* tagDecl = clang::dyn_cast<clang::TagDecl>(decl))
    {
        // Lookup in the definition
        const clang::TagDecl* def = tagDecl->getDefinition();
        if(def && def != decl)
        {
            return isReflected(def);
        }
    }
    else if (const clang::ClassTemplateDecl* tplDecl = clang::dyn_cast<clang::ClassTemplateDecl>(decl))
    {
        // Lookup the attribute in the record decl.
        if (const clang::CXXRecordDecl* recDecl = getCXXRecordFromTemplate(tplDecl))
        {
            return isReflected(recDecl);
        }
    }
    else if(const clang::FieldDecl* field = clang::dyn_cast<clang::FieldDecl>(decl))
    {
        // True if it is the field of a reflected class with reflected fields.
        if(isReflected(field->getParent()))
        {
            // True if there is no hk::ReflectDetails or there is a hk::ReflectDetails(fields=true).
            auto reflDetails = getAttributeValue<Optional<Attribute>>(field->getParent(), "hk::ReflectDetails");
            if (!reflDetails || (*reflDetails)["fields"]->as<bool>())
            {
                return true;
            }
        }
    }
    else if(const clang::CXXMethodDecl* methDecl = clang::dyn_cast<clang::CXXMethodDecl>(decl))
    {
        // True if the class is reflected and has hkReflectDetails(methods=true)
        if(isReflected(methDecl->getParent()))
        {
            if (auto reflDetails = getAttributeValue<Optional<Attribute>>(methDecl->getParent(), "hk::ReflectDetails"))
            {
                if ((*reflDetails)["methods"]->as<bool>())
                {
                    return true;
                }
            }
        }
    }
    else if(const clang::EnumConstantDecl* enumConstDecl = clang::dyn_cast<clang::EnumConstantDecl>(decl))
    {
        // True if the parent enum is reflected.
        clang::EnumDecl* enumDecl = clang::dyn_cast<clang::EnumDecl>(clang::Decl::castFromDeclContext(enumConstDecl->getDeclContext()));
        assert(enumDecl);

        return isReflected(enumDecl);
    }

    return false;
}

bool Database::isReflected(const clang::Type* type) const
{
    type = resolveElaboratedType(clang::QualType(type, 0)).getTypePtr();

    if( clang::dyn_cast<clang::FunctionType>(type) ||
        clang::dyn_cast<clang::MemberPointerType>(type) )
    {
        return false;
    }
    else if(const clang::TagType* tagType = clang::dyn_cast<clang::TagType>(type))
    {
        // Records and enums.
        const clang::TagDecl* tagDef = tagType->getDecl()->getDefinition();
        if(tagDef != NULL)
        {
            return isReflected(tagDef);
        }
        else
        {
            return false;
        }
    }
    else if(const clang::TypedefType* typedefType = clang::dyn_cast<clang::TypedefType>(type))
    {
        if(isReflected(typedefType->getDecl()))
        {
            return true;
        }
        else
        {
            return isReflected(typedefType->getDecl()->getUnderlyingType());
        }
    }
    else if(const clang::ParenType* parenType = clang::dyn_cast<clang::ParenType>(type))
    {
        return isReflected(parenType->getInnerType());
    }
    else if(const clang::PointerType* pointerType = clang::dyn_cast<clang::PointerType>(type))
    {
        return isReflected(pointerType->getPointeeType());
    }
    else if(const clang::ReferenceType* referenceType = clang::dyn_cast<clang::ReferenceType>(type))
    {
        // todo.rt this should be false in general, true only for function parameters
        return isReflected(referenceType->getPointeeType());
    }
    else if(const clang::InjectedClassNameType* injType = clang::dyn_cast<clang::InjectedClassNameType>(type))
    {
        return isReflected(injType->getInjectedTST());
    }
    else if(const clang::TemplateSpecializationType* tsType = clang::dyn_cast<clang::TemplateSpecializationType>(type))
    {
        // check if the main template is reflected
        return isReflected(getCXXRecordFrom(tsType));
    }
    else if (clang::isa<clang::BuiltinType>(type))
    {
        return true;
    }
    else if (auto dType = clang::dyn_cast<clang::DependentNameType>(type))
    {
        const clang::NestedNameSpecifier* scope = dType->getQualifier();
        const clang::IdentifierInfo* id = dType->getIdentifier();
        if (const clang::TypeDecl* decl = getDependentTypeDefinition(scope, id))
        {
            return isReflected(astContext.getTypeDeclType(decl).getTypePtr());
        }
    }
    else if (auto dtType = clang::dyn_cast<clang::DependentTemplateSpecializationType>(type))
    {
        const clang::NestedNameSpecifier* scope = dtType->getQualifier();
        const clang::IdentifierInfo* id = dtType->getIdentifier();
        if (const clang::TypeDecl* decl = getDependentTypeDefinition(scope, id))
        {
            return isReflected(astContext.getTypeDeclType(decl).getTypePtr());
        }
    }
    return false;
}

std::string Database::getAttrName(const clang::AnnotateAttr* annotation) const
{
    return ::getAttrName(annotation, astContext.getSourceManager());
}

std::string Database::getAttrNameOrEmpty(const clang::AnnotateAttr* annotation) const
{
    return ::getAttrNameOrEmpty(annotation, astContext.getSourceManager());
}

const clang::AnnotateAttr* Database::getAttribute(const clang::Decl* decl, const std::string& attrName) const
{
    const std::vector<const clang::AnnotateAttr*>& attrs = getAllAttributes(decl, attrName);

    if(attrs.empty())
        return 0;
    else if(attrs.size() == 1)
        return attrs.front();
    else
        throw RuntimeError(decl, "Attribute '%s' is specified multiple times.", attrName.c_str());
}

std::vector<const clang::AnnotateAttr*> Database::getAllAttributes(const clang::Decl* decl) const
{
    std::vector<const clang::AnnotateAttr*> attrs;

    if (const clang::ClassTemplateDecl* tplDecl = clang::dyn_cast<clang::ClassTemplateDecl>(decl))
        if (const clang::CXXRecordDecl* recDecl = getCXXRecordFromTemplate(tplDecl))
            return getAllAttributes(recDecl);

    if(decl->hasAttr<clang::AnnotateAttr>())
    {
        const clang::AttrVec& attrVec = decl->getAttrs();
        typedef clang::specific_attr_iterator<clang::AnnotateAttr> SpecificIterator;
        for( SpecificIterator iterator = clang::specific_attr_begin<clang::AnnotateAttr>(attrVec), end_iterator = clang::specific_attr_end<clang::AnnotateAttr>(attrVec);
            iterator != end_iterator;
            ++iterator )
        {
            attrs.push_back(*iterator);
        }
    }


    return attrs;

}

std::vector<const clang::AnnotateAttr*> Database::getAllAttributes(const clang::Decl* decl, const std::string& attrName) const
{
    std::vector<const clang::AnnotateAttr*> attrs = getAllAttributes(decl);
    std::vector<const clang::AnnotateAttr*> res;

    for (auto iterator = attrs.begin(); iterator != attrs.end(); ++iterator)
    {
        if(getAttrNameOrEmpty(*iterator) == attrName)
        {
            res.push_back(*iterator);
        }
    }
    return res;
}

const clang::AnnotateAttr* Database::getAttributeInHierarchy(const clang::Decl* decl, const std::string& attrName) const
{
    if(const clang::AnnotateAttr* attr = getAttribute(decl, attrName))
        return attr;

    const clang::CXXRecordDecl* recDecl = 0;

    if(const clang::CXXRecordDecl* r = clang::dyn_cast<clang::CXXRecordDecl>(decl))
        recDecl = r;
    else if(const clang::ClassTemplateDecl* tpl = clang::dyn_cast<clang::ClassTemplateDecl>(decl))
        recDecl = getCXXRecordFromTemplate(tpl);

    if(recDecl == 0 || !recDecl->isThisDeclarationADefinition() || recDecl->bases_begin() == recDecl->bases_end())
        return 0;

    if(const clang::CXXRecordDecl* parent = getCXXRecordFrom(recDecl->bases_begin()->getType()))
        return getAttributeInHierarchy(parent, attrName);

    return 0;
}

std::vector<Attribute> Database::getAllRuntimeAttributes(const clang::Decl* decl) const
{
    const std::vector<const clang::AnnotateAttr*>& attrs = getAllAttributes(decl);
    std::vector<Attribute> objs;

    for (std::vector<const clang::AnnotateAttr*>::const_iterator it = attrs.begin(); it != attrs.end(); ++it)
    {
        const std::string& name = getAttrNameOrEmpty(*it);
        if (!name.empty())
        {
            if (getRuntimeAttributeDecl(name))
            {
                try
                {
                    objs.push_back(Attribute(*this, *it));
                }
                catch (const std::exception& e)
                {
                    hkLog.error("%s", e.what());
                    // skip the invalid attribute
                }
            }
        }
    }
    return objs;
}


std::vector<Attribute> Database::getAllStaticAttributes(const clang::Decl* decl, const char* attrName) const
{
    const std::vector<const clang::AnnotateAttr*>& attrs = getAllAttributes(decl);
    std::vector<Attribute> objs;

    for(std::vector<const clang::AnnotateAttr*>::const_iterator it = attrs.begin(); it != attrs.end(); ++it)
    {
        const std::string& name = getAttrNameOrEmpty(*it);
        if(!name.empty() && (attrName==nullptr || name==attrName) )
        {
            if(getStaticAttributeDecl(name))
            {
                try
                {
                    objs.push_back(Attribute(*this, *it));
                }
                catch(const std::exception&)
                {
                    // skip the invalid attribute
                }
            }
        }
    }
    return objs;
}

std::string Database::tkbms(const std::string& product, const std::string& platform, const std::string& visibility) const
{
    return
        "// TKBMS v1.0 -----------------------------------------------------\n"
        "//\n"
        "// PLATFORM    : " + platform + "\n"
        "// PRODUCT     : " + product + "\n"
        "// VISIBILITY  : " + visibility + "\n"
        "//\n"
        "// ------------------------------------------------------TKBMS v1.0";
}

std::string Database::getMangledName(const clang::NamedDecl* decl, NameType nt) const
{
    std::string cxxName;

    llvm::raw_string_ostream o(cxxName);
    o << decl->getNameAsString();

    if(const clang::ClassTemplateSpecializationDecl* tplDecl = clang::dyn_cast<clang::ClassTemplateSpecializationDecl>(decl))
    {
        // todo.nt6 need to add template disambiguator?
        const clang::TemplateArgumentList& args = tplDecl->getTemplateArgs();

        o << "< ";

        const clang::TemplateParameterList* params = nullptr;
        if(const clang::ClassTemplatePartialSpecializationDecl* pspecDecl = clang::dyn_cast<clang::ClassTemplatePartialSpecializationDecl>(tplDecl))
            params = pspecDecl->getTemplateParameters();

        // print arguments
        printSpecializationArguments(*this, args.asArray(), o, params, nt);

        o << " >";
    }
    else if(const clang::ClassTemplateDecl* ctplDecl = getTemplateDecl(decl))
    {
        const clang::TemplateParameterList* parameters = ctplDecl->getTemplateParameters();

        o << "< ";
        for(clang::TemplateParameterList::const_iterator cit = parameters->begin(); cit != parameters->end(); ++cit)
        {
            const std::string& paramName = (*cit)->getNameAsString();
            if (nt == NAME_RESOLVE_TAGS && clang::isa<clang::TemplateTypeParmDecl>(*cit))
            {
                o << "HK_REFLECT_RESOLVE(" << paramName << ")";
            }
            else
            {
                o << paramName;
            }

            if(cit + 1 != parameters->end())
                o << ", ";
        }
        o << " >";
    }
    o.flush();

    return cxxName;
}

std::string Database::getScopeName(const clang::NamedDecl* decl, NameType nt) const
{
    if(const clang::NamedDecl* namedCtx = clang::dyn_cast<clang::NamedDecl>(decl->getDeclContext()))
    {
        if (const clang::TypeDecl* typeDecl = clang::dyn_cast<clang::TypeDecl>(namedCtx))
        {
            return getScopedMangledNameNoPrefix(astContext.getTypeDeclType(typeDecl), nt);
        }
        return concatWithColon(getScopeName(namedCtx, nt), getMangledName(namedCtx, nt));
    }
    return "";
}

std::string Database::getScopedMangledName(const clang::NamedDecl* decl, NameType nt) const
{
    if (const clang::TypeDecl* typeDecl = clang::dyn_cast<clang::TypeDecl>(decl))
    {
        return getScopedMangledName(astContext.getTypeDeclType(typeDecl), nt);
    }

    return concatWithColon(getScopeName(decl, nt), getMangledName(decl, nt));
}

std::string Database::getScopedMangledNameNoPrefix(clang::QualType type, NameType nt) const
{
    return getScopedMangledNameInternal(type, nt).first;
}

std::pair<std::string, const clang::TypeDecl*> Database::getScopedMangledNameInternal(clang::QualType type, NameType nt) const
{
    clang::PrintingPolicy policy(astContext.getLangOpts());
    policy.SuppressTagKeyword = 1;
    policy.SuppressUnwrittenScope = 0;
    policy.SuppressScope = 0;
    policy.Bool = 1;

    type = resolveElaboratedType(type);

    
#if 0
    if (nt == NAME_REFLECTION && isReflected(type))
    {
        // If the type is nested in a template type (e.g. A<T>::B) and reflected,
        // returns a placeholder which is not a dependent name (A_B<T>)
        const std::string& ret = getNestedTypePlaceholder(*this, type.getTypePtr());

        if (!ret.empty())
        {
            return std::make_pair(ret, nullptr);
        }
    }
#endif

    const clang::TypeDecl* typeDeclOut = nullptr;
    std::string name;
    llvm::raw_string_ostream o(name);
    if(auto typedefType = clang::dyn_cast<clang::TypedefType>(type))
    {
        typeDeclOut = typedefType->getDecl();

        std::string tdefName = concatWithColon(getScopeName(typeDeclOut, nt), typeDeclOut->getNameAsString());

        if (nt == NAME_REFLECTION)
        {
            if (isReflected(typeDeclOut))
            {
                // replace with a placeholder
                const std::string& tagname = getAttributeValue<std::string>(typeDeclOut, "hk::Type", "");
                if( tagname.size() )
                {
                    tdefName = tagname;
                }
                else
                {
                    hkLog.error(typeDeclOut, "The typedef %s is marked for reflection with an hk::Reflect attribute, but instead it should be reflected with HK_REFLECT_TYPEDEF.", typeDeclOut->getNameAsString().c_str());
                }
            }
            else if (auto customAttr = getAttributeValue< Optional<std::string> >(typeDeclOut, "hk::ReflectAsTypedef"))
            {
                tdefName = *customAttr;
            }
            else if (shouldBeExpanded(typedefType->getDecl()->getUnderlyingType()))
            {
                // The typedefs in the name must be replaced by their tags.
                tdefName = getScopedMangledName(typedefType->getDecl()->getUnderlyingType(), NAME_REFLECTION);
            }
        }

        o << tdefName;
    }
    else if(auto enumType = clang::dyn_cast<clang::EnumType>(type))
    {
        typeDeclOut = enumType->getDecl();
        o << concatWithColon(getScopeName(typeDeclOut, nt), typeDeclOut->getName());
    }
    else if(auto ptrType = clang::dyn_cast<clang::PointerType>(type))
    {
        o << getScopedMangledName(ptrType->getPointeeType(), nt) << "*";
    }
    else if(auto refType = clang::dyn_cast<clang::ReferenceType>(type))
    {
        o << getScopedMangledName(refType->getPointeeType(), nt) << "&";
    }
    else if(auto arrType = clang::dyn_cast<clang::ConstantArrayType>(type))
    {
        o << getScopedMangledName(arrType->getElementType(), nt);
        o << formatString("[%d]", arrType->getSize().getZExtValue());
    }
    else if(auto tsType = clang::dyn_cast<clang::TemplateSpecializationType>(type))
    {
        // We print arguments directly instead of using the same path as ClassTemplateSpecializationDecl
        // because that seems to lose information about typedef template arguments.

        typeDeclOut = getCXXRecordFrom(type);

        std::string specName;
        if (nt != NAME_DECLARATION && tsType->isDependentType() && clang::isa<clang::TypeDecl>(typeDeclOut->getDeclContext()))
        {
            specName = "template ";
        }
        specName += typeDeclOut->getName();
        o << concatWithColon(getScopeName(typeDeclOut, nt), specName);

        // print arguments
        o << "< ";
        printSpecializationArguments(*this, getArgs(tsType), o, nullptr, nt);
        o << " >";
    }
    else if(auto t = clang::dyn_cast<clang::TemplateTypeParmType>(type))
    {
        llvm::StringRef paramName;
        if (const clang::IdentifierInfo* id = t->getIdentifier())
        {
            paramName = id->getName();
        }
        else if(const clang::TemplateTypeParmDecl* d = t->getDecl())
        {
            paramName = d->getName();
        }
        if (!paramName.empty())
        {
            if (nt == NAME_RESOLVE_TAGS)
            {
                o << "HK_REFLECT_RESOLVE(" << paramName << ")";
            }
            else
            {
                o << paramName;
            }
        }
    }
    else if(auto dType = clang::dyn_cast<clang::DependentNameType>(type))
    {
        clang::NestedNameSpecifier* scope = dType->getQualifier();
        const clang::IdentifierInfo* id = dType->getIdentifier();
        scope->print(o, policy);
        // todo.rt if the dependent name is a reflected typedef the typedef info is lost here
        o << id->getName();
    }
    else if(auto dtType = clang::dyn_cast<clang::DependentTemplateSpecializationType>(type))
    {
        clang::NestedNameSpecifier* scope = dtType->getQualifier();
        const clang::IdentifierInfo* id = dtType->getIdentifier();
        scope->print(o, policy);
        o << "template " << id->getName();
        // print arguments
        o << "< ";
        printSpecializationArguments(*this, getArgs(dtType), o, nullptr, nt);
        o << " >";
    }
    else if(auto daType = clang::dyn_cast<clang::DependentSizedArrayType>(type))
    {
        o << getScopedMangledName(daType->getElementType(), nt);
        o << "[";
        {
            const clang::Expr* size = daType->getSizeExpr();
            if (auto declRef = clang::dyn_cast<clang::DeclRefExpr>(size))
            {
                std::string declName = getScopedMangledName(declRef->getDecl(), nt);
                if (stringStartsWith(declName, "typename "))
                {
                    declName = declName.substr(9);
                }
                o << declName;
            }
            else
            {
                size->printPretty(o, 0, policy);
            }
        }
        o << "]";
    }
    else if (const clang::CXXRecordDecl* recDecl = getCXXRecordFrom(type))
    {
        typeDeclOut = recDecl;

        // might be a ClassTemplateSpecializationDecl even if the type is not a TemplateSpecializationType
        std::string declName;
        if (nt != NAME_DECLARATION && getTemplateDecl(typeDeclOut) && type->isDependentType() && clang::isa<clang::TypeDecl>(typeDeclOut->getDeclContext()))
        {
            declName = "template ";
        }
        declName += getMangledName(typeDeclOut, nt);
        o << concatWithColon(getScopeName(typeDeclOut, nt), declName);
    }
    else
    {
        // fallback
        type.removeLocalConst();
        o << type.getAsString(policy);
    }
    o.flush();
    return std::make_pair(name, typeDeclOut);
}

std::string Database::getScopedMangledName(clang::QualType type, NameType nt /* = NAME_NORMAL */) const
{
    std::string name;
    const clang::TypeDecl* typeDecl;
    std::tie(name, typeDecl) = getScopedMangledNameInternal(type, nt);

    // add typename if the type is dependent
    if (nt != NAME_DECLARATION &&
        (clang::isa<clang::DependentNameType>(type) ||
        clang::isa<clang::DependentTemplateSpecializationType>(type) ||
        (typeDecl && isDependentTypeDecl(typeDecl) && clang::isa<clang::CXXRecordDecl>(typeDecl->getDeclContext())) ) &&
        !stringStartsWith(name, "typename "))
    {
        name = "typename " + name;
    }

    name = type.isConstQualified() ?
        (type->isPointerType() ? (name + " const") : ("const " + name))
        : name;

    return name;
}

bool Database::isDependentTypeDecl(const clang::TypeDecl* decl) const
{
    if (const clang::TypeDecl* context = clang::dyn_cast<clang::TypeDecl>(decl->getDeclContext()))
    {
        clang::QualType type = astContext.getTypeDeclType(context);
        return type->isDependentType();
    }
    return false;
}

std::string Database::getSourceFileName(const clang::Decl* decl)
{
    const clang::ASTContext& astCtx = decl->getASTContext();

    clang::FullSourceLoc fullLoc = astCtx.getFullLoc(decl->getLocation());

    if(fullLoc.isMacroID())
        fullLoc = astCtx.getFullLoc(astCtx.getSourceManager().getExpansionLoc(fullLoc));

    if (fullLoc.isValid())
    {
        if(const clang::FileEntry* fentry = astCtx.getSourceManager().getFileEntryForID(fullLoc.getFileID()))
        {
            return Filesystem::absoluteCanonicalPath(fentry->getName());
        }
    }

    return std::string();
}

bool Database::isOpaqueTypeAllowed(const clang::FieldDecl* field) const
{
    return getAttributeValue<bool>(field, "hk::OpaqueType");
}

bool Database::isSerializable(const clang::Decl* field) const
{
    return getAttributeValue<bool>(field, "hk::Serialize", true);
}

const AttributeDecl* Database::getRuntimeAttributeDecl(const std::string& name) const
{
    auto cit = runtimeAttributeDecl.find(name);
    if(cit != runtimeAttributeDecl.end())
        return &cit->second;
    else
        return 0;
}

const AttributeDecl* Database::getStaticAttributeDecl(const std::string& name) const
{
    auto cit = staticAttributeDecl.find(name);
    if(cit != staticAttributeDecl.end())
        return &cit->second;
    else
        return 0;
}

const AttributeDecl* Database::getAttributeDecl(const std::string& name) const
{
    // todo.ntm this should just check if the name identifies a valid type
    if (const AttributeDecl* decl = getStaticAttributeDecl(name))
    {
        return decl;
    }
    return getRuntimeAttributeDecl(name);
}

const clang::CXXRecordDecl* Database::getMessageTypeDecl(const std::string& name) const
{
    std::map<std::string, const clang::CXXRecordDecl*>::const_iterator cit = messageTypeDecl.find(name);
    if(cit != messageTypeDecl.end())
        return cit->second;
    else
        return 0;
}

const clang::NamedDecl* Database::findDecl(const std::string& name) const
{
    const std::vector<std::string>& nameParts = splitString(name, ':');

    const auto endLoc = astContext.getTranslationUnitDecl()->getLocEnd();
    clang::DeclContext* ctx = astContext.getTranslationUnitDecl();
    const clang::NamedDecl* decl = 0;

    for(auto it = nameParts.begin(); ctx && it != nameParts.end(); ++it)
    {
        const std::string& n = *it;
        if(n.empty())
            continue;

        clang::IdentifierInfo* identInfo = &astContext.Idents.get(n);
        clang::DeclarationName&& declName = astContext.DeclarationNames.getIdentifier(identInfo);

        clang::LookupResult lookupResult(sema, declName, endLoc, clang::Sema::LookupAnyName);

        if(sema.LookupQualifiedName(lookupResult, ctx))
        {
            decl = lookupResult.getAsSingle<clang::NamedDecl>();
            ctx = lookupResult.getAsSingle<clang::DeclContext>();
        }
        else
        {
            return 0;
        }
    }

    
//  if(ctx == 0)
//  {
//      return 0;
//  }
//  else
//  {
    return decl;
/*  }*/
}
const clang::CXXRecordDecl* Database::findRecordDefinition(const std::string& name) const
{
    if(const clang::NamedDecl* namedDecl = findDecl(name))
    {
        if(const clang::CXXRecordDecl* recDecl = clang::dyn_cast<clang::CXXRecordDecl>(namedDecl))
        {
            return recDecl->getDefinition();
        }
    }

    return 0;
}


int Database::getOptionValue(clang::SourceLocation srcLoc, const std::string& option) const
{
    std::string cleanOption = option;

    const char ns1[] = "hkReflect::";
    if(stringStartsWith(cleanOption, ns1))
        cleanOption = cleanOption.substr(strlen(ns1), std::string::npos);

    const char ns2[] = "Opt::";
    if (stringStartsWith(cleanOption, ns2))
        cleanOption = cleanOption.substr(strlen(ns2), std::string::npos);

    std::map<std::string, int>::const_iterator cit = optionEnumValues.find(cleanOption);

    if(cit == optionEnumValues.end())
        throw RuntimeError(astContext.getSourceManager(), srcLoc,
        "Optional '%s' was not found in hkReflect::Opt::Values enum.", option.c_str());

    return cit->second;
}

void Database::addAttribute(clang::Decl* decl, const std::string& hkAttrString, clang::SourceRange attrRange)
{
    std::string annotation = Database::attributePrefix;
    annotation.append(hkAttrString);

    clang::ASTContext& context = decl->getASTContext();
    if (attrRange.isInvalid())
    {
        attrRange = decl->getSourceRange();
    }

    if (decl->hasAttrs())
    {
        const clang::AttrVec& attrVec = decl->getAttrs();
        typedef clang::specific_attr_iterator<clang::AnnotateAttr> SpecificIterator;
        for (SpecificIterator iterator = clang::specific_attr_begin<clang::AnnotateAttr>(attrVec), end_iterator = clang::specific_attr_end<clang::AnnotateAttr>(attrVec);
            iterator != end_iterator;
            ++iterator)
        {
            if (iterator->getAnnotation() == annotation) // Do we already have an identical attribute?
            {
                return;
            }
        }
    }

    decl->addAttr(::new(context) clang::AnnotateAttr(attrRange, context, annotation));
}

void Database::replaceAttribute(clang::Decl* decl, const clang::AnnotateAttr* orig, const std::string& hkAttrString, clang::SourceRange attrRange)
{
    std::string annotation = Database::attributePrefix;
    annotation.append(hkAttrString);

    clang::ASTContext& context = decl->getASTContext();
    if (attrRange.isInvalid())
    {
        attrRange = decl->getSourceRange();
    }

    auto& vec = decl->getAttrs();
    auto it = std::find(vec.begin(), vec.end(), orig);
    assert(it != vec.end());

    auto newAttr = ::new(context) clang::AnnotateAttr(attrRange, context, annotation);
    *it = newAttr;
}

Attribute Database::getReflectDetails(const clang::Decl* decl) const
{
    auto explicitAttr = getAttributeValue< Optional<Attribute> >(decl, "hk::ReflectDetails");
    return explicitAttr ? *explicitAttr : Attribute(*this, "hk::ReflectDetails()", clang::SourceLocation());
}

bool Database::shouldBeExpanded(clang::QualType type) const
{
    // Check if the name underlying type actually needs to be expanded because it can contain typedef names.
    // Not really necessary but we gain readability by printing the original typedef name rather than the expansion.
    type = resolveElaboratedType(type);
    if (auto nestedTdef = clang::dyn_cast<clang::TypedefType>(type))
    {
        return isReflected(nestedTdef->getDecl()) || shouldBeExpanded(nestedTdef->getDecl()->getUnderlyingType());
    }
    else if (auto spec = clang::dyn_cast<clang::TemplateSpecializationType>(type))
    {
        if (isReflected(spec))
        {
            for (unsigned i = 0; i < spec->getNumArgs(); ++i)
            {
                if (spec->getArg(i).getKind() == clang::TemplateArgument::Type &&
                    shouldBeExpanded(spec->getArg(i).getAsType()))
                {
                    return true;
                }
            }
        }
    }
    return false;
}

const char* Database::allowedOldAttributes[] = {"hk.", "dotnet::"};

bool Database::isOldStyleAttribute(const std::string& name)
{
    for(unsigned i=0; i < (sizeof(allowedOldAttributes) / sizeof(allowedOldAttributes[0])); i++)
    {
        if(name.find(allowedOldAttributes[i]) == 0)
        {
            return true;
        }
    }
    return false;
}

bool Database::hasOwnVtable(const clang::CXXRecordDecl* recDecl) const
{
    if(isTemplateRecursive(recDecl) == false) 
    {
        const clang::ASTRecordLayout& layout = astContext.getASTRecordLayout(recDecl);
        if(layout.hasVBPtr())
        {
            hkLog.error(recDecl, "Reflected classes cannot use virtual inheritance.", getScopedMangledName(recDecl).c_str());
        }
        if(layout.hasOwnVFPtr())
        {
            if(!recDecl->field_empty())
            {
                hkLog.error(recDecl, "Reflected classes which add a vtable may not have fields. This is to ensure sensible layout.", getScopedMangledName(recDecl).c_str());
            }
            return true;
        }
    }

    return false;
}


bool Database::hasTrackerHandle(const clang::TypeDecl* decl) const
{
    if (auto recDecl = clang::dyn_cast<clang::CXXRecordDecl>(decl))
    {
        // Look for the trackerHandle declaration.
        auto handle = findMemberDecl<clang::VarDecl>(recDecl, "trackerHook");
        if (handle)
        {
            // Templates are tracked only if marked with "NewTemplate", which declares trackerHook as void*.
            bool refersToTemplate = hasTemplateParamsRecursive(recDecl);

            if (!refersToTemplate)
            {
                if (auto specDecl = clang::dyn_cast<clang::ClassTemplateSpecializationDecl>(decl))
                {
                    if (!specDecl->isExplicitSpecialization())
                    {
                        refersToTemplate = true;
                    }
                }
            }
            if (refersToTemplate)
            {
                return handle->getType()->isPointerType();
            }
            else
            {
                if (handle->getType()->isPointerType())
                {
                    hkLog.error(handle, "NewTemplate used on a non-template type (full specializations should use New)");
                }
                return true;
            }
        }

    }
    return false;
}

bool Database::trackContent(const clang::Decl* decl) const
{
    if (auto trackerAttr = getAttributeValue< Optional<Attribute> >(decl, "hk::MemoryTracker"))
    {
        return !(*trackerAttr)["opaque"]->as<bool>();
    }
    return true;
}

std::string Database::getReflectedTypeAddress(const clang::QualType& type, bool tracker) const
{
    if (tracker)
    {
        // Special-case here rather than using the generic trait to speedup compilation.
        if (type->isPointerType())
        {
            // Flatten pointers, we don't really need this information so we can skip a few sfinae checks.
            return "&hkReflect::ReflectionOf<void*>::typeData";
        }
        else if (auto recDecl = getCXXRecordFrom(type))
        {
            // Must be tracked or execution will not get here
            assert(hasTrackerHandle(recDecl));
            return formatString("HK_TRACKER_GET_HANDLE_UNCHECKED(typename %s::_OldMacroHack::_TrackerType)",
                getScopedMangledName(type.getUnqualifiedType()).c_str());
        }
    }

    return getReflectedTypeAddress(getScopedMangledName(type, Database::NAME_REFLECTION), tracker);
}

std::string Database::getReflectedTypeAddress(const clang::NamedDecl* decl, bool tracker) const
{
    if (tracker)
    {
        if (auto recDecl = clang::dyn_cast<clang::CXXRecordDecl>(decl))
        {
            // Must be tracked or execution will not get here
            assert(hasTrackerHandle(recDecl));
            return formatString("HK_TRACKER_GET_HANDLE_UNCHECKED(typename %s::_OldMacroHack::_TrackerType)", getScopedMangledName(decl).c_str());
        }
    }
    return getReflectedTypeAddress(getScopedMangledName(decl, Database::NAME_REFLECTION), tracker);
}

namespace
{
    /// A string which convinces Intellisense and VisualAssist to ignore file contents below it, so that
    /// auto-generated definitions are not used for source code navigation/completion.
    const std::string exclusionPreamble = R"incantation(
#if 0
R\
"exclude1(";
R"exclude2()exclude1";
#endif
)incantation";

    /// A string which ends the exclusion begun by makeExclusionPreamble.
    const std::string exclusionPostamble = R"incantation(
#if 0
//)exclude2";
#endif
)incantation";
}


std::string Database::getPreamble(const std::string& outputName, const std::string& headerName) const
{
    std::string res;
    llvm::raw_string_ostream str(res);

    // Get file extension.
    const std::string& ext = outputName.substr(outputName.rfind('.'));
    if (ext.empty())
        throw FatalError("Output file without extension: " + outputName);

    if (m_outputTkbms)
    {
        // Add TKBMS
        if (headerName.length())
        {
            // Header TKBMS
            TkbmsRegistry::Entry entry;
            if (tkbmsRegistry.tryGetTkbmsEntry(headerName, entry))
            {
                str << entry.fullTkbms;
            }
        }
        else
        {
            // Project TKBMS
            bool isHeader = ext == ".h" || ext == ".hxx";
            str << tkbms(tkbmsRegistry.getProjectProduct(), isHeader ? "ALL !REFLECT" : "ALL");
        }

        // Intellisense/VisualAssist exclusion
        str << exclusionPreamble;
    }

    // Also add any custom preamble
    if (preamble.length())
    {
        // Custom preamble
        str << preamble;
    }

    str << "\n";

    if (ext == ".cpp")
    {
        if (preambleCpp.length())
        {
            str << preambleCpp << "\n";
        }

        // Add forced and default includes to cpps.
        str << forceIncludesCode << "\n";
    }
    str.flush();
    return res;
}

std::string Database::getPostamble(const std::string& outputName) const
{
    std::string res;
    llvm::raw_string_ostream str(res);

    // Intellisense/VisualAssist exclusion
    str << exclusionPostamble;

    if (stringEndsWith(outputName, ".cpp") && postambleCpp.length())
    {
        str << "\n" << postambleCpp << "\n";
    }
    if (postamble.length())
    {
        str << "\n" << postamble << "\n";
    }
    str.flush();
    return res;
}

std::string Database::getMangledMethodName( const clang::CXXMethodDecl* method ) const
{
    std::string newName;
    {
        llvm::raw_string_ostream str( newName );

        // Name.
        if ( clang::dyn_cast<clang::CXXConstructorDecl>(method) )
        {
            str << "new";
        }
        else
        {
            method->printName( str );
        }

        // Args.
        if ( method->param_size() == 0 )
        {
            str << "__void";
        }
        else
        {
            str << "_";
            for ( clang::CXXMethodDecl::param_const_iterator it = method->param_begin(); it != method->param_end(); ++it )
            {
                str << "_" << pyNameFromCxxName( getScopedMangledName( (*it)->getType(), Database::NAME_DECLARATION ) );
            }
        }

        // Return type.
        if ( !clang::isa<clang::CXXConstructorDecl>( method ) )
        {
            str << "__" << pyNameFromCxxName( getScopedMangledName( method->getResultType(), Database::NAME_DECLARATION ) );
        }

        // Constness.
        if ( method->isConst() )
        {
            str << "__const";
        }
    }
    return newName;
}

const char* const Database::attributePrefix = "hk_attr.";

const int Database::attributePrefixLength = 8;

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
