// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
#include "Attribute.h"
#include "utils/StlUtils.h"
#include "utils/RuntimeError.h"
#include "utils/Database.h"
#include "utils/ElementUtils.h"

namespace
{
    hkAttrData::FieldDecl parseField(const Database& db,
        const clang::CXXRecordDecl* attrDecl,
        const clang::FieldDecl* currentField,
        const std::string& fieldName,
        hkAttrData::FieldDecl::Flags flags)
    {
        clang::QualType fieldType = resolveType(currentField->getType());

        if (const clang::CXXRecordDecl* recDecl = getCXXRecordFrom(fieldType))
        {
            if (recDecl->getQualifiedNameAsString() == "hk::ValueArray")
            {
                return hkAttrData::FieldDecl::variableArray(fieldName.c_str(), flags);
            }
            else
            {
                checkIsDefinition(recDecl);

                // If the record has just one field, use that field.
                clang::CXXRecordDecl::field_iterator fit = recDecl->field_begin();
                if (fit == recDecl->field_end())
                {
                    throw RuntimeError(attrDecl, "Field '%s' of attribute '%s' must be a record containing just one field.",
                        fieldName.c_str(), db.getScopedMangledName(attrDecl).c_str());
                }
                const clang::FieldDecl* firstField = *fit;
                if (++fit != recDecl->field_end())
                {
                    throw RuntimeError(attrDecl, "Field '%s' of attribute '%s' must be a record containing just one field.",
                        fieldName.c_str(), db.getScopedMangledName(attrDecl).c_str());
                }
                return parseField(db, attrDecl, firstField, fieldName, flags);
            }
        }
        else if (const clang::ConstantArrayType* arrayType = clang::dyn_cast<clang::ConstantArrayType>(fieldType))
        {
            return hkAttrData::FieldDecl::fixedArray(fieldName.c_str(),
                static_cast<int>(arrayType->getSize().getZExtValue()), flags);
        }
        else
        {
            return hkAttrData::FieldDecl::simple(fieldName.c_str(), flags);
        }
    }
}

void AttributeDecl::parse(const clang::CXXRecordDecl* currentDecl) const
{
    // hack to avoid infinite recursion (hk::Optional has an optional field)
    if (m_recDecl->getQualifiedNameAsString() == "hk::Optional")
    {
        m_fields.push_back(Field(hkAttrData::FieldDecl::simple("optional", hkAttrData::FieldDecl::FLAG_OPTIONAL), std::vector<std::string>(1, "true")));
        m_fields.push_back(Field(hkAttrData::FieldDecl::simple("implicit", hkAttrData::FieldDecl::FLAG_OPTIONAL | hkAttrData::FieldDecl::FLAG_EXPLICIT), std::vector<std::string>(1, "false")));
        return;
    }

    if (currentDecl->bases_begin() != currentDecl->bases_end())
    {
        parse(currentDecl->bases_begin()->getType()->getAsCXXRecordDecl());
    }

    const std::string& fieldPrefix = m_db.getAttributeValue<std::string>(currentDecl, "hk::FieldPrefix");
    for (clang::CXXRecordDecl::field_iterator fit = currentDecl->field_begin(); fit != currentDecl->field_end(); ++fit)
    {
        m_fieldNames.push_back(fieldNameWithoutPrefix(fieldPrefix, fit->getNameAsString()));
        std::vector<std::string> defaultVal;

        // check if the field is optional
        hkAttrData::FieldDecl::Flags flags = hkAttrData::FieldDecl::FLAG_NONE;
        if (Optional<Attribute> optionalAttr = m_db.getAttributeValue< Optional<Attribute> >(*fit, "hk::Optional"))
        {
            if ((*optionalAttr)["optional"]->as<bool>())
            {
                flags.orWith(hkAttrData::FieldDecl::FLAG_OPTIONAL);
            }
            if (!(*optionalAttr)["implicit"] || !(*optionalAttr)["implicit"]->as<bool>())
            {
                flags.orWith(hkAttrData::FieldDecl::FLAG_EXPLICIT);
            }

            // gets the default values
            if (Optional<Attribute> defaultAttr = m_db.getAttributeValue< Optional<Attribute> >(*fit, "hk::Default"))
            {
                for (Attribute::PosValuesIterator vit = defaultAttr->posBegin(); vit != defaultAttr->posEnd(); ++vit)
                {
                    defaultVal.push_back(*vit);
                }
            }
        }

        m_fields.push_back(Field(parseField(m_db, m_recDecl, *fit, m_fieldNames.back(), flags), defaultVal));
    }
}

const std::vector<AttributeDecl::Field>& AttributeDecl::getFields() const
{
    // Parsing is done lazily because it needs all the attributes to be already in the Database.
    if (!m_parsed)
    {
        const clang::CXXRecordDecl* parsingDecl = m_recDecl;

        // char* in the decl fields will point to strings in this vector, must prevent reallocations.
        unsigned maxFieldNum = 32;
        m_fieldNames.reserve(maxFieldNum);

        // check if the type overrides the attribute declaration
        Optional<Attribute> parsingDeclAttr = m_db.getAttributeValue< Optional<Attribute> >(m_recDecl, "hk::ParsingDecl");
        if (parsingDeclAttr)
        {
            parsingDecl = m_db.findRecordDefinition(parsingDeclAttr->get("decl"));
            if (parsingDecl == 0)
                throw RuntimeError(m_db.astContext.getSourceManager(), parsingDeclAttr->getLocation(),
                "Could not find a C++ declaration for the attribute '%s'", parsingDeclAttr->get("decl").c_str());
        }

        parse(parsingDecl);

        if (m_fieldNames.size() > maxFieldNum)
        {
            throw RuntimeError(m_db.astContext.getSourceManager(), m_recDecl->getLocation(), "Attribute has more than the maximum supported number of fields (%d)", maxFieldNum);
        }

        m_parsed = true;
    }
    return m_fields;
}

Attribute::Attribute(const Database& db, const clang::AnnotateAttr* attrVal)
    : m_db(&db), m_srcMgr(&db.astContext.getSourceManager()), m_location(attrVal->getLocation()), m_fullAttr(attrVal->getAnnotation()), m_hasValidation(false)
{
    // remove the attribute prefix.
    if(m_fullAttr.find(Database::attributePrefix) == std::string::npos)
    {
        throw RuntimeError(getSrcMgr(), m_location, "Trying to get the value of a non-Havok attribute.");
    }
    m_fullAttr = m_fullAttr.substr(Database::attributePrefixLength, std::string::npos);
    parse();
}

Attribute::Attribute(const Database& db, const std::string& attrString, clang::SourceLocation location)
    : m_db(&db), m_srcMgr(&db.astContext.getSourceManager()), m_location(location), m_fullAttr(attrString), m_hasValidation(false)
{
    parse();
}

void Attribute::parse()
{
    assert(m_db != nullptr);

    hkAttributeParser parser(m_fullAttr.c_str());

    try
    {
        if (!parser.advance())
            throw RuntimeError(getSrcMgr(), m_location, "No valid attributes found");

        hkStringBuf nameBuf;
        if (!parser.currentAttrName(nameBuf).isSuccess())
            throw RuntimeError(getSrcMgr(), m_location, nameBuf.cString());

        m_attrName = nameBuf;

        // todo.ntm Silently ignore some old-style attributes (hk.*, dotnet.*, ...). Should be removed eventually.
        // Run the clang parser with -v=4 to trigger debug prints for those attributes.
        if(Database::isOldStyleAttribute(m_attrName))
        {
            return;
        }

        // Look the attribute declaration up.
        const AttributeDecl* attrDecl = m_db->getAttributeDecl(m_attrName);

        if(attrDecl == 0)
            throw RuntimeError(getSrcMgr(), m_location, "Could not find a C++ declaration for the attribute '%s'", m_attrName.c_str());

        if (attrDecl->getFields().empty())
        {
            // No further parsing necessary.
            return;
        }

        if (m_db->getAttributeValue(attrDecl->getRec(), "hk::GenerateValidationForAttribute", false))
        {
            m_hasValidation = true;
        }

        // Parse the arguments.
        hkAttrData attrData;
        for (auto field : attrDecl->getFields())
        {
            attrData.addDecl(field.m_decl);
        }

        hkStringBuf errorBuf;
        if (!parser.currentAttrValue(attrData, errorBuf).isSuccess())
            throw RuntimeError(getSrcMgr(), m_location, errorBuf.cString());

        for (int fieldIdx = 0; fieldIdx < attrData.getNumFields(); ++fieldIdx)
        {
            hkAttrData::Field field = attrData.getField(fieldIdx);
            const char* fieldName = field.getDecl().getName();

            if(field.isSet())
            {
                if (field.getDecl().isSimple())
                {
                    m_namedValues[fieldName] = std::string(field.getValue().begin(), field.getValue().getSize());
                }
                else // array
                {
                    hkArrayView<const hkStringView> elements = field.getValuesArray();// fill the attribute with the array elements
                    for (int arrayIdx = 0; arrayIdx < elements.getSize(); ++arrayIdx)
                    {
                        m_arrayValues.push_back(std::string(elements[arrayIdx].begin(), elements[arrayIdx].getSize()));
                    }
                }
            }
            else
            {
                const std::vector<std::string>& defaultVal = attrDecl->getFields()[fieldIdx].m_default;
                if (defaultVal.size())
                {
                    if (field.getDecl().isSimple())
                    {
                        if (defaultVal.size() != 1)
                        {
                            throw RuntimeError(getSrcMgr(), m_location,
                                "Field '%s' of attribute '%s' is simple but its default is an array",
                                fieldName, m_attrName.c_str());
                        }
                        m_namedValues[fieldName] = defaultVal[0];
                    }
                    else // array
                    {
                        if (field.getDecl().isFixedArray() && (int)defaultVal.size() != field.getDecl().getFixedSize())
                        {
                            throw RuntimeError(getSrcMgr(), m_location,
                                "Field '%s' of attribute '%s' has length %d but its default has length %d",
                                fieldName, m_attrName.c_str(), field.getDecl().getFixedSize(), defaultVal.size());
                        }
                        m_arrayValues = defaultVal;
                    }
                }
            }
        }

        // todo.ntm add support for multiple attributes
        if (parser.advance())
            throw RuntimeError(getSrcMgr(), m_location,
                "Multiple attributes in a single annotation are not supported");
    }
    catch (const RuntimeError& e)
    {
        throw RuntimeError(e.getLine(), e.getFileName(),
            "%s while evaluating attribute '%s'", e.getMessage(), m_fullAttr.c_str());
    }
}

Optional<AttributeValue> Attribute::operator[](const std::string& name) const
{
    std::map<std::string, std::string>::const_iterator cit = m_namedValues.find(name);

    if(cit != m_namedValues.end())
        return Optional<AttributeValue>(AttributeValue(this, cit->second));

    return Optional<AttributeValue>();
}

std::string Attribute::get(const std::string& name) const
{
    return get(name, std::string());
}

std::string Attribute::get(const std::string& name, const std::string& ifNotFound) const
{
    if(const Optional<AttributeValue>& opt = (*this)[name])
        return *opt;

    return ifNotFound;
}

std::string Attribute::get(unsigned int i) const
{
    if(i < m_arrayValues.size())
        return m_arrayValues[i];

    return std::string();
}

std::string Attribute::getPosValuesString(bool forceBraces) const
{
    if (m_arrayValues.size() == 1 && !forceBraces)
    {
        return m_arrayValues[0];
    }

    std::string res = "{ ";
    for (unsigned i = 0; i < m_arrayValues.size(); ++i)
    {
        res += m_arrayValues[i];
        if (i < m_arrayValues.size() - 1)
        {
            res += ", ";
        }
    }
    return res + " }";
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
