// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
PRAGMA_WARNING_PUSH
# include <sstream>
# include <ctime>
# include <clang/AST/DeclVisitor.h>
# include <llvm/Support/ManagedStatic.h>
# include <llvm/Support/system_error.h>
# include <llvm/Support/FileSystem.h>
PRAGMA_WARNING_POP

#include "Generators.h"
#include "utils/Database.h"
#include "utils/Filesystem.h"
#include "utils/Logger.h"
#include "utils/TextBuilder.h"
#include "utils/StlUtils.h"
#include "utils/ElementUtils.h"
#include "utils/TkbmsRegistry.h"
#include "utils/RuntimeError.h"
#include "utils/Optional.h"
#include "utils/TypeOut.h"
#include "AttributeRegistry.h"

#define FUNC_REFL_INCLUDE "Common/Base/Reflect/Core/Detail/hkCallableDetail.h"

namespace
{
    typedef std::pair<std::string, std::string> AttributeVariant;

    TextBuilder cxxTextBuilder(const Database& db)
    {
        TextBuilder t;
        t.addSection("errors", new TextBuilder::VerbatimSection());
        t.addSection("warnings", new TextBuilder::VerbatimSection());
        t.addSection("include", new TextBuilder::IncludeSection(db.includePaths));
        t.addSection("auto", new TextBuilder::CodeSection());
        t.addSection("verbatim", new TextBuilder::VerbatimSection());
        t.addSection("extern", new TextBuilder::DeclarationSection());
        t.addSection("declare", new TextBuilder::GuardedSection("HK_DEFINED_"));
        t.addSection("append", new TextBuilder::CodeSection());
        return t;
    }

    void generateBypassConstructor(const Database& db, const clang::CXXRecordDecl* recDecl, const std::string& cxxName, TextBuilder::Section& section)
    {
        if(getTemplateDecl(recDecl))
        {
            hkLog.error(recDecl, "%s is a template and its bypass constructor must be implemented manually.", recDecl->getNameAsString().c_str());
            return;
        }

        std::vector<std::string> toConstruct;

        clang::PrintingPolicy policy(db.astContext.getLangOpts());
        policy.SuppressTagKeyword = 1;
        policy.Bool = 1;

        // We only reflect construct the firs base since we assume the other ones are abstract interfaces for which
        // the default ctor can be used as bypass ctor.
        if(recDecl->getNumBases() > 0 && hasBypassConstructor(recDecl->bases_begin()->getType()))
        {
            toConstruct.push_back(recDecl->bases_begin()->getType().getAsString(policy) + "(f)");
        }

        // Forward reflect flag to fields
        for(clang::CXXRecordDecl::field_iterator cit = recDecl->field_begin(); cit != recDecl->field_end(); ++cit)
        {
            if( hasBypassConstructor(cit->getType()) &&  !db.getAttributeValue<bool>(*cit, "hk::OpaqueType", false) )
            {
                toConstruct.push_back(cit->getNameAsString() + "(f)");
            }
            else if( cit->getType().isConstQualified() && cit->getType()->isPointerType() ) // const pointers must be initialized also to avoid warnings
            {
                toConstruct.push_back(cit->getNameAsString() + "(HK_NULL)");
            }
        }

        section.append("#if !defined(HK_REFLECT_TYPE_DATA_ONLY)");
        section.append("%s::%s(hkReflect::BypassCtorFlag f)", cxxName.c_str(), recDecl->getNameAsString().c_str());

        if(!toConstruct.empty())
        {
            std::ostringstream initList;
            initList << ": " << toConstruct.front();
            for(std::vector<std::string>::const_iterator cit = toConstruct.begin() + 1; cit != toConstruct.end(); ++cit)
                initList << ", " << *cit;
            section.append(initList.str().c_str());
        }

        section.append("{\n}");
        section.append("#endif");
    }

    int generateAttributes(const Database& db,
                            const clang::NamedDecl* decl,
                            TypeOut& typeOut,
                            TypeOut& recordTypeOut,
                            const std::vector< AttributeVariant >& extraAttrs = std::vector< AttributeVariant >(),
                            std::vector< std::pair<std::string, std::string> >* attributesWithValidation = nullptr)
    {
        std::vector<Attribute> hkAttributes = db.getAllRuntimeAttributes(decl);

        const Optional<Attribute>& addAttributeAttr = db.getAttributeValue< Optional<Attribute> >(decl, "hk::AddAttribute");

        int nbAttrs = 0;

        // we do this in one pass
        // output the VALUES directly (e.g. const int Foo = 10;)
        // Buffer the attribute VARIANTS in attrpairs which gets printed after all VALUES at the end of this block
        TextBuilder::VerbatimSection attrpairs;
        const char* attrSeparator = "";

        // If the attributes are being written into a different type's typeOut, assume they need a scope.
        std::string localScope;
        std::string attrprefix;
        if (&typeOut != &recordTypeOut)
        {
            localScope = pyNameFromCxxName(decl->getNameAsString());
            attrprefix = localScope + "_";
        }

        for(std::vector<Attribute>::const_iterator cit = hkAttributes.begin(); cit != hkAttributes.end(); ++cit)
        {
            AttributeRegistry::AttributeFunction attrFun = AttributeRegistry::getAttribute(cit->getAttrName());
            assert(attrFun); // We should always have an attribute handler, either the attribute's one or the generic one.

            try
            {
                std::string attrSym, attrType, attrStruct;
                std::tie(attrSym, attrType, attrStruct) = (*attrFun)(*cit, db, decl, recordTypeOut, attrprefix);

                attrpairs.append("%s{ &%s, HK_REFLECT_GET_TYPE( %s ) }", attrSeparator, attrSym.c_str(), attrType.c_str());
                attrSeparator = ", ";
                ++nbAttrs;

                if (attributesWithValidation && cit->hasValidation())
                {
                    attributesWithValidation->push_back(std::make_pair(attrStruct, attrType));
                }

            }
            catch (RuntimeError e)
            {
                hkLog.error("%s", e.what());
            }
        }

        for (auto it = extraAttrs.begin(); it != extraAttrs.end(); ++it)
        {
            attrpairs.append("%s{ %s, HK_REFLECT_GET_TYPE( %s ) }", attrSeparator, it->first.c_str(), it->second.c_str());
            ++nbAttrs;
        }

        if(addAttributeAttr)
        {
            const std::string& attrName = addAttributeAttr->get("attrName");
            const std::string& attrVal = addAttributeAttr->get("attrVal");
            attrpairs.append("%s{ %s, HK_REFLECT_GET_TYPE( %s ) }", attrSeparator, attrVal.c_str(), attrName.c_str());
            ++nbAttrs;
        }

        if (nbAttrs)
        {
            std::string attrpairsStr; attrpairs.format(attrpairsStr);
            std::string attrScope;
            std::string scope;
            if (!localScope.empty())
            {
                attrScope = localScope + "_AttrValues";
                scope = attrScope + "::";
            }

            recordTypeOut.addData(
                formatString("%s_attributes", pyNameFromCxxName(decl->getNameAsString()).c_str()),
                formatString("const hkReflect::Detail::FixedArrayStorage<hkUlong, hkReflect::Detail::AttributeItem,%i>", nbAttrs),
                formatString("\n{\n%d,\n{\n%s}\n}", nbAttrs, attrpairsStr.c_str()),
                attrScope
                );

            std::string context = recordTypeOut.getDataQualifier();
            std::string symbol = formatString("%s%s_attributes", scope.c_str(), pyNameFromCxxName(decl->getNameAsString()).c_str());
            const std::string& qualifiedName = formatString("reinterpret_cast<const hkReflect::Detail::AttributeArray*>(&%s%s)", context.c_str(), symbol.c_str());

            typeOut.addOpt(decl->getLocation(), "Opt::ATTRIBUTES", qualifiedName);
        }
        return nbAttrs;
    }


    void generateDefault(const Database& db,
        const clang::Decl* decl,
        const std::string& cxxName,
        TypeOut& output,
        Optional<Attribute> defaultAttr = Optional<Attribute>())
    {
        if(!defaultAttr)
            defaultAttr =  db.getAttributeValue< Optional<Attribute> >(decl, "hk::Default");

        if(defaultAttr)
        {
            const Attribute& attr = defaultAttr->as<Attribute>();

            const std::string& defaultType = formatString("const %s", getAggregateTypeFor(cxxName, output.hasTemplateParams()).c_str());

            if (stringStartsWith(cxxName, "hk::ValueArray"))
            {
                if (attr.getNumPosValues())
                {
                    // Add an array containing the default elements.
                    const std::string& elemName = formatString("const %s::ElementType", cxxName.c_str());
                    output.addData("defaultElems", elemName + "[]", attr.getPosValuesString(true));

                    // Write beginning and end in the ValueArray.
                    output.addOpt(decl->getLocation(), "Opt::DEFAULT", "defaultVal", defaultType,
                        formatString("{ defaultElems, defaultElems + %d }", attr.getNumPosValues()).c_str());
                }
                else
                {
                    output.addOpt(decl->getLocation(), "Opt::DEFAULT", "defaultVal", defaultType, "{ HK_NULL, HK_NULL }");
                }
            }
            else
            {
                std::string defaultVal = attr.getPosValuesString();
                if (auto typeDecl = getTypeDeclFrom(db, decl))
                {
                    if (auto defaultStringType = db.getAttributeValue< Optional<std::string> >(typeDecl, "hk::DefaultStringType"))
                    {
                        output.addData("defaultStringValue", *defaultStringType, defaultVal);
                        defaultVal = output.getDataQualifier() + "defaultStringValue";
                    }
                }
                output.addOpt(decl->getLocation(), "Opt::DEFAULT", "defaultVal", defaultType, defaultVal);
            }
        }
    }

    bool hasTrackedPointers(const Database& db, const clang::CXXRecordDecl* recDecl);
    bool hasTrackedPointers(const Database& db, clang::QualType type)
    {
        if (clang::isa<clang::PointerType>(type) || clang::isa<clang::TemplateTypeParmType>(type) ||
            clang::isa<clang::DependentNameType>(type) || clang::isa<clang::DependentTemplateSpecializationType>(type))
        {
            return true;
        }
        if (auto arr = clang::dyn_cast<clang::ArrayType>(type))
        {
            return hasTrackedPointers(db, arr->getElementType());
        }

        auto recDecl = getCXXRecordFrom(type);
        return recDecl != nullptr && hasTrackedPointers(db, recDecl);
    }

    bool hasTrackedPointers(const Database& db, const clang::CXXRecordDecl* recDecl)
    {
        if (db.hasTrackerHandle(recDecl))
        {
            auto trackerAttr = db.getAttributeValue< Optional<Attribute> >(recDecl, "hk::MemoryTracker");
            if (trackerAttr)
            {
                // If it has a manual handler then it contains pointers.
                if ((*trackerAttr)["handler"]->as<std::string>() != "0")
                {
                    return true;
                }
            }

            // Check if opaque to tracker.
            if (!trackerAttr || (*trackerAttr)["opaque"]->as<bool>() == false)
            {
                if (auto reflectAs = db.getAttributeValue< Optional<Attribute> >(recDecl, "hk::ReflectAs"))
                {
                    // ReflectAs type.
                    const std::string& format = reflectAs->get("format");
                    if (format == "KIND_POINTER" || format == "KIND_STRING" || format == "KIND_ARRAY" || format == "KIND_CONTAINER")
                    {
                        return true;
                    }
                }
                else
                {
                    // Normal record, traverse parent and fields.
                    if (recDecl->getNumBases() > 0)
                    {
                        
                        if (hasTrackedPointers(db, recDecl->bases_begin()->getType()))
                        {
                            return true;
                        }
                    }

                    for (auto fieldIt = recDecl->field_begin(); fieldIt != recDecl->field_end(); ++fieldIt)
                    {
                        if (hasTrackedPointers(db, fieldIt->getType()))
                        {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    }

    static std::string getRequestedAlignment(const Database& db, const clang::Decl* decl)
    {
        if(db.getAttributeValue(decl, "hk::AlignReal", false))
        {
            return "hkReflect::Detail::ALIGN_REQ_REAL4";
        }
        else if(unsigned abits = decl->getMaxAlignment())
        {
            assert( (abits % 8) == 0 ); // clang alignment is specified in bits?!
            unsigned abytes = abits / 8;
            return formatString("hkReflect::Detail::ALIGN_REQ_%i", abytes);
        }
        return "0";
    }

    static std::string getEffectiveAlignment(const Database& db, const clang::Decl* decl)
    {
        if(db.getAttributeValue(decl, "hk::AlignReal", false))
        {
            return "sizeof(hkReal)*4";
        }
        else if(unsigned abits = decl->getMaxAlignment())
        {
            assert((abits % 8) == 0); // clang alignment is specified in bits?!
            unsigned abytes = abits / 8;
            return formatString("%i", abytes);
        }
        return "0";
    }

    void addExtraOptsAndFlags(const Database& db, const clang::Decl* decl, TypeOut& output)
    {
        const std::vector<Attribute>& extraOpts = db.getAllAttributesValues<Attribute>(decl, "hk::AddOptional");
        for (std::vector<Attribute>::const_iterator it = extraOpts.begin(); it != extraOpts.end(); ++it)
            output.addOpt(it->getLocation(), it->get("optName"), it->get("optValue"));
        const std::vector<Attribute>& extraFlags = db.getAllAttributesValues<Attribute>(decl, "hk::AddFlags");
        for (std::vector<Attribute>::const_iterator it = extraFlags.begin(); it != extraFlags.end(); ++it)
            output.addFlag(it->getLocation(), it->get("flags"));
    }

    // Generate the field type. Appends the data items to the program listing directly.
    // Returns (reflected field name, field type symbol, fieldValidator or empty)
    std::tuple<std::string, std::string, std::string> generateFieldType(
        const Database& db,
        const clang::CXXRecordDecl* recDecl,
        const std::string& cxxName,
        const clang::FieldDecl* fieldDecl,
        const std::string& fieldPrefix,
        TypeOut& typeOut,
        bool trackerType)
    {
        const std::string& fieldName = fieldDecl->getNameAsString();

        // In general is the same as the C++ field type but may be overridden by attributes.
        std::string fieldTypeStr;

        if(const Optional<std::string>& overriddenType = db.getAttributeValue< Optional<std::string> >(fieldDecl, "hk::Type"))
        {
            fieldTypeStr = *overriddenType;
        }

        if(db.getAttributeValue<bool>(fieldDecl, "hk::OpaqueType", false) && !trackerType)
        {
            const std::string ftName = fieldTypeStr.empty() ? db.getScopedMangledName(fieldDecl->getType(), Database::NAME_REFLECTION) : fieldTypeStr;
            fieldTypeStr = formatString("hkReflect::Detail::MarkAsOpaque< %s >", ftName.c_str());
        }

        if (trackerType && (fieldDecl->getType()->isPointerType()))
        {
            // Early out; use the helper template class
            const std::string& sym = formatString("hkReflect::Detail::PtrFieldHelper< HK_DETAIL_SIMPLE_OFFSET_OF(%s, %s) >::typeData",
                fieldName.c_str(), cxxName.c_str());
            return std::make_tuple("", sym, "");
        }

        TypeOut fieldTypeOut(db, fieldDecl);

        if(!fieldTypeStr.empty())
        {
            fieldTypeOut.setParent(db.getReflectedTypeAddress(fieldTypeStr, trackerType));
        }
        else
        {
            fieldTypeOut.setParent(db.getReflectedTypeAddress(fieldDecl->getType(), trackerType));
        }


        const std::string& cleanFieldName = fieldNameWithoutPrefix(fieldPrefix, fieldName);

        const std::string& fieldDeclName = db.getAttributeValue(fieldDecl, "hk::Rename", formatString("\"%s\"", cleanFieldName.c_str()));
        std::vector< std::pair<std::string, std::string> > attributesWithValidation;

        if (!trackerType)
        {
            // Attributes.
            std::vector<AttributeVariant> extraAttrs;
            // Put attributes in the main typeOut so they can access symbols in the containing record.
            generateAttributes(db, fieldDecl, fieldTypeOut, typeOut, extraAttrs, &attributesWithValidation);

            // Default.
            const Optional<Attribute>& defaultValue = db.getAttributeValue< Optional<Attribute> >(fieldDecl, "hk::Default");
            if (defaultValue)
            {
                
                generateDefault(db, fieldDecl, db.getScopedMangledName(fieldDecl->getType()), fieldTypeOut, defaultValue);
            }

            // Decl context.
            fieldTypeOut.addOpt(fieldDecl->getLocation(), "Opt::DECL_CONTEXT", db.getReflectedTypeAddress(getTypeFrom(db, recDecl)));

            // Name.
            fieldTypeOut.addOpt(fieldDecl->getLocation(), "Opt::DECL_NAME", formatString("%s", fieldDeclName.c_str()));

            // Check for extra alignment on field
            const std::string& reqAlign = getRequestedAlignment(db, fieldDecl);
            if(reqAlign!="0")
            {
                const std::string& tname = db.getScopedMangledName(fieldDecl->getType());
                std::string alignOf = formatString("HK_MAX2(HK_ALIGN_OF(%s),%s)", tname.c_str(), getEffectiveAlignment(db, fieldDecl).c_str() );
                std::string sizeOf = "sizeof(" + tname + ")";
                fieldTypeOut.addSizeAlign(recDecl->getLocation(), sizeOf, alignOf, reqAlign);
            }
        }

        // Flags.
        std::string fieldFlags("hkReflect::Decl::DECL_DATA_FIELD");
        if(!db.isSerializable(fieldDecl) || db.isOpaqueTypeAllowed(fieldDecl))
        {
            fieldFlags.append(" | hkReflect::Decl::DECL_NOT_SERIALIZABLE");
        }
        //todo.nt if the field is serializable we should check that the type of the field is serializable too

        // Much of this logic is duplicated in FieldSetterLinkFilter.cpp, since we need to figure out if it would have a setter to set the hk::SetsField
        // attribute. Eventually we should cache this
        clang::AccessSpecifier access = fieldDecl->getAccess();
        if (auto setterAttr = db.getAttributeValue< Optional<Attribute> >(fieldDecl, "hk::FieldSetter"))
        {
            const std::string& lookFor = (*setterAttr)["value"] ? *(*setterAttr)["value"] : "set" + cleanFieldName;

            bool found = false;
            int candidates = 0;

            hkLog.info(fieldDecl, "Searching setter for %s::%s (%s)",
                cxxName.c_str(), fieldName.c_str(), lookFor.c_str());
            for (auto it = recDecl->method_begin(); it != recDecl->method_end(); ++it)
            {
                const clang::CXXMethodDecl* method = *it;
                const std::string& methodName = method->getNameAsString();
                // Check the name
                if (llvm::StringRef(methodName).equals_lower(lookFor))
                {
                    ++candidates;

                    if (method->isStatic())
                    {
                        hkLog.info(method, "Skipping %s because it is static", method->getNameAsString().c_str());
                        continue;
                    }

                    clang::QualType resultType = method->getResultType();
                    if (!(resultType->isVoidType() || resultType->isBooleanType() || db.getScopedMangledName(resultType) == "hkResult"))
                    {
                        // Difficult to handle arbitrary return types safely, so setters are limited to void and bool return types.
                        hkLog.info(method, "Skipping %s because it does not return void|bool|hkResult", method->getNameAsString().c_str());
                        continue;
                    }

                    if (method->getNumParams() != 1)
                    {
                        hkLog.info(method, "Skipping %s because it does not have one parameter", method->getNameAsString().c_str());
                        continue;
                    }

                    clang::QualType paramType = method->getParamDecl(0)->getType();

                    clang::QualType subType;
                    if (auto asPtr = clang::dyn_cast<clang::PointerType>(paramType))
                    {
                        subType = asPtr->getPointeeType();
                    }
                    else if (auto asRef = clang::dyn_cast<clang::ReferenceType>(paramType))
                    {
                        subType = asRef->getPointeeType();
                    }

                    if (!subType.isNull() && !subType.isConstQualified() &&
                        subType.getCanonicalType().getUnqualifiedType() == fieldDecl->getType().getCanonicalType().getUnqualifiedType())
                    {
                        hkLog.info(method, "Skipping %s because its param is a non-const pointer/reference.",
                            methodName.c_str());
                        continue;
                    }

                    // Valid setter found.
                    found = true;
                    hkLog.info(method, "FieldSetter %s::%s.", cxxName.c_str(),methodName.c_str());

                    const std::string& fieldType = db.getScopedMangledName(fieldDecl->getType());
                    const std::string& setValueType = db.getScopedMangledName(paramType);
                    const std::string& setMethReturnType = db.getScopedMangledName(method->getResultType());

                    const std::string& implName = fieldName + "_impl";
                    const std::string& implType = formatString(
                        "const %shkReflect::Detail::SetterImpl<typename hkReflect::Detail::ReflectedKindOf< %s >::Type>::Type",
                        typeOut.hasTemplateParams() ? "typename " : "",
                        fieldType.c_str());
                    const std::string& args = formatString(
                        "\n&hkReflect::Detail::erasedFieldSetterCall< %s >::call< "
                        "HK_REFLECT_RESOLVE(%s), HK_REFLECT_RESOLVE(%s), HK_REFLECT_RESOLVE(%s), "
                        "&HK_REFLECT_RESOLVE2(%s)::%s >, HK_ROFFSET_OF(%s, %s)\n",
                        setMethReturnType.c_str(),
                        cxxName.c_str(), fieldType.c_str(), setValueType.c_str(),
                        cxxName.c_str(), methodName.c_str(), fieldName.c_str(), cxxName.c_str());


                    typeOut.addData(implName, implType, args, "", TypeOut::FLAG_CONSTRUCTOR);
                    fieldTypeOut.addOpt(fieldDecl->getLocation(), "Opt::IMPL", formatString("&%s%s", typeOut.getDataQualifier().c_str(), implName.c_str()));
                    fieldFlags.append(" | hkReflect::Decl::DECL_CUSTOM_SETTER");

                    // The access level of the reflected field is the most exposed of the levels of the field itself and its setter.
                    access = std::min(method->getAccess(), access);
                    break;
                }
            }


            if (!found)
            {
                hkLog.error(db.astContext.getSourceManager(), setterAttr->getLocation(),
                    "No valid setter named '%s' found for %s::%s (%d candidates)",
                    lookFor.c_str(), cxxName.c_str(), fieldName.c_str(), candidates);
            }
        }

        if (access == clang::AS_protected)
        {
            fieldFlags.append(" | hkReflect::Decl::DECL_PROTECTED");
        }
        else if (access == clang::AS_private)
        {
            fieldFlags.append(" | hkReflect::Decl::DECL_PRIVATE");
        }

        fieldTypeOut.addOpt(fieldDecl->getLocation(), "Opt::DECL_FORMAT", formatString("((%s)<<16) | HK_ROFFSET_OF(%s, %s)", fieldFlags.c_str(), fieldName.c_str(), cxxName.c_str()).c_str());

        addExtraOptsAndFlags(db, fieldDecl, fieldTypeOut);

        
        fieldTypeOut.appendDataTo(typeOut.getRawCode());

        const std::string& fieldTypeSym = fieldDecl->getNameAsString()+"::typeData";

        std::string fieldValidationFunction;
        if (!trackerType)
        {
            //bool hasDeclaredValidation = false;
            bool recordDoesntWantValidation = false;
            bool fieldHasDeclaredValidation = false;
            // Record false -> clear all fields
            // record true -> continue and check each field
            if (const clang::AnnotateAttr* attr = db.getAttribute(recDecl, "hk::Validate"))
            {
                // Use any attribute on the field declaration first
                bool hasDeclaredValidation = db.getAttributeValue<bool>(attr);
                if (!hasDeclaredValidation)
                {
                    // If we explicitly said no validate, don't generate anything
                    attributesWithValidation.clear();
                    recordDoesntWantValidation = true;
                }
            }

            // Validate(false) on the record overrides anything else
            if (!recordDoesntWantValidation)
            {
                if (const clang::AnnotateAttr* attr = db.getAttribute(fieldDecl, "hk::Validate"))
                {
                    // Use any attribute on the field declaration first
                    fieldHasDeclaredValidation = db.getAttributeValue<bool>(attr);
                    if (!fieldHasDeclaredValidation)
                    {
                        // If we explicitly said no validate, don't generate anything
                        attributesWithValidation.clear();
                    }
                }
                else if (auto fieldRecDecl = fieldDecl->getType()->getAsCXXRecordDecl())
                {
                    // If nothing on the field directly, use the type of the field
                    if (const clang::AnnotateAttr* validateAttr = db.getAttribute( fieldRecDecl, "hk::Validate"))
                    {
                        fieldHasDeclaredValidation = db.getAttributeValue<bool>(validateAttr);
                        if (!fieldHasDeclaredValidation)
                        {
                            // If we explicitly said no validate, don't generate anything
                            attributesWithValidation.clear();
                        }
                    }
                }
            }

            // The attributes check is only really needed to check the user has correctly declared hk::Validate
            if ((attributesWithValidation.size() > 0) || fieldHasDeclaredValidation)
            {
                fieldValidationFunction = "isValid_"; fieldValidationFunction.append(fieldDecl->getNameAsString());
                std::string validationFunctionBody;
                std::string separator;
                for (auto& s : attributesWithValidation)
                {
                    if (s.second == "hk::Presets")
                    {
                        // These are generated differently to the other attributes
                        validationFunctionBody.append(formatString("%s%s.isValid(fieldValue)", separator.c_str(), s.first.c_str()));
                    }
                    else
                    {
                        validationFunctionBody.append(formatString("%sreinterpret_cast<const %s*>(&%s::attr)->isValid(fieldValue)", separator.c_str(), s.first.c_str(), s.first.c_str()));
                    }
                    separator = " && ";
                }
                if (fieldHasDeclaredValidation)
                {
                    validationFunctionBody.append(formatString("%sfieldValue.reflectValidate()", separator.c_str()/*, db.getScopedMangledName(fieldDecl->getType()).c_str()*/));
                    separator = " && ";
                }
                typeOut.addData(formatString("static inline bool %s(const void* valuePtr) { const decltype(%s::%s)& fieldValue = *reinterpret_cast<const decltype(%s::%s)*>(valuePtr); return %s; }", fieldValidationFunction.c_str(), cxxName.c_str(), fieldName.c_str(), cxxName.c_str(), fieldName.c_str(), validationFunctionBody.c_str()));

                // We can't partially specialize a template function. Ideally we would use a struct with an inlined method, but
                // for now we will just disallow that

                fieldTypeOut.addOpt(fieldDecl->getLocation(), "Opt::VALIDATE", formatString("%s::_Auto::isValid_%s", cxxName.c_str(), fieldDecl->getNameAsString().c_str()).c_str(), TypeOut::FLAG_IS_CODE);
            }
        }

        TextBuilder::VerbatimSection section;
        fieldTypeOut.appendTypeContentTo(section);

        // Fields of tracker types are not const, they need to be fixed up at runtime.
        const char* type = trackerType ? "hkReflect::Detail::TypeData" : "const hkReflect::Detail::TypeData";
        typeOut.addData("typeData", type, section.toString(), fieldDecl->getNameAsString());

        return std::make_tuple(fieldDeclName, fieldTypeSym, fieldValidationFunction);
    }


    // Generate the property type. Appends the data items to the program listing directly.
    // Returns (property name, property type symbol)
    std::pair<std::string, std::string> generatePropertyType(
        const Database& db,
        const clang::CXXRecordDecl* recDecl,
        const clang::CXXRecordDecl* propertyDecl,
        const std::string& cxxName,
        TypeOut& typeOut)
    {
        TypeOut propTypeOut(db, propertyDecl);

        std::string name;
        const clang::CXXMethodDecl* getmeth, *setmeth;
        std::tie(name, getmeth, setmeth) = getPropertyData(db, propertyDecl);

        // todo.nt6 forbid use of types nested in templates as property type
        const std::string& getterValuetype = db.getScopedMangledName(getmeth->getResultType());
        const std::string& propImpl = formatString("const %shkReflect::ReflectionOf< %s >::PropertyImpl",
            typeOut.hasTemplateParams() ? "typename " : "",
            getterValuetype.c_str());

        std::string setterImpl = formatString("static_cast<%shkReflect::Detail::MethodTypes< %s >::SetterType>(nullptr)", typeOut.hasTemplateParams() ? "typename " : "", getterValuetype.c_str());
        const std::string& cxxNameNoTypename = db.getScopedMangledNameNoPrefix(db.astContext.getTypeDeclType(recDecl));

        if(setmeth)
        {
            const std::string& setMethName = setmeth->getNameAsString();
            const std::string& setValueType = db.getScopedMangledName(setmeth->getParamDecl(0)->getType());
            const std::string& setMethReturnType = db.getScopedMangledName(setmeth->getResultType());
            setterImpl = formatString("&hkReflect::Detail::erasedSetterCall< %s >::call< %s, %s, &%s::%s >", setMethReturnType.c_str(), cxxName.c_str(), setValueType.c_str(), cxxNameNoTypename.c_str(), setMethName.c_str());
        }

        std::string parentType = getterValuetype;
        if(const clang::ReferenceType* refType = clang::dyn_cast<clang::ReferenceType>(getmeth->getResultType().getTypePtr()))
        {
            const clang::QualType pointeeType = clang::QualType(refType->getPointeeType().getTypePtr(), 0);
            parentType = db.getScopedMangledName(pointeeType);
        }
        propTypeOut.setParent("HK_REFLECT_GET_TYPE("+parentType+")");
        propTypeOut.addFlag(propertyDecl->getLocation(), "hkReflect::Type::TYPE_PROPERTY");
        propTypeOut.addOpt(propertyDecl->getLocation(), "Opt::DECL_NAME", "\"" + name + "\"");
        //          propTypeOut.addItem(ret.getLocation(), "Opt::SIZE_ALIGN", "0"); // hide underlying size/align?

        std::vector<AttributeVariant> extraAttrs;
        generateAttributes(db, propertyDecl, propTypeOut, typeOut, extraAttrs);

        std::string fieldFlags("hkReflect::Decl::DECL_PROPERTY_FIELD");
        if( db.getAttributeValue<bool>(propertyDecl, "hk::Serialize", false) == false )
        {
            fieldFlags += "| hkReflect::Decl::DECL_NOT_SERIALIZABLE";
        }

        propTypeOut.addOpt(propertyDecl->getLocation(), "Opt::DECL_FORMAT", formatString("((%s)<<16)", fieldFlags.c_str()));
        propTypeOut.addOpt(propertyDecl->getLocation(), "Opt::DECL_CONTEXT", db.getReflectedTypeAddress(recDecl).c_str());

        const std::string& implSection = formatString("%s_impl", name.c_str());
        const std::string& implName= typeOut.getDataQualifier() + implSection + "::impl";
        typeOut.addData("impl", propImpl,
            formatString("\n&hkReflect::Detail::erasedGetterCall< %s, %s, &%s::%s >,\n%s",
            cxxName.c_str(), getterValuetype.c_str(), cxxNameNoTypename.c_str(), getmeth->getNameAsString().c_str(),
            setterImpl.c_str()), implSection, TypeOut::FLAG_CONSTRUCTOR );
        propTypeOut.addOpt(propertyDecl->getLocation(), "Opt::IMPL", "&"+implName);

        addExtraOptsAndFlags(db, propertyDecl, propTypeOut);

        propTypeOut.appendDataTo(typeOut.getRawCode());

        const std::string& propSection = formatString("%s_type", name.c_str());

        TextBuilder::VerbatimSection content;
        propTypeOut.appendTypeContentTo(content);
        std::string propTypeSym = typeOut.getDataQualifier() + name+"_type::typeData";
        typeOut.addData("typeData", "const hkReflect::Detail::TypeData", content.toString(), propSection);

        return std::pair<std::string, std::string>('"' + name + '"', propTypeSym);
    }

    void generateFieldsAndProperties(
        const Database& db,
        const clang::CXXRecordDecl* recDeclOrig,
        const clang::CXXRecordDecl* recDecl,
        const std::string& cxxName,
        TypeOut& typeOut,
        std::vector< std::pair<std::string, std::string> >& fieldsWithValidation,
        bool forceReflected=false,
        bool trackerType = false)
    {
        std::string fieldsArrayName("HK_NULL");

        std::vector<std::string> fields;
        std::vector<std::string> properties;
        std::set<std::string> memberNames;

        if (!trackerType)
        {
            // Fill the set with the reflected fields in the ancestors (to check for hidden fields).
            // Does not work if the parent is a dependent type but it is useful in the common case.
            const clang::CXXRecordDecl* curr = recDecl;
            while (curr && curr->getNumBases())
            {
                const Attribute& details = db.getReflectDetails(curr);
                if (forceReflected || details["parents"]->as<bool>())
                {
                    curr = getCXXRecordFrom(curr->bases_begin()->getType());
                    if (curr)
                    {
                        checkIsDefinition(curr);
                        const std::string& fieldPrefix = db.getAttributeValue<std::string>(recDecl, "hk::FieldPrefix");
                        for (auto it = curr->field_begin(); it != curr->field_end(); ++it)
                        {
                            if (forceReflected || db.isReflected(*it))
                            {
                                memberNames.insert(fieldNameWithoutPrefix(fieldPrefix, it->getNameAsString()));
                            }
                        }
                    }
                }
                else
                {
                    break;
                }
            }
        }

        const std::string& fieldPrefix = db.getAttributeValue<std::string>(recDecl, "hk::FieldPrefix");
        for(clang::CXXRecordDecl::field_iterator it = recDecl->field_begin(); it != recDecl->field_end(); ++it)
        {
            // Keep only reflected fields
            if(db.isReflected(*it) || forceReflected)
            {
                if (clang::isa<clang::ReferenceType>(it->getType()))
                {
                    if (!trackerType)
                    {
                        hkLog.error(*it, "Reference fields are not allowed in reflected types.");
                    }
                    continue;
                }

                if (auto recField = it->getType()->getAsCXXRecordDecl())
                {
                    if (recField->isUnion() && !db.getAttribute(*it, "hk::Type") && !db.getAttribute(*it, "hk::OpaqueType"))
                    {
                        if (!trackerType)
                        {
                            hkLog.error(*it, "Union fields are not allowed in reflected types.");
                        }
                        continue;
                    }
                }

                if (trackerType)
                {
                    // Skip all fields which do not contain links/blocks information.
                    if (!db.trackContent(*it) || !hasTrackedPointers(db, it->getType()))
                    {
                        continue;
                    }
                }

                std::string fieldReflectedName, fieldTypeSym, fieldValidationFunction;
                std::tie(fieldReflectedName, fieldTypeSym, fieldValidationFunction) = generateFieldType(db, recDeclOrig, cxxName, *it, fieldPrefix, typeOut, trackerType);

                // This could happen even for fields since we remove the field prefix from it's name. So if a record set the field prefix to "m_" and had two
                // fields called "buffer" and "m_buffer" this would fire since they would be indistinguishable for reflection.
                if(!trackerType && stringStartsWith(fieldReflectedName, "\"") && !memberNames.insert(fieldReflectedName).second)
                    hkLog.error(*it, "Field '%s' in class '%s' hides a field with the same name in its parent.",
                        fieldReflectedName.c_str(), cxxName.c_str());

                if (fieldValidationFunction.size())
                {
                    fieldsWithValidation.push_back(std::make_pair((*it)->getNameAsString(), fieldValidationFunction));
                }
                fields.push_back(fieldTypeSym);
            }
        }

        if (!trackerType)
        {
            if (db.hasOwnVtable(recDecl))
            {
                typeOut.addFlag(recDecl->getLocation(), "hkReflect::Type::TYPE_OWN_VTABLE");
            }

            // generate record properties
            {
                // check that the hk::PropertyInternal attribute is not used on the record directly
                std::vector<Attribute> attrs = db.getAllAttributesValues<Attribute>(recDecl, "hk::PropertyInternal");
                for (unsigned i = 0; i < attrs.size(); ++i)
                {
                    hkLog.error(db.astContext.getSourceManager(), attrs[i].getLocation(), "Attribute hk::PropertyInternal is used explicitly on class %s; must use HK_PROPERTY instead.",
                        db.getScopedMangledName(recDecl).c_str());
                }
            }

            for (auto innerDeclIt = clang::CXXRecordDecl::specific_decl_iterator<clang::CXXRecordDecl>(recDecl->decls_begin());
                innerDeclIt != clang::CXXRecordDecl::specific_decl_iterator<clang::CXXRecordDecl>(recDecl->decls_end()); ++innerDeclIt)
            {
                if (const clang::CXXRecordDecl* innerDecl = isPropertyDecl(*innerDeclIt))
                {
                    try
                    {
                        std::string propertyName, propertyTypeSym;
                        std::tie(propertyName, propertyTypeSym) = generatePropertyType(db, recDecl, innerDecl, cxxName, typeOut);
                        if (!memberNames.insert(propertyName).second)
                            hkLog.error(innerDecl, "Property '%s' in class '%s' hides a field with the same name.",
                            propertyName.c_str(), cxxName.c_str());
                        properties.push_back(propertyTypeSym);
                    }
                    catch(RuntimeError e)
                    {
                        hkLog.error("%s", e.what());
                    }
                }
            }
        }

        std::string numberOfFields("0"), numberOfProps("0"), numberOfOtherDecls("0");

        // Skip the DECLS optional for tracker types with no relevant fields.
        // todo.rt do we always need DECLS in reflected types or could we just skip it?
        if(!trackerType || fields.size())
        {
            int numTotal = int(fields.size()+properties.size());
            numberOfFields = formatString("%d", int(fields.size()));
            numberOfProps = formatString("%d", int(properties.size()));

            TextBuilder::VerbatimSection fieldsArrayContents;
            fieldsArrayContents.append("\n{");
            fieldsArrayContents.append( formatString("{ %i, %i, %i, 0 },%s", fields.size(), properties.size(), numTotal, numTotal?" {":"") );

            for(auto cit = fields.begin(); cit != fields.end(); ++cit)
            {
                fieldsArrayContents.append("reinterpret_cast<const hkReflect::Type*>(&%s),", cit->c_str());
            }
            for(auto cit = properties.begin(); cit != properties.end(); ++cit)
            {
                fieldsArrayContents.append("reinterpret_cast<const hkReflect::Type*>(&%s),", cit->c_str());
            }

            fieldsArrayContents.append("%s}", numTotal?" }":"");

            typeOut.addOptCast(recDecl->getLocation(), "Opt::DECLS", "decls",
                formatString("const hkReflect::Detail::FixedArrayStorage<hkReflect::Detail::DeclsArray::Head, const hkReflect::Type*,%i>", numTotal),
                "reinterpret_cast<const hkReflect::Detail::DeclsArray*>",
                fieldsArrayContents.toString(), "Decls");
        }
    }

    // Return true if any validation was generated
    bool generateValidationChecks(
        const Database& db,
        const clang::CXXRecordDecl* recDecl,
        const std::string& cxxName,
        TypeOut& typeOutAuto,
        std::vector< std::pair<std::string, std::string> >& fieldsWithValidation)
    {
        bool foundValidateContract = false;
        bool parentRequiresValidation = false;
        std::string parentTypeName;
        for (auto mit = recDecl->method_begin(); mit != recDecl->method_end(); ++mit)
        {
            if ((*mit)->getNameAsString() == "validateContract")
            {
                foundValidateContract = true;
                break;
            }
        }

        if (recDecl->bases_begin())
        {
            if (const clang::CXXRecordDecl* parentRecord = recDecl->bases_begin()->getType()->getAsCXXRecordDecl())
            {
                parentRequiresValidation = db.getAttributeValue(parentRecord, "hk::Validate", false);
                parentTypeName = db.getScopedMangledName(parentRecord);
            }
        }

        // If it declares custom validation, the user will provide it
        if (!db.getAttributeValue(recDecl, "hk::HasCustomValidation", false))
        {
            // Emit this if any fields are validated, or there us a user-defined validation
            if ((fieldsWithValidation.size() > 0) || foundValidateContract || parentRequiresValidation)
            {
                {
                    std::string validationBody = formatString("static inline bool isValid(const %s& value) { return ", cxxName.c_str());
                    std::string separator;
                    if (parentRequiresValidation)
                    {
                        validationBody.append(formatString("%sreinterpret_cast<const %s&>(value).reflectValidate()", separator.c_str(), parentTypeName.c_str()));
                        separator = " && ";
                    }
                    for (auto&p : fieldsWithValidation)
                    {
                        validationBody.append(formatString("%s%s(&value.%s)", separator.c_str(), p.second.c_str(), p.first.c_str()));
                        separator = " && ";
                    }
                    if (foundValidateContract)
                    {
                        validationBody.append(formatString("%svalue.validateContract().isSuccess()", separator.c_str()));
                        separator = " && ";
                    }

                    if (!separator.size())
                    {
                        // We always emit something
                        validationBody.append("true");
                    }
                    validationBody.append("; }");
                    typeOutAuto.addData(validationBody);
                    typeOutAuto.addData(formatString("static inline bool isValidPtr(const void* valuePtr) { return isValid(*reinterpret_cast<const %s*>(valuePtr)); }", cxxName.c_str()));

                    typeOutAuto.addOpt(recDecl->getLocation(), "Opt::VALIDATE", formatString("%s::_Auto::isValidPtr", cxxName.c_str()), TypeOut::FLAG_IS_CODE);

                    if (!typeOutAuto.hasTemplateParams())
                    {
                        typeOutAuto.getRawCodeTypeDeclarations().append("#if defined(HK_INCLUDE_VALIDATE_INLINES) || defined(HK_DETAIL_REFLECT_DEFINITIONS)");
                    }
                    typeOutAuto.getRawCodeTypeDeclarations().append(formatString("%sHK_REFLECT_VALIDATE_INLINE bool %s::reflectValidate() const { return %sisValid(*this); };", typeOutAuto.getTemplateHeader().c_str(), cxxName.c_str(), typeOutAuto.getDataQualifier().c_str()));
                    if (!typeOutAuto.hasTemplateParams())
                    {
                        typeOutAuto.getRawCodeTypeDeclarations().append("#endif");
                    }
                    return true;
                }
            }
            else
            {
                if (!typeOutAuto.hasTemplateParams())
                {
                    typeOutAuto.getRawCodeTypeDeclarations().append("#if defined(HK_INCLUDE_VALIDATE_INLINES) || defined(HK_DETAIL_REFLECT_DEFINITIONS)");
                }
                typeOutAuto.getRawCodeTypeDeclarations().append(formatString("%sHK_REFLECT_VALIDATE_INLINE bool %s::reflectValidate() const { return true; };", typeOutAuto.getTemplateHeader().c_str(), cxxName.c_str(), cxxName.c_str(), typeOutAuto.getDataQualifier().c_str()));
                if (!typeOutAuto.hasTemplateParams())
                {
                    typeOutAuto.getRawCodeTypeDeclarations().append("#endif");
                }
            }
        }
        return false;
    }

    std::string flattenTypeName(const std::string& typeName, bool dependent)
    {
        std::string res = std::string(dependent ? "typename " : "") + "hkReflect::Detail::Flatten<" + typeName + " >::T";
        std::replace(res.begin(), res.end(), '&', '*');
        return res;
    }

    // Iterates in order over the fields of a message type, skips non-public fields and ancestors
    struct MessageFieldIterator
    {
    public:
        MessageFieldIterator(const clang::CXXRecordDecl* type)
        {
            parents.push_back(type);
            while (type->getNumBases() > 0 && type->bases_begin()->getAccessSpecifier() == clang::AS_public)
            {
                type = type->bases_begin()->getType()->getAsCXXRecordDecl();
                parents.push_back(type);
            }
            currentField = parents.back()->field_begin();
            skipEmptyAndNonPublic();
        }

        bool isValid() { return !parents.empty(); }

        const clang::FieldDecl* operator*() { return *currentField; }

        const clang::FieldDecl* operator->() { return *currentField; }

        MessageFieldIterator& operator++()
        {
            ++currentField;
            skipEmptyAndNonPublic();
            return *this;
        }

        protected:
            void skipEmptyAndNonPublic()
            {
                for(;;)
                {
                    while (currentField != parents.back()->field_end())
                    {
                        // skip the current field if non-public
                        if ((*currentField)->getAccess() == clang::AS_public)
                        {
                            // a public field has been found
                            return;
                        }
                        ++currentField;
                    }
                    // skip the current parent if it has no public fields
                    parents.pop_back();
                    if (parents.empty())
                    {
                        break;
                    }
                    currentField = parents.back()->field_begin();
                }
            }

            clang::CXXRecordDecl::field_iterator currentField;
            std::vector<const clang::CXXRecordDecl*> parents;
    };

    void getFunctionSignatureParamList(const Database& db, const clang::CXXMethodDecl* m, std::vector<std::string>& params, bool flatten, Database::NameType nt = Database::NAME_NORMAL)
    {
        for(clang::CXXMethodDecl::param_const_iterator it = m->param_begin(); it != m->param_end(); ++it)
        {
            clang::QualType type = (*it)->getType();
            const std::string& paramType = db.getScopedMangledName(type, nt);
            params.push_back(flatten ? flattenTypeName(paramType, type->isDependentType()) : paramType);
        }
    }

    std::string getSharedInvokerFunc(const Database& db, const clang::CXXRecordDecl* recDecl, const clang::CXXMethodDecl* m, bool isMethod)
    {
        std::vector<std::string> params;
        clang::QualType resultType = m->getResultType();
        params.push_back(flattenTypeName(db.getScopedMangledName(resultType), resultType->isDependentType()));

        if (isMethod)
        {
            // add flattened class type
            const std::string& classType = formatString("hkReflect::Detail::%s",
                recDecl->getNumBases() <= 1 ? "SingleInheritanceClass" : "MultipleInheritanceClass"); 
            params.push_back(classType);
        }

        getFunctionSignatureParamList(db, m, params, true);

        bool stringOut = db.getAttributeValue(m, "hk::StringOut", false);
        if (stringOut)
        {
            // last param should be unflattened for type checking
            const std::string& paramType = db.getScopedMangledName(m->getParamDecl(m->getNumParams()-1)->getType());
            params.back() = formatString("hkReflect::Detail::RefToPtr< %s >::Type", paramType.c_str());
        }

        return formatString("hkReflect::Detail::%s%sInvoker< %s >::invoke",
            stringOut ? "StringOut" : "",
            isMethod ? "Method" : "Function",
            params.empty() ? "" : joinStrings(params, ", ").c_str());
    }

    struct HandlerData
    {
        const clang::CXXRecordDecl* msgType;
        bool methodTakesMessagePointer;
    };

    std::string getSignature(const std::vector<std::string>& params)
    {
        return formatString("HK_REFLECT_CALLABLE_SIGNATURE_OF(%s)", joinStrings(params, ", ").c_str());
    }

    std::string getSignature(const Database& db, const clang::CXXMethodDecl* m, clang::QualType returnType = clang::QualType())
    {
        std::vector<std::string> params;

        if(returnType.isNull())
            returnType = m->getResultType();

        params.push_back(db.getScopedMangledName(returnType, Database::NAME_REFLECTION));

        // todo.nt4m prevent reflection of functions which take non-const pointer to values/pointers;
        // scripts are not handling this kind of parameters correctly
        for(clang::CXXMethodDecl::param_const_iterator it = m->param_begin(); it != m->param_end(); ++it)
        {
            const clang::QualType paramType = (*it)->getType();
            const clang::PointerType* p = clang::dyn_cast<clang::PointerType>(paramType);
            const clang::ReferenceType* r = clang::dyn_cast<clang::ReferenceType>(paramType);
            clang::QualType targetType = p ?  p->getPointeeType() :
                r ? r->getPointeeType() : clang::QualType((clang::Type*)NULL, 0);

            if (targetType.getTypePtrOrNull() && !targetType.isConstQualified())
            {
                bool typeIsInvalid = false;
                if (targetType->isBooleanType() || targetType->isIntegerType() || targetType->isFloatingType() || targetType->isPointerType())
                {
                    typeIsInvalid = true;
                }
                else
                {
                    
                    if (const clang::CXXRecordDecl* rec = getCXXRecordFrom(targetType))
                    {
                        if(const Optional<Attribute>& reflectAs = db.getAttributeValue< Optional<Attribute> >(rec, "hk::ReflectAs") )
                        {
                            const std::string& tag = reflectAs->get("format");
                            if (tag == "KIND_BOOL" || tag == "KIND_INT" || tag == "KIND_FLOAT" || tag == "KIND_POINTER"
                                || (tag == "KIND_STRING" && (!db.getAttributeValue(m, "hk::StringOut", false) || (it+1) != m->param_end())))
                            {
                                typeIsInvalid = true;
                            }
                        }
                        else if(Optional<Attribute> reflectAsInt = db.getAttributeValue< Optional<Attribute> >(rec, "hk::ReflectAsInteger") )
                        {
                            typeIsInvalid = true;
                            //throw RuntimeError(rec, "Fixme");
                        }

                        // hk::ReflectAsRepeat should be fine.
                    }
                }

                if (typeIsInvalid)
                {
                    hkLog.warning(m, "Cannot reflect method %s: parameter %d is non-const pointer to value",
                        db.getScopedMangledName(m).c_str(), (*it)->getFunctionScopeIndex() + 1);
                }
            }
        }
        getFunctionSignatureParamList(db, m, params, false, Database::NAME_REFLECTION);

        if (db.getAttributeValue(m, "hk::StringOut", false))
        {
            // return type must be void or char*
            if (!returnType->isVoidType())
            {
                const clang::PointerType* pointerType = clang::dyn_cast<clang::PointerType>(returnType);
                if (!pointerType || !pointerType->getPointeeType().getTypePtr()->isCharType())
                {
                    hkLog.error(m,
                        "Method %s is marked as hk::StringOut so its return type must be 'void' or '(const) char*'",
                        db.getScopedMangledName(m).c_str());
                }
            }

            // change return type to StringOut
            params[0] = "hkReflect::Detail::StringOut";
            params.pop_back();
        }

        return getSignature(params);
    }

    HandlerData getHandlerData(const Database& db, const clang::CXXMethodDecl* method)
    {
        HandlerData res;
        res.msgType = NULL;
        res.methodTakesMessagePointer = false;

        if (Optional<std::string> attr = db.getAttributeValue< Optional<std::string> >(method, "hk::HandledMessage"))
        {
            // search for the attribute type
            res.msgType = db.getMessageTypeDecl(*attr);
            if (!res.msgType)
            {
                hkLog.error(method, "Cannot find the message type handled by %s (%s)", db.getScopedMangledName(method).c_str(), attr->c_str());
                return res;
            }
        }

        // check if the method takes a message pointer/ref in input
        if (method->getResultType()->isVoidType() && method->getNumParams() == 1)
        {
            const clang::PointerType* pointerType = clang::dyn_cast<clang::PointerType>(method->getParamDecl(0)->getType().getTypePtr());
            const clang::ReferenceType* refType = clang::dyn_cast<clang::ReferenceType>(method->getParamDecl(0)->getType().getTypePtr());

            if (const clang::Type* pointedType = pointerType ? pointerType->getPointeeType().getTypePtr() :
                refType ? refType->getPointeeType().getTypePtr() : NULL)
            {
                if (const clang::CXXRecordDecl* recordType = pointedType->getAsCXXRecordDecl())
                {
                    const clang::CXXRecordDecl* recordTypeDefinition = recordType->getDefinition();
                    if (recordTypeDefinition && db.getAttribute(recordTypeDefinition, "hk::MessageType"))
                    {
                        // check that the message type corresponds to the one declared in the attribute
                        if (res.msgType && res.msgType != recordTypeDefinition)
                        {
                            hkLog.error(method, "%s is marked as a handler of %s but handles %s instead",
                                db.getScopedMangledName(method).c_str(),
                                db.getScopedMangledName(res.msgType).c_str(),
                                db.getScopedMangledName(recordTypeDefinition).c_str());
                            return res;
                        }
                        res.msgType = recordTypeDefinition;
                        res.methodTakesMessagePointer = true;
                    }
                }
            }
        }

        if (res.msgType && !res.methodTakesMessagePointer)
        {
            // the method is a handler with a generic signature
            // check if the parameter names correspond to the message fields
            const std::string& fieldPrefix = db.getAttributeValue<std::string>(res.msgType, "hk::FieldPrefix", "");
            for (clang::CXXMethodDecl::param_const_iterator mIt = method->param_begin(); mIt != method->param_end(); ++mIt)
            {
                const std::string& fieldName = fieldPrefix + (*mIt)->getNameAsString();
                bool found = false;
                for (MessageFieldIterator fIt(res.msgType); fIt.isValid(); ++fIt)
                {
                    if ((*fIt)->getNameAsString() == fieldName)
                    {
                        found = true;
                        break;
                    }
                }
                if (!found)
                {
                    hkLog.error((*mIt), "Parameter '%s' in '%s' is not valid: message type '%s' has no field called '%s' or the field is not public",
                        (*mIt)->getNameAsString().c_str(), db.getScopedMangledName(method).c_str(),
                        db.getScopedMangledName(res.msgType).c_str(), fieldName.c_str());
                }
            }
        }

        return res;
    }

    std::string generateMethodTypedefDecl(const Database& db, const clang::CXXMethodDecl* m, const clang::CXXRecordDecl* recDecl, const std::string& methodTdefName)
    {
        const std::string& returnType = db.getScopedMangledName(m->getResultType());

        std::vector<std::string> paramTypes;
        for (clang::CXXMethodDecl::param_const_iterator p = m->param_begin(); p != m->param_end(); ++p)
            paramTypes.push_back(db.getScopedMangledName((*p)->getType()));
        const std::string& paramList = joinStrings(paramTypes, ", ");

        return formatString("typedef %s (%s%s* const %s) (%s) %s;",
            returnType.c_str(),
            m->isStatic() ? "" : db.getScopedMangledName(recDecl, Database::NAME_DECLARATION).c_str(),
            m->isStatic() ? "" : "::",
            methodTdefName.c_str(), paramList.c_str(), m->isConst() ? "const" : "");
    }

    std::string generateParamNamesArray(const clang::CXXMethodDecl* method, const std::string& methodName, TypeOut& typeOut)
    {
        if (method->getNumParams() > 0)
        {
            std::string names;
            {
                llvm::raw_string_ostream os(names);
                os << "{ ";
                for (auto it = method->param_begin(); it != method->param_end(); ++it)
                {
                    os << '"' << (*it)->getNameAsString() << "\", ";
                }
                os << "}";
            }
            const std::string& arrayName = methodName + "_params";
            const std::string& arrayType = formatString("const char* const[%d]", method->getNumParams());
            typeOut.addData(arrayName, arrayType, names);
            return formatString("%s%s", typeOut.getDataQualifier().c_str(), arrayName.c_str());
        }
        return "HK_NULL";
    }

    std::string generateMethodAttributes(const Database& db, const clang::CXXMethodDecl* m, const std::string& methodName, TypeOut& typeOut)
    {
        TypeOut auxType(db, m);
        if (generateAttributes(db, m, auxType, auxType))
        {
            TextBuilder::VerbatimSection section;
            auxType.appendDataTo(typeOut.getRawCode());
            auxType.appendTypeContentTo(section);
            typeOut.addData("typeData", "const hkReflect::Detail::TypeData", section.toString(), "meth_" + methodName);
            return formatString("%smeth_%s::typeData", typeOut.getDataQualifier().c_str(), methodName.c_str());
        }
        return "HK_NULL";
    }

    std::string generateDefaultArgs(const Database& db, const clang::CXXMethodDecl* method, const std::string& methodName, TypeOut& typeOut)
    {
        if (method->getNumParams() > 0)
        {
            std::string args;
            bool hasArgs = false;
            {
                llvm::raw_string_ostream os(args);
                os << "{ ";
                for (auto it = method->param_begin(); it != method->param_end(); ++it)
                {
                    if (const clang::Expr* defArg = (*it)->getDefaultArg())
                    {
                        hasArgs = true;

                        // todo.rt this is wrong if the expr is non-const (should be invoked every time the function is
                        // called rather than just once on initialization).
                        std::string defArgStorageSection =
                            formatString("DefaultArg_%s_%s", methodName.c_str(), (*it)->getNameAsString().c_str());

                        // Get argument type.
                        const std::string& defHelper = formatString(
                            "hkReflect::Detail::DefaultArg< HK_REFLECT_RESOLVE(%s) >",
                            db.getScopedMangledName((*it)->getType()).c_str()
                        );
                        const std::string& defArgTypeName = formatString("const %s%s::Type",
                            typeOut.hasTemplateParams() ? "typename " : "", defHelper.c_str());

                        // Get argument value.
                        std::string defArgValue;
                        {
                            llvm::raw_string_ostream vos(defArgValue);
                            vos << defHelper << "::convert(";
                            defArg->printPretty(vos, nullptr, db.astContext.getPrintingPolicy());
                            vos << ")";
                        }

                        typeOut.addData("value", defArgTypeName, defArgValue, defArgStorageSection);

                        os << "&" << defArgStorageSection << "::value, ";
                    }
                    else
                    {
                        os << "HK_NULL, ";
                    }
                }
                os << "}";
            }
            if (hasArgs)
            {
                const std::string& arrayName = methodName + "_defaultArgs";
                const std::string& arrayType = formatString("const void* const[%d]", method->getNumParams());
                typeOut.addData(arrayName, arrayType, args);
                return formatString("%s%s", typeOut.getDataQualifier().c_str(), arrayName.c_str());
            }
        }
        return "HK_NULL";
    }

    std::string generateCallablesArray(
        const Database& db,
        const clang::CXXRecordDecl* recDecl,
        const std::vector<const clang::CXXMethodDecl*>& methods,
        const std::vector<const clang::CXXConstructorDecl*>& ctors,
        const std::vector<const clang::CXXMethodDecl*>& staticFunctions,
        TypeOut& typeOut,
        int& adaptersOut,
        const std::map<const void*, std::string>& nameMap)
    {
        const std::string& cxxNameNoTypename = db.getScopedMangledNameNoPrefix(db.astContext.getTypeDeclType(recDecl));

        TextBuilder::VerbatimSection callables;
        callables.append("\n{");

        // Hack: associate a symbol to every position in the Callable array.
        // This should go away if Callables become Decls (since every Decl has its own named symbol).
        int callableArrayIndex = 0;

        // keep a map of the handled messages for consistency checks
        typedef std::map<const clang::CXXRecordDecl*, const clang::CXXMethodDecl*> MethodFromMessageMap;
        MethodFromMessageMap methodFromMessageMap;

        const char methodInitializerTemplate[] =
            "{\n"
                "{\n"
                    "&%s,\n" // callAdapterName
                    "%s\n" // methodPointerLocation
                "},\n"
                "%s,\n" // sigInitName
                "HK_REFLECT_GET_TYPE(%shkReflect::Detail::RawWhenPointer< %s* >::Type),\n" // typename spec, cxxName
                "%s,\n" // paramNames
                "\"%s\",\n" // methodName
                "%s,\n" // attributes
                "%s,\n" // default args
            "},";

        adaptersOut = 0;

        typedef std::vector<const clang::CXXMethodDecl*>::const_iterator MethodIterator;
        for(MethodIterator it = methods.begin(); it != methods.end(); ++it)
        {
            const clang::CXXMethodDecl* m = *it;
            const std::string& mName = nameMap.find(m)->second;
            const std::string& paramNames = generateParamNamesArray(m, mName, typeOut);

            // check if the method is a message handler
            HandlerData handlerData = getHandlerData(db, m);
            if (handlerData.msgType)
            {
                MethodFromMessageMap::iterator messageIt = methodFromMessageMap.find(handlerData.msgType);
                if (messageIt == methodFromMessageMap.end())
                {
                    methodFromMessageMap[handlerData.msgType] = m;
                }
                else
                {
                    hkLog.error(m, "Class '%s' handles message '%s' with more than one method ('%s', '%s')",
                        cxxNameNoTypename.c_str(), db.getScopedMangledName(handlerData.msgType).c_str(),
                        mName.c_str(), nameMap.find(messageIt->second)->second.c_str());
                }
            }

            const std::string& methodAttrs = generateMethodAttributes(db, m, mName, typeOut);
            const std::string& defaultArgs = generateDefaultArgs(db, m, mName, typeOut);

            // check if this method is a handler with a generic signature
            if (handlerData.msgType && !handlerData.methodTakesMessagePointer)
            {
                // generate a Callable using a custom invoker
                const std::string& fieldPrefix = db.getAttributeValue<std::string>(recDecl, "hk::FieldPrefix");
                const std::string& msgName = db.getScopedMangledName(handlerData.msgType);

                std::vector<std::string> args;
                for(clang::CXXMethodDecl::param_const_iterator p = m->param_begin(); p != m->param_end(); ++p)
                {
                    args.push_back("(*reinterpret_cast<" + msgName + "**>(packedArgs))->" + fieldPrefix + (*p)->getNameAsString());
                }

                const std::string& callAdapterName = "handlerInvoker_" + pyNameFromCxxName(db.getScopedMangledName(handlerData.msgType));

                // the signature will contain just a pointer to the message
                std::vector<std::string> params;
                params.push_back("void");
                params.push_back(msgName + "*");
                const std::string& signature = getSignature(params);

                typeOut.addData(formatString(
                    "static void %s(void* thisPtr, void* packedArgs, void* returnBuffer, const void* invokerData)\n"
                    "{\n"
                        "reinterpret_cast<%s*>(thisPtr)->%s(%s);\n"
                    "}",
                    callAdapterName.c_str(), cxxNameNoTypename.c_str(), m->getNameAsString().c_str(), joinStrings(args, ", ").c_str()));

                callables.append(methodInitializerTemplate,
                    callAdapterName.c_str(),
                    "HK_NULL", 
                    signature.c_str(),
                    typeOut.hasTemplateParams() ? "typename " : "",
                    db.getScopedMangledName(recDecl, Database::NAME_REFLECTION).c_str(),
                    paramNames.c_str(),
                    (mName + "__ADAPTER__").c_str(),
                    "HK_NULL",
                    "HK_NULL"); 
                ++adaptersOut;

                typeOut.addData( formatString( "enum { INDEX_%s__ADAPTER__ = %d }; ", db.getMangledMethodName( m ).c_str(), callableArrayIndex ) );
                ++callableArrayIndex;
            }

            // generate regular Callable
            {
                const std::string& methodTdefName = "t_" + mName;
                const std::string& methodPtrSection = "s_" + mName;

                // add a typedef of the method type
                typeOut.addData(generateMethodTypedefDecl(db, m, recDecl, methodTdefName));

                // add a pointer to the method, which will be used in the shared invoker
                typeOut.addData("ptr", methodTdefName, formatString("&%s::%s", cxxNameNoTypename.c_str(), m->getNameAsString().c_str()), methodPtrSection, TypeOut::FLAG_QUALIFY_TYPE);

                callables.append(methodInitializerTemplate,
                    getSharedInvokerFunc(db, recDecl, m, true).c_str(),
                    formatString("(const void*)&%s::ptr", methodPtrSection.c_str()).c_str(),
                    getSignature(db, m).c_str(),
                    typeOut.hasTemplateParams() ? "typename " : "",
                    db.getScopedMangledName(recDecl, Database::NAME_REFLECTION).c_str(),
                    paramNames.c_str(),
                    mName.c_str(),
                    methodAttrs.c_str(),
                    defaultArgs.c_str() );

                typeOut.addData( formatString( "enum { INDEX_%s = %d }; ", db.getMangledMethodName( m ).c_str(), callableArrayIndex ) );
                ++callableArrayIndex;
            }
        }

        // warn if the class handles a message type and its parent with different methods
        for (MethodFromMessageMap::iterator messageIt = methodFromMessageMap.begin();
            messageIt != methodFromMessageMap.end(); ++messageIt)
        {
            const clang::CXXRecordDecl* current = messageIt->first;
            while (current->bases_begin() != current->bases_end())
            {
                const clang::CXXRecordDecl* parent = getCXXRecordFrom(current->bases_begin()->getType());
                MethodFromMessageMap::iterator parentIt = methodFromMessageMap.find(parent);
                if (parentIt != methodFromMessageMap.end())
                {
                    const std::string& msgName = db.getScopedMangledName(messageIt->first).c_str();
                    const std::string& parentMsgName = db.getScopedMangledName(parentIt->first).c_str();
                    hkLog.warning(messageIt->second, "Method '%s::%s(%s)' overrides method '%s::%s(%s)' when handling message type '%s'",
                        cxxNameNoTypename.c_str(), nameMap.find(messageIt->second)->second.c_str(), msgName.c_str(),
                        cxxNameNoTypename.c_str(), nameMap.find(parentIt->second)->second.c_str(), parentMsgName.c_str(),
                        msgName.c_str());
                }
                current = parent;
            }
        }

        for(std::vector<const clang::CXXConstructorDecl*>::const_iterator it = ctors.begin(); it != ctors.end(); ++it)
        {
            const clang::CXXConstructorDecl* m = *it;
            const std::string mName = nameMap.find(m)->second;

            std::string ctorInvokerName = formatString("hkReflect::Detail::ConstructorInvoker< %s", db.getScopedMangledName(recDecl).c_str());
            std::vector<std::string> params;
            getFunctionSignatureParamList(db, m, params, false);
            if (!params.empty())
            {
                ctorInvokerName += ", " + joinStrings(params, ", ") + " ";
            }
            ctorInvokerName +=  " >::invoke";

            const std::string& sigInitName = getSignature(db, m, db.astContext.getTypeDeclType(recDecl));

            const char constructorInitializerTemplate[] =
                "{\n"
                    "{\n"
                        "&%s,\n" // ctorInvokerName
                        "HK_NULL\n" // (invokerData)
                    "},\n"
                    "%s,\n" // sigInitName
                    "HK_NULL,\n" // (thisType)
                    "%s,\n" // paramNames
                    "\"%s\",\n" // ctorName
                    "%s,\n" // attributes
                    "%s,\n" // default args
                "},";

            callables.append( constructorInitializerTemplate,
                ctorInvokerName.c_str(),
                sigInitName.c_str(),
                generateParamNamesArray(m, mName, typeOut).c_str(),
                mName.c_str(),
                generateMethodAttributes(db, m, mName, typeOut).c_str(),
                generateDefaultArgs(db, m, mName, typeOut).c_str() );

            typeOut.addData( formatString( "enum { INDEX_%s = %d }; ", db.getMangledMethodName( m ).c_str(), callableArrayIndex ) );
            ++callableArrayIndex;
        }

        for(MethodIterator it = staticFunctions.begin(); it != staticFunctions.end(); ++it)
        {
            const clang::CXXMethodDecl* m = *it;
            const std::string mName = nameMap.find(m)->second;

            const std::string& invokerName =  getSharedInvokerFunc(db, recDecl, m, false);
            const std::string& sigInitName = getSignature(db, m);

            // add a typedef of the method type
            const std::string& methodTdefName = "t_" + mName;
            typeOut.addData(generateMethodTypedefDecl(db, m, recDecl, methodTdefName));

            const char staticFunctionsTemplate[] =
                "{\n"
                    "{\n"
                        "&%s,\n"  // invokername
                        "(const void*)(%s)&%s::%s\n"  // (methodTdefName)className :: methodName
                    "},\n"
                    "%s,\n"  // sigInitName
                    "HK_NULL,\n"  // (thisType)
                    "%s,\n" // paramNames
                    "\"%s\",\n"  // functionName
                    "%s,\n" // attributes
                    "%s,\n" // default args
                "},";

            callables.append(staticFunctionsTemplate,
                invokerName.c_str(),
                methodTdefName.c_str(), cxxNameNoTypename.c_str(), m->getNameAsString().c_str(),
                sigInitName.c_str(),
                generateParamNamesArray(m, mName, typeOut).c_str(),
                mName.c_str(),
                generateMethodAttributes(db, m, mName, typeOut).c_str(),
                generateDefaultArgs(db, m, mName, typeOut).c_str());

            typeOut.addData( formatString( "enum { INDEX_%s_adapter = %d }; ", db.getMangledMethodName( m ).c_str(), callableArrayIndex ) );
            ++callableArrayIndex;
        }

        callables.append("}");

        return callables.toString();
    }

    // Is the string usable as a C identifier, and thus a valid name for reflection.
    bool isValidCIdentifier(const std::string& name)
    {
        if ( name.length() == 0 )
        {
            return false;
        }
        const char& fc = name[0];
        if ( fc >= '0' && fc <= '9' )
        {
            return false;
        }
        for (unsigned int i = 0; i < name.length(); ++i)
        {
            const char& c = name[i];
            if ( !( ( c >= 'a' && c <= 'z' ) || ( c >= 'A' && c <= 'Z' ) || ( c >= '0' && c <= '9' ) || ( c == '_' ) ) )
            {
                return false;
            }
        }
        return true;
    }

    // Check that a method has a usable reflected name, and add it to the name map.
    // Return false if the name cannot be used.
    template<typename MethodOrCtorDecl>
    bool checkNameAndAddToMap(const Database& db, const MethodOrCtorDecl* method, std::map<const void*, std::string>& nameMap, std::set<std::string>& nameSet)
    {
        const bool isConstructor = clang::dyn_cast<clang::CXXConstructorDecl>(method) != 0;

        const std::string oldName = isConstructor ? "new" : method->getNameAsString();
        const std::string logName = isConstructor ? "constructor" : "element " + method->getNameAsString();

        std::string newName = db.getAttributeValue<std::string>(method, "hk::Rename");
        if(!newName.empty())
        {
            if(nameSet.count(newName))
            {
                hkLog.error(method, "Cannot rename \"%s\" to \"%s\" because it duplicates another name.", logName.c_str(), newName.c_str());
                return false;
            }
            else if (!isValidCIdentifier(newName))
            {
                hkLog.error(method, "Cannot rename \"%s\" to \"%s\" because the latter is not a valid identifier.", logName.c_str(), newName.c_str());
                return false;
            }
        }
        else if (!isValidCIdentifier(oldName))
        {
            // It's an operator.
            hkLog.warning(method, "The name \"%s\" cannot be used as a reflected method name. Use the hk::Rename attribute to give it a suitable name.", oldName.c_str());
            return false;
        }
        else
        {
            newName = oldName;
            if(nameSet.count(newName))
            {
                newName = db.getMangledMethodName(method);
                hkLog.debug(method, "The overloaded %s has been automatically given the name %s to avoid duplicates. We recommend you use the hk::Rename attribute.",
                    logName.c_str(), newName.c_str());
            }
        }

        nameSet.insert(newName);
        nameMap[method] = newName;

        return true;
    }

    void generateFunctionsPod(const Database& db, const clang::CXXRecordDecl* recDecl, TypeOut& output)
    {
        std::vector<const clang::CXXMethodDecl*> methods;
        std::vector<const clang::CXXConstructorDecl*> ctors;
        std::vector<const clang::CXXMethodDecl*> staticFunctions;

        std::set<std::string> nameSet;
        std::map<const void*, std::string> nameMap;

        for(clang::CXXRecordDecl::method_iterator it = recDecl->method_begin(); it != recDecl->method_end(); ++it)
        {
            const clang::CXXMethodDecl* method = *it;

            if(!method->isImplicit() && db.isReflected(method))
            {
                if(const clang::CXXConstructorDecl* c = clang::dyn_cast<clang::CXXConstructorDecl>(method))
                {
                    if ( checkNameAndAddToMap(db, c, nameMap, nameSet) )
                    {
                        ctors.push_back(c);
                    }
                }
                else if(!clang::isa<clang::CXXDestructorDecl>(method))
                {
                    if ( checkNameAndAddToMap(db, method, nameMap, nameSet) )
                    {
                        if(method->isStatic())
                            staticFunctions.push_back(method);
                        else
                            methods.push_back(method);
                    }
                }
            }
        }

        const bool hasReflectedMethods = !methods.empty() || !ctors.empty() || !staticFunctions.empty();
        if(!hasReflectedMethods)
            return;

        int handlerAdapters;
        const std::string& callablesArray = generateCallablesArray(db, recDecl, methods, ctors, staticFunctions, output, handlerAdapters, nameMap);
        output.addData("values", "const hkReflect::Detail::NamedCallablePod[]", callablesArray, "Callables");

        const char* functionTemplate = "{\n"
            "Callables::values,\n"
            "%d, // num methods\n"
            "%d, // num constructors\n"
            "%d, // num functions\n"
            "}";

        output.addOpt(recDecl->getLocation(), "Opt::FUNCTIONS", "functions", "const hkReflect::Detail::FunctionsPod",
            formatString(functionTemplate, int(methods.size()) + handlerAdapters, int(ctors.size()), int(staticFunctions.size())),
            "", TypeOut::FLAG_IS_CODE);
    }

    void generateDynamicType(const Database& db, const clang::CXXRecordDecl* recDecl, const std::string& cxxName, TypeOut& output)
    {
        if(db.getAttributeValueInHierarchy(recDecl, "hk::GenerateExactTypeMethod", false))
        {
            output.getRawCode().append(output.getTemplateHeader());
            output.getRawCode().append("hkReflect::Detail::AddrAndType %s::getExactType() const\n{", cxxName.c_str());
            // todo.rt we should check the object vtable so that we detect this issue even with a concrete parent
            if(isAbstract(recDecl))
            {
                output.getRawCode().append(
                    "HK_WARN(0x34fdd456, \"Instances of a non-reflected class inheriting from %s have been passed to "
                    "reflection and will be treated as instances of their reflected ancestor. Mark subclasses of %s as "
                    "Reflect or NoReflect to suppress this warning.\");",
                    cxxName.c_str(), cxxName.c_str());
            }
            output.getRawCode().append("return hkReflect::Detail::AddrAndType(this, %s);\n}",
                db.getReflectedTypeAddress(recDecl).c_str());
        }
    }

    bool canBeImplicit(const Database& db, const clang::Decl* decl)
    {
        if (auto recDecl = clang::dyn_cast<clang::CXXRecordDecl>(decl))
        {
            // The method can be implicit only if parent and fields are all reflected.
            const Attribute& details = db.getReflectDetails(recDecl);
            if (!details["parents"]->as<bool>() || !details["fields"]->as<bool>())
            {
                return false;
            }

            for (auto fieldIt = recDecl->field_begin(); fieldIt != recDecl->field_end(); ++fieldIt)
            {
                if (!db.getAttributeValue<bool>(*fieldIt, "hk::Reflect", true) ||
                    db.getAttributeValue<bool>(*fieldIt, "hk::OpaqueType", false) ||
                    db.getAttribute(*fieldIt, "hk::Type"))
                {
                    return false;
                }
            }
        }
        return true;
    }

    void generateSpecialMethodOptional(
        const Database& db,
        const clang::Decl* decl,
        SpecialMethod method,
        MethodType type,
        const std::string& cxxName,
        TypeOut& output)
    {
        const SpecialMethodsData& smData = getSpecialMethodsData(method);
        const char* opt = smData.optional;

        const int flags = ((method == DESTRUCTOR) || (method == DEF_CONSTRUCTOR)) ? TypeOut::FLAG_NEEDED_FOR_CONSTRUCTION : TypeOut::FLAG_IS_CODE;

        switch (type)
        {
            case METHOD_UNAVAILABLE:
            {
                if (method == DESTRUCTOR)
                {
                    // The destructor should always be available.
                    output.addOpt(decl->getLocation(), opt, "&hkReflect::Detail::assertOnInvalidDestruction", flags);
                }
                // no optional if method is unavailable
                break;
            }
            case METHOD_UNKNOWN:
            {
                // hk::FixupReflectedMethods forces the Unknown optional even if the class is not fully reflected.
                if (db.getAttribute(decl, "hk::FixupReflectedMethods") || canBeImplicit(db, decl))
                {
                    // Assume that the destructor is always available.
                    const char* methodImpl = (method == DESTRUCTOR) ? "Implicit" : "Unknown";
                    output.addOpt(decl->getLocation(), opt, formatString("&hkReflect::Detail::%s< hkReflect::%s >::func", methodImpl, smData.optional), flags);
                    break;
                }
                else
                {
                    // Use the native wrapper.
                    output.addOpt(decl->getLocation(), opt, formatString("&hkReflect::Detail::ExplicitWrapper< hkReflect::%s, %s >::func", smData.optional, cxxName.c_str()), flags);
                }
            }
            case METHOD_IMPLICIT:
            {
                if (canBeImplicit(db, decl))
                {
                    output.addOpt(decl->getLocation(), opt, formatString("&hkReflect::Detail::Implicit< hkReflect::%s >::func", smData.optional), flags);
                    break;
                }
                // Fall through and use the native wrapper.
            }
            case METHOD_EXPLICIT:
            {
                output.addOpt(decl->getLocation(), opt, formatString("&hkReflect::Detail::ExplicitWrapper< hkReflect::%s, %s >::func", smData.optional, cxxName.c_str()), flags);
                break;
            }
            case METHOD_TRIVIAL:
            {
                if (method != DESTRUCTOR)
                {
                    output.addOpt(decl->getLocation(), opt, formatString("&hkReflect::Detail::Trivial< hkReflect::%s >::func", smData.optional), flags);
                }
                // no optional if destructor is trivial
                break;
            }
            default: { assert(false); }
        }
    }

    MethodType generateRecordSpecialMethod(
        const Database& db,
        const clang::CXXRecordDecl* recDecl,
        SpecialMethod method,
        const std::string& cxxName,
        TypeOut& output)
    {
        MethodType methodType = _METHOD_COUNT;
        if (method != DEF_CONSTRUCTOR && method != COPY_CONSTRUCTOR)
        {
            // If the method is virtual, the method must be explicit.
            const SpecialMethodsData& smData = getSpecialMethodsData(method);

            // Look for the method decl.
            const clang::CXXRecordDecl* curr = recDecl;
            const clang::CXXMethodDecl* methodDecl = smData.findMethodFunc(curr);

            if (methodDecl == nullptr || methodDecl->getAccess() == clang::AS_public)
            {
                while (methodDecl == nullptr && curr->getNumBases() != 0)
                {
                    curr = getCXXRecordFrom(curr->bases_begin()->getType());
                    if (curr == nullptr) { break; } 
                    methodDecl = smData.findMethodFunc(curr);
                }

                if (methodDecl != nullptr && methodDecl->isVirtual())
                {
                    methodType = METHOD_EXPLICIT;
                }
                // todo.nt6 this should always use the wrapper of the method in the virtual root to save memory
            }
        }

        if (methodType == _METHOD_COUNT)
        {
            methodType = getMethodType(db, recDecl, method);
        }

        generateSpecialMethodOptional(db, recDecl, method, methodType, cxxName, output);
        return methodType;
    }

    void generateSpecialMethods( const Database& db, const clang::CXXRecordDecl* recDecl, const std::string& cxxName, TypeOut& output )
    {
        const std::string& resolvedName = output.hasTemplateParams() ? formatString("HK_REFLECT_RESOLVE(%s)", cxxName.c_str()) : cxxName;

        std::string reason;
        if (!hasBypassConstructor(recDecl) && needsBypassConstructor(db, recDecl, reason))
        {
            hkLog.error(recDecl,
                "'%s' does not declare a bypass constructor. Not declaring a bypass constructor causes the "
                "default constructor of '%s' to be called during clone/load instead of its bypass constructor. "
                "Either add a bypass constructor to '%s' or mark this type as hk::Serialize(false).",
                cxxName.c_str(), reason.c_str(), cxxName.c_str());
        }

        // Produce bypass constructor when requested.
        if(db.getAttributeValue(recDecl, "hk::GenerateBypassCtor", false))
            generateBypassConstructor(db, recDecl, resolvedName, output.getRawCode());

        if (!isAbstract(recDecl))
        {
            // Default constructor
            MethodType defCtorType = generateRecordSpecialMethod(db, recDecl, DEF_CONSTRUCTOR, resolvedName, output);

            if( db.isSerializable(recDecl) )
            {
                // Generate the bypass constructor optional.
                if ( hasBypassConstructor(recDecl) )
                {
                    // Use the explicit bypass constructor.
                    output.addOpt(recDecl->getLocation(), "Opt::REFLECT_CONSTRUCTOR", formatString("hkReflect::Detail::ReflectConstructionWrapper< %s >::func", resolvedName.c_str()).c_str(), TypeOut::FLAG_NEEDED_FOR_CONSTRUCTION);
                }
                else if(defCtorType == METHOD_UNAVAILABLE)
                {
                    // Must have a default constructor.
                    hkLog.error(recDecl, "No constructor available for serializable type '%s'. Either add a default constructor, a bypass constructor or mark this type as hk::Serialize(false).", cxxName.c_str());
                }
            }

            // Copy constructor
            generateRecordSpecialMethod(db, recDecl, COPY_CONSTRUCTOR, resolvedName, output);

            // Destructor
            generateRecordSpecialMethod(db, recDecl, DESTRUCTOR, resolvedName, output);
        }

        // Copy assignment
        generateRecordSpecialMethod(db, recDecl, COPY_ASSIGNMENT, resolvedName, output);
    }

    // Generate the sequence of PODs common to both template and non-template record reflections.
    void generateRecordReflectionCommon(const Database& db, const clang::CXXRecordDecl* recDecl, TypeOut& output, bool& hasFunctionRefl)
    {
        checkIsDefinition(recDecl);
        const std::string& cxxName = db.getScopedMangledName(recDecl);
        std::string declName = db.getScopedMangledName(recDecl, Database::NAME_DECLARATION);

        // Add name.
        output.addOpt(recDecl->getLocation(), "Opt::NAME", "\"" + db.getAttributeValue<std::string>(recDecl, "hk::Rename", recDecl->getQualifiedNameAsString()) + "\"");

        // Add attributes on type.
        generateAttributes(db, recDecl, output, output);

        // Mark this type with a flag if it defines an attribute
        if(db.getAttributeDecl(recDecl->getQualifiedNameAsString()))
        {
            output.addFlag(recDecl->getLocation(), "hkReflect::Type::TYPE_IS_ATTRIBUTE");
        }

        // Early out if the type is reflected as a typedef of another type.
        if(const Optional<Attribute>& reflectAs = db.getAttributeValue< Optional<Attribute> >(recDecl, "hk::ReflectAsTypedef"))
        {
            const Attribute& attr = *reflectAs;
            output.setParent( "HK_REFLECT_GET_TYPE( " + attr.get("type") + " )");
            return;
        }

        const Attribute& details = db.getReflectDetails(recDecl);

        bool ignoreParents = !details["parents"]->as<bool>();


        std::vector< std::pair<std::string, std::string> > fieldsWithValidation;

        if(const Optional<Attribute>& reflectAs = db.getAttributeValue< Optional<Attribute> >(recDecl, "hk::ReflectAs"))
        {
            // Reflect as a record.
            if( details["fields"]->as<bool>() )
            {
                if( details["parents"]->as<bool>() )
                {
                    generateFieldsAndProperties(db, recDecl, recDecl, cxxName, output, fieldsWithValidation, false, false);
                }
                else
                {
                    // Some types don't have their true inheritance reflected because we only allow
                    // inheritance on record types. (e.g hkArrayBase is ignored.)
                    // Flatten their fields into our reflection instead
                    for( auto cur = recDecl; cur; /**/ )
                    {
                        if( cur->field_empty() )
                        {
                            cur = cur->getNumBases()
                                ? getCXXRecordFrom(cur->bases_begin()->getType())
                                : nullptr;
                        }
                        else
                        {
                            checkIsDefinition(cur);
                            generateFieldsAndProperties(db, recDecl, cur, cxxName, output, fieldsWithValidation, true, false);
                            break;
                        }
                    }
                }
            }

            // Use the ReflectAs attribute to determine format.
            const Attribute& attr = *reflectAs;
            output.setFormat(attr.getLocation(), attr.get("format"));
            if(attr["subtype"])
            {
                auto subtype = attr["subtype"]->as<std::string>();
                if ( subtype == "0" || subtype == "nullptr")
                {
                    // todo.rt do we need to add a null opt or could we just skip it?
                    output.addOpt(attr.getLocation(), "Opt::SUBTYPE", "HK_NULL");
                }
                else
                {
                    output.addOpt(attr.getLocation(), "Opt::SUBTYPE", formatString("HK_REFLECT_GET_TYPE( %s )", subtype.c_str()));
                }
            }
            output.addOpt(attr.getLocation(), "Opt::IMPL", *attr["impl"]);
        }
        else
        {
            // Reflect as a record.
            output.setFormat(recDecl->getLocation(), "hkReflect::Format::OfRecord::Value");
            generateFieldsAndProperties(db, recDecl, recDecl, cxxName, output, fieldsWithValidation, false, false);

            if (const Optional<std::string> implAttr = db.getAttributeValue< Optional<std::string> >(recDecl, "hk::ReflectionImpl"))
            {
                output.addOpt(recDecl->getLocation(), "Opt::IMPL", *implAttr);
            }
            else if (recDecl->getNumBases() == 0 || ignoreParents)
            {
                output.addOpt(recDecl->getLocation(), "Opt::IMPL", "&hkReflect::Detail::HavokRecordImpl::s_instance");
            }

            if(details["inheritance"]->as<bool>() && !hasFields(recDecl))
                output.addFlag(recDecl->getLocation(), "hkReflect::Type::TYPE_INTERFACE");
        }


        std::string alignOf = isAbstract(recDecl) ? "hkReflect::Detail::ALIGN_UNSET" : "HK_ALIGN_OF(" + cxxName + ")";
        std::string sizeOf = "sizeof(" + cxxName + ")";
        std::string reqAlign = getRequestedAlignment(db, recDecl);
        output.addSizeAlign( recDecl->getLocation(), sizeOf, alignOf, reqAlign );

        bool recIsAbstract = isAbstract(recDecl);
        if(recIsAbstract)
            output.addFlag(recDecl->getLocation(), "hkReflect::Type::TYPE_ABSTRACT");

        if(recIsAbstract || !db.isSerializable(recDecl))
            output.addFlag(recDecl->getLocation(), "hkReflect::Type::TYPE_NOT_SERIALIZABLE");
        if(!db.getAttributeValue(recDecl, "hk::Retargetable", true))
            output.addFlag(recDecl->getLocation(), "hkReflect::Type::TYPE_NOT_RETARGETABLE");

        // Add parent and interfaces.
        if(!ignoreParents && recDecl->bases_begin() != recDecl->bases_end())
        {
            // Add the primary parent and check that it is reflected.
            {
                const clang::QualType parent = recDecl->bases_begin()->getType();
                const std::string& addrParent = db.getReflectedTypeAddress(parent);
                // An unreflected parent is caught by the compiler in the IsReflectedCheck, but we can give a nicer error here.
                const clang::CXXRecordDecl* parentRecDecl = parent->getAsCXXRecordDecl();
                // parentRecDecl may be NULL if the parent is a template argument. We shouldn't issue an error in that case.
                if (parentRecDecl && !db.isReflected(parentRecDecl))
                {
                    const std::string& parentCxxName = db.getScopedMangledName(parentRecDecl);
                    hkLog.error(recDecl, "Class %s is reflected, but its parent class %s is not. "
                        "Add the attribute \"hk::ReflectDetails(parents=false)\" to make reflection ignore parents "
                        "(all contents of parents will be ignored and %s will not be retargetable).",
                        cxxName.c_str(), parentCxxName.c_str(), cxxName.c_str());
                }
                output.setParent(addrParent);
            }

            std::vector< std::pair<std::string, std::string> > interfaces;
            // Add the interfaces skipping the first "main" parent
            const Attribute& reflectDetails = db.getReflectDetails(recDecl);
            for(clang::CXXRecordDecl::base_class_const_iterator it = recDecl->bases_begin() + 1; it != recDecl->bases_end(); ++it)
            {
                const std::string iCxxName = db.getScopedMangledName(it->getType());

                bool reflected = true;
                if(const clang::CXXRecordDecl* baseRecDecl = it->getType()->getAsCXXRecordDecl())
                {
                    const bool allowMultipleInheritance = reflectDetails["allowMultipleDataInheritance"]->as<bool>(); //todo.merge Remove when hkeBreakableBodyNode is sorted out
                    reflected = db.isReflected(baseRecDecl);

                    if(allowMultipleInheritance == false && !db.isReflected(baseRecDecl))
                    {
                        hkLog.error(recDecl, "Class %s is reflected, but its parent class %s is not. "
                            "Add the attribute \"hk::ReflectDetails(parents=false)\" to make reflection ignore parents "
                            "(all contents of parents will be ignored and %s will not be retargetable).",
                            cxxName.c_str(), iCxxName.c_str(), cxxName.c_str());
                    }
                    // Ensure that the interface is actually an interface (ie. no data).
                    // This is only really a problem for serialization, we could therefore allow it for "Serialize false" classes.
                    if(hasFields(baseRecDecl) || db.getAttributeValue< Optional<Attribute> >(baseRecDecl, "hk::ReflectAs"))
                    {
                        if(allowMultipleInheritance == false)
                        {
                            hkLog.error(recDecl, "Parent %s of type %s is not the primary base and must "
                                        "be an interface (reflected as Record, no fields).", iCxxName.c_str(), cxxName.c_str());
                        }
                    }
                }

                if(reflected)
                {
                    interfaces.push_back(std::make_pair(db.getReflectedTypeAddress(it->getType()), formatString("HK_OFFSET_TO_BASE((%s*), (%s*))", cxxName.c_str(), iCxxName.c_str())));
                }
            }

            if(!interfaces.empty())
            {
                std::string res = formatString("\n{\n%i,\n", interfaces.size());
                if(interfaces.size())
                {
                    res += "{\n";
                    for(auto it = interfaces.begin(); it != interfaces.end(); ++it)
                    {
                        res += formatString("{ %s, %s },\n", it->first.c_str(), it->second.c_str());
                    }
                    res += "}\n";
                }
                res += "}";

                output.addOptCast(recDecl->getLocation(), "Opt::INTERFACES", "Interfaces",
                    formatString("const hkReflect::Detail::FixedArrayStorage<hkUlong, hkReflect::Detail::Interface, %i>", interfaces.size()),
                    "reinterpret_cast<const hkReflect::Detail::InterfaceArray*>",
                    res );
            }
        }

        if(const Optional<std::string>& attrVersion = db.getAttributeValue< Optional<std::string> >(recDecl, "hk::Version"))
            output.addOpt(recDecl->getLocation(), "Opt::VERSION", *attrVersion);

        generateDefault(db, recDecl, cxxName, output);

        {
            TextBuilder::Section& out = output.getRawCode();
            auto attrs = db.getAllStaticAttributes(recDecl, "hk::EmitQualifiedData");
            for( auto& attr : attrs )
            {
                const std::string& tname = attr["type"]->as<std::string>();
                const std::string& vname = attr["name"]->as<std::string>();
                const std::string& value = attr.get("value");
                out.append("%s %s::%s%s%s;", tname.c_str(), cxxName.c_str(), vname.c_str(), value.size()?" = ":"", value.c_str());
            }
        }

        if (details["specials"]->as<bool>())
        {
            // add allocation impl to hierarchy roots unless the user asked us not to
            if(!db.getAttributeValue<bool>(recDecl, "hk::Allocatable", true))
            {
                output.addOpt(recDecl->getLocation(), "Opt::ALLOC_IMPL", "HK_NULL");
            }
            else if (recDecl->getNumBases() == 0 || ignoreParents)
            {
                output.addOpt(recDecl->getLocation(), "Opt::ALLOC_IMPL", "&hkReflect::Detail::HeapAllocImpl::s_instance");
            }

            generateSpecialMethods(db, recDecl, cxxName, output);
        }

        if (details["inheritance"]->as<bool>())
        {
            if( db.getAttributeValue<bool>(recDecl, "hk::RegisterType", true) )
            {
                output.addOpt(recDecl->getLocation(), "Opt::INHERITANCE", "0");
            }

            if(const Optional<std::string>& exactTypeOf = db.getAttributeValue< Optional<std::string> >(recDecl, "hk::ExactTypeFunction"))
            {
                const std::string& eto = formatString(
                    "(hkReflect::Detail::ExactTypeFunction)"
                    "static_cast< hkReflect::Detail::AddrAndType(*)(const %s*) >( %s )",
                    cxxName.c_str(), exactTypeOf->c_str() );
                output.addOpt(recDecl->getLocation(), "Opt::EXACT_TYPE", eto, TypeOut::FLAG_IS_CODE);
            }

            generateDynamicType(db, recDecl, cxxName, output);
        }

        generateFunctionsPod(db, recDecl, output);

        addExtraOptsAndFlags(db, recDecl, output);

        if( const clang::CXXMethodDecl* method = hasAfterReflectNew(recDecl) )
        {
            output.addData(formatString(
                "#if !defined(HK_REFLECT_TYPE_DATA_ONLY) || defined(HK_REFLECT_TYPE_CONSTRUCTABLE)\n"
                "static void afterReflectNew(void* thisPtr)\n"
                "{\n"
                "reinterpret_cast<%s*>(thisPtr)->afterReflectNew();\n"
                "}\n"
                "#endif",
                cxxName.c_str()));

            const std::string& resolvedName = output.hasTemplateParams() ?
                formatString("hkReflect::Typedef::Resolve< %s >::Type", cxxName.c_str()) : cxxName;

            output.addOpt(method->getLocation(), "Opt::AFTER_REFLECT_NEW", "&" + resolvedName + "::_Auto::afterReflectNew", TypeOut::FLAG_NEEDED_FOR_CONSTRUCTION );
        }

        generateValidationChecks(db, recDecl, cxxName, output, fieldsWithValidation);

        // checks if the record has function reflection
        hasFunctionRefl |= output.hasOpt("Opt::FUNCTIONS");
    }


    void getTemplateParamsAndArgs(const clang::ClassTemplateDecl* tplDecl, const clang::ClassTemplateDecl*& tplOut,
        llvm::ArrayRef<clang::NamedDecl*>& paramsOut, llvm::ArrayRef<clang::TemplateArgument>& argsOut)
    {
        tplOut = tplDecl;
        paramsOut = tplDecl->getTemplateParameters()->asArray();
        argsOut = llvm::ArrayRef<clang::TemplateArgument>();
    }

    void getTemplateParamsAndArgs(const clang::ClassTemplateSpecializationDecl* specDecl,
        const clang::ClassTemplateDecl*& tplOut,
        llvm::ArrayRef<clang::NamedDecl*>& paramsOut, llvm::ArrayRef<clang::TemplateArgument>& argsOut)
    {
        tplOut = getTemplateDecl(specDecl);
        paramsOut = tplOut->getTemplateParameters()->asArray();
        argsOut = specDecl->getTemplateArgs().asArray();
    }

    // returns (pod type, pod value)
    std::pair<std::string, std::string> getTemplateParametersPodString(const Database& db,
        const clang::NamedDecl* decl, const clang::ClassTemplateDecl* tplDecl,
        llvm::ArrayRef<clang::NamedDecl*>& params, llvm::ArrayRef<clang::TemplateArgument>& args)
    {

        std::string paramStr;
        int numTypeParams = 0;
        int numValueParams = 0;

        {
            llvm::raw_string_ostream paramStream(paramStr);
            paramStream << "{ ";

            const Optional<Attribute>& ignoredParams = db.getAttributeValue< Optional<Attribute> >(tplDecl, "hk::IgnoreTemplateParams");

            for (unsigned i = 0; i < params.size(); ++i)
            {
                const clang::NamedDecl* param = params[i];

                // Here we need to check if this parameter is in the IgnoreTemplateParam list
                std::string paramName = param->getNameAsString();

                if (ignoredParams)
                {
                    bool skip = false;
                    for (auto it = ignoredParams->posBegin(), end = ignoredParams->posEnd(); it != end; it++)
                    {
                        if (paramName.compare(*it) == 0)
                        {
                            skip = true;
                            break;
                        }
                    }

                    if (skip)
                    {
                        // Yes it was, skip it
                        continue;
                    }
                }

                if (const clang::TemplateTypeParmDecl* typeParam = clang::dyn_cast<clang::TemplateTypeParmDecl>(param))
                {
                    if(numTypeParams || numValueParams)
                    {
                        paramStream << ", ";
                    }

                    paramStream << "{ hkUlong(HK_REFLECT_GET_TYPE( ";
                    if (args.empty() || args[i].getAsType()->isTemplateTypeParmType())
                    {
                        paramStream << *typeParam; 
                    }
                    else
                    {
                        paramStream << db.getScopedMangledName(args[i].getAsType(), Database::NAME_REFLECTION);
                    }
                    paramStream << " )), \"t" << typeParam->getName() << "\" }";

                    numTypeParams++;
                }
                else if (const clang::NonTypeTemplateParmDecl* nonTypeParam = clang::dyn_cast<clang::NonTypeTemplateParmDecl>(param))
                {
                    if(numTypeParams || numValueParams)
                    {
                        paramStream << ", ";
                    }

                    clang::IdentifierInfo* name = nonTypeParam->getIdentifier();
                    paramStream << "{ hkUlong(";
                    if (args.empty() || args[i].getKind() == clang::TemplateArgument::Expression)
                    {
                        paramStream << name->getName(); 
                    }
                    else
                    {
                        args[i].print(db.astContext.getPrintingPolicy(), paramStream);
                    }
                    paramStream << "), \"v" << nonTypeParam->getName() << "\" }";

                    numValueParams++;
                }
                else if (clang::dyn_cast<clang::TemplateTemplateParmDecl>(param))
                {
                    throw RuntimeError((param), "Template template parameter are not supported by reflection");
                }
            }
            paramStream << " }";
        }

        std::string ret;
        llvm::raw_string_ostream retStr(ret);
        retStr << "{ ";

        
        retStr << "HK_REFLECT_GET_TYPE(HK_REFLECT_RESOLVE(" << db.getScopedMangledName( decl ) << ")), {";

        if (numTypeParams || numValueParams)
        {
            retStr << "((" << numTypeParams << " + " << numValueParams << ") & 0xffff)";
        }
        else
        {
            retStr << "0";
        }
        if (numTypeParams || numValueParams)
        {
            retStr << ", " << paramStr;
        }
        retStr << "} }";

        retStr.flush();

        const std::string& podType = formatString("const hkReflect::Detail::TemplatePod<%d + %d>", numTypeParams, numValueParams);

        return std::pair<std::string, std::string>(podType, ret);
    }

    void generateTrackerRegNode(const Database& db, const clang::CXXRecordDecl* recDecl, const std::string& cxxName,
        const std::string& featureString, bool generateClassList, TextBuilder& cxxBody, TextBuilder& cxxTracker)
    {
        if (hasTrackedPointers(db, recDecl))
        {
            bool usesOldMacro = findMemberDecl<clang::TypedefDecl>(clang::dyn_cast<clang::CXXRecordDecl>(recDecl), "hkUsesOldMacro") != nullptr;
            const std::string& pyName = pyNameFromCxxName(cxxName);
            cxxBody["append"]
                .append("#if defined(HK_MEMORY_TRACKER_ENABLE) && defined(HK_DETAIL_REFLECT_DEFINITIONS)")
                .append("hkReflect::Detail::TrackerRegNode %s_trackerRegNode"
                "( hkReflect::ReflectionOf< %s >::Holder::typeData, HK_TRACKER_GET_HANDLE_UNCHECKED( %s%s ), false );",
                pyName.c_str(), cxxName.c_str(), cxxName.c_str(), usesOldMacro ? "::_OldMacroHack" : "")
                .append("#endif\n");

            if (generateClassList)
            {
                cxxTracker["registerDecl"].append("extern hkReflect::Detail::TrackerRegNode %s_trackerRegNode;", pyName.c_str());
                if (featureString.size())
                {
                    cxxTracker["registerFunc"].append(featureString);
                }
                cxxTracker["registerFunc"].append("res ^= (hkUlong)&%s_trackerRegNode;", pyName.c_str());
                if (featureString.size())
                {
                    cxxTracker["registerFunc"].append("#endif");
                }
            }
        }
    }

    void generateNonTemplateRecordReflection( const Database& db, const clang::CXXRecordDecl* recDecl,
        TextBuilder& cxxBody, TextBuilder& inlBody, bool& hasFunctionRefl)
    {
        TypeOut curBase(db, recDecl);
        generateRecordReflectionCommon(db, recDecl, curBase, hasFunctionRefl);
        curBase.appendTo(cxxBody, inlBody);
    }

    template<typename T>
    void generateTemplateRecordReflection(const Database& db, const T* tplDecl,
        TextBuilder& cxxBody, TextBuilder& inlBody, bool& hasFunctionRefl)
    {
        // Generate a single template record reflection, in a .inl file to be
        // included by the user just after the template class declaration, at global scope.

        const clang::CXXRecordDecl* recDecl = getCXXRecordFromTemplate(tplDecl);

        TypeOut curBase(db, recDecl);

        generateRecordReflectionCommon(db, recDecl, curBase, hasFunctionRefl);

        const clang::ClassTemplateDecl* origTemplate;
        llvm::ArrayRef<clang::NamedDecl*> params;
        llvm::ArrayRef<clang::TemplateArgument> args;
        getTemplateParamsAndArgs(tplDecl, origTemplate, params, args);

        std::string tplParamPodType, tplParamPodString;
        std::tie(tplParamPodType, tplParamPodString) =
            getTemplateParametersPodString(db, tplDecl, origTemplate, params, args);

        curBase.addOptCast(tplDecl->getLocation(), "Opt::TEMPLATE", "params",
            tplParamPodType,
            "reinterpret_cast<const hkReflect::Detail::TemplateParameterArray*>",
            tplParamPodString);

        curBase.appendTo(cxxBody, inlBody);
    }

    void generateEnum(const Database& db, const clang::EnumDecl* enumDecl, TextBuilder& cxxBody, TextBuilder& inlBody)
    {
        TypeOut base(db, enumDecl);

        const std::string& cxxName = db.getScopedMangledName(enumDecl);
        const std::string& reflectedName = db.getScopedMangledName(enumDecl, Database::NAME_REFLECTION);

        const Attribute& details = db.getReflectDetails(enumDecl);

        base.addOpt(enumDecl->getLocation(), "Opt::NAME", "\"" + enumDecl->getQualifiedNameAsString() + "\"");
        base.setFormat(enumDecl->getLocation(), formatString("hkReflect::Format::OfInt< %s >::Value", cxxName.c_str()));
        base.addSizeAlign( enumDecl->getLocation(), formatString("sizeof(%s)", cxxName.c_str()), formatString("HK_ALIGN_OF(%s)", cxxName.c_str()) );
        base.addOpt(enumDecl->getLocation(), "Opt::IMPL",
            formatString("&hkReflect::Detail::IntImplN< %shkTrait::CorrespondingIntType< %s >::Type >::s_instance",
            base.hasTemplateParams() ? "typename " : "", cxxName.c_str()));

        if (details["inheritance"]->as<bool>() && db.getAttributeValue<bool>(enumDecl, "hk::RegisterType", true))
            base.addOpt(enumDecl->getLocation(), "Opt::INHERITANCE", "0");

        if (details["specials"]->as<bool>())
        {
            base.addOpt(enumDecl->getLocation(), "Opt::ALLOC_IMPL", "&hkReflect::Detail::HeapAllocImpl::s_instance");
            // todo.nt6 special methods
        }

        Presets presets;
        presets.strict = true;

        clang::QualType declType = getTypeFrom(db, enumDecl);
        presets.valueType = db.getScopedMangledName(declType);
        presets.type = db.getReflectedTypeAddress(declType);

        for(clang::EnumDecl::enumerator_iterator it = enumDecl->enumerator_begin(); it != enumDecl->enumerator_end(); ++it)
        {
            Presets::Val value;
            value.name = it->getNameAsString();
            value.value = formatString("%d", int(*it->getInitVal().getRawData()));
            value.attribs = db.getAllRuntimeAttributes(*it);
            presets.values.push_back(value);
        }

        PresetsData presetsData = generatePresetsData(presets, db, enumDecl, base, "");
        const std::string& presetsContents = formatString("{ %s, %s, %d, %s%s, %s%s, %s }",
            presetsData.strict.c_str(), presetsData.type.c_str(), presetsData.count,
            base.getDataQualifier().c_str(), presetsData.names.c_str(),
            base.getDataQualifier().c_str(), presetsData.values.c_str(),
            presetsData.hasAttribs ? (base.getDataQualifier()+ presetsData.attribArrays).c_str() : "0");

        generateSpecialMethodOptional(db, enumDecl, DEF_CONSTRUCTOR, METHOD_TRIVIAL, cxxName, base);
        generateSpecialMethodOptional(db, enumDecl, COPY_CONSTRUCTOR, METHOD_TRIVIAL, cxxName, base);
        generateSpecialMethodOptional(db, enumDecl, COPY_ASSIGNMENT, METHOD_TRIVIAL, cxxName, base);

        std::vector<AttributeVariant> extraAttrs;
        extraAttrs.push_back(AttributeVariant(formatString("&hkReflect::Detail::PresetsOf< %s >::presets", reflectedName.c_str()), "hk::Presets"));
        generateAttributes(db, enumDecl, base, base, extraAttrs);

        generateDefault(db, enumDecl, cxxName, base);

        base.addData(formatString("hkReflect::Detail::PresetsOf< %s >::presets", reflectedName.c_str()),
            "const hk::Presets", presetsContents, "", TypeOut::FLAG_EXTERN_DECL);
        base.appendTo(cxxBody, inlBody);
    }

    void generateTypedef(const Database& db, const clang::TypedefDecl* tdefDecl, TextBuilder& cxxBody, TextBuilder& inlBody)
    {
        TypeOut base(db, tdefDecl);

        const std::string& cxxName = db.getScopedMangledName(tdefDecl);

        std::string parentName = cxxName;
        if (auto customAttr = db.getAttributeValue< Optional<std::string> >(tdefDecl, "hk::ReflectAsTypedef"))
        {
            parentName = *customAttr;
        }
        else if (db.shouldBeExpanded(tdefDecl->getUnderlyingType()))
        {
            // The typedefs in the name must be replaced by their tags.
            parentName = db.getScopedMangledName(tdefDecl->getUnderlyingType(), Database::NAME_REFLECTION);
        }


        base.setParent(db.getReflectedTypeAddress(parentName));
        base.addOpt(tdefDecl->getLocation(), "Opt::NAME", "\"" + tdefDecl->getQualifiedNameAsString() + "\"");
        if (const Optional<std::string>& attrVersion = db.getAttributeValue< Optional<std::string> >(tdefDecl, "hk::Version"))
        {
            base.addOpt(tdefDecl->getLocation(), "Opt::VERSION", *attrVersion);
        }

        generateAttributes(db, tdefDecl, base, base);
        generateDefault(db, tdefDecl, cxxName, base);

        base.appendTo(cxxBody, inlBody);
    }

    void generateTrackerTypeCommon(const Database& db, const clang::CXXRecordDecl* recDecl, TypeOut& trackerType, bool addAnyway)
    {
        // Generate a name opt only when necessary (nested class, old macro).
        if (clang::isa<clang::CXXRecordDecl>(recDecl->getDeclContext()) ||
            clang::isa<clang::NamespaceDecl>(recDecl->getDeclContext()) ||
            findMemberDecl<clang::TypedefDecl>(recDecl, "hkUsesOldMacro"))
        {
            trackerType.addOpt(recDecl->getLocation(), "Opt::NAME", "\"" + recDecl->getQualifiedNameAsString() + "\"");
        }

        if (auto attr = db.getAttributeValue< Optional<Attribute> >(recDecl, "hk::MemoryTracker"))
        {
            const std::string& handler = (*attr)["handler"]->as<std::string>();
            if (handler != "0" || addAnyway)
            {
                trackerType.addData("memoryTracker", "hk::MemoryTracker",
                    formatString("{%s, %s}", (*attr)["opaque"]->as<bool>() ? "true" : "false", handler.c_str()),
                    "MemoryTracker");

                trackerType.addOptCast(recDecl->getLocation(), "Opt::ATTRIBUTES",
                    "attributes",
                    "const hkReflect::Detail::FixedArrayStorage<hkUlong, hkReflect::Detail::AttributeItem, 1>",
                    "reinterpret_cast<const hkReflect::Detail::AttributeArray*>",
                    formatString("\n{\n1, { { &%sMemoryTracker::memoryTracker, HK_REFLECT_GET_TYPE( hk::MemoryTracker ) } }\n}", trackerType.getDataQualifier().c_str()),
                    "Attributes");
            }
        }
    }

    void generateTemplateTrackerType(const Database& db, const clang::CXXRecordDecl* recDecl, TextBuilder& cxxBody, TextBuilder& inlBody)
    {
        
        TrackerTypeOut trackerType(db, recDecl);

        bool trackContent = db.trackContent(recDecl);
        // Generate type description.
        if (auto reflectAs = db.getAttributeValue< Optional<Attribute> >(recDecl, "hk::ReflectAs"))
        {
            // Use the ReflectAs attribute to determine format.
            // Generate subtype and impl even if hk::MemoryTracked(opaque) is specified. They will not be used by
            // automatic exploration but are very convenient when writing manual handlers.
            const Attribute& attr = *reflectAs;
            trackerType.setFormat(attr.getLocation(), attr.get("format"));
            if (attr["subtype"])
            {
                if (attr["subtype"]->as<std::string>() == "0")
                {
                    // todo.rt do we need to add a null opt or could we just skip it?
                    trackerType.addOpt(attr.getLocation(), "Opt::SUBTYPE", "HK_NULL");
                }
                else
                {
                    trackerType.addOpt(attr.getLocation(), "Opt::SUBTYPE",
                        formatString("(const void*)HK_TRACKER_GET_HANDLE( %s )",
                        attr["subtype"]->as<std::string>().c_str()));
                }
            }
            trackerType.addOpt(attr.getLocation(), "Opt::IMPL", *attr["impl"]);
        }
        else
        {
            // Normal record.
            trackerType.setFormat(recDecl->getLocation(), "hkReflect::Format::OfOpaque::Value");
            if (trackContent)
            {
                if (recDecl->bases_begin() != recDecl->bases_end())
                {
                    const clang::QualType parent = recDecl->bases_begin()->getType();
                    const std::string& addrParent = db.getReflectedTypeAddress(parent, true);
                    trackerType.setParent(addrParent);
                }

                std::vector<std::pair<std::string, std::string> > fieldsWithValidation;
                generateFieldsAndProperties(db, recDecl, recDecl, db.getScopedMangledName(recDecl), trackerType, fieldsWithValidation, true, true);
            }
        }

        // Name and attribute.
        generateTrackerTypeCommon(db, recDecl, trackerType, true);

        // Size (does not need alignment).
        const std::string& cxxName = db.getScopedMangledName(recDecl);
        trackerType.addSizeAlign(recDecl->getLocation(), "sizeof(" + cxxName + ")", "1");

        // Append the tracker type.
        inlBody["append"].append("#if defined(HK_MEMORY_TRACKER_ENABLE)");

        // If this is a full specialization it will add code to the cxx, so add the guards there too.
        bool fullSpec = !getTemplateParams(getTemplateDecl(recDecl));
        if (fullSpec)
        {
            cxxBody["append"].append("#if defined(HK_MEMORY_TRACKER_ENABLE)");
        }
        trackerType.appendTo(cxxBody, inlBody);
        if (fullSpec)
        {
            cxxBody["append"].append("#endif");
        }
        inlBody["append"].append("#endif");
    }

    void generateNonTemplateTrackerType(const Database& db, const clang::CXXRecordDecl* recDecl,
        const std::string& featureString, bool generateClassList, TextBuilder& cxxBody, TextBuilder& inlBody)
    {
        const std::string& cxxName = db.getScopedMangledName(recDecl);
        TrackerTypeOut trackerType(db, recDecl);

        if (!isPubliclyAccessible(recDecl))
        {
            
            return;
        }

        bool tracked = db.trackContent(recDecl);

        bool pointersInParent = false;
        if (tracked)
        {
            // Parent.
            
            if (recDecl->getNumBases() > 0)
            {
                clang::QualType parentType = recDecl->bases_begin()->getType();
                auto parent = getCXXRecordFrom(parentType);
                if (db.hasTrackerHandle(parent))
                {
                    const std::string& addrParent = db.getReflectedTypeAddress(parent, true);
                    trackerType.setParent(addrParent);
                    pointersInParent = hasTrackedPointers(db, parentType);
                }
            }

            // Fields.
            std::vector<std::pair<std::string, std::string> > fieldsWithValidation;
            generateFieldsAndProperties(db, recDecl, recDecl, cxxName, trackerType, fieldsWithValidation, true, true);
        }

        generateTrackerTypeCommon(db, recDecl, trackerType, false);

        // Generate a tracker type only if it contains valuable info.
        if (trackerType.hasOpts() || pointersInParent)
        {
            
            if (featureString.size())
            {
                cxxBody["include"].append(featureString);
                cxxBody["append"].append(featureString);
            }

            cxxBody["include"].append(db.getSourceFileName(recDecl));

            if (trackerType.hasOpts())
            {
                trackerType.setFormat(recDecl->getLocation(), "hkReflect::Format::OfOpaque::Value");
                
                trackerType.addSizeAlign(recDecl->getLocation(), "sizeof(" + cxxName + ")", "1");
                trackerType.appendTo(cxxBody, inlBody);
            }
            else if (pointersInParent)
            {
                
                
                trackerType.appendTo(cxxBody, inlBody);
            }


            if (featureString.size())
            {
                cxxBody["include"].append("#endif");
                cxxBody["append"].append("#endif");
            }


            if (generateClassList)
            {
                // Append the reference to the regNode to the list.
                if (featureString.size())
                {
                    cxxBody["registerFunc"].append(featureString.c_str());
                }
                cxxBody["registerFunc"].append("res ^= (hkUlong)&%s_trackerRegNode;", pyNameFromCxxName(cxxName).c_str());

                if (featureString.size())
                {
                    cxxBody["registerFunc"].append("#endif");
                }
            }
        }
    }

    const clang::CXXRecordDecl* hasReflectedParent(const Database& db, const clang::CXXRecordDecl* recDecl)
    {
        for(clang::CXXRecordDecl::base_class_const_iterator it = recDecl->bases_begin(); it != recDecl->bases_end(); ++it)
        {
            if(const clang::CXXRecordDecl* parent = getCXXRecordFrom(it->getType()))
            {
                const Optional<bool>& reflAttr = db.getAttributeValue< Optional<bool> >(parent, "hk::Reflect");

                if(reflAttr)
                {
                    if(*reflAttr)
                        return parent;
                }
                else if(parent != recDecl) // This may happen for template specializations such as template<int N> struct A : A<N+1> {};
                {
                    if(const clang::CXXRecordDecl* reflectedParent = hasReflectedParent(db, parent))
                        return reflectedParent;
                }
            }
        }

        return 0;
    }
}

void checkIfSimilarNameExists(const Database& db, const clang::NamedDecl* decl, std::map<std::string, std::string>& typeNames)
{
    // todo.ntm currently if two types have the same name except for special characters and case
    // their names will clash in the type registry, so we trigger an error if this happens
    // (note: this check will not detect clashing types declared in different projects)
    const std::string& name = db.getScopedMangledName(decl);
    std::string mangledName = name;
    std::transform(mangledName.begin(), mangledName.end(), mangledName.begin(), ::tolower);
    mangledName.erase( std::remove_if(mangledName.begin(), mangledName.end(),
        [&](char c)
        {
            return (c == ':') || (c == '<') || (c == '>') || (c == ',') || (c == ' ');
        }),
        mangledName.end()
    );
    std::map<std::string, std::string>::iterator it = typeNames.find(mangledName);
    if (it != typeNames.end())
    {
        hkLog.error(decl, "Cannot reflect type '%s'; a type named '%s' exists, will be considered as a duplicate",
            name.c_str(), it->second.c_str());
    }
    else
    {
        typeNames[mangledName] = name;
    }
}

// Exported, this should be in a utility
std::string getFeatureString(const Database& db, const clang::Decl* decl, TkbmsRegistry::Entry tkbmsEntry, const std::string& filename)
{
    std::vector<std::string> tkbmsPlatforms = tkbmsEntry.platform;
    std::vector<std::string> tkbmsProducts = tkbmsEntry.product;

    // Exclude "REFLECT" from the list of platforms.
    auto reflectPos = std::find_if(tkbmsPlatforms.begin(), tkbmsPlatforms.end(),
        [](const std::string& p) { return p == "REFLECT" || p == "!REFLECT"; });
    if (reflectPos != tkbmsPlatforms.end())
    {
        tkbmsPlatforms.erase(reflectPos);
    }

    // Exclude COMMON and VISION product
    bool foundProductToExclude = true;
    while(foundProductToExclude)
    {
        auto commonProductPos = std::find_if(tkbmsProducts.begin(), tkbmsProducts.end(),
            [](const std::string& p) { return p == "COMMON" || p == "VISION"; });
        if (commonProductPos != tkbmsProducts.end())
        {
            tkbmsProducts.erase(commonProductPos);
        }
        else
        {
            foundProductToExclude = false;
        }
    }
    std::string featureStr;
    {
        if (const Optional<std::string>& feature = db.getAttributeValue< Optional<std::string> >(decl, "hk::Feature"))
        {
            featureStr = *feature;
            // strip quotes if present
            if (featureStr.length() >= 2 && featureStr.front() == '"' && featureStr.back() == '"')
            {
                featureStr = featureStr.substr(1, featureStr.length() - 2);
            }
            featureStr = "!defined(HK_EXCLUDE_FEATURE_" + featureStr + ")";
            // If one defines this, the others must be the same anyway
        }
    }

    bool containsAll = false;
    bool containsNegative = false;
    bool containsPositive = false;

    for (const std::string& platform : tkbmsPlatforms)
    {
        if (platform == "ALL")
        {
            containsAll = true;
        }
        else if (platform.substr(0, 1) == "!")
        {
            containsNegative = true;
        }
        else if (platform != "REFLECT")
        {
            containsPositive = true;
        }
    }

    if (containsAll && containsPositive)
    {
        hkLog.errorWithoutDatabase(0, filename.c_str(), "File contains both 'ALL' and positive platform TKBMS");
        return "";
    }

    if (!containsAll && !containsPositive)
    {
        hkLog.errorWithoutDatabase(0, filename.c_str(), "File must contain either 'ALL' or at least one positive platform TKBMS");
        return "";
    }

    if (containsAll && containsNegative)
    {
        // ALL !X !Y
        int numAdded = 0;
        for (const std::string& platform : tkbmsPlatforms)
        {
            if (platform == "REFLECT" || platform == "!REFLECT")
            {
                continue;
            }

            if ((platform != "ALL"))
            {
                if (numAdded)
                {
                    featureStr += " && ";
                }
                else
                {
                    if (featureStr.length())
                    {
                        featureStr += " && ";
                    }
                    featureStr += "(";
                }
                numAdded++;
                featureStr += "!defined(HK_PLATFORM_" + platform.substr(1) + ")";
            }
        }

        if (numAdded)
        {
            featureStr += ")";
        }
    }
    else if (!containsAll)
    {
        // X Y !Z
        std::string positive;
        std::string negative;

        {
            llvm::raw_string_ostream posStr(positive);
            llvm::raw_string_ostream negStr(negative);
            bool firstPositiveMet = false;
            for (const std::string& platform : tkbmsPlatforms)
            {
                if ((platform != "ALL"))
                {
                    if ( platform.substr(0, 1) == "!" )
                    {
                        negStr << " && !defined(HK_PLATFORM_" << platform.substr(1) << ")";
                    }
                    else
                    {
                        if ( firstPositiveMet )
                        {
                            posStr << " || ";
                        }
                        else
                        {
                            if ( featureStr.length() )
                            {
                                posStr << " && ";
                            }
                            posStr << "(";
                            firstPositiveMet = true;
                        }
                        posStr << "defined(HK_PLATFORM_" << platform << ")";
                    }
                }
            }
            posStr << ")";
        }
        featureStr += positive;
        featureStr += negative;
    }

    if(tkbmsProducts.size())
    {
        if(featureStr.length())
        {
            featureStr += " && ";
        }
        if(tkbmsEntry.requireAllProducts)
        {
            featureStr += "(";
            bool first = true;
            for(auto& s: tkbmsProducts)
            {
                if(!first) { featureStr += " && "; }
                featureStr += "defined(HK_FEATURE_PRODUCT_" + s + ")";
                first = false;
            }
            featureStr += ")";
        }
        else
        {
            featureStr += "(";
            bool first = true;
            for(auto& s: tkbmsProducts)
            {
                if(!first) { featureStr += " || "; }
                featureStr += "defined(HK_FEATURE_PRODUCT_" + s + ")";
                first = false;
            }
            featureStr += ")";
        }
    }

    if (featureStr.length())
    {
        return "#if " + featureStr;
    }
    return "";
}

namespace
{
    /// Due to include order between high-up reflect headers, a few specific headers cannot include their
    /// inl files. We hard-code those exceptions here in order to explicitly suppress the error.
    bool shouldWarnAboutTemplateIncludes(const std::string& filePathIn)
    {
        std::vector<std::string> specialCases;
        {
            specialCases.push_back("hkBaseTypes_Types.inl");
            specialCases.push_back("hkReflectValue_Types.inl");
            specialCases.push_back("hkPtrAndInt_Types.inl");
            specialCases.push_back("hkArrayView_Types.inl");
        };

        std::string filePathUnused;
        std::string fileNameIn;
        Filesystem::getFileNameAndPath(filePathIn, fileNameIn, filePathUnused);

        for ( auto ex = specialCases.begin(); ex != specialCases.end(); ++ex )
        {
            if (*ex == fileNameIn)
            {
                return false;
            }
        }

        return true;
    }

    bool containsVtables(const Database& db, const clang::CXXRecordDecl* recDecl)
    {
        if (recDecl->isPolymorphic())
        {
            return true;
        }

        // todo.nt6 as usual this does not work with dependent types, so we need to add a compile-time/runtime check on top
        for (auto fieldIt = recDecl->field_begin(); fieldIt != recDecl->field_end(); ++fieldIt)
        {
            if (const clang::CXXRecordDecl* fieldRec = getCXXRecordFrom(fieldIt->getType()))
            {
                checkIsDefinition(fieldRec);
                if (containsVtables(db, fieldRec))
                {
                    return true;
                }
            }
        }

        return false;
    }
}

static void checkUniquelyGenerated(const std::string& output, std::map<std::string, std::string>& inputFromOutputFilename, const std::string& input)
{
    if( inputFromOutputFilename.find(output) != inputFromOutputFilename.end() )
    {
        const std::string& previous = inputFromOutputFilename[output];
        throw FatalError("Two files produce the same output file '%s' generated by '%s' and by '%s'",
            output.c_str(), input.c_str(), previous.c_str());
    }
    else
    {
        inputFromOutputFilename[output] = input;
    }
}

TextBuilder cppTextBuilder(const Database& db)
{
    TextBuilder cppFile;
    cppFile.addSection("verbatim", new TextBuilder::VerbatimSection());
    cppFile.addSection("include", new TextBuilder::IncludeSection(db.includePaths));
    cppFile.addSection("allCxxs", new TextBuilder::VerbatimSection());

    cppFile["include"].append("Common/Base/KeyCode.h");

    for (auto& fn : db.inputFiles)
    {
        // Hack to #include header files in _Auto - they declare specializations which are needed
#ifdef _WIN32
        if (fn.find("\\_ExtraPreBuildIncludes\\") != std::string::npos)
#else
        if (fn.find("/_ExtraPreBuildIncludes/") != std::string::npos)
#endif
        {
            hkLog.debug("Adding extra include %s", fn.c_str());
            cppFile["include"].append(fn);
            continue;
        }
    }

    return cppFile;
}


void generateTypes(const Database& db, bool generateTracker, bool generateClassList)
{
    const std::string& outputDir = db.getOutputDir();
    static const std::string s_dataOnlyBoilerPlateStart =
        "#if defined(HK_COMPILER_GCC)\n"
        "#pragma GCC diagnostic push\n"
        "#pragma GCC diagnostic ignored \"-Wmissing-field-initializers\"\n"
        "#endif\n\n"
        "#define UNPACK_OPT_PARAM(...) __VA_ARGS__\n"
        "#if defined(HK_FILTER_REFLECT_OPT)\n"
        "#   undef HK_FILTER_REFLECT_OPT\n"
        "#endif\n"
        "#if defined(HK_EXCLUDE_FROM_DATA_REFLECTION)\n"
        "#   undef HK_EXCLUDE_FROM_DATA_REFLECTION\n"
        "#endif\n"
        "#if defined(HK_INCLUDE_IN_CONSTRUCTABLE_REFLECTION)\n"
        "#   undef HK_INCLUDE_IN_CONSTRUCTABLE_REFLECTION\n"
        "#endif\n"
        "#if defined(HK_REFLECT_TYPE_DATA_ONLY)\n"
        "#   define HK_FILTER_REFLECT_OPT(x) ((x) & hkReflect::Opt::DATA_ONLY_MASK)\n"
        "#   define HK_EXCLUDE_FROM_DATA_REFLECTION(x)\n"
        "#   define HK_INCLUDE_IN_CONSTRUCTABLE_REFLECTION(x)\n"
        "#elif defined(HK_REFLECT_TYPE_CONSTRUCTABLE)\n"
        "#   define HK_FILTER_REFLECT_OPT(x) ((x) & hkReflect::Opt::DATA_CONSTRUCTIBLE_MASK)\n"
        "#   define HK_EXCLUDE_FROM_DATA_REFLECTION(x)\n"
        "#   define HK_INCLUDE_IN_CONSTRUCTABLE_REFLECTION(x) UNPACK_OPT_PARAM x\n"
        "#else\n"
        "#   define HK_FILTER_REFLECT_OPT(x) (x)\n"
        "#   define HK_EXCLUDE_FROM_DATA_REFLECTION(x) UNPACK_OPT_PARAM x\n"
        "#   define HK_INCLUDE_IN_CONSTRUCTABLE_REFLECTION(x) UNPACK_OPT_PARAM x\n"
        "#endif\n\n";
    static const std::string s_dataOnlyBoilerPlateEnd =
        "#if defined(HK_COMPILER_GCC)\n"
        "#pragma GCC diagnostic pop\n"
        "#endif\n";

    bool existed;
    const std::string& typesOutputDir = outputDir + "/Types/";
    if(llvm::sys::fs::create_directories(typesOutputDir, existed) != llvm::errc::success)
        throw FatalError("Could not create " + typesOutputDir);

    const std::string& templateTypesOutputDir = outputDir + "/TemplateTypes/";
    if(llvm::sys::fs::create_directories(templateTypesOutputDir, existed) != llvm::errc::success)
        throw FatalError("Could not create " + templateTypesOutputDir);

    // Memory tracker cpp
    TextBuilder cppTracker = cppTextBuilder(db);
    cppTracker.addSection("append", new TextBuilder::CodeSection);

    cppTracker["verbatim"].append("#if !defined(HK_MEMORY_TRACKER_EXCLUDE_PROJECT)");

    if (generateClassList)
    {
        cppTracker.addSection("registerDecl", new TextBuilder::CodeSection);
        cppTracker.addSection("registerFunc", new TextBuilder::CodeSection);
        cppTracker["registerFunc"].append("hkUlong register%sTracker()\n{\nhkUlong res = 0;", db.getProjectNameRoot().c_str())
            .append("#if !defined(HK_MEMORY_TRACKER_EXCLUDE_PROJECT)");
    }

    // Reflection cpp
    TextBuilder cxxCombined = cppTextBuilder(db);

    std::vector<std::pair<std::string, std::string> > includesWithFeature;
    std::vector<std::pair<std::string, std::string> > cxxIncludesWithFeature;

    std::map<std::string, std::string> typeNames;


    // Map of "_Auto/bar_Types.h" -> "foo/bar.h" to check for multiple "bar.h"s in different folders
    std::map<std::string, std::string> inputFromOutputFilename;

    for (auto& fileDeclIt : db.inputFilesDecls)
    {
        const std::string& curFileName = fileDeclIt.first;
        // a/b/c/foo.h -> foo
        const std::string::size_type first = curFileName.find_last_of(Filesystem::pathSeparator) + 1;
        const std::string& curFileStem = curFileName.substr(first, curFileName.size() - first - 2);
        const Database::DeclListType& declList = fileDeclIt.second.decls;
        TextBuilder cxxBody = cxxTextBuilder(db);
        TextBuilder inlBody = cxxTextBuilder(db);

        cxxBody["include"].append(curFileName);

        bool hasFunctionRefl = false;
        bool foundAutomaticRegisteredType = false;
        std::string featureString;
        for(Database::DeclListType::const_iterator cit = declList.begin(); cit != declList.end(); ++cit)
        {
            try
            {
                const clang::TypeDecl* decl = *cit;

                TkbmsRegistry::Entry entry;
                // Only Havok files have TKBMS entries
                if(db.tkbmsRegistry.tryGetTkbmsEntry(curFileName, entry))
                {
                    featureString = getFeatureString(db, decl, entry, fileDeclIt.first);
                }
                else
                {
                    featureString = "";
                }
                //todo declaredName = checkAndGetDeclaredName(sym)
                const std::string& declaredName = decl->getNameAsString();
                if(!db.isReflected(decl))
                {
                    if(const clang::CXXRecordDecl* recDecl = clang::dyn_cast<clang::CXXRecordDecl>(decl))
                    {
                        // If this record isn't reflected and it has a parent forcing children reflection we issue
                        // an error.
                        if(const clang::CXXRecordDecl* reflectedParent = hasReflectedParent(db, recDecl))
                        {
                            if(db.getAttributeValueInHierarchy<bool>(reflectedParent, "hk::ReflectChildren", false))
                            {
                                const std::string& childName = db.getScopedMangledName(recDecl);
                                const std::string& parentName = db.getScopedMangledName(reflectedParent);

                                hkLog.error(recDecl, "The record '%s' must be reflected since it has at least one reflected parent (%s) forcing children reflection.",
                                    childName.c_str(), parentName.c_str());
                            }
                        }

                        // Generate a tracker type.
                        if (generateTracker && !db.isDependentTypeDecl(recDecl) && db.hasTrackerHandle(decl))
                        {
                            if (getTemplateDecl(recDecl))
                            {
                                generateTemplateTrackerType(db, recDecl, cxxBody, inlBody);
                            }
                            else
                            {
                                generateNonTemplateTrackerType(db, recDecl, featureString, generateClassList, cppTracker, cppTracker);
                            }
                        }
                    }
                }
                else
                {
                    
                    if (auto outerDecl = clang::dyn_cast<clang::CXXRecordDecl>(decl->getDeclContext()))
                    {
                        if (getTemplateDecl(outerDecl))
                        {
                            hkLog.error(decl, "Reflection of types declared inside a template class is not supported");
                            continue;
                        }
                    }

                    if(const clang::ClassTemplateSpecializationDecl* specDecl = clang::dyn_cast<clang::ClassTemplateSpecializationDecl>(decl))
                    {
                        if (specDecl->isExplicitSpecialization())
                        {
                            // This type is a record, that is not manually registered -> this file should be in the cxx list
                            if (!db.getAttributeValue<bool>(specDecl, "hk::ManualTypeRegistration", false))
                            {
                                foundAutomaticRegisteredType = true;
                            }

                            generateTemplateRecordReflection(db, specDecl, cxxBody, inlBody, hasFunctionRefl);
                            if (generateTracker && db.hasTrackerHandle(decl))
                            {
                                if (clang::isa<clang::ClassTemplatePartialSpecializationDecl>(decl))
                                {
                                    generateTemplateTrackerType(db, specDecl, cxxBody, inlBody);
                                }
                                else
                                {
                                    // Full reflected template specializations do not need an additional tracker type,
                                    // they will register the reflected type as non-templates.
                                    generateTrackerRegNode(db, specDecl, db.getScopedMangledName(specDecl), featureString, generateClassList, cxxBody, cppTracker);
                                }
                            }
                        }
                    }
                    else if(const clang::CXXRecordDecl* recDecl = clang::dyn_cast<clang::CXXRecordDecl>(decl))
                    {
                        const clang::ClassTemplateDecl* tplDecl = getTemplateDecl(recDecl);
                        if (tplDecl)
                        {
                            generateTemplateRecordReflection(db, tplDecl, cxxBody, inlBody, hasFunctionRefl);
                            if (generateTracker && db.hasTrackerHandle(recDecl))
                            {
                                generateTemplateTrackerType(db, recDecl, cxxBody, inlBody);
                            }
                        }
                        else
                        {
                            generateNonTemplateRecordReflection(db, recDecl, cxxBody, inlBody, hasFunctionRefl);

                            // This type is a record, that is not manually registered -> this file should be in the cxx list
                            if (!db.getAttributeValue<bool>(recDecl, "hk::ManualTypeRegistration", false))
                            {
                                foundAutomaticRegisteredType = true;
                            }

                            // Generate a TrackerRegNode for the type.
                            if (generateTracker && db.hasTrackerHandle(recDecl))
                            {
                                
                                const std::string& cxxName = recDecl->getQualifiedNameAsString();
                                generateTrackerRegNode(db, recDecl, cxxName, featureString, generateClassList, cxxBody, cppTracker);
                            }
                        }
                        checkIfSimilarNameExists(db, decl, typeNames);
                    }
                    else if(const clang::TypedefDecl* tdefDecl = clang::dyn_cast<clang::TypedefDecl>(decl))
                    {
                        // This type is a typedef, that is not manually registered -> this file should be in the cxx list
                        if (!db.getAttributeValue<bool>(tdefDecl, "hk::ManualTypeRegistration", false))
                        {
                            foundAutomaticRegisteredType = true;
                        }

                        generateTypedef(db, tdefDecl, cxxBody, inlBody);
                        checkIfSimilarNameExists(db, decl, typeNames);
                    }
                    else if(const clang::EnumDecl* enumDecl = clang::dyn_cast<clang::EnumDecl>(decl))
                    {
                        // This type is an enum, that is not manually registered -> this file should be in the cxx list
                        if (!db.getAttributeValue<bool>(enumDecl, "hk::ManualTypeRegistration", false))
                        {
                            foundAutomaticRegisteredType = true;
                        }

                        generateEnum(db, enumDecl, cxxBody, inlBody);
                        checkIfSimilarNameExists(db, decl, typeNames);
                    }
                    else
                    {
                        hkLog.debug(decl, "Skipping %s (type = %s)", declaredName.c_str(), decl->getDeclKindName());
                    }
                }

                // todo.nt6 this should be in a sanity pass and should run on cpps too, but for the moment here is
                // better than nowhere
                if (auto recDecl = clang::dyn_cast<clang::CXXRecordDecl>(decl))
                {
                    // Check that it does not contain vtables if marked as POD.
                    if (db.getAttributeValue(recDecl, "hk::Pod", false) && containsVtables(db, recDecl))
                    {
                        hkLog.error(recDecl, "Class '%s' contains vtables and cannot be marked as POD.",
                            recDecl->getNameAsString().c_str());
                    }
                }
            }
            catch (const RuntimeError& e)
            {
                hkLog.errorWithoutDatabase(e.getLine(), e.getFileName(), e.getMessage());
            }
        }

        if (hasFunctionRefl)
        {
            inlBody["include"].append(FUNC_REFL_INCLUDE);
            cxxBody["include"].append(FUNC_REFL_INCLUDE);
        }

        // Add extra includes to inl and cpp.
        for (auto& extraInclude : fileDeclIt.second.extraIncludes)
        {
            inlBody["include"].append(extraInclude);
            cxxBody["include"].append(extraInclude);
        }

        if(!cxxBody["append"].isEmpty())
        {
            const std::string& cxxBodyStr = cxxBody.format();
            const std::string& outFileName = outputDir + "/Types/" + curFileStem + "_Types.cxx";

            checkUniquelyGenerated(outFileName, inputFromOutputFilename, curFileName);
            Filesystem::FileHandle out = db.fs.openRW(outFileName);
            if ( !out.stream )
            {
                throw FatalError("Cannot open %s for writing", outFileName.c_str());
            }
            out << db.getPreamble(outFileName, curFileName)
                << "#pragma once\n\n"
                << s_dataOnlyBoilerPlateStart
                << cxxBodyStr
                << s_dataOnlyBoilerPlateEnd
                << db.getPostamble(outFileName);

            // We have found at least one automatically registered type, include this cxx
            if (foundAutomaticRegisteredType)
            {
                cxxIncludesWithFeature.push_back(std::make_pair("Types/" + curFileStem + "_Types.cxx", featureString));
            }
            includesWithFeature.push_back(std::make_pair(curFileName, featureString));
        }

        if(!inlBody["append"].isEmpty())
        {
            const std::string& inlBodyStr = inlBody.format();
            // Only emit these if we have a cxxbody
            // Some errors can get generated during parse but we don't
            // want to generate an output file just for the errors
            if (!hkLog.loggedErrors().empty())
            {
                cxxBody["errors"].append("").append(std::string(80, '/'))
                    .append("// Reflection errors:\n//\n");

                cxxBody["errors"].append(hkLog.loggedErrors());
                hkLog.clearLoggedErrors();
            }

            if (!hkLog.loggedWarnings().empty())
            {
                cxxBody["warnings"].append("").append(std::string(80, '/'))
                    .append("// Reflection warnings:\n//\n")
                    .append("#ifdef HK_REFLECTION_WARNINGS_ARE_ERRORS\n")
                    .append(hkLog.loggedWarnings())
                    .append("#endif\n");
                hkLog.clearLoggedWarnings();
            }

            const std::string& outFileName = curFileStem + "_Types.inl";
            const std::string& pathToFile = Filesystem::absoluteCanonicalPath(Filesystem::absoluteCanonicalPath(outputDir) + "/TemplateTypes/");
            const std::string& filePath = pathToFile + Filesystem::pathSeparator + outFileName;

            // Open it for write anyway.
            checkUniquelyGenerated(filePath, inputFromOutputFilename, curFileName);
            Filesystem::FileHandle out = db.fs.openRW(filePath);
            hkLog.info("Writing %s, include it in the template class header.", filePath.c_str());

            out << db.getPreamble(outFileName, curFileName)
                << "#pragma once\n\n"
                << "#ifndef HK_REFLECT_SKIP_INL_BODY\n"
                << s_dataOnlyBoilerPlateStart
                << inlBodyStr
                << s_dataOnlyBoilerPlateEnd
                << "#endif // HK_REFLECT_SKIP_INL_BODY\n"
                << db.getPostamble(outFileName);

            // Try to issue an error if the header doesn't contain an include for this file.
            {
                bool suppressError = false;
                std::vector<std::string> filesIncludingTheInlFile;

                std::string inFileName;
                std::string inPathUnused;
                Filesystem::getFileNameAndPath(curFileName, inFileName, inPathUnused);

                auto range = db.directIncludes.equal_range(outFileName);
                for ( Database::DirectIncludes::iterator it = range.first; it != range.second; ++it )
                {
                    const Database::DirectInclude& di = it->second;
                    std::string dbFileName;
                    std::string dbPathUnused;
                    Filesystem::getFileNameAndPath(di.m_sourceFile, dbFileName, dbPathUnused);
                    if (dbFileName == inFileName)
                    {
                        filesIncludingTheInlFile.push_back(di.m_sourceFile);
                    }
                }

                if (!suppressError)
                {
                    if (std::find(filesIncludingTheInlFile.begin(), filesIncludingTheInlFile.end(), curFileName) == filesIncludingTheInlFile.end())
                    {
                        if (shouldWarnAboutTemplateIncludes(filePath))
                        {
                            // If the inline file is included somewhere else, issue a warning.
                            std::string curFileNameRel;
                            std::string inlFileNameRel;
                            Filesystem::makePathRelativeToIncludes(curFileName, curFileNameRel);
                            Filesystem::makePathRelativeToIncludes(filePath, inlFileNameRel);
                            hkLog.warningWithoutDatabase(1, curFileName.c_str(), "\"%s\" contains a reflected template but does not #include the generated template types file \"%s\".", curFileNameRel.c_str(), inlFileNameRel.c_str());
                        }
                    }
                }
            }
            if (!hkLog.loggedErrors().empty())
            {
                out << std::string(80, '/') << "\n// Reflection errors:\n//\n" << hkLog.loggedErrors();
                hkLog.clearLoggedErrors();
            }
            if (!hkLog.loggedWarnings().empty())
            {
                out << std::string(80, '/') << "\n// Reflection warnings:\n//\n" << "#ifdef HK_REFLECTION_WARNINGS_ARE_ERRORS\n" << hkLog.loggedWarnings() << "#endif\n";
                hkLog.clearLoggedWarnings();
            }
        }
    }

    if (generateTracker)
    {
        // Always write, even if there is no content.
        cppTracker["append"].append("#endif // !defined(HK_MEMORY_TRACKER_EXCLUDE_PROJECT)");

        if (generateClassList)
        {
            cppTracker["registerFunc"].append("#endif // !defined(HK_MEMORY_TRACKER_EXCLUDE_PROJECT)\nreturn res;\n}");
        }

        const std::string& trackerName = db.getOutputDir() + "/" + db.getProjectNameRoot() + "_Tracker.cxx";

        
        db.fs.openRW(trackerName)
            << db.getPreamble(trackerName)
            << "#if defined(HK_MEMORY_TRACKER_ENABLE)\n"
            << cppTracker.format()
            << "#endif // !defined(HK_MEMORY_TRACKER_ENABLE)\n"
            << db.getPostamble(trackerName);
    }

    // Always write, even if we have no files
    {
        const std::string& cxxCombinedName = db.getOutputDir() + "/" + db.getProjectNameRoot() + "_IncludeAllTypes.cxx";

        // This needs to be deterministic
        std::map<std::string, TextBuilder::IncludeSection*> includeFeatureSets;
        std::map<std::string, TextBuilder::IncludeSection*> cxxIncludeFeatureSets;

        for (auto& includes: includesWithFeature)
        {
            TextBuilder::IncludeSection* sec = includeFeatureSets[includes.second];
            if (!sec)
            {
                sec = new TextBuilder::IncludeSection(db.includePaths);
                includeFeatureSets[includes.second] = sec;
            }
            sec->append(includes.first);
        }

        for (auto& cxxIncludes : cxxIncludesWithFeature)
        {
            TextBuilder::IncludeSection* sec = cxxIncludeFeatureSets[cxxIncludes.second];
            if (!sec)
            {
                // These paths are "Types/...", they are relative to the cpp not any other path
                sec = new TextBuilder::IncludeSection(db.includePaths, "", TextBuilder::IncludeSection::USE_QUOTES);
                cxxIncludeFeatureSets[cxxIncludes.second] = sec;
            }
            sec->append(cxxIncludes.first);
        }

        for (auto iter = includeFeatureSets.begin(); iter != includeFeatureSets.end(); ++iter)
        {
            const std::string& featureName = iter->first;
            const TextBuilder::VerbatimSection* section = iter->second;

            if (featureName.length())
            {
                cxxCombined["allCxxs"].append(featureName);
            }

            std::string outStr;
            section->format(outStr);
            cxxCombined["allCxxs"].append(outStr);
            delete section;

            if (featureName.length())
            {
                cxxCombined["allCxxs"].append("#endif");
            }
        }
        includeFeatureSets.clear();

        for (auto iter = cxxIncludeFeatureSets.begin(); iter != cxxIncludeFeatureSets.end(); ++iter)
        {
            const std::string& featureName = iter->first;
            const TextBuilder::IncludeSection* section = iter->second;

            if (featureName.length())
            {
                cxxCombined["allCxxs"].append(featureName);
            }

            std::string outStr;
            section->format(outStr);
            cxxCombined["allCxxs"].append(outStr);
            delete section;

            if (featureName.length())
            {
                cxxCombined["allCxxs"].append("#endif");
            }
        }
        cxxIncludeFeatureSets.clear();

        db.fs.openRW(cxxCombinedName)
            << db.getPreamble(cxxCombinedName)
            << "#define HK_DETAIL_REFLECT_DEFINITIONS\n"
            << cxxCombined.format()
            << db.getPostamble(cxxCombinedName);
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
