// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
#include "Generators.h"
#include "utils/Database.h"
#include "utils/ElementUtils.h"
#include "utils/Filesystem.h"
#include "utils/TkbmsRegistry.h"
#include "utils/Logger.h"

std::string getFeatureString(const Database& db, const clang::Decl* decl, TkbmsRegistry::Entry tkbmsEntry, const std::string& filename);

namespace
{
    const clang::CXXRecordDecl* isInterestingSymbol(const Database& db, const clang::Decl* decl)
    {
        if(const clang::CXXRecordDecl* record = clang::dyn_cast<clang::CXXRecordDecl>(decl))
        {
            if (!record->isCompleteDefinition() ||
                hasTemplateParamsRecursive(decl) ||
                !db.isReflected(decl) ||
                !db.getAttributeValue<bool>(decl, "hk::RegisterType", true) ||
                db.getAttributeValue<bool>(decl, "hk::ManualTypeRegistration", false))
            {
                return 0;
            }

            return record;
        }

        return 0;
    }

    void dumpClassList(std::ostream& defaultOut, const Database& db, std::map<std::string, std::unique_ptr<std::ostringstream> >& featureDecls)
    {
        for(auto file : db.inputFilesDecls)
        {
            const std::string& curFileName = file.first;
            for (auto decl : file.second.decls)
            {
                std::ostream* out = &defaultOut;

                if (const clang::CXXRecordDecl* record = isInterestingSymbol(db, decl))
                {
                    TkbmsRegistry::Entry tkbmsEntry;
                    if(db.tkbmsRegistry.tryGetTkbmsEntry(curFileName, tkbmsEntry))
                    {
                        std::string featureStr = getFeatureString(db, decl, tkbmsEntry, curFileName);

                        // Empty string is just the default list that is always included
                        if(featureStr.length())
                        {
                            out = featureDecls[featureStr].get();
                            if (out == nullptr)
                            {
                                auto o = new std::ostringstream();
                                featureDecls[featureStr].reset(o);
                                out = o;
                            }
                        }
                    }

                    const std::string& cxxName = db.getScopedMangledName(record);
                    const std::string& pyName = pyNameFromCxxName(cxxName);
                    (*out) << "HK_RECORD_TYPE(" << record->getKindName() << ", " << pyName << ", " << cxxName << ")\n";
                }
            }
        }
    }

    void generateClassListFile(const Database& db, bool generateTracker)
    {
        const std::string& fname = db.getOutputDir() + "/" + db.getProjectNameRoot() + "ClassList.cxx";

        Filesystem::FileHandle of = db.fs.openRW(fname);

        of << db.getPreamble(fname)
            << "\n// This file describes all the types generated in this project.\n"
            << "#ifndef HK_RECORD_TYPE\n"
            << "#error This file must be included from hkProductFeatures.cxx or hkRegisterClasses.cxx. Do not use this file directly.\n"
            << "#endif\n"
            << "#ifndef HK_TRACKER_FUNC\n#define HK_TRACKER_FUNC(P)\n#endif\n"
            << "\n";

        std::map<std::string, std::unique_ptr<std::ostringstream> > features;
        dumpClassList(*of.stream, db, features);

        for(std::map<std::string, std::unique_ptr<std::ostringstream> >::const_iterator cit = features.begin(); cit != features.end(); ++cit)
        {
            of << cit->first << "\n"
                << cit->second.get()->str()
                << "#endif\n";
        }

        if (generateTracker)
        {
            of << "HK_TRACKER_FUNC(" << db.getProjectNameRoot() << ")\n";
        }
        of << db.getPostamble(fname);
    }


    void forwardDecls(std::vector<std::string>& out, const clang::Decl* decl, const Database& db, const std::string& prefix, const std::string& scope);
    void forwardDecls2(std::vector<std::string>& out, const clang::DeclContext* dc, const Database& db, const std::string& prefix, const std::string& scope)
    {
        for(clang::DeclContext::decl_iterator declIt = dc->decls_begin(), end = dc->decls_end();
            declIt != end;
            ++declIt)
        {
            forwardDecls(out, *declIt, db, prefix, scope);
        }
    }
    void forwardDecls(std::vector<std::string>& out, const clang::Decl* decl, const Database& db, const std::string& prefix, const std::string& scope)
    {
        if(!db.isDeclaredInInputs(decl))
        {
            return;
        }
        if(const clang::CXXRecordDecl* record = isInterestingSymbol(db, decl))
        {
            out.push_back( formatString("%s%s %s%s;", prefix.c_str(), record->getKindName(), scope.c_str(), record->getNameAsString().c_str() ) );
            forwardDecls2(out, record, db, prefix + "    ", record->getNameAsString() + "_");
        }
        else if(const clang::DeclContext* dc = clang::dyn_cast<clang::DeclContext>(decl))
        {
            if(const clang::NamespaceDecl* ns = clang::dyn_cast<clang::NamespaceDecl>(dc))
            {
                std::vector<std::string> decls;
                forwardDecls2(decls, ns, db, prefix+"    ", "");
                if( decls.size() )
                {
                    out.push_back( formatString("%snamespace %s {", prefix.c_str(), ns->getNameAsString().c_str() ) );
                    out.insert( out.end(), decls.begin(), decls.end());
                    out.push_back( formatString("%s}", prefix.c_str() ) );
                }
            }
            else if(const clang::TranslationUnitDecl* tu = clang::dyn_cast<clang::TranslationUnitDecl>(decl))
            {
                forwardDecls2(out, tu, db, prefix, "");
            }
        }
    }

    void specDecls(std::ostream& out, const clang::Decl* decl, const Database& db, const std::string& scope)
    {
        if(!db.isDeclaredInInputs(decl))
        {
            return;
        }
        if( const clang::CXXRecordDecl* record = clang::dyn_cast<clang::CXXRecordDecl>(decl) )
        {
            if(isInterestingSymbol(db, record))
            {
                const std::string& cxxName = record->getNameAsString();
                out << formatString("template<> struct ReflectionOf< %s%s > { static HK_EXPORT_COMMON TypeData typeData; };\n", scope.c_str(), cxxName.c_str());
            }
        }
        if(const clang::DeclContext* dc = clang::dyn_cast<clang::DeclContext>(decl))
        {
            std::string scope2;
            if(const clang::NamespaceDecl* ns = clang::dyn_cast<clang::NamespaceDecl>(decl))
            {
                scope2 = scope + ns->getNameAsString() + "::";
            }
            else if(const clang::CXXRecordDecl* cx = clang::dyn_cast<clang::CXXRecordDecl>(decl))
            {
                scope2 = scope + cx->getNameAsString() + "_";
            }
            else
            {
                scope2 = scope;
            }

            for(clang::DeclContext::decl_iterator declIt = dc->decls_begin(), end = dc->decls_end();
                declIt != end;
                ++declIt)
            {
                specDecls(out, *declIt, db,  scope2);
            }
        }
    }
}

void generateLists(const Database& db, bool generateTracker)
{
    generateClassListFile(db, generateTracker);
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
