// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
PRAGMA_WARNING_PUSH
# include <sstream>
# include <ctime>
# include <clang/AST/DeclVisitor.h>
# include <llvm/Support/ManagedStatic.h>
# include <llvm/Support/system_error.h>
# include <llvm/Support/FileSystem.h>
PRAGMA_WARNING_POP

#include "Generators.h"
#include "utils/Database.h"
#include "utils/Filesystem.h"
#include "utils/Logger.h"
#include "utils/TextBuilder.h"
#include "utils/ElementUtils.h"
#include "utils/TkbmsRegistry.h"
#include "utils/HavokUtils.h"

namespace
{
    void generateEnumDecl(const Database& db, const clang::EnumDecl* enumDecl, TextBuilder::Section& classDeclarations, TextBuilder::Section& nativeMapDeclarations)
    {
        bool definesFlags = false;

        if(const clang::AnnotateAttr* attr = db.getAttribute(enumDecl, "dotnet::DefineFlags"))
        {
            if(attr->getAnnotation().endswith("(true)"))
            {
                definesFlags = true;
            }
        }

        const char* enumType = definesFlags ? "FLAG" : "ENUM";

        classDeclarations.append("HK_MANAGED_%s(%s)\n{", enumType, enumDecl->getNameAsString().c_str());
        for(clang::EnumDecl::enumerator_iterator it = enumDecl->enumerator_begin(); it != enumDecl->enumerator_end(); ++it)
            classDeclarations.append("HK_MANAGED_%s_VALUE(%s, %d),", enumType, it->getNameAsString().c_str(), int(*it->getInitVal().getRawData()));

        classDeclarations.append("};\n");

        std::string fullNameStr = db.getScopedMangledName(enumDecl);
        nativeMapDeclarations.append("template<> struct NativeFromManaged<%s> { typedef ::%s Type; };", fullNameStr.c_str(), fullNameStr.c_str());
        nativeMapDeclarations.append("template<> struct ManagedFromNative<::%s> { typedef %s Type; };", fullNameStr.c_str(), fullNameStr.c_str());
    }

    bool shouldSkipAttribute(const std::string& fullName)
    {
        // Old exclusive list for reference
        //return (fullName == "hk::Type") || (fullName == "hk::AbsMin") || (fullName == "hk::AbsMax")
        //  || (fullName == "hk::UI_Navigable") || (fullName == "hke::ConstantData") || (fullName == "hk::Optional")
        //  || (fullName == "hk::AlignReal") || (fullName == "hk::PointerType") || (fullName == "hk::Ui_Type")
        //  || ( fullName == "hk::Format" ) || ( fullName == "hk::Ui_Order" ) || ( fullName == "hk::LeafObject" ) ||
        //  ( fullName == "hk::Validate" );

        // New inclusive list
        return
            ( hkString::strNcasecmp( fullName.c_str(), "hkb", 3 ) != 0 ) &&
            ( fullName != "hk::OpaqueType" ) && ( fullName != "hk::Serialize" ) && ( fullName != "hk::Untracked" )
            && ( fullName != "hkDescriptionAttribute" ) && ( fullName != "hkRangeRealAttribute" ) && ( fullName != "hkSemanticsAttribute" )
            && ( fullName != "hkUiAttribute" ) && ( fullName != "hkRangeInt32Attribute" ) && ( fullName != "hkLinkAttribute" );
    }

    bool shouldSkipClassAttribute(const std::string& fullName)
    {
        return
            ( hkString::strNcasecmp( fullName.c_str(), "hkb", 3 ) != 0 ) &&
            (fullName != "hkUiAttribute");
    }

    std::string getAttributes(const Database& db, const clang::FieldDecl* fieldDecl, const std::string& className, int memIndex)
    {
        std::stringstream ret;
        int attrIndex = 0;
        if(fieldDecl->getAccess() == clang::AS_public)
        {
            ret << "[hkPublicAttribute()] ";
            attrIndex++;
        }

        if(fieldDecl->hasAttrs())
        {
            const clang::AttrVec& attrVec = fieldDecl->getAttrs();
            typedef clang::specific_attr_iterator<clang::AnnotateAttr> SpecificIterator;
            for( SpecificIterator iterator = clang::specific_attr_begin<clang::AnnotateAttr>(attrVec), end_iterator = clang::specific_attr_end<clang::AnnotateAttr>(attrVec);
                iterator != end_iterator;
                ++iterator)
            {
                if(iterator->getAnnotation().startswith(Database::attributePrefix)) // Check that the attribute starts with hk_attr (Havok attribute).
                {
                    const std::string& attrName = db.getAttrNameOrEmpty(*iterator);
                    //hkLog.debug("%s::%s has attr %s", recDecl->getNameAsString().c_str(), fieldDecl->getNameAsString().c_str(), iterator->getAnnotation().str().c_str());
                    if(attrName.compare("hk::Default") == 0)
                    {
                        std::string fieldName(fieldDecl->getName());
                        if(strncmp(fieldName.data(), "m_", 2) == 0)
                        {
                            fieldName.erase(fieldName.begin(), fieldName.begin() + 2);
                        }
                        ret << "[hkDefaultAttribute(" << className << "::typeid, \"" << fieldName << "\", " << memIndex << ")] ";
                        attrIndex++;
                    }
                    else
                    {
                        //attrName.
                        std::string shortenedName;
                        std::string fullName;
                        for(auto c: attrName)
                        {
                            if(c == '(')
                            {
                                // Stop when we get to the end of the name
                                break;
                            }
                            if(c != ':')
                            {
                                shortenedName.push_back(c);
                            }
                            fullName.push_back(c);
                        }

                        if(!shouldSkipAttribute(fullName))
                        {
                            ret << "[" << shortenedName <<  "(" << className << "::typeid, " << "\"" << fieldDecl->getNameAsString() << "\", \"" << fullName << "\")] ";
                            attrIndex++;
                        }
                    }
                }
            }
        }

        if(attrIndex == 0)
        {
            ret << "/* no attributes */";

        }
        return ret.str();
    }

    void recursivelyCreateDataFromDeclContext(const Database& db, const clang::DeclContext* thisContext, TextBuilder::Section& classDeclarations, TextBuilder::Section& classForwardDeclaration, TextBuilder::Section& nativeMap, std::set<std::string>& headersToInclude, TextBuilder::Section& classDefinitions, TextBuilder::Section& propertyOffsetDefinitions, bool isTopLevel);

    bool isSpecialHandledRecord(clang::CXXRecordDecl* rd)
    {
        return (rd->getName().find("hkStringPtr") == 0) || (rd->getName().find("hkArray") == 0)  || (rd->getName().find("hkEnum") == 0)
            || (rd->getName().find("hkFlags") == 0) || (rd->getName().find("hkInplaceArray") == 0) || (rd->getName().find("hkSimpleArray") == 0)
            || (rd->getName().find("hkRelArray") == 0) || (rd->getName().find("hkPtr") == 0) || (rd->getName().find("hkRefPtr") == 0)
            || (rd->getName().find("hkMatrix") == 0);
    }

    std::string getTypeNameConvertPointers(const Database& db, clang::QualType t)
    {
        t = resolveType(t);

        if(const clang::PointerType* pt = clang::dyn_cast<clang::PointerType>(t))
        {
            std::string cxxTypeName = getTypeNameConvertPointers(db, pt->getPointeeType());
            cxxTypeName.append("^");
            return cxxTypeName;
        }
        else
        {
            std::string cxxTypeName = db.getScopedMangledName(t);
            return cxxTypeName;
        }
    }

    void getPropertyForType(const Database& db, clang::QualType fieldType, const std::string& fieldTypeName, const std::string& fieldAttributes, const std::string& thisClassScopedName, std::string& declarationOut, std::string& definitionOut)
    {
        std::string cxxTypeName = getTypeNameConvertPointers(db, fieldType);

        if((cxxTypeName.find("hkReflect::Type") != std::string::npos) || (cxxTypeName.find("hkReflect::Var") != std::string::npos))
        {
            // Skip!
            std::stringstream def;
            def << "// Skipping type " << cxxTypeName << "\n";
            declarationOut = def.str();
            return;
        }

        // Skip carrays
        if(cxxTypeName.find("[") != std::string::npos)
        {
            return;
        }

        std::stringstream ret;

        ret << "public:\n";

        if ( clang::CXXRecordDecl* rd = fieldType->getAsCXXRecordDecl() )
        {
            if(!isSpecialHandledRecord(rd))
            {
                if(clang::dyn_cast<clang::ClassTemplateSpecializationDecl>(rd))
                {
                    return; // Templates that we don't handle specially get ignored
                }
                ret << "[Havok::hkStructAttribute()]\n";
            }

            std::string arrayType;
            if((rd->getNameAsString().find("hkArray") == 0)) { arrayType = "hkArray"; }
            else if((rd->getNameAsString().find("hkSimpleArray") == 0)) { arrayType = "hkSimpleArray"; }
            else if((rd->getNameAsString().find("hkInplaceArray") == 0)) { arrayType = "hkInplaceArray"; }
            else if((rd->getNameAsString().find("hkRelArray") == 0)) { arrayType = "hkArray"; }
            if(arrayType.length() > 0)
            {
                std::string arrayMemberType;
                clang::QualType arrayElemType;
                if(clang::ClassTemplateSpecializationDecl* tempDecl = clang::dyn_cast<clang::ClassTemplateSpecializationDecl>(rd))
                {
                    // todo
                    const clang::TemplateArgumentList& tal = tempDecl->getTemplateArgs();
                    if(tal.size() > 0)
                    {
                        const clang::TemplateArgument& arg = tal.get(0);
                        arrayElemType = arg.getAsType();
                        arrayMemberType = getTypeNameConvertPointers(db, arrayElemType);
                    }
                }

                if((arrayMemberType == "") || (arrayMemberType.find("hkArray")) != std::string::npos)
                {
                    // Things we don't recognise, arrays of arrays
                    return;
                }
                ret << fieldAttributes << "\n";
                ret << "property HavokManaged::" << arrayType << "< typename PropertyOf< " << arrayMemberType << " >::Type >^ " << fieldTypeName << "\n";
                ret << "{\n";
                ret << "HavokManaged::" << arrayType << "< typename PropertyOf< " << arrayMemberType << " >::Type >^ get();\n";
                ret << "};";
                declarationOut = ret.str();

                {
                    std::stringstream fieldTypeStr;
                    fieldTypeStr << "HK_HA_PROP_TYPE_OF(" << fieldTypeName << ", " << thisClassScopedName << ")";

                    std::stringstream def;
                    def << "HavokManaged::" << arrayType << "< typename PropertyOf< " << arrayMemberType << " >::Type >^ " << thisClassScopedName << "::" << fieldTypeName << "::get()\n";
                    def << "{\n";
                    def << "return HavokManaged::PropertyAccessor< HavokManaged::" << arrayType << "< typename PropertyOf< " << arrayMemberType << " >::Type >^ >::get(getPointer(), HK_HA_PROP_OFFSET_OF(" << fieldTypeName << ", " << thisClassScopedName << "), " << fieldTypeStr.str() << ", m_wrapper);\n";
                    def << "}\n";

                    definitionOut = def.str();
                }
                return;
            }
        }

        // Skip, we seem to have made quite a few +nosave pointers into reflected void*, these won't work with the binding
        if(cxxTypeName == "void^")
        {
            return;
        }

        ret << fieldAttributes << "\n";
        ret << "property PropertyValue<PropertyOf< " << cxxTypeName << " >::Type>::Type " << fieldTypeName << " { void set(PropertyValue<PropertyOf< " << cxxTypeName << " >::Type>::Type value); PropertyValue<PropertyOf< " << cxxTypeName << " >::Type>::Type get(); };";

        declarationOut = ret.str();

        {
            std::stringstream fieldTypeStr;
            fieldTypeStr << "HK_HA_PROP_TYPE_OF(" << fieldTypeName << ", " << thisClassScopedName << ")";

            std::stringstream def;
            def << "PropertyValue<PropertyOf< " << cxxTypeName << " >::Type>::Type " << thisClassScopedName << "::" << fieldTypeName << "::get()\n";
            def << "{\n";
            def << "return HavokManaged::PropertyAccessor< typename PropertyOf< " << cxxTypeName << " >::Type >::get(getPointer(), HK_HA_PROP_OFFSET_OF(" << fieldTypeName << ", " << thisClassScopedName << "), " << fieldTypeStr.str() << ", m_wrapper);\n";
            def << "}\n";

            def << "void " << thisClassScopedName << "::" << fieldTypeName << "::set(PropertyValue<PropertyOf< " << cxxTypeName << " >::Type>::Type value)\n";
            def << "{\n";
            def << "HavokManaged::PropertyAccessor< typename PropertyOf< " << cxxTypeName << " >::Type >::set(value, getPointer(),  HK_HA_PROP_OFFSET_OF(" << fieldTypeName << ", " << thisClassScopedName << "), " << fieldTypeStr.str() << ", m_wrapper);\n";
            //def << "PropertyChanged(this, gcnew PropertyChangedEventArgs(\"" << fieldTypeName << "\"));\n"; // when should we emit this??
            def << "}\n";
            definitionOut = def.str();
        }


        return;
    }

    
    void generateFieldPropertyDecl(const Database& db, const clang::FieldDecl* fieldDecl, const std::string& thisClassName, const std::string& thisClassScopedName, int memIndex, std::string& declarationOut, std::string& definitionOut, TextBuilder::Section& propertyOffsetDefinitions)
    {
        const std::string& fieldTypeName = fieldDecl->getNameAsString();

        std::string const &fieldAttributes = getAttributes(db, fieldDecl, thisClassName, memIndex);

        if(db.isSerializable(fieldDecl) && !db.isOpaqueTypeAllowed(fieldDecl))
        {
            clang::QualType fieldType;
            if(const Optional<std::string>& overriddenType = db.getAttributeValue< Optional<std::string> >(fieldDecl, "hk::Type"))
            {
                if(const clang::NamedDecl* fieldTypeDecl = db.findDecl(*overriddenType))
                {
                    if(const clang::TypeDecl* typeDecl = clang::dyn_cast<clang::TypeDecl>(fieldTypeDecl))
                    {
                        fieldType = clang::QualType(typeDecl->getTypeForDecl(), 0);
                    }
                }

                // Some types are just too hard or even impossible to resolve.
                // For example: "hk::Type(hkUint8[3])", "hk::Type(hkRelArray<hkVector4>)", ...
                // In that case, we just don't add this member to the binding
            }
            else
            {
                fieldType = fieldDecl->getType();
            }

            propertyOffsetDefinitions.append("enum { offset_%s = HK_ROFFSET_OF(%s, ::%s) };", fieldTypeName.c_str(), fieldTypeName.c_str(), thisClassScopedName.c_str());
            // TODO: Need special handling for override types, currently we ignore them here

            if(const Optional<std::string>& overriddenType = db.getAttributeValue< Optional<std::string> >(fieldDecl, "hk::Type"))
            {
                // Need to do a little processing on this
                std::string overrideType = *overriddenType;
                std::string cArrayPart = "";
                int splitIndex = (int)overrideType.find("[");
                if(splitIndex >= 0)
                {
                    cArrayPart = overrideType.substr(splitIndex, std::string::npos);
                    overrideType = overrideType.substr(0, splitIndex);
                }
                propertyOffsetDefinitions.append("typedef %s type_%s%s;", overrideType.c_str(), fieldTypeName.c_str(), cArrayPart.c_str());
            }
            else
            {
                propertyOffsetDefinitions.append("typedef decltype(((::%s*)0)->%s) type_%s;", thisClassScopedName.c_str(), fieldTypeName.c_str(), fieldTypeName.c_str());
            }

            if(!fieldType.isNull())
            {
                getPropertyForType(db, fieldType, fieldTypeName, fieldAttributes, thisClassScopedName, declarationOut, definitionOut);
                return;
            }
        }

//      std::stringstream ret;
//      ret << "HK_PROPERTY_ZERO(" << fieldTypeName << ", /* no attributes */);";
//      return ret.str();
        // No need for PROPERTY_ZERO, it evaluates to nothing anyway
    }

    // If we have a public default constructor and a public parameterless operator new we can generate a managed constructor
    bool shouldGenerateMgdConstructor(const clang::CXXRecordDecl* decl)
    {
        if(decl->hasDefaultConstructor())
        {
            for(auto iter = decl->ctor_begin(); iter != decl->ctor_end(); ++iter)
            {
                if(iter->isDefaultConstructor() && (iter->getAccess() != clang::AS_public)/* && (iter->isUserProvided())*/)
                {
                    return false;
                }
            }

            bool anyExplicitNew = false;

            for(clang::CXXRecordDecl::method_iterator it = decl->method_begin(); it != decl->method_end(); ++it)
            {
                const clang::CXXMethodDecl* method = *it;
                if ( method->getOverloadedOperator() == clang::OO_New && (!method->isImplicit()))
                {
                    // We have at least one explicit operator new
                    anyExplicitNew = true;
                    if((method->getNumParams() == 1) && (method->getAccess() == clang::AS_public))
                    {
                        // This operator new is public and has only the size_t parameter
                        return true;
                    }

                }
            }

            // If we found at least one explicit new, but no normal (non-placement) new, then we can't create it
            if(anyExplicitNew)
            {
                return false;
            }

            return true;
        }
        else
        {
            return false;
        }
    }

    bool shouldExcludeClass(const Database& db, const clang::CXXRecordDecl* decl)
    {
        const std::string& name = decl->getName();
        const Optional<bool>& mgdAttr = db.getAttributeValue< Optional<bool> >(decl, "hk::IncludeInMgd");

        // Copied from the old generation, including the fact that two of them don't even start with hknp
        if((name.find("hknp") == 0))
        {
            if (mgdAttr)
            {
                return !(*mgdAttr); // Include = true -> exclude = false
            }
            return true;
        }
        else
        {
            if (mgdAttr)
            {
                return !(*mgdAttr); // Include = true -> exclude = false
            }
            return false;
        }

        //for (auto it = decl->method_begin(); it != decl->method_end(); ++it)
        //{
        //  if ((*it)->isCopyAssignmentOperator() && ((*it)->getAccess() != clang::AS_public))
        //  {
        //      return true;
        //  }
        //}
    }

    
    void generateRecordDecl(const Database& db, const clang::CXXRecordDecl* recDecl, TextBuilder::Section& classPreDeclarations, TextBuilder::Section& classDeclarations, TextBuilder::Section& classForwardDeclarations, TextBuilder::Section& nativeMapDeclarations, std::set<std::string>& headersToInclude, TextBuilder::Section& classDefinitions, TextBuilder::Section& propertyOffsetDefinitions, bool isTopLevel)
    {
        // TODO: Is scope of parent correct?

        const std::string thisClassScopedName = db.getScopedMangledName(recDecl); //recDecl->getName();
        const std::string thisClassName = recDecl->getName();
        if ( thisClassName == "hkaBoneAttachment" )
        {
            bool b = true;
            b = b;
            b;
        }
        std::string thisClassScope = db.getScopeName(recDecl);
        if(!thisClassScope.empty())
        {
            thisClassScope += ", "; // This is convenient for constructing strings below
        }

        // Temp, exclude some specific classes
        if (shouldExcludeClass(db, recDecl))
        {
            return;
        }

        std::string cxxParent;
        bool derived = false;

        if(recDecl->getNumBases() > 0)
        {
            std::string pyName;
            const clang::Type* parentType = recDecl->bases_begin()->getType()->getUnqualifiedDesugaredType();
            if(const clang::Decl* parentDecl = parentType->getAsCXXRecordDecl())
            {
                if(clang::dyn_cast<clang::ClassTemplateSpecializationDecl>(parentDecl) || clang::dyn_cast<clang::ClassTemplateDecl>(parentDecl))
                {
                    return;
                }

            }

            cxxParent = db.getScopedMangledName(clang::QualType(parentType, 0));

            derived = true;
        }

        const clang::Type* c = recDecl->getTypeForDecl();
        while(c)
        {
            if(clang::dyn_cast<clang::TemplateSpecializationType>(c))
            {
                return;
            }
            if(clang::CXXRecordDecl* rdl = c->getAsCXXRecordDecl())
            {
                c = rdl->getNumBases() ? rdl->bases_begin()->getType().getTypePtr() : 0;
            }
            else
            {
                c = 0;
            }
        }

        bool abstract = recDecl->isAbstract();
        bool definesAttribute = false;
        bool scoped = !thisClassScope.empty();

        hkLog.debug("Adding reflected class %s to HavokAssembly", thisClassScopedName.c_str());

        // A bit simple?
        if(thisClassScopedName == thisClassName)
        {
            classForwardDeclarations.append("ref class %s;", thisClassName.c_str());
        }

        // Always pre-declare it in the current context, only global if we can (can't pre-declare nested struct)
        classPreDeclarations.append("ref class %s;", thisClassName.c_str());

        nativeMapDeclarations.append("template<> struct NativeFromManaged<%s> { typedef ::%s Type; };", thisClassScopedName.c_str(), thisClassScopedName.c_str());
        nativeMapDeclarations.append("template<> struct ManagedFromNative<::%s> { typedef %s Type; };", thisClassScopedName.c_str(), thisClassScopedName.c_str());

        if(const clang::AnnotateAttr* attr = db.getAttribute(recDecl, "dotnet::DefineAttribute"))
        {
            if(attr->getAnnotation().endswith("(true)"))
            {
                definesAttribute = true;
            }
        }
        else if(recDecl->hasAttrs())
        {
            const clang::AttrVec& attrVec = recDecl->getAttrs();
            typedef clang::specific_attr_iterator<clang::AnnotateAttr> SpecificIterator;
            for( SpecificIterator iterator = clang::specific_attr_begin<clang::AnnotateAttr>(attrVec), end_iterator = clang::specific_attr_end<clang::AnnotateAttr>(attrVec);
                iterator != end_iterator;
                ++iterator)
            {
                if(iterator->getAnnotation().startswith(Database::attributePrefix)) // Check that the attribute starts with hk_attr (Havok attribute).
                {
                    const std::string& attrName = db.getAttrNameOrEmpty(*iterator);

                    if(db.getRuntimeAttributeDecl(attrName))
                    {
                        std::string shortenedName;
                        std::string fullName;
                        for(auto ch : attrName)
                        {
                            if(ch == '(')
                            {
                                // Stop when we get to the end of the name
                                break;
                            }
                            if(ch != ':')
                            {
                                shortenedName.push_back(ch);
                            }
                            fullName.push_back(ch);
                        }

                        if(!shouldSkipClassAttribute(fullName))
                        {
                            classDeclarations.append("[%s(%s::typeid, nullptr, \"%s\")]", shortenedName.c_str(), recDecl->getNameAsString().c_str(), fullName.c_str());
                        }
                    }
                }
            }
        }

        //std::string classAttributes = "/* no attributes */";
        std::string publicString(isTopLevel ? "public " : "");

        if(!derived)
        {
            if(abstract)
            {
                classDeclarations.append(publicString);
                classDeclarations.append("ref class %s abstract{\npublic:\n", thisClassName.c_str());
                classDefinitions.append("HK_MANAGED_CLASS_DEFINE%s(%s %s);", (scoped ? "_SCOPED" : ""), thisClassScope.c_str(), thisClassName.c_str());
            }
            else
            {
                if(definesAttribute)
                {
                    classDeclarations.append("[AttributeUsage(AttributeTargets::Class | AttributeTargets::Property, AllowMultiple=false)]\n");
                    classDeclarations.append(publicString);
                    classDeclarations.append("ref class %s : public Attribute\n{\npublic:\n", thisClassName.c_str()); // Can't be scoped, can't have attributes??
                    classDefinitions.append("HK_MANAGED_ATTRIBUTE_CLASS_DEFINE(%s);", thisClassName.c_str());
                }
                else
                {
                    classDeclarations.append(publicString);
                    classDeclarations.append("ref class %s\n{\npublic:\n", thisClassName.c_str());
                    classDefinitions.append("HK_MANAGED_CLASS_DEFINE%s(%s %s);", (scoped ? "_SCOPED" : ""), thisClassScope.c_str(), thisClassName.c_str());
                }
            }
            classDeclarations.append("HK_MANAGED%s_CLASS_BASE_SCOPED(%s, %s);\n", (definesAttribute ? "_ATTRIBUTE" : ""), thisClassScopedName.c_str(), thisClassName.c_str());

//          if(recDecl->getNameAsString() == "hkFloat16")
//          {
//              std::string tmp = recDecl->getNameAsString();
//          }

            if ((!abstract) && (!definesAttribute) && shouldGenerateMgdConstructor(recDecl))
            {
                classDeclarations.append("HK_MANAGED_CLASS_CTOR_SCOPED(%s, %s, ());", thisClassScopedName.c_str(), thisClassName.c_str());
                classDefinitions.append("HK_MANAGED_CLASS_CTOR_DEFINE_START%s(%s %s, ())", (scoped ? "_SCOPED" : ""), thisClassScope.c_str(), thisClassName.c_str());
                classDefinitions.append("HK_MANAGED_CLASS_CTOR_DEFINE_END();");
            }
        }
        else
        {
            if(abstract)
            {
                classDeclarations.append(publicString);
                classDeclarations.append("ref class %s abstract : public %s\n{\npublic:\n", thisClassName.c_str(), cxxParent.c_str());
            }
            else
            {
                classDeclarations.append(publicString);
                classDeclarations.append("ref class %s : public %s\n{\npublic:\n", thisClassName.c_str(), cxxParent.c_str());
            }

            classDefinitions.append("HK_MANAGED_CLASS_DEFINE_DERIVED%s(%s %s, %s);", (scoped ? "_SCOPED" : ""), thisClassScope.c_str(), thisClassName.c_str(), cxxParent.c_str());

            classDeclarations.append("HK_MANAGED_CLASS_BASE_DERIVED_SCOPED(%s, %s, %s);", thisClassScopedName.c_str(), thisClassName.c_str(), cxxParent.c_str());

            if ((!abstract) && shouldGenerateMgdConstructor(recDecl))
            {
                classDeclarations.append("HK_MANAGED_CLASS_CTOR_DERIVED_SCOPED(%s, %s, (), %s);", thisClassScopedName.c_str(), thisClassName.c_str(), cxxParent.c_str());
                
                classDefinitions.append("HK_MANAGED_CLASS_CTOR_DEFINE_START_DERIVED%s(%s %s, (), %s)", (scoped ? "_SCOPED" : ""), thisClassScope.c_str(), thisClassName.c_str(), cxxParent.c_str());
                classDefinitions.append("HK_MANAGED_CLASS_CTOR_DEFINE_END();");
            }
        }

        if(clang::dyn_cast<clang::DeclContext>(recDecl)) // Can this ever be false??
        {
            recursivelyCreateDataFromDeclContext(db, recDecl, classDeclarations, classForwardDeclarations, nativeMapDeclarations, headersToInclude, classDefinitions, propertyOffsetDefinitions, false);
        }

        propertyOffsetDefinitions.append("struct ::%s::_HABinding {", thisClassScopedName.c_str());

        // If we are hk::ReflectDetails(fields=false), skip them
        const Attribute& details = db.getReflectDetails(recDecl);
        if(details["fields"]->as<bool>())
        {
            // TODO: Standardise all this
            int memIndex = 0;
            for(clang::CXXRecordDecl::field_iterator it = recDecl->field_begin(); it != recDecl->field_end(); ++it, ++memIndex)
            {
                const clang::FieldDecl* fieldDecl = *it;

                const Optional<bool>& reflAttr = db.getAttributeValue< Optional<bool> >(fieldDecl, "hk::Reflect");
                const Optional<bool>& includeInMgdAttr = db.getAttributeValue< Optional<bool> >(fieldDecl, "hk::IncludeInMgd");

                // Skip hk::Reflect(false) and hk::IncludeInMgd(false)
                if ((!reflAttr || *reflAttr) && (!includeInMgdAttr || *includeInMgdAttr))
                {
                    std::string declaration;
                    std::string definition;
                    generateFieldPropertyDecl(db, fieldDecl, thisClassName, thisClassScopedName, memIndex, declaration, definition, propertyOffsetDefinitions);
                    classDeclarations.append(declaration);
                    classDefinitions.append(definition);

                }
            }

            for(auto it = recDecl->decls_begin(); it != recDecl->decls_end(); ++it)
            {

                const clang::Decl* thisDecl = *it;
                if(const clang::TypedefDecl* tdd = clang::dyn_cast<clang::TypedefDecl>(thisDecl))
                {
                    if(db.isReflected(tdd))
                    {
                        // Nested typedef
                        std::string dn = tdd->getDeclName().getAsString();

                        clang::QualType resolvedType = resolveTypedefType(tdd->getUnderlyingType());
                        if(const clang::NamedDecl* dd = resolvedType->getAsCXXRecordDecl())
                        {
                            const std::string& ddn = db.getScopedMangledName(dd);
                            classDeclarations.append("typedef %s %s;\n", ddn.c_str(), dn.c_str());
                        }
                    }
                }
            }
        }
        else
        {
            classDeclarations.append("// Skipping fields of %s due to hk::ReflectDetails", thisClassScopedName.c_str());
        }
        propertyOffsetDefinitions.append("};");
        classDeclarations.append("};");
    }

    // Generate the HA binding for this project only, these need to be combined to generate the overall binding file
    void recursivelyCreateDataFromDeclContext(const Database& db, const clang::DeclContext* thisContext, TextBuilder::Section& classDeclarationsSplit, TextBuilder::Section& classForwardDeclarations, TextBuilder::Section& nativeMapDeclarations, std::set<std::string>& headersToInclude, TextBuilder::Section& classDefinitions, TextBuilder::Section& propertyOffsetDefinitions, bool isTopLevel)
    {
        TextBuilder::CodeSection classPreDeclarations;
        TextBuilder::CodeSection classActualDeclarations;

        for(auto it = clang::DeclContext::specific_decl_iterator<clang::NamedDecl>(thisContext->decls_begin());
            it != clang::DeclContext::specific_decl_iterator<clang::NamedDecl>(thisContext->decls_end()); it++)
        {
            const clang::NamedDecl* baseDecl = *it;
            {
                const std::string& curFileName = Database::getSourceFileName(baseDecl);
                if(curFileName.empty() || curFileName.find("UnitTest") != std::string::npos ||
                    db.inputFilesDecls.find(curFileName) == db.inputFilesDecls.end())
                {
                    continue;
                }

                if (auto decl = clang::dyn_cast<clang::TypeDecl>(baseDecl))
                {
                    if (clang::isa<clang::TagDecl>(decl) && !clang::dyn_cast<clang::TagDecl>(decl)->isThisDeclarationADefinition())
                    {
                        continue;
                    }
                    if(db.isReflected(decl))
                    {
                        hkLog.debug(decl, "Considering %s (type = %s)", decl->getNameAsString().c_str(), decl->getDeclKindName());

                        if(curFileName.length() == 0)
                        {
                            continue;
                        }

                        TkbmsRegistry::Entry tkbms;
                        bool hasTkbms = db.tkbmsRegistry.tryGetTkbmsEntry(curFileName, tkbms);
                        if(hasTkbms && (tkbms.visibility == "INTERNAL"))
                        {
                            continue;
                        }

                        //const std::string& declaredName = decl->getNameAsString();
                        // Reflected declaration
                        if(const clang::ClassTemplateSpecializationDecl* specDecl = clang::dyn_cast<clang::ClassTemplateSpecializationDecl>(decl))
                        {
                            //hkLog.debug("Ignoring %s (type = %s)", declaredName.c_str(), decl->getDeclKindName());
                        }
                        else if(clang::dyn_cast<clang::ClassTemplateDecl>(decl))
                        {
                            // ignore
                            //hkLog.debug("Ignoring %s (type = %s)", declaredName.c_str(), decl->getDeclKindName());
                        }
                        else if(const clang::CXXRecordDecl* recDecl = clang::dyn_cast<clang::CXXRecordDecl>(decl))
                        {
                            for (const clang::DeclContext* context = recDecl->getDeclContext(); context->isRecord(); context = context->getParent())
                            {
                                if (getTemplateDecl(clang::dyn_cast<clang::CXXRecordDecl>(context)))
                                {
                                    throw RuntimeError(recDecl, "In %s: nested struct in template struct '%s' is reflected, this is not supported",
                                        curFileName.c_str(), db.getScopedMangledName(recDecl).c_str());
                                }
                            }

                            const clang::ClassTemplateDecl* tplDecl = getTemplateDecl(recDecl);
                            if (tplDecl)
                            {
                                //hkLog.debug("Ignoring %s (type = %s)", declaredName.c_str(), decl->getDeclKindName());
                            }
                            else
                            {
                                // Normal reflected record
                                headersToInclude.insert(curFileName);
                                generateRecordDecl(db, recDecl, classPreDeclarations, classActualDeclarations, classForwardDeclarations, nativeMapDeclarations, headersToInclude, classDefinitions, propertyOffsetDefinitions, isTopLevel);
                            }
                        }
                        // Don't need to do this for now
//                      else if(const clang::TypedefDecl* tdefDecl = clang::dyn_cast<clang::TypedefDecl>(decl))
//                      {
//                          // Don't reflect typedefs at the top level for now
//                          if (!isTopLevel && db.isReflected(tdefDecl))
//                          {
//                              //hkLog.debug("Typedef %s (type = %s)", declaredName.c_str(), decl->getDeclKindName());
//                              std::string tdefName(tdefDecl->getName());
//                              std::string tdefPointsTo(db.getScopedMangledName(tdefDecl->getUnderlyingType()));
//                              classPreDeclarations.append("typedef %s %s;", tdefPointsTo.c_str(), tdefName.c_str()); // , db.getScopedMangledName(tdefDecl->getUnderlyingType()));
//                          }
//                      }
                        else if(const clang::EnumDecl* enumDecl = clang::dyn_cast<clang::EnumDecl>(decl))
                        {
                            const std::string& declaredName = decl->getNameAsString();
                            hkLog.debug("Enum %s (type = %s)", declaredName.c_str(), decl->getDeclKindName());
                            headersToInclude.insert(curFileName);
                            generateEnumDecl(db, enumDecl, classActualDeclarations, nativeMapDeclarations);
                        }
                        else
                        {
                            //hkLog.debug(decl, "Skipping %s (type = %s)", declaredName.c_str(), decl->getDeclKindName());
                        }
                    }
                    else
                    {
                        hkLog.debug(decl, "Skipping %s (type = %s) NOT IN INPUTS", decl->getNameAsString().c_str(), decl->getDeclKindName());
                    }
                }
                else if (const clang::NamespaceDecl* nsDecl = clang::dyn_cast<clang::NamespaceDecl>(baseDecl))
                {
                    TextBuilder::CodeSection classDeclTemp;
                    recursivelyCreateDataFromDeclContext(db, nsDecl, classDeclTemp, classForwardDeclarations, nativeMapDeclarations, headersToInclude, classDefinitions, propertyOffsetDefinitions, false);

                    if (!classDeclTemp.isEmpty())
                    {
                        classActualDeclarations.append("namespace %s\n{\n", nsDecl->getName().str().c_str());
                        classActualDeclarations.append(classDeclTemp.toString());
                        classActualDeclarations.append("}");
                    }
                }
                else
                {
                    if(baseDecl)
                    {
                        hkLog.debug(baseDecl, "Skipping %s (type = %s) NOT TAG/NAMESPACE", baseDecl->getNameAsString().c_str(), baseDecl->getDeclKindName());
                    }
                }
            }
        }

        // Don't output an empty namespace, if we have content, pre-declare the nested
        // types (so we can refer to them from other nested types) then output the rest
        if (!classPreDeclarations.isEmpty() || !classActualDeclarations.isEmpty())
        {
            classDeclarationsSplit.append(classPreDeclarations.toString());
            classDeclarationsSplit.append(classActualDeclarations.toString());
        }
    }
}

void generateHABinding(const Database& db)
{
    {
        const std::string& outputDir = db.getOutputDir();

        bool existed;
        const std::string& typesOutputDir = outputDir + "/HABinding/";
        if(llvm::sys::fs::create_directories(typesOutputDir, existed) != llvm::errc::success)
            throw FatalError("Could not create " + typesOutputDir);
    }

    TextBuilder cxxBinding;
    TextBuilder headerBinding;

    {
        cxxBinding.addSection("startBoilerPlate", new TextBuilder::VerbatimSection());
        cxxBinding.addSection("includes", new TextBuilder::IncludeSection(db.includePaths));
        cxxBinding.addSection("propertyOffsetDefinitions", new TextBuilder::CodeSection());
        cxxBinding.addSection("definitions", new TextBuilder::CodeSection());
    }

    {
        headerBinding.addSection("includePrefix", new TextBuilder::VerbatimSection());
        headerBinding.addSection("includeDecls", new TextBuilder::IncludeSection(db.includePaths));
        headerBinding.addSection("declarations", new TextBuilder::VerbatimSection());
    }

    {
        cxxBinding["startBoilerPlate"].append("#include <HavokAssembly/hkHavokAssembly.h>");
        cxxBinding["startBoilerPlate"].append("#pragma unmanaged");
        cxxBinding["startBoilerPlate"].append("#include <HavokAssembly/hkHavokUnmanaged.h>");
        cxxBinding["startBoilerPlate"].append("#pragma managed");
        cxxBinding["startBoilerPlate"].append("#include <HavokAssembly/hkHavokManagedTypes.h>");
        cxxBinding["startBoilerPlate"].append("#include <HavokAssembly/hkHavokManagedUtils.h>");

        cxxBinding["startBoilerPlate"].append("#include \"" + db.getProjectNameRoot() + "_HABinding.hxx\"");
    }

    headerBinding["includePrefix"].append("#ifndef " + db.getProjectNameRoot() + "_HABinding_h_INCLUDED");
    headerBinding["includePrefix"].append("#define " + db.getProjectNameRoot() + "_HABinding_h_INCLUDED");

    TextBuilder::CodeSection classDeclarations;
    TextBuilder::CodeSection classForwardDecls;
    TextBuilder::CodeSection nativeMapDeclarations;

    std::set<std::string> headersToInclude;
    clang::TranslationUnitDecl* dbRoot = db.getRoot();

    classDeclarations.append("namespace Havok");
    classDeclarations.append("{");

    classForwardDecls.append("namespace Havok");
    classForwardDecls.append("{");

    nativeMapDeclarations.append("namespace Havok");
    nativeMapDeclarations.append("{");

    cxxBinding["definitions"].append("namespace Havok");
    cxxBinding["definitions"].append("{");

    cxxBinding["propertyOffsetDefinitions"].append("#pragma unmanaged");

    recursivelyCreateDataFromDeclContext(db, dbRoot, classDeclarations, classForwardDecls, nativeMapDeclarations, headersToInclude, cxxBinding["definitions"], cxxBinding["propertyOffsetDefinitions"], true);

    classDeclarations.append("}");
    classForwardDecls.append("}");
    nativeMapDeclarations.append("}");
    cxxBinding["definitions"].append("}");

    cxxBinding["propertyOffsetDefinitions"].append("#pragma managed");

    {
        std::string forwardDecls;
        classForwardDecls.format(forwardDecls);
        std::string decls;
        classDeclarations.format(decls);
        std::string nativeMap;
        nativeMapDeclarations.format(nativeMap);
        headerBinding["declarations"].append("#pragma managed");
        headerBinding["declarations"].append(forwardDecls);
        headerBinding["declarations"].append(decls);
        headerBinding["declarations"].append(nativeMap);

        headerBinding["declarations"].append("#endif // include guard");
    }

    headerBinding["includePrefix"].append("//\n\
// All reflected class headers\n\
//\n\
#pragma unmanaged\n\
");
    for(auto it = headersToInclude.begin(); it != headersToInclude.end(); ++it)
    {
        headerBinding["includeDecls"].append(it->c_str());
    }

    {
        const std::string& cppOutputFileName = db.getOutputDir() + "/HABinding/" + db.getProjectNameRoot() + "_HABinding.cxx";
        const std::string& headerOutputFileName = db.getOutputDir() + "/HABinding/" + db.getProjectNameRoot() + "_HABinding.hxx";

        db.fs.openRW(cppOutputFileName)
            << db.getPreamble(cppOutputFileName)
            << cxxBinding.format()
            << db.getPostamble(cppOutputFileName);
        db.fs.openRW(headerOutputFileName)
            << db.getPreamble(headerOutputFileName)
            << headerBinding.format()
            << db.getPostamble(headerOutputFileName);
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
