// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
#include "utils/Logger.h"
#include "utils/Database.h"
#include "utils/StlUtils.h"
#include "utils/RuntimeError.h"
#include "utils/ElementUtils.h"
#include "AttributeRegistry.h"
#include "utils/Attribute.h"
#include "utils/HavokUtils.h"
#include "utils/TypeOut.h"

PRAGMA_WARNING_PUSH
#include <clang/Sema/Sema.h>
PRAGMA_WARNING_POP
namespace
{
    bool isAutoTypedAttr(const clang::CXXRecordDecl* attrDecl)
    {
        const clang::CXXRecordDecl* currDecl = attrDecl;
        // go up through the hierarchy until we find hk::AutoType or we get to the root
        for (;;)
        {
            if (currDecl->getQualifiedNameAsString() == "hk::AutoType")
            {
                return true;
            }
            if (currDecl->bases_begin() == currDecl->bases_end())
            {
                return false;
            }
            currDecl = getCXXRecordFrom(currDecl->bases_begin()->getType());
            checkIsDefinition(currDecl);
        }
    }

    std::string getAttrValue(const Database& db, const std::string& value, const clang::FieldDecl* field)
    {
        if (auto castAttr = db.getAttributeValue< Optional<Attribute> >(field, "hk::Cast"))
        {
            return formatString("%s( %s )", castAttr->as<std::string>().c_str(), value.c_str());
        }
        else if (auto ptr = clang::dyn_cast<clang::PointerType>(resolveType(field->getType())))
        {
            // Check if it is a type pointer.
            if (db.getScopedMangledName(ptr->getPointeeType().getLocalUnqualifiedType()) == "hkReflect::Type")
            {
                return formatString("HK_REFLECT_GET_TYPE( %s )", value.c_str());
            }
        }

        return value;
    }

    std::string addConst(const std::string& type)
    {
        // todo.ntm this is very hackish, use hkTrait::AddConst?
        return llvm::StringRef(type).rtrim().endswith("*") ? (type + " const") : ("const " + type);
    }
}

PresetsData generatePresetsData(const Attribute& attr, const Database& db, const clang::NamedDecl* decl, TypeOut& typeOut, const std::string& sectionScope)
{
    Presets presets;

    std::string strict = attr["strict"]->as<std::string>();
    if(strict == "true")
    {
        presets.strict = true;
    }
    else if(strict == "false")
    {
        presets.strict = false;
    }
    else
    {
        hkLog.error(db.getSourceManager(), attr.getLocation(), "Invalid strict value.");
    }

    if (Optional<AttributeValue> t = attr["type"])
    {
        presets.valueType = *t;
        presets.type = formatString("HK_REFLECT_GET_TYPE( %s )", t->as<std::string>().c_str());
    }
    else
    {
        clang::QualType declType = getTypeFrom(db, decl);
        presets.valueType = db.getScopedMangledName(declType);
        presets.type = db.getReflectedTypeAddress(declType);
    }

    for(Attribute::PosValuesIterator it = attr.posBegin(); it != attr.posEnd(); ++it)
    {
        const std::vector<std::string>& vals = splitString(*it, '=');
        if(vals.size() != 2)
        {
            throw RuntimeError(decl, "Invalid preset %s", it);
        }

        Presets::Val presetVal;
        presetVal.name = vals[0];
        presetVal.value = vals[1];
        presets.values.push_back(presetVal);
    }

    return generatePresetsData(presets, db, decl, typeOut, sectionScope);
}

PresetsData generatePresetsData(const Presets& presets, const Database& db, const clang::NamedDecl* decl, TypeOut& typeOut, const std::string& sectionScope)
{
    PresetsData res;

    res.strict = presets.strict ? "true" : "false";
    res.type = presets.type;

    bool hasAttribs = false;
    for(const Presets::Val& val : presets.values)
    {
        if(!val.attribs.empty())
        {
            hasAttribs = true;
            break;
        }
    }

    //  names and values
    std::ostringstream names, valuesArray, attribsArrays;
    names << "{";
    valuesArray << "{";

    if(hasAttribs)
    {
        attribsArrays << "{";
    }

    res.count = 0;
    for(std::vector<Presets::Val>::const_iterator it = presets.values.begin();
        it != presets.values.end(); ++it)
    {
        names << "\"" << it->name << "\"";

        std::string valueSym = formatString("%shkPresets_%s", sectionScope.c_str(), it->name.c_str());
        std::string valueContent = formatString("static_cast< %s >(%s)", presets.valueType.c_str(), it->value.c_str());
        typeOut.addData(valueSym, addConst(presets.valueType), valueContent);

        valuesArray << " &" << sectionScope << "hkPresets_" << it->name;

        if(it->attribs.size())
        {
            std::string attribsArraySym = formatString("%shkPresets_%s_attribsArray", sectionScope.c_str(), it->name.c_str());
            std::string attribsArrayType = formatString("const hkReflect::Detail::FixedArrayStorage<hkUlong, hkReflect::Detail::AttributeItem, %i>", it->attribs.size());
            std::string attribPrefix = it->name + "_";

            std::ostringstream attribsArrayContent;
            attribsArrayContent << "\n";
            attribsArrayContent << "{\n";
            attribsArrayContent << it->attribs.size() << ",\n";
            attribsArrayContent << "{\n";

            for(int i = 0; i < (int)it->attribs.size(); i++)
            {
                const Attribute& attrib = it->attribs[i];

                AttributeRegistry::AttributeFunction attrFun = AttributeRegistry::getAttribute(attrib.getAttrName());
                assert(attrFun); // We should always have an attribute handler, either the attribute's one or the generic one.

                std::string attrSym, attrType, attrStruct;
                std::tie(attrSym, attrType, attrStruct) = (*attrFun)(attrib, db, decl, typeOut, attribPrefix);

                attribsArrayContent << formatString("{ &%s, HK_REFLECT_GET_TYPE( %s ) }%s",
                    attrSym.c_str(), attrType.c_str(), i == (int)it->attribs.size() - 1 ? "" : ", ") << std::endl;
            }

            attribsArrayContent << "}\n";
            attribsArrayContent << "};";

            typeOut.addData(attribsArraySym, attribsArrayType, attribsArrayContent.str());

            attribsArrays << "reinterpret_cast<const hkReflect::Detail::AttributeArray*>(&" << sectionScope << "hkPresets_" << it->name << "_attribsArray)";
        }
        else if(hasAttribs)
        {
            attribsArrays << " 0";
        }

        if(it + 1 != presets.values.end())
        {
            names << ',';
            valuesArray << ',';

            if(hasAttribs)
            {
                attribsArrays << ',';
            }
        }
        ++res.count;
    }

    names << " }";
    valuesArray << " }";

    if(hasAttribs)
    {
        attribsArrays << " }";
    }

    std::string presets_names = sectionScope + "hkPresets_names";
    std::string presets_values = sectionScope + "hkPresets_value";

    typeOut.addData(presets_names, "const char* const[]", names.str());
    typeOut.addData(presets_values, "const void* const[]", valuesArray.str());

    res.names = presets_names;
    res.values = presets_values;
    res.hasAttribs = hasAttribs;

    if(hasAttribs)
    {
        std::string presets_attribsArrays = sectionScope + "hkPresets_attribsArrays";
        typeOut.addData(presets_attribsArrays, "const hkReflect::Detail::AttributeArray* const[]", attribsArrays.str());
        res.attribArrays = presets_attribsArrays;
    }

    return res;
}

namespace
{
    // Get the signature of a method in the form "name(args)" (used to distinguish overloaded methods).
    std::string getNameAndParams( const Database& db, const clang::CXXMethodDecl* m )
    {
        std::string res;
        {
            llvm::raw_string_ostream str( res );
            m->printName( str );
            str << '(';
            for ( auto pit = m->param_begin(); pit != m->param_end(); ++pit )
            {
                if ( pit != m->param_begin() )
                {
                    str << ",";
                }
                // Normalize type name.
                std::string paramType = db.getScopedMangledName( (*pit)->getType(), Database::NAME_DECLARATION );
                paramType.erase( std::remove( paramType.begin(), paramType.end(), ' ' ), paramType.end() );
                str << paramType;
            }
            str << ')';
        }
        return res;
    }

    void lookupMethod(
        const Database& db,
        clang::SourceLocation loc,
        std::string lookFor,
        const char* methodType,
        const clang::CXXRecordDecl* recDecl,
        llvm::raw_string_ostream& strOut )
    {
        if ( lookFor == "default" )
        {
            strOut << "HK_NULL";
            return;
        }

        bool explicitSignature = lookFor.find( '(' ) != std::string::npos;

        if ( explicitSignature )
        {
            // Normalize.
            lookFor.erase( std::remove( lookFor.begin(), lookFor.end(), ' ' ), lookFor.end() );
        }

        const clang::CXXMethodDecl* result = nullptr;
        for ( auto mit = recDecl->method_begin(); mit != recDecl->method_end(); ++mit )
        {
            const clang::CXXMethodDecl* method = *mit;

            // Check the name
            const std::string& methodName = method->getNameAsString();
            if ( (!explicitSignature && llvm::StringRef( methodName ).equals_lower( lookFor )) ||
                (explicitSignature && getNameAndParams( db, method ) == lookFor) )
            {
                if ( db.isReflected( method ) )
                {
                    if ( result == nullptr )
                    {
                        result = method;
                    }
                    else
                    {
                        
                        hkLog.error( db.getSourceManager(), loc, "More than one %s method named '%s'", methodType, lookFor.c_str() );
                        strOut << "HK_NULL";
                        return;
                    }
                }
            }
        }
        if ( !result )
        {
            
            hkLog.error( db.getSourceManager(), loc, "No valid %s method named '%s'", methodType, lookFor.c_str() );
            strOut << "HK_NULL";
        }
        else
        {
            strOut << "reinterpret_cast<const hkReflect::Callable*>("
                "&Callables::values[INDEX_" << db.getMangledMethodName( result ) << "])";
        }
    }

    std::tuple<std::string, std::string, std::string> generateExposeAttr(
        const Attribute& attr,
        const Database& db,
        const clang::NamedDecl* decl,
        TypeOut& typeOut,
        const std::string& sectionScope,
        const char* defaultGetFmt,
        const char* defaultSetFmt,
        bool isMap)
    {
        const std::string& attrTypeName = attr.getAttrName();
        const std::string& attrSection = sectionScope + pyNameFromCxxName( attrTypeName ) + "_attr";

        // Get the name of the field w/o prefix.
        const clang::FieldDecl* field = clang::dyn_cast<clang::FieldDecl>(decl);
        if ( !field )
        {
            throw RuntimeError( db.getSourceManager(), attr.getLocation(), "hk::Expose can be only used on fields." );
        }
        auto recDecl = clang::dyn_cast<clang::CXXRecordDecl>(field->getDeclContext());
        assert( recDecl );
        const std::string& fieldPrefix = db.getAttributeValue<std::string>( recDecl, "hk::FieldPrefix" );
        const std::string& fieldName = decl->getNameAsString();
        const std::string& cleanFieldName = fieldNameWithoutPrefix( fieldPrefix, fieldName );

        std::string values;
        {
            llvm::raw_string_ostream valuesStr( values );

            valuesStr << "{ " << *attr["access"] << ", " << *attr["scope"] << ", ";

            if ( auto valueType = attr["valueType"] )
            {
                valuesStr << "HK_REFLECT_GET_TYPE( " << *valueType << " )";
            }
            else
            {
                valuesStr << "HK_NULL";
            }

            valuesStr << ", ";

            bool canGet = llvm::StringRef( *attr["access"] ).find( "GET" ) != llvm::StringRef::npos;
            bool canSet = llvm::StringRef( *attr["access"] ).find( "SET" ) != llvm::StringRef::npos;

            if ( canGet )
            {
                lookupMethod( db, attr.getLocation(),
                    attr["getter"] ? *attr["getter"] : formatString( defaultGetFmt, cleanFieldName.c_str() ),
                    "getter", recDecl, valuesStr );
                
            }
            else
            {
                valuesStr << "HK_NULL";
            }

            valuesStr << ", ";

            if ( canSet )
            {
                lookupMethod( db, attr.getLocation(),
                    attr["setter"] ? *attr["setter"] : formatString( defaultSetFmt, cleanFieldName.c_str() ),
                    "setter", recDecl, valuesStr );
                
            }
            else
            {
                valuesStr << "HK_NULL";
            }

            if ( isMap )
            {
                valuesStr << ", ";
                if ( auto keysMeth = attr["listKeys"] )
                {
                    lookupMethod( db, attr.getLocation(), *keysMeth, "listKeys", recDecl, valuesStr );
                }
                else
                {
                    valuesStr << "HK_NULL";
                }

                valuesStr << ", ";
                if ( auto keysMeth = attr["cacheKey"] )
                {
                    lookupMethod( db, attr.getLocation(), *keysMeth, "cacheKey", recDecl, valuesStr );
                }
                else
                {
                    valuesStr << "HK_NULL";
                }
            }

            valuesStr << " }";
        }

        // Make the attribute itself a parent of the generated struct.
        const std::string& sectionParentString = formatString( ": public %s ", attrTypeName.c_str() );
        typeOut.addData( "attr", getAggregateTypeFor( attrTypeName, false ), values, attrSection, 0, sectionParentString );
        return std::make_tuple( typeOut.getDataQualifier() + attrSection + "::attr", attrTypeName, attrSection );
    }
}

namespace hk
{
    std::tuple<std::string, std::string, std::string> Presets(
        const Attribute& attr,
        const Database& db,
        const clang::NamedDecl* decl,
        TypeOut& typeOut,
        const std::string& sectionScope)
    {
        PresetsData presets = generatePresetsData(attr, db, decl, typeOut, sectionScope);

        const std::string& content = formatString("{ %s, %s, %d, %s, %s, HK_NULL }",
            presets.strict.c_str(), presets.type.c_str(), presets.count, presets.names.c_str(), presets.values.c_str());

        std::string attr_presets = sectionScope + "attr_hkPresets";

        typeOut.addData(attr_presets, "hk::Presets", content);

        return std::make_tuple(attr_presets, "hk::Presets", attr_presets);
    }
    REGISTER_ATTRIBUTE(hk::Presets);

    std::tuple<std::string, std::string, std::string> Expose(
        const Attribute& attr,
        const Database& db,
        const clang::NamedDecl* decl,
        TypeOut& typeOut,
        const std::string& sectionScope )
    {
        return generateExposeAttr( attr, db, decl, typeOut, sectionScope, "get%s", "set%s", false );
    }
    REGISTER_ATTRIBUTE( hk::Expose );

    std::tuple<std::string, std::string, std::string> ExposeAsMap(
        const Attribute& attr,
        const Database& db,
        const clang::NamedDecl* decl,
        TypeOut& typeOut,
        const std::string& sectionScope )
    {
        return generateExposeAttr( attr, db, decl, typeOut, sectionScope, "getIn%s", "setIn%s", true );
    }
    REGISTER_ATTRIBUTE( hk::ExposeAsMap );

    std::tuple<std::string, std::string, std::string> ExposeAsFlags(
        const Attribute& attr,
        const Database& db,
        const clang::NamedDecl* decl,
        TypeOut& typeOut,
        const std::string& sectionScope )
    {
        return generateExposeAttr( attr, db, decl, typeOut, sectionScope, "get%sbits", "set%sbits", true );
    }
    REGISTER_ATTRIBUTE( hk::ExposeAsFlags );

    std::tuple<std::string, std::string, std::string> defaultHandler(
        const Attribute& attr,
        const Database& db,
        const clang::NamedDecl* decl,
        TypeOut& typeOut,
        const std::string& sectionScope)
    {
        const std::string& attrTypeName = attr.getAttrName();
        const std::string& attrSection = sectionScope + pyNameFromCxxName(attrTypeName) + "_attr";

        const AttributeDecl* attrDecl = db.getRuntimeAttributeDecl(attrTypeName);

        if(attrDecl == 0)
            throw RuntimeError(decl, "Could not find a C++ declaration for the attribute '%s'.", attrTypeName.c_str());

        const clang::CXXRecordDecl* recAttrDecl = attrDecl->getRec();

        std::string storageTypeName;
        std::string value;
        if(isAutoTypedAttr(recAttrDecl))
        {
            clang::QualType baseType = getTypeFrom(db, decl);
            const std::string& cxxName = db.getScopedMangledName(baseType);
            storageTypeName = getAggregateTypeFor(cxxName, typeOut.hasTemplateParams());
            value = attr.getPosValuesString();
        }
        else
        {
            storageTypeName = getAggregateTypeFor( attrTypeName, false );

            std::ostringstream ostr;
            ostr << "{ ";

            // Traverse fields from parent to child.
            std::vector<const clang::CXXRecordDecl*> hierarchy;
            for (const clang::CXXRecordDecl* currDecl = recAttrDecl; currDecl;
                currDecl = (currDecl->getNumBases() ? getCXXRecordFrom(currDecl->bases_begin()->getType()) : nullptr))
            {
                checkIsDefinition(currDecl);
                hierarchy.push_back(currDecl);
            }

            for (auto rit = hierarchy.rbegin(); rit != hierarchy.rend(); ++rit)
            {
                auto currDecl = *rit;
                const std::string& fieldPrefix = db.getAttributeValue<std::string>( currDecl, "hk::FieldPrefix" );
                for(clang::CXXRecordDecl::field_iterator fit = currDecl->field_begin(); fit != currDecl->field_end(); ++fit)
                {
                    const std::string& fieldName = fieldNameWithoutPrefix(fieldPrefix, fit->getNameAsString());
                    clang::QualType fieldType = resolveType(fit->getType());
                    const clang::CXXRecordDecl* fieldAsRecDecl = getCXXRecordFrom(fieldType);

                    if (Optional<AttributeValue> val = attr[fieldName])
                    {
                        ostr << getAttrValue(db, *val, *fit);
                    }
                    else if (fieldAsRecDecl &&
                        fieldAsRecDecl->getQualifiedNameAsString() == "hk::ValueArray")
                    {
                        if (attr.getNumPosValues())
                        {
                            // get the element type
                            const std::string& elemName = formatString("const %s::ElementType",
                                db.getScopedMangledName(fieldAsRecDecl).c_str());

                            const std::string& arraySection = pyNameFromCxxName(db.getScopedMangledName(decl)) + "_" + pyNameFromCxxName(attrTypeName) + "_attrValues";
                            const std::string& arraySym = typeOut.getDataQualifier() + arraySection + "::values";

                            typeOut.addData("values", elemName + "[]", attr.getPosValuesString(true), arraySection);
                            ostr << "{" << arraySym << ", " << arraySym << " + " << attr.getNumPosValues() << "}";
                        }
                        else
                        {
                            ostr << "{ HK_NULL, HK_NULL }";
                        }
                    }
                    else
                    {
                        throw RuntimeError(decl, "Evaluation of attribute '%s' didn't produce a value for field '%s'.", attrTypeName.c_str(), fit->getName().str().c_str());
                    }

                    clang::CXXRecordDecl::field_iterator next(fit);
                    ++next;
                    if(next != currDecl->field_end())
                        ostr << ", ";
                }
            }

            ostr << " }";
            value = ostr.str();
        }

        // Make the attribute itself a parent of the generated struct.
        std::string sectionParentString = formatString(": public %s ", db.getScopedMangledName(recAttrDecl).c_str());
        typeOut.addData("attr", addConst(storageTypeName), value, attrSection, 0, sectionParentString);
        const std::string& attrSym = typeOut.getDataQualifier() + attrSection + "::attr";
        return std::make_tuple(attrSym, attrTypeName, attrSection);
    }
    REGISTER_ATTRIBUTE(defaultHandler);
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
