// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
PRAGMA_WARNING_PUSH
    #include <clang/Basic/SourceManager.h>
    #include <clang/Sema/Sema.h>
    #include <llvm/Support/ManagedStatic.h>
PRAGMA_WARNING_POP

#include "rawDump.h"
#include "utils/Logger.h"
#include "utils/StlUtils.h"
#include "utils/Database.h"
#include "utils/ElementUtils.h"
#include "utils/Filesystem.h"

using namespace clang;

#define printf poisoned
#ifdef _WIN32
    inline int local_stricmp(const char* a, const char* b) { return _stricmp(a,b); }
#else
    #include <stdio.h>
    inline int local_stricmp(const char* a, const char* b) { return strcasecmp(a,b); }
#endif


////////////////////////////////////////////////////////////////////////////
////////////////////// TEMPORARY ATTRIBUTES HANDLER ////////////////////////
/////// TO BE REMOVED WHEN WE USE THE PROPER SYNTAX ALL OVER THE SDK ///////
////////////////////////////////////////////////////////////////////////////

#include <cctype>
#include <iostream>

#include <llvm/Support/raw_os_ostream.h>
#include <llvm/Support/Path.h>

/// OLD ATTRIBUTE SUPPORT MIGHT BE EXTENDED ADDING NEW OLD FORMAT ATTRIBUTES
/// Follow the steps marked as <step 1> and <step 2>
namespace
{
    // old attribute
    struct OldAttribute
    {
        const char* m_name;
        int m_nameLength;

        OldAttribute(const char* name)
            : m_name(name)
        {
            m_nameLength = (int)strlen(m_name);
        }
    };

    // This array represents the old attributes registry
    // <step 1> Add a entry in this list with the appropriate name
    static const OldAttribute g_oldSupportedAttributes[] =
    {
        OldAttribute("version"), // 0
        OldAttribute("serialized"), // 1
        OldAttribute("nosave"), // 2
        OldAttribute("overridetype"), // 3
        OldAttribute("default"), // 4
        OldAttribute("reflected"), // 5
        OldAttribute("owned"), // 6
        OldAttribute("serializable"), // 7
        OldAttribute("defineattribute"), // 8
        OldAttribute("hk.MemoryTracker"), // 9
        OldAttribute("defineflags"), // 10
        OldAttribute("hk.ReflectedFile"), // 11
        OldAttribute("absmin"), // 12
        OldAttribute("absmax"), // 13
        OldAttribute("hk.ModelerNodeType"), // 14
        OldAttribute("hk.Ui"), // 15
        OldAttribute("hk.Link"), // 16
        OldAttribute("hk.RangeReal"), // 17
        OldAttribute("hk.RangeInt32"), // 18
        OldAttribute("hk.Semantics"), // 19
        OldAttribute("hk.Gizmo"), // 20
        OldAttribute("hk.Description"), // 21
        OldAttribute("hkb.RoleAttribute"), // 22
        OldAttribute("hk.Documentation"), // 23
    };
    static const int g_numOldSupportedAttributes = sizeof(g_oldSupportedAttributes)/sizeof(OldAttribute);

    // Old attributes translation structures
    struct OldAttributeToAssociate
    {
        int m_attrIndex;
        unsigned int m_line;
        clang::SourceRange m_range;
        std::string m_arguments;
        clang::Decl* m_closestDeclaration;
        unsigned int m_closestDeclarationLine;
        /// The line number of a non-attribute comment which precede this attribute.
        /// An attribute does not apply to a declaration if there is a comment between them.
        unsigned int m_precedingCommentLine;

        OldAttributeToAssociate(int attrIndex, int line, const clang::SourceRange& range, const std::string& arguments, unsigned int procedingCommentLine)
            : m_attrIndex(attrIndex), m_line(line), m_range(range), m_arguments(arguments),
              m_closestDeclaration(NULL), m_closestDeclarationLine(0), m_precedingCommentLine(procedingCommentLine)
        {}

        OldAttributeToAssociate()
            : m_closestDeclaration(NULL), m_closestDeclarationLine(0), m_precedingCommentLine(0)
        {}

        bool operator<(const OldAttributeToAssociate& other) const
        {
            // compare only using the line number
            return (m_line < other.m_line);
        }
    };

    static llvm::StringMap< std::vector<OldAttributeToAssociate> > g_attributesToAssociate;

    // A doc comment line which still needs to be associated with its decl.
    struct DocCommentToAssociate
    {
        /// The line in the source file where the doc comment was declared.
        unsigned int m_line;

        /// The source range of the doc comment.
        clang::SourceRange m_range;

        /// Whether this comment is one which is placed before or after its
        /// declaration. This depends on whether the comment tag used iwas
        /// /// (for comments coming before the declaration) or ///< (for
        /// comments coming after the declaration).
        bool m_afterDeclaration;

        bool operator < (const DocCommentToAssociate& b) const
        {
            return m_line < b.m_line;
        }
    };

    static llvm::StringMap< std::vector<DocCommentToAssociate> > g_docCommentsToAssociate;

    std::string buildRoleAttributeString(std::string const& stringIn)
    {
        std::string workString = stringIn;
        size_t quoteIndex = workString.find("\"");
        while(quoteIndex != std::string::npos)
        {
            workString.erase(quoteIndex, 1);
            quoteIndex = workString.find("\"");
        }

        std::vector<std::string> splitStrings = splitString(workString, '|');
        for(unsigned i=0; i < splitStrings.size(); i++)
        {
            splitStrings[i] = "hkbRoleAttribute::" + splitStrings[i];
        }

        workString = joinStrings(splitStrings, "|");
        return workString;
    }

    std::string lowerBools(const std::string& str)
    {
        std::string args = str;
        size_t index = args.find("True");
        while (index != std::string::npos)
        {
            args[index] = 't';
            index = args.find("True");
        }
        index = args.find("False");
        while (index != std::string::npos)
        {
            args[index] = 'f';
            index = args.find("False");
        }
        return args;
    }

    std::string removeQuotes(const std::string& str)
    {
        if (str.length() >= 2 && str.front() == '"' && str.back() == '"')
            return str.substr(1, str.length() - 2);
        return str;
    }
}

// <step 2> Add a new case to the if-else chaing inside this function, handling the new
// old-style attribute. This function should append to the 'text' argument, the same
// content you would put inside the HK_ATTR(<content>) for the new attribute syntax
static void s_newFromOldAttribute(const SourceManager& sman, const SourceLocation& loc, std::string& text, const OldAttributeToAssociate& oldAttribute)
{
    switch(oldAttribute.m_attrIndex)
    {
        // the index identifies the exact old attribute
    case 0: // version
            text = text + Database::attributePrefix + "hk::Version(" + oldAttribute.m_arguments + ")";
            break;
    case 1: // serialized
    case 7: // serializable
            text = text + Database::attributePrefix + "hk::Serialize(" + oldAttribute.m_arguments + ")";
            break;
    case 2: // nosave
            text = text + Database::attributePrefix + "hk::OpaqueType(true)";
            break;
    case 3: // overridetype
            text = text + Database::attributePrefix + "hk::Type(" + oldAttribute.m_arguments + ")"; // simple semantic, try to get the type by prepending hkType
            break;
    case 4: // default
            text = text + Database::attributePrefix + "hk::Default(" + oldAttribute.m_arguments + ")"; // simple semantic, try to get the type by prepending hkType
            break;
    case 5: // reflected
            text = text + Database::attributePrefix + "hk::Reflect(" + toLower(oldAttribute.m_arguments) + ")";
            break;
    case 6: // owned
            text = text + Database::attributePrefix + "hk::OwnedAttributeNotSupportedAnyMore";
            break;
    // case 7: - see case 1
    case 8: // defineattribute
        {
            text = text + Database::attributePrefix + "dotnet::DefineAttribute(";
            if(local_stricmp(oldAttribute.m_arguments.c_str(), "false") == 0)
                text += "false";
            else text += "true";
            text += ")";
        }
        break;
    case 9: // hk.MemoryTracker
        {
            text = text + Database::attributePrefix + "hk::MemoryTracker(";
            std::string args = lowerBools(oldAttribute.m_arguments);
            args = replaceAll(args, "ignore", "opaque");
            text += args;
            text += ")";
        }
        break;
    case 10: // defineflags
        {
            text = text + Database::attributePrefix + "dotnet::DefineFlags(";
            if(local_stricmp(oldAttribute.m_arguments.c_str(), "false") == 0)
                text += "false";
            else text += "true";
            text += ")";
        }
        break;
    case 11: // hk.ReflectedFile
            text = text + Database::attributePrefix + "hk::Feature(" + oldAttribute.m_arguments + ")";
            break;
    case 12: // absmin
            text = text + Database::attributePrefix + "hk::AbsMin(" + oldAttribute.m_arguments + ")";
            break;
    case 13: // absmax
            text = text + Database::attributePrefix + "hk::AbsMax(" + oldAttribute.m_arguments + ")";
            break;
    case 14: // hk.ModelerNodeType
        {
            text = text + Database::attributePrefix + "hkModelerNodeTypeAttribute(";
            // remove quotes, add prefix
            text += "hkModelerNodeTypeAttribute::";
            text += removeQuotes(oldAttribute.m_arguments);
            text += ")";
        }
        break;
    case 15: // hk.Ui
        {
            text = text + Database::attributePrefix + "hkUiAttribute(";
            std::string args = lowerBools(oldAttribute.m_arguments);

            size_t index = args.find("hideCriteria");
            if (index != std::string::npos)
            {
                // remove quotes from argument value
                index += strlen("hideCriteria");
                if ((index + 2) < args.length() && args[index] == '=' && args[index + 1] == '"')
                {
                    size_t begin = index + 1;
                    args.erase(begin, 1);
                    size_t end = args.find('"', begin);
                    if (end != std::string::npos)
                        args.erase(end, 1);
                }
            }
            text += args + ")";
        }
        break;
    case 16: // hk.Link
        {
            text = text + Database::attributePrefix + "hkLinkAttribute(";
            // remove quotes, add prefix
            text += "hkLinkAttribute::";
            text += removeQuotes(oldAttribute.m_arguments);
            text += ")";
        }
        break;
    case 17: // hk.RangeReal
        {
            std::string args = oldAttribute.m_arguments;
            std::replace(args.begin(), args.end(), '"', ' ');
            text = text + Database::attributePrefix + "hkRangeRealAttribute(" + args + ")";
        }
        break;
    case 18: // hk.RangeInt
        {
            text = text + Database::attributePrefix + "hkRangeInt32Attribute(" + oldAttribute.m_arguments + ")";
        }
        break;
    case 19: // hk.Semantics
        {
            text = text + Database::attributePrefix + "hkSemanticsAttribute(";
            // remove quotes from argument value, add prefix
            std::string args = oldAttribute.m_arguments;
            size_t index = args.find("type");
            if (index != std::string::npos && (index + 1) < args.length())
            {
                args = args.substr(strlen("type="));
            }
            text += "hkSemanticsAttribute::" + removeQuotes(args);
            text += ")";
        }
        break;
    case 20: // hk.Gizmo
        {
            text = text + Database::attributePrefix + "hkGizmoAttribute(";
            std::string args = lowerBools(oldAttribute.m_arguments);

            // remove quotes from argument value, add prefix
            size_t index = args.find("type");
            if (index != std::string::npos)
            {
                index += strlen("type");
                if ((index + 2) < args.length() && args[index] == '=' && args[index + 1] == '"')
                {
                    size_t begin = index + 1;
                    args.replace(begin, 1, "hkGizmoAttribute::");
                    size_t end = args.find('"', begin);
                    if (end != std::string::npos)
                        args.erase(end, 1);
                }
            }

            text += args + ")";
        }
        break;
    case 21: // hk.Description
        {
            text = text + Database::attributePrefix + "hkDescriptionAttribute(" + oldAttribute.m_arguments + ")";
        }
        break;
    case 22: // hkb.RoleAttribute
        {
            text = text + Database::attributePrefix + "hkbRoleAttribute(";
            std::vector<std::string> argParts = splitString(oldAttribute.m_arguments, ',');
            std::string roleVal = argParts.size() > 0 ? argParts[0] : "ROLE_DEFAULT";
            std::string flagVal = argParts.size() > 1 ? argParts[1] : "FLAG_NONE";

            text += "role=" + buildRoleAttributeString(roleVal) + ", flags=" + buildRoleAttributeString(flagVal);
            text += ")";
        }
        break;
    case 23: // hk.Documentation
        {
            text = text + Database::attributePrefix + "hkDocumentationAttribute(";
            text += "&" + removeQuotes(oldAttribute.m_arguments);
            text += ")";
        }
        break;
    default:
        {
            hkLog.error(sman, loc, "Conversion of old attribute %s is not supported.", g_oldSupportedAttributes[oldAttribute.m_attrIndex].m_name);
            std::abort();
        }
    }
}

void s_associateWithAttribute(const Decl* decl, const ASTContext* context)
{
    typedef llvm::StringMap< std::vector<OldAttributeToAssociate> >::iterator MapIt;
    typedef std::vector<OldAttributeToAssociate>::iterator VecIt;

    if( !isa<CXXRecordDecl>(decl) &&
        !isa<FieldDecl>(decl) &&
        !isa<NamespaceDecl>(decl) &&
        !isa<EnumDecl>(decl))
        return;

    if( const NamespaceDecl* namespaceDecl = dyn_cast<NamespaceDecl>(decl) )
    {
        DeclContext::decl_iterator it;
        for(it = namespaceDecl->decls_begin(); it != namespaceDecl->decls_end(); ++it)
        {
            s_associateWithAttribute(*it, context);
        }
    }
    else
    {
        if( const CXXRecordDecl* recordDecl = dyn_cast<CXXRecordDecl>(decl) )
        {
            if(!recordDecl->isCompleteDefinition())
                return; // only consider full definitions concerning attributes
            // for record declaration, we recur on the members
            DeclContext::decl_iterator it;
            for(it = recordDecl->decls_begin(); it != recordDecl->decls_end(); ++it)
            {
                s_associateWithAttribute(*it, context);
            }
        }
        clang::PresumedLoc loc = context->getSourceManager().getPresumedLoc(decl->getLocation());
        const char* file = loc.getFilename();
        std::string canonBufName = ::Filesystem::absoluteCanonicalPath(std::string(file, strlen(file)));
        MapIt mapIt = g_attributesToAssociate.find(canonBufName);
        if(mapIt != g_attributesToAssociate.end())
        {
            // found the file
            OldAttributeToAssociate comparisonAttribute;
            unsigned int declarationLine = loc.getLine();

            comparisonAttribute.m_line = declarationLine;
            std::vector<OldAttributeToAssociate>& attVec = mapIt->second;
            // first element having line number equal or greater than comparisonAttribute
            VecIt vecIt = std::lower_bound(attVec.begin(), attVec.end(), comparisonAttribute);
            if(vecIt != attVec.end() && vecIt->m_closestDeclarationLine < declarationLine)
            {
                vecIt->m_closestDeclarationLine = declarationLine;
                vecIt->m_closestDeclaration = const_cast<clang::Decl*>(decl);
            }
        }
    }
}

Havok::RawDumpAttributeHandler::RawDumpAttributeHandler(const SourceManager& sourceManager)
    : m_sourceManager(sourceManager)
    , m_lastSeenNonAttributeCommentLine(0)
{}

bool Havok::RawDumpAttributeHandler::HandleComment(clang::Preprocessor&, clang::SourceRange comment)
{
    bool invalid = false;
    clang::SourceLocation begin = comment.getBegin();
    clang::SourceLocation end = comment.getEnd();
    const char* commentBegin = m_sourceManager.getCharacterData(begin, &invalid);
    const char* commentEnd = m_sourceManager.getCharacterData(end, &invalid);
    const char* commentIt = commentBegin;

    // Handle intervening comments.
    bool foundAttribute = false;
    if ( m_lastSeenFile != m_sourceManager.getFileID( begin ) )
    {
        m_lastSeenFile = clang::FileID::getSentinel();
        m_lastSeenNonAttributeCommentLine = 0;
    }

    // We don't want to match any old "+". e.g. // Note that self <= +epsilon
    // We skip the line entirely unless + is the first nonwhite character after the /

    for(const char* cit = commentBegin; cit < commentEnd; ++cit )
    {
        int c = *cit;
        if(c == '/' || c == ' ' || c == '\t')
        {
            continue;
        }
        else if(c == '+')
        {
            break; // ok!
        }
        else
        {
            #if 0 // debugging, show skipped comments
            for(const char* p = commentBegin; p < commentEnd; ++p)
            {
                if(*p == '+')
                {
                    std::string s(commentBegin, commentEnd);
                    std::cerr << "SKIP: '" << c << "' " << s << '\n';
                    return false;
                }
            }
            #endif
            return false;
        }
    }

    // Now try to match old attrs

    while(commentIt < commentEnd)
    {
        if(commentIt[0] == '+')
        {
            ++commentIt;
            if( (commentIt[0] >= 'A' && commentIt[0] <= 'Z') ||
                (commentIt[0] >= 'a' && commentIt[0] <= 'z') )
            {
                int i;
                for(i = 0; i < g_numOldSupportedAttributes; ++i )
                {
                    const OldAttribute& attribute = g_oldSupportedAttributes[i];
                    if(strncmp(commentIt, attribute.m_name, attribute.m_nameLength) == 0)
                    {
                        clang::SourceLocation loc = begin.getLocWithOffset(int(commentIt - commentBegin));
                        #if 0 && defined(_DEBUG)
                        std::cerr << "hkPreBuild: found old attribute '" <<
                            attribute.m_name << "' in " <<
                            llvm::sys::path::filename(m_sourceManager.getBufferName(loc)).data() << ", line " <<
                            m_sourceManager.getPresumedLineNumber(loc) << ", column " <<
                            m_sourceManager.getPresumedColumnNumber(loc) << std::endl;
                        #endif
                        commentIt += attribute.m_nameLength;
                        std::string arguments("");
                        if(commentIt[0] == '(')
                        {
                            unsigned nestLevel = 1;
                            // has arguments
                            ++commentIt;
                            while(nestLevel > 0)
                            {
                                if(commentIt[0] == '(')
                                {
                                    nestLevel++;
                                }
                                else if(commentIt[0] == ')')
                                {
                                    nestLevel--;
                                }
                                if(nestLevel > 0)
                                {
                                    arguments.push_back(commentIt[0]);
                                    ++commentIt;
                                }
                            }
                        }

                        unsigned int attributeLineNumber = m_sourceManager.getPresumedLineNumber(loc);
                        OldAttributeToAssociate oldAttribute(i, attributeLineNumber,
                            clang::SourceRange(loc.getLocWithOffset(-1), begin.getLocWithOffset(int(commentIt - commentBegin))), arguments, m_lastSeenNonAttributeCommentLine );

                        const char* bufName = m_sourceManager.getBufferName(loc);
                        std::string canonBufName = ::Filesystem::absoluteCanonicalPath(std::string(bufName, strlen(bufName)));
                        g_attributesToAssociate[canonBufName].push_back(oldAttribute);

                        foundAttribute = true;
                        break;
                    }
                }
                #if defined(_DEBUG) || 1
                if(i == g_numOldSupportedAttributes)
                {
                    clang::SourceLocation loc = begin.getLocWithOffset(int(commentIt - commentBegin));
                    size_t numchar = std::min<size_t>(strlen(commentIt), 16);
                    std::string token(commentIt, commentIt+numchar);
                    std::replace(token.begin(), token.end(), '\n', '\0');
                    std::replace(token.begin(), token.end(), '\r', '\0');
                    hkLog.warningWithoutDatabase(m_sourceManager.getSpellingLineNumber(loc), m_sourceManager.getBufferName(loc),
                        "hkPreBuild skipping old attribute-like comment starting with '%s'", token.c_str());
                }
                #endif
            }
        }
        else
        {
            ++commentIt;
        }
    }
    if ( !foundAttribute )
    {
        m_lastSeenNonAttributeCommentLine = m_sourceManager.getPresumedLineNumber( begin );
        m_lastSeenFile = m_sourceManager.getFileID( begin );
    }
    return false;
}

Havok::RawDumpAttributeHandler::~RawDumpAttributeHandler()
{}

Havok::RawDumpDocCommentHandler::RawDumpDocCommentHandler
    ( const SourceManager& sourceManager
    , const std::vector<std::string>& inputFileNames )
    : m_sourceManager(sourceManager)
    , m_inputFileNames(inputFileNames)
{
    std::sort(m_inputFileNames.begin(), m_inputFileNames.end(), &Filesystem::sortFileNames);
}

bool Havok::RawDumpDocCommentHandler::isInteresting(clang::FileID fid)
{
    auto it = m_interesting.find(fid);
    if (it != m_interesting.end())
    {
        return it->second;
    }
    if( auto* entry = m_sourceManager.getFileEntryForID(fid) )
    {
        std::string file = Filesystem::fixSeparatorsAndRemoveLeadingDot(entry->getName());
        bool interesting = std::binary_search(m_inputFileNames.begin(), m_inputFileNames.end(), file, &Filesystem::sortFileNames);
        m_interesting[fid] = interesting;
        return interesting;
    }
    return false;
}


bool Havok::RawDumpDocCommentHandler::HandleComment(clang::Preprocessor&, clang::SourceRange range)
{
    auto fid = m_sourceManager.getFileID(range.getBegin());
    if (isInteresting(fid)==false)
    {
        return false;
    }
    auto* com0 = m_sourceManager.getCharacterData(range.getBegin());
    auto* com1 = m_sourceManager.getCharacterData(range.getEnd());
    llvm::StringRef str(com0, com1 - com0);
    if (str.startswith("///") && (str.size() <= 3 || str[3] != '/'))
    {
        auto loc = m_sourceManager.getPresumedLoc(range.getBegin());

        DocCommentToAssociate docComment;
        docComment.m_line = loc.getLine();
        docComment.m_range = range;
        docComment.m_afterDeclaration = str.size() > 3 && str[3] == '<';
        g_docCommentsToAssociate[m_sourceManager.getBufferName(range.getBegin())].push_back(docComment);
    }
    return false;
}

void Havok::RawDumpASTConsumer::fixDeclarationAttributes()
{
    typedef llvm::StringMap< std::vector<OldAttributeToAssociate> >::iterator MapIt;
    typedef std::vector<OldAttributeToAssociate>::iterator VecIt;

    // 1: sort all vectors in the map (each one corresponds to a different file)
    for( MapIt mapIt = g_attributesToAssociate.begin();
         mapIt != g_attributesToAssociate.end();
         ++mapIt )
    {
        std::vector<OldAttributeToAssociate>& elem = mapIt->second;
        std::sort(elem.begin(), elem.end());
    }

    // 2: run through all declarations and save the closest interesting one to each attribute
    for( std::vector<const clang::Decl*>::const_iterator declIt = m_decls.begin();
         declIt != m_decls.end();
         ++declIt )
    {
        s_associateWithAttribute(*declIt, m_context);
    }

    // 3: change the closest declaration attribute lists to include new annotations
    // Note: when associating the declaration with its closest attribute defined
    // after it, we are not associating it with any other attribute defined on it.
    // When an attribute doesn't have any declaration associated, we know that we can
    // use the attribute before it to understand to what declaration it is associated.
    for( MapIt mapIt = g_attributesToAssociate.begin();
        mapIt != g_attributesToAssociate.end();
        ++mapIt )
    {
        std::vector<OldAttributeToAssociate>& elem = mapIt->second;
        for( VecIt vecIt = elem.begin();
             vecIt != elem.end();
             ++vecIt )
        {
            if(vecIt->m_closestDeclaration == NULL && vecIt != elem.begin())
            {
                // If this attribute has not closest declaration associated, we
                // use the previous attribute to recover the declaration.
                // The declaration might remain NULL if this attribute is the first
                // one encountered/
                VecIt prevIt = vecIt-1;
                vecIt->m_closestDeclaration = prevIt->m_closestDeclaration;
            }
            // Check that there is no non-attribute comment line between the attribute and its nearest declaration.
            if(Decl* decl = vecIt->m_closestDeclaration)
            {
                std::string text;
                unsigned int declarationLineNumber = m_context->getSourceManager().getPresumedLineNumber( decl->getLocation() );
                if ( declarationLineNumber < vecIt->m_precedingCommentLine )
                {
                    hkLog.errorWithoutDatabase( vecIt->m_precedingCommentLine, m_context->getSourceManager().getFilename( decl->getLocation() ).str().c_str(), "Comments are not allowed between old attributes and the declarations to which they apply.");
                    // This suppresses the error for subsequent attribute comments.
                    vecIt->m_closestDeclaration = NULL;
                }
                else
                {
                    AnnotateAttr* attr = new(*m_context) AnnotateAttr(vecIt->m_range, *m_context, "");
                    s_newFromOldAttribute(m_context->getSourceManager(), decl->getLocation(), text, *vecIt);
                    attr->setAnnotation(*m_context, text);
                    decl->addAttr(attr);
                }
            }
            #if 0 && defined(_DEBUG)
            if(vecIt->m_closestDeclaration != NULL)
            {
                std::cerr << "hkPreBuild: Associated old attribute '" << g_oldSupportedAttributes[vecIt->m_attrIndex].m_name
                    << "' with declaration '";
                const clang::NamedDecl* namedDecl = dyn_cast<NamedDecl>(vecIt->m_closestDeclaration);
                std::cerr << namedDecl->getNameAsString() << "'" << std::endl;
            }
            else
            {
                std::cerr << "hkPreBuild: Old attribute '" << g_oldSupportedAttributes[vecIt->m_attrIndex].m_name
                    << "' could not be associated with any declaration, file '" <<
                    mapIt->first().data() << "' line '" << vecIt->m_line << "'" << std::endl;
            }
            #endif
        }
    }

    for( std::vector<const clang::Decl*>::const_iterator declIt = m_decls.begin();declIt != m_decls.end(); ++declIt )
    {
        clang::Decl* decl = const_cast<clang::Decl*>(*declIt);
        fixDeclarationAttributesRecursive(decl);
    }
}

void Havok::RawDumpASTConsumer::fixSpecializationAttributesRecursive(clang::NamedDecl* spec, const clang::NamedDecl* tpl)
{
    if (auto tplSpec = clang::dyn_cast<clang::ClassTemplateDecl>(spec))
    {
        auto tplTpl = clang::dyn_cast<clang::ClassTemplateDecl>(tpl);
        assert(tplTpl);
        fixSpecializationAttributesRecursive(tplSpec->getTemplatedDecl(), tplTpl->getTemplatedDecl());

        // recurse on the implicit specializations
        for (clang::ClassTemplateDecl::spec_iterator spec_it = tplSpec->spec_begin();
            spec_it != tplSpec->spec_end(); ++spec_it)
        {
            if (!spec_it->isExplicitInstantiationOrSpecialization())
            {
                fixSpecializationAttributesRecursive(*spec_it, tplTpl->getTemplatedDecl());
            }
        }
    }
    else
    {
        if (tpl->hasAttr<AnnotateAttr>())
        {
            // skip all the attributes which the specialization already has
            const AttrVec& tplAttrs = tpl->getAttrs();
            const AttrVec& specAttrs = spec->hasAttrs() ? spec->getAttrs() : AttrVec();

            auto attrIt = specific_attr_begin<AnnotateAttr>(tplAttrs);
            auto attrEnd = specific_attr_end<AnnotateAttr>(tplAttrs);

            for( auto specIt = specific_attr_begin<AnnotateAttr>(specAttrs),
                specEnd = specific_attr_end<AnnotateAttr>(specAttrs);
                specIt != specEnd; ++specIt )
            {
                assert(attrIt != attrEnd);
                ++attrIt;
            }

            // add all the other attributes to the specialization
            for( ; attrIt != attrEnd; ++attrIt )
            {
                spec->addAttr(*attrIt);
            }
        }

        if (auto specCtx = clang::dyn_cast<clang::CXXRecordDecl>(spec))
        {
            if (specCtx->isThisDeclarationADefinition())
            {
                auto tplCtx = clang::dyn_cast<clang::CXXRecordDecl>(tpl);
                assert(tplCtx);

                // recurse on the inner decls
                typedef clang::DeclContext::specific_decl_iterator<clang::NamedDecl> NamedIterator;
                for(auto innerSpecIt = NamedIterator(specCtx->decls_begin()),
                    innerSpecEnd = NamedIterator(specCtx->decls_end());
                    innerSpecIt != innerSpecEnd; ++innerSpecIt)
                {
                    clang::NamedDecl* innerSpec = *innerSpecIt;

                    // todo.nt6 this recurses on templates/types only, since they can be easily
                    // matched by name; methods are skipped
                    if (clang::isa<clang::ClassTemplateDecl>(innerSpec) ||
                        clang::isa<clang::TypeDecl>(innerSpec))
                    {
                        for (auto innerTplIt = NamedIterator(tplCtx->decls_begin()),
                            innerTplEnd = NamedIterator(specCtx->decls_end());
                            innerTplIt != innerTplEnd; ++innerTplIt)
                        {
                            clang::NamedDecl* innerTpl = *innerTplIt;

                            if (innerSpec->getKind() == innerTpl->getKind() &&
                                innerSpec->getNameAsString() == innerTpl->getNameAsString().substr(0, innerTpl->getNameAsString().find('<')))
                            {
                                fixSpecializationAttributesRecursive(innerSpec, innerTpl);
                                break;
                            }

                        }
                    }
                }
            }
        }
    }
}

void Havok::RawDumpASTConsumer::fixDeclarationAttributesRecursive(clang::Decl* decl)
{
//  std::string declName;
//  if(clang::NamedDecl* namedDecl = dyn_cast<clang::NamedDecl>(decl)) declName = namedDecl->getNameAsString();
//  const clang::SourceLocation& loc = decl->getLocation();
//  std::cerr << "Seeing " << declName << " from " << loc.printToString(this->m_context->getSourceManager()) << std::endl;
//  const char* cname = declName.c_str();

    if(clang::ClassTemplateDecl* tplDecl = clang::dyn_cast<clang::ClassTemplateDecl>(decl))
    {
        //std::cerr << "Seeing " << tplDecl->getNameAsString() << std::endl;
        if(clang::CXXRecordDecl* recDecl = clang::dyn_cast<clang::CXXRecordDecl>(tplDecl->getTemplatedDecl())->getDefinition())
            return fixDeclarationAttributesRecursive(recDecl);
    }
    else if(clang::DeclContext* declContext = dyn_cast<clang::DeclContext>(decl))
    {
        std::vector<clang::Decl*> toRemove;

        for(clang::DeclContext::decl_iterator innerDeclIt = declContext->decls_begin(); innerDeclIt != declContext->decls_end(); ++innerDeclIt)
        {
            if(clang::DeclContext* innerCtx = dyn_cast<clang::DeclContext>(*innerDeclIt))
            {
                if(clang::NamedDecl* innerRec = dyn_cast<clang::NamedDecl>(innerCtx))
                {
                    std::string name;
                    {
                        llvm::raw_string_ostream tmp(name);
                        innerRec->printName(tmp);
                    }
                    if(stringStartsWith(name, "ReflectionOptions"))
                    {
                        if(innerRec->hasAttr<AnnotateAttr>())
                        {
//                          std::string declName;
//                          if(clang::NamedDecl* namedDecl = dyn_cast<clang::NamedDecl>(decl)) declName = namedDecl->getNameAsString();
//                          std::cerr << "Moving attributes of " << name << " to " << declName << std::endl;

                            // attach all the attributes to the declaration
                            const AttrVec& attrVec = innerRec->getAttrs();
                            for( specific_attr_iterator<AnnotateAttr> iterator = specific_attr_begin<AnnotateAttr>(attrVec), end_iterator = specific_attr_end<AnnotateAttr>(attrVec);
                                iterator != end_iterator;
                                ++iterator )
                            {
                                decl->addAttr(*iterator);
                            }

                            innerRec->dropAttr<AnnotateAttr>();
                            //innerRec->dropAttrs();
                        }
                        toRemove.push_back(innerRec);
                    }
                    else if (stringStartsWith(name, "PropertyPlaceholder"))
                    {
                        if(innerRec->hasAttr<AnnotateAttr>())
                        {
                            AttrVec& attrVec = innerRec->getAttrs();
                            for (auto iterator = attrVec.begin(); iterator != attrVec.end(); ++iterator)
                            {
                                if (auto attr = clang::dyn_cast<AnnotateAttr>(*iterator))
                                {
                                    const std::string& attrName = getAttrNameOrEmpty(attr, m_context->getSourceManager());
                                    if (attrName == "hk::Reflect")
                                    {
                                        // Marking a property as hk::Reflect will cause the placeholder to be processed as reflected;
                                        // remove the attribute and output a warning.
                                        attrVec.erase(iterator);
                                        hkLog.warning(m_context->getSourceManager(), (*iterator)->getLocation(),
                                            "Attribute 'hk::Reflect' on a property will be ignored.");
                                    }
                                }
                            }
                        }
                    }
                }
            }

            fixDeclarationAttributesRecursive(*innerDeclIt);
        }
        if (const clang::CXXRecordDecl* recDecl = clang::dyn_cast<clang::CXXRecordDecl>(decl))
        {
            if (const clang::ClassTemplateDecl* templ = recDecl->getDescribedClassTemplate())
            {
                // attach all the attributes to the implicit specializations
                for (clang::ClassTemplateDecl::spec_iterator spec_it = templ->spec_begin();
                    spec_it != templ->spec_end(); ++spec_it)
                {
                    if (!spec_it->isExplicitInstantiationOrSpecialization())
                    {
                        fixSpecializationAttributesRecursive(*spec_it, recDecl);
                    }
                }
            }
        }

        for(std::vector<clang::Decl*>::const_iterator cit = toRemove.begin(); cit != toRemove.end(); ++cit)
        {
            declContext->removeDecl(*cit);
        }
    }
    else
    {
//      std::string declName;
//      if(clang::NamedDecl* namedDecl = dyn_cast<clang::NamedDecl>(decl)) declName = namedDecl->getNameAsString();
//      std::cerr << "Unhandled decl : " << decl->getDeclKindName() << " (" << declName << ")" << std::endl;
    }
}

void Havok::RawDumpASTConsumer::fixDocComments()
{
    /// Sort the doc comments of each file.
    for(auto& mapEntry : g_docCommentsToAssociate)
    {
        std::sort(mapEntry.second.begin(), mapEntry.second.end());
    }

    // Go over all declarations, and associate them
    for(const clang::Decl* decl : m_decls)
    {
        associateDocCommentsRecursive(const_cast<clang::Decl*>(decl));
    }
}

/// Returns whether the source between the given two clang::SourceLocation's
/// consists of only whitespace.
static bool s_isWhiteSpace(const clang::SourceManager& srcMgr, clang::SourceLocation begin, clang::SourceLocation end)
{
    auto beginTxt = srcMgr.getCharacterData(begin);
    auto endTxt = srcMgr.getCharacterData(end);

    for(auto c = beginTxt; c != endTxt; c++)
    {
        if(!std::isspace(*c))
        {
            return false;
        }
    }

    return true;
}

static void s_escapeString(const std::string& str, std::stringstream& out)
{
    for(char c : str)
    {
        switch(c)
        {
        case '\t':
            out << "\\t";
            break;

        case '\n':
            out << "\\n";
            break;

        case '\r':
            out << "\\r";
            break;

        case '\\':
            out << "\\\\";
            break;

        case '\"':
            out << "\\";
            break;

        default:
            out << c;
            break;
        }
    }
}

static inline void s_trimWhiteSpace(const char*& begin, const char*& end)
{
    while(begin < end && std::isspace(*begin))
    {
        begin++;
    }

    while(begin < end && std::isspace(*(end - 1)))
    {
        end--;
    }
}

/// Creates a hk::DocString attribute from the given range of comments.
static void s_createAttributeFromDocComments(ASTContext* context, clang::Decl* decl,
    DocCommentToAssociate* docCommentsBegin, DocCommentToAssociate* docCommentsEnd)
{
    assert(docCommentsBegin < docCommentsEnd);

    auto& srcMgr = context->getSourceManager();
    std::stringstream docStr;
    hkBool paragraphStarted = false;
    for(DocCommentToAssociate* docComment = docCommentsBegin; docComment != docCommentsEnd; docComment++)
    {
        const char* rangeBegin = srcMgr.getCharacterData(docComment->m_range.getBegin());
        const char* rangeEnd = srcMgr.getCharacterData(docComment->m_range.getEnd());

        // Strip the /// or ///< and the white space before the first character.
        rangeBegin += docComment->m_afterDeclaration ? 4 : 3;
        assert(rangeBegin <= rangeEnd);

        s_trimWhiteSpace(rangeBegin, rangeEnd);

        if(rangeBegin != rangeEnd)
        {
            if(paragraphStarted)
            {
                docStr << ' ';
            }

            docStr.write(rangeBegin, rangeEnd - rangeBegin);
            paragraphStarted = true;
        }
        else
        {
            if(paragraphStarted)
            {
                docStr << std::endl << std::endl;
                paragraphStarted = false;
            }
        }
    }

    clang::SourceRange srcRange(docCommentsBegin->m_range.getBegin(),
        (docCommentsEnd - 1)->m_range.getEnd());

    std::stringstream attrStr;
    attrStr << Database::attributePrefix << "hk::DocString(\"";
    s_escapeString(docStr.str(), attrStr);
    attrStr << "\")";
    AnnotateAttr* attr = new(*context) AnnotateAttr(srcRange, *context, attrStr.str());
    decl->addAttr(attr);
}

/// Returns whether we need to create hk::DocString attributes from doc comments
/// for the given decl.
static bool s_hasAttachDocStringAttribute(const clang::SourceManager& srcMgr, const clang::TypeDecl* decl)
{
    for(auto it = decl->attr_begin(); it != decl->attr_end(); ++it)
    {
        auto attr = clang::dyn_cast<clang::AnnotateAttr>(*it);
        if(attr)
        {
            if(getAttrNameOrEmpty(attr, srcMgr) == "hk::AttachDocString")
            {
                return true;
            }
        }
    }

    return false;
}

static clang::SourceRange s_getDeclSourceRange(const clang::SourceManager& srcMgr, const clang::Decl* decl)
{
    auto range = decl->getSourceRange();
    auto rangeEnd = range.getEnd();

    auto rangeEndDecomposed = srcMgr.getDecomposedLoc(rangeEnd);
    auto file = srcMgr.getBufferData(rangeEndDecomposed.first);

    auto tokenBegin = file.data() + rangeEndDecomposed.second;
    Lexer lexer(srcMgr.getLocForStartOfFile(rangeEndDecomposed.first), clang::LangOptions(), file.begin(), tokenBegin, file.end());

    Token tok;
    if(lexer.LexFromRawLexer(tok))
    {
        assert(!"Missing semicolon");
    }

    while(tok.isNot(tok::semi))
    {
        if(lexer.LexFromRawLexer(tok))
        {
            assert(!"Missing semicolon");
        }
    }

    rangeEnd = Lexer::getLocForEndOfToken(tok.getLocation(), 0, srcMgr, LangOptions());

    range.setEnd(rangeEnd);
    return range;
}

void Havok::RawDumpASTConsumer::associateDocCommentsRecursive(clang::Decl* decl)
{
    auto& srcMgr = m_context->getSourceManager();

    if(auto typeDecl = clang::dyn_cast<clang::TypeDecl>(decl))
    {
        bool attachDocStrings = s_hasAttachDocStringAttribute(srcMgr, typeDecl);
        if(attachDocStrings)
        {
            associateDocComments(typeDecl);

            auto declCtx = clang::dyn_cast<clang::DeclContext>(decl);
            if(declCtx)
            {
                for(auto it = declCtx->decls_begin(); it != declCtx->decls_end(); ++it)
                {
                    if(clang::isa<clang::FieldDecl>(*it))
                    {
                        associateDocComments(*it);
                    }
                }
            }
        }
    }

    {
        auto declCtx = clang::dyn_cast<clang::DeclContext>(decl);
        if(declCtx)
        {
            for(auto it = declCtx->decls_begin(); it != declCtx->decls_end(); ++it)
            {
                if(clang::isa<clang::NamespaceDecl>(*it) ||
                    clang::isa<clang::TypeDecl>(*it))
                {
                    associateDocCommentsRecursive(*it);
                }
            }
        }
    }

}

void Havok::RawDumpASTConsumer::associateDocComments(clang::Decl* decl)
{
    auto& srcMgr = m_context->getSourceManager();

    if(decl->isImplicit())
    {
        return;
    }

    {
        auto loc = srcMgr.getPresumedLoc(decl->getLocation());
        auto file = loc.getFilename();
        auto declSrcRange = s_getDeclSourceRange(srcMgr, decl);
        auto mapIt = g_docCommentsToAssociate.find(file);
        if(mapIt != g_docCommentsToAssociate.end())
        {
            auto& docComments = mapIt->second;

            // Process adjacent doc comments of the /// type (that is, those
            // which have their m_afterDeclaration flag set to false).
            {
                // Find the last doc comment which comes before the current  declaration.
                auto docCommentsEnd = std::upper_bound(&*docComments.begin(), &*docComments.begin() + docComments.size(), declSrcRange.getBegin(),
                    [&srcMgr](const clang::SourceLocation& declBegin, const DocCommentToAssociate& docComment)
                    {
                        return srcMgr.isBeforeInTranslationUnit(declBegin, docComment.m_range.getEnd());
                    });

                if(docCommentsEnd != &*docComments.begin())
                {
                    // If the preceding doc comment is of the type /// (as
                    // opposed to ///<) and if there's only whitespace between
                    // the end of the doc comment and the beginning of the
                    // declaration, then this comment line is the last one of
                    // the documentation block which belongs to the current
                    // declaration.
                    auto& docComment = *(docCommentsEnd - 1);
                    if(!docComment.m_afterDeclaration && s_isWhiteSpace(srcMgr, docComment.m_range.getEnd(), declSrcRange.getBegin()))
                    {
                        // Find the beginning of the documentation block.
                        unsigned int expectedLine = docComment.m_line - 1;
                        DocCommentToAssociate* it;
                        auto prevCommentBegin = docComment.m_range.getBegin();
                        for(it = docCommentsEnd - 2; it >= &*docComments.begin(); --it)
                        {
                            if(it->m_afterDeclaration || it->m_line != expectedLine ||
                                !s_isWhiteSpace(srcMgr, it->m_range.getEnd(), prevCommentBegin))
                            {
                                break;
                            }

                            expectedLine--;
                            prevCommentBegin = it->m_range.getBegin();
                        }

                        auto docCommentsBegin = it + 1;
                        s_createAttributeFromDocComments(m_context, decl, docCommentsBegin, docCommentsEnd);
                    }
                }
            }

            // Process adjacent doc comments of the ///< type (that is, those
            // which have their m_afterDeclaration flag set to true).
            {
                // Find the first doc comment which comes after the current declaration.
                auto allDocCommentsEnd = &*docComments.begin() + docComments.size();
                auto docCommentsBegin = std::lower_bound(&*docComments.begin(), allDocCommentsEnd, declSrcRange.getEnd(),
                    [&srcMgr](const DocCommentToAssociate& docComment, const clang::SourceLocation& declEnd)
                    {
                        bool ret = srcMgr.isBeforeInTranslationUnit(docComment.m_range.getBegin(), declEnd);
                        return ret;
                    });

                if(docCommentsBegin != allDocCommentsEnd)
                {
                    // If the first doc comment is of the type ///< (as opposed
                    // to ///) and if there's only whitespace between the end of
                    // the declaration and the beginning of the doc comment,
                    // then the current comment line is the first one of the
                    // documentation block which belongs to the current
                    // declaration.
                    auto& docComment = *docCommentsBegin;
                    if(docComment.m_afterDeclaration && s_isWhiteSpace(srcMgr, declSrcRange.getEnd(), docComment.m_range.getBegin()))
                    {
                        unsigned int expectedLine = docComment.m_line + 1;
                        DocCommentToAssociate* it;
                        auto prevCommentEnd = docComment.m_range.getEnd();
                        for(it = docCommentsBegin + 1; it <= allDocCommentsEnd; ++it)
                        {
                            if(!it->m_afterDeclaration || it->m_line != expectedLine ||
                                !s_isWhiteSpace(srcMgr, prevCommentEnd, it->m_range.getBegin()))
                            {
                                break;
                            }

                            expectedLine++;
                            prevCommentEnd = it->m_range.getEnd();
                        }

                        auto docCommentsEnd = it;
                        s_createAttributeFromDocComments(m_context, decl, docCommentsBegin, docCommentsEnd);
                    }
                }
            }
        }
    }
}

namespace
{
    const clang::AnnotateAttr* getAttribute(const clang::Decl* decl, const std::string& attrName)
    {
        if(decl->hasAttr<clang::AnnotateAttr>())
        {
            const std::string& fullName = attrName + "(";

            const clang::AttrVec& attrVec = decl->getAttrs();
            for( clang::specific_attr_iterator<clang::AnnotateAttr> iterator = clang::specific_attr_begin<clang::AnnotateAttr>(attrVec), end_iterator = clang::specific_attr_end<clang::AnnotateAttr>(attrVec);
                iterator != end_iterator;
                ++iterator )
            {
                if(iterator->getAnnotation().startswith(fullName))
                {
                    return *iterator;
                }
            }
        }

        return 0;
    }
}

////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

// ------------------- RawDumpASTConsumer Implementation -------------------- //

// Initialize the database object with its global state. Each consumer object is only expected to be used once
Havok::RawDumpASTConsumer::RawDumpASTConsumer()
    : m_context(0), m_sema(0)
{
}

Havok::RawDumpASTConsumer::~RawDumpASTConsumer()
{
}

// Called by the parser at the start of the process
void Havok::RawDumpASTConsumer::Initialize(ASTContext &Context)
{
    // Remember the context so we can use it to find file names and locations
    m_context = &Context;
}

void Havok::RawDumpASTConsumer::InitializeSema(Sema& sema)
{
    // Remember the sema instance so we can use it to perform semantic analysis
    m_sema = &sema;
}

// Called by the parser for each top-level declaration encountered.
// This includes top-level classes, namespaces, typedefs, constants, enums, functions
// in every included header file. Contained elements are children of a top-level element.
// Every element that is parsed is in here somewhere
bool Havok::RawDumpASTConsumer::HandleTopLevelDecl(DeclGroupRef declGroupIn)
{
    // If there are multiple declarations with the same semantic type in the same scope they are
    // returned as a group. [ e.g. class A { ... } B; ] Usually each group only contains one declaration.
    for (DeclGroupRef::iterator iter = declGroupIn.begin(), iterEnd = declGroupIn.end(); iter != iterEnd; ++iter)
    {
        m_decls.push_back(*iter);
    }

    return true;
}

// -------------------------------------------------------------------------- //

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
