// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
PRAGMA_WARNING_PUSH
#include <llvm/Support/ManagedStatic.h>
#include <llvm/Support/FileSystem.h>
#include <llvm/Support/CommandLine.h>
#include <llvm/Support/raw_ostream.h>
#include <llvm/Support/Path.h>
PRAGMA_WARNING_POP

#include <signal.h>

#include "utils/Filesystem.h"
#include "utils/TkbmsRegistry.h"
#include "utils/StlUtils.h"
#include "utils/Logger.h"
#include "utils/ProfilePoint.h"
#include "utils/HavokUtils.h"
#include "utils/RuntimeError.h"

std::string projectOutputPathStr;
std::string projectLogPathStr;
void onAbortCallback(int)
{
    std::ofstream of(projectLogPathStr + "/hkReflect_clang_crashlog.txt");
    if(!of.is_open())
        return;
    of << hkLog.loggedMessages() << std::flush;
}

void scanDirectoryForTkbms(const std::string& project, TkbmsRegistry& tkbms,
    const std::set<std::string>& enabledProducts, const std::set<std::string>& enabledPlatforms,
    bool explicitReflect, bool excludeInternal, const std::vector<std::string>& excludeDirs, std::vector<std::string>& sourceFileNames);

int clangMain(TkbmsRegistry& tkbms, Filesystem& fs, const std::string& trackingFileName, bool printStats, bool binding
              , const llvm::cl::list<std::string>& cppDefines, const llvm::cl::list<std::string>& includePaths, const llvm::cl::list<std::string>& excludeFileNames
              , const llvm::cl::list<std::string>& excludeFilePatterns, const llvm::cl::list<std::string>& passAtttributes, const llvm::cl::list<std::string>& forceIncludes, bool generateMemoryTracker
              , bool generateClassLists, const std::vector<std::string>& inputFileNames, const std::string& progName, const std::string& outputFileDir
              , const std::string& preamble, const std::string& postamble, const std::string& cppPreamble, const std::string& cppPostamble, const llvm::cl::list<std::string>& addAttributes
              , bool noRawRefObjPtr, const llvm::cl::list<std::string>& remapInclude, bool outputTkbms);

class ArgsStringSaver : public llvm::cl::StringSaver
{
public:
    ArgsStringSaver() {}
    virtual void anchor() {}
    virtual const char *SaveString(const char *Str)
    {
        const size_t length = strlen(Str);
        char* ret = m_alloc.Allocate<char>(length + 1);
        memcpy(ret, Str, length);
        ret[length] = 0;
        return ret;
    }
    virtual ~ArgsStringSaver() {};

    llvm::BumpPtrAllocator m_alloc;
};

static void expandDollarEnvironmentVariable(std::string& inout)
{
    for (;;)
    {
        auto start = inout.find("$(");
        if (start == std::string::npos)
        {
            return;
        }
        auto end = inout.find(")", start);
        if (end == std::string::npos)
        {
            hkLog.warning("Found '$(' without a matching ')'");
            return;
        }
        if (end - start <= 3)
        {
            hkLog.info("Found empty variable '$()'");
            return;
        }
        auto var = inout.substr(start + 2, end - start - 2);
        const char* replacement = getenv(var.c_str());
        if (replacement == nullptr)
        {
            hkLog.info(" $(%s) is not set in the environment", var.c_str());
            return;
        }
        inout.replace(start, end - start + 1, replacement);
    }
}

static llvm::cl::opt<bool> o_quiet(llvm::cl::Optional, "quiet", llvm::cl::desc("Reduce verbosity to minimum (only output errors)"), llvm::cl::initializer<bool>(false));
static llvm::cl::opt<bool> o_clangVersion(llvm::cl::Optional, "clang-version", llvm::cl::desc("Print the LLVM/Clang version string and exit"), llvm::cl::initializer<bool>(false));
static llvm::cl::opt<bool> o_clangStats(llvm::cl::Optional, "clang-stats", llvm::cl::desc("Print some clang statistics"), llvm::cl::initializer<bool>(false));
static llvm::cl::opt<bool> o_profile(llvm::cl::Optional, "profile", llvm::cl::desc("Print profiling information"), llvm::cl::initializer<bool>(false));
static llvm::cl::opt<int> o_verbosity(llvm::cl::ZeroOrMore, "verbose", llvm::cl::desc("Set the verbosity level"), llvm::cl::initializer<int>(Logger::LOG_WARNING));
static llvm::cl::opt<bool> o_dryRun(llvm::cl::Optional, "dry-run", llvm::cl::desc("Print what would be done, but don't do it"), llvm::cl::initializer<bool>(false));
static llvm::cl::opt<bool> o_binding(llvm::cl::Optional, "HATBinding", llvm::cl::desc("Generate HAT binding files"), llvm::cl::initializer<bool>(false));
static llvm::cl::opt<bool> o_excludeInternal(llvm::cl::Optional, "exclude-internal", llvm::cl::desc("Exclude headers marked TKBMS internal"), llvm::cl::initializer<bool>(false));
static llvm::cl::opt<bool> o_generateTracker(llvm::cl::Optional, "tracker", llvm::cl::desc("Generate memory tracker data"), llvm::cl::initializer<bool>(false));
static llvm::cl::opt<bool> o_explicitReflect(llvm::cl::Optional, "explicit-reflect", llvm::cl::desc("Only parse headers whose TKBMS has PLATFORM : REFLECT (ignored if --tkbms-scan is not used)"), llvm::cl::initializer<bool>(false));
static llvm::cl::opt<bool> o_classList(llvm::cl::Optional, "classlist", llvm::cl::desc("Generate Class Lists"), llvm::cl::initializer<bool>(false));
static llvm::cl::opt<std::string> o_outputPathArg(llvm::cl::Required, "output-path", llvm::cl::desc("Path for output files"));
static llvm::cl::opt<std::string> o_logPathArg(llvm::cl::Optional, "log-path", llvm::cl::desc("Path for log files"));
static llvm::cl::list<std::string> o_preprocessorDefines(llvm::cl::ZeroOrMore, "D", llvm::cl::desc("Predefined preprocessor constants"));
static llvm::cl::list<std::string> o_includePaths(llvm::cl::ZeroOrMore, "I", llvm::cl::desc("Include Paths"));
static llvm::cl::list<std::string> o_attributes(llvm::cl::ZeroOrMore, "A", llvm::cl::desc("Attributes to pass to the output stages"));
static llvm::cl::list<std::string> o_forceIncludeFiles(llvm::cl::ZeroOrMore, "include", llvm::cl::desc("Force Include Files"));
static llvm::cl::list<std::string> o_excludeFiles(llvm::cl::ZeroOrMore, "exclude", llvm::cl::desc("Exclude Files"));
static llvm::cl::list<std::string> o_excludePatterns(llvm::cl::ZeroOrMore, "exclude-pattern", llvm::cl::desc("Exclude Patterns"));
static llvm::cl::list<std::string> o_excludeDirs(llvm::cl::ZeroOrMore, "exclude-dir", llvm::cl::desc("Exclude Directories"));
static llvm::cl::list<std::string> o_tkbmsScanDirs(llvm::cl::ZeroOrMore, "tkbms-scan", llvm::cl::desc("Scan a directory using TKBMS to find files"));
static llvm::cl::opt<std::string> o_tkbmsProducts(llvm::cl::ZeroOrMore, "tkbms-products", llvm::cl::desc("The Havok products being built (e.g. 'PHYSICS ANIMATION'). For internal use"));
static llvm::cl::opt<std::string> o_tkbmsPlatforms(llvm::cl::ZeroOrMore, "tkbms-platforms", llvm::cl::desc("Platforms in current build (e.g. 'WIN32 X64'). For internal use"));
static llvm::cl::opt<std::string> o_preamble(llvm::cl::Optional, "preamble", llvm::cl::desc("Preamble added to the start of all generated files"));
static llvm::cl::opt<std::string> o_postamble(llvm::cl::Optional, "postamble", llvm::cl::desc("Postamble added to the end of all generated files"));
static llvm::cl::opt<std::string> o_preambleCpp(llvm::cl::Optional, "preamble-cpp", llvm::cl::desc("Preamble added to the start of generated cpp files. Will be added after generic preamble") );
static llvm::cl::opt<std::string> o_postambleCpp(llvm::cl::Optional, "postamble-cpp", llvm::cl::desc("Postamble added to the end of generated cpp files. Will be added before generic postamble") );
static llvm::cl::opt<std::string> o_timeStampFile(llvm::cl::Optional, "timestamp-file", llvm::cl::desc("File that gets written to after a successful build"));
static llvm::cl::opt<std::string> o_projectProduct(llvm::cl::Optional, "tkbms-project-product", llvm::cl::desc("The Havok product(s) which the processed project belongs to. For internal use"));
static llvm::cl::opt<bool> o_outputTkbms(llvm::cl::Optional, "--output-tkbms", llvm::cl::desc("Output tkbms headers to generated files"), llvm::cl::initializer<bool>(true));
static llvm::cl::opt<bool> o_cleanOutputPath(llvm::cl::Optional, "clean-output-path", llvm::cl::desc("Clean any old files from the output directory"));
static llvm::cl::opt<bool> o_disableLog(llvm::cl::Optional, "disable-log", llvm::cl::desc("Disables writing to the log file."));
static llvm::cl::opt<std::string> o_includeTrackingFile(llvm::cl::Optional, "include-tracking-file", llvm::cl::desc("Write include dependency tracking information to this file (disabled by default)"));
static llvm::cl::list<std::string> o_addAttributes(llvm::cl::ZeroOrMore, "add-attribute", llvm::cl::desc("Add additional attributes to every declaration"));
static llvm::cl::opt<bool> o_noRawRefObjPtr(llvm::cl::Optional, "no-raw-referencedobject", llvm::cl::desc("Disallow raw pointers to hkReferencedObject"));
// This is required while generated files still exist in the SDK _Auto dirs, it will eventually be removed
static llvm::cl::list<std::string> o_remapInclude(llvm::cl::ZeroOrMore, llvm::cl::CommaSeparated, "remap-include", llvm::cl::desc("Remap includes from one directory to appear to come from another"));

// Anything else goes into this
static llvm::cl::list<std::string> o_additionalInputFiles(llvm::cl::Positional, llvm::cl::desc("Additional input files"));

int main(int argc, char* argv[])
{
    signal(SIGABRT, onAbortCallback);
#ifdef _WIN32
    signal(SIGBREAK, onAbortCallback);
#endif

    PROFILE_POINT(main);

    const char desc[] = "This program will generate the reflection files of a Havok project\n";

    llvm::SmallVector<const char*, 100> args;
    for(int i=0; i < argc; i++)
    {
        args.push_back(argv[i]);
    }

    ArgsStringSaver sv;
#if defined(_WIN32)
    llvm::cl::ExpandResponseFiles(sv, llvm::cl::TokenizeWindowsCommandLine, args);
#else
    llvm::cl::ExpandResponseFiles(sv, llvm::cl::TokenizeGNUCommandLine, args);
#endif

    llvm::cl::ParseCommandLineOptions(static_cast<hkUlong>(args.size()), args.begin(), desc);

    // Sort out the output path
    const std::string& outputPath = Filesystem::absoluteCanonicalPath(o_outputPathArg);
    projectOutputPathStr.assign(outputPath);
    if (o_logPathArg.getValue().empty())
    {
        projectLogPathStr.assign(outputPath);
    }
    else
    {
        const std::string& logPath = Filesystem::absoluteCanonicalPath(o_logPathArg);
        projectLogPathStr.assign(logPath);
    }

    std::vector<std::string> products;
    if(o_tkbmsProducts.length())
        products = splitString(o_tkbmsProducts, ' ');
    std::vector<std::string> platforms;
    if (o_tkbmsPlatforms.length())
        platforms = splitString(o_tkbmsPlatforms, ' ');

    if(o_quiet)
        o_verbosity = 0;

    if(o_clangVersion)
    {
        llvm::cl::PrintVersionMessage();
        llvm::outs().flush();
        return 0;
    }

    replaceAllInplace(o_preamble.getValue(), "\r\n", "\n");
    replaceAllInplace(o_postamble.getValue(), "\r\n", "\n");
    replaceAllInplace(o_preambleCpp.getValue(), "\r\n", "\n");
    replaceAllInplace(o_postambleCpp.getValue(), "\r\n", "\n");

    hkLog.setLevel(o_verbosity);

    TkbmsRegistry tkbms(o_projectProduct);
    Filesystem fs(&Filesystem::simpleOpen);
    fs.setDryRun(o_dryRun);

    std::set<std::string> enabledProducts(products.begin(), products.end());
    std::set<std::string> enabledPlatforms(platforms.begin(), platforms.end());

    std::vector<std::string> inputFileNames;
    try
    {
        for (auto& s : o_additionalInputFiles)
        {
            // Split any file;file inputs
            tkbms.scanSingleFile(s);

            inputFileNames.push_back(s);
        }

        for(auto& directoryToScan: o_tkbmsScanDirs)
        {
            scanDirectoryForTkbms(directoryToScan, tkbms, enabledProducts, enabledPlatforms, o_explicitReflect.getValue(), o_excludeInternal, o_excludeDirs, inputFileNames);
            o_includePaths.push_back(directoryToScan);
        }

        if(inputFileNames.size() == 0)
        {
            if (o_tkbmsScanDirs.getNumOccurrences())
            {
                hkLog.warning("No files with valid TKBMS found.");
            }
            return 0;
        }

        if (!o_includeTrackingFile.empty())
        {
            Filesystem::ensureContainingDirectoryExists(o_includeTrackingFile);
        }


        for (auto& includePath : o_includePaths)
        {
            expandDollarEnvironmentVariable(includePath);
        }

        int ret = clangMain(tkbms, fs, o_includeTrackingFile, o_clangStats, o_binding, o_preprocessorDefines, o_includePaths, o_excludeFiles
            , o_excludePatterns, o_attributes, o_forceIncludeFiles, o_generateTracker, o_classList, inputFileNames, argv[0], projectOutputPathStr
            , o_preamble, o_postamble, o_preambleCpp, o_postambleCpp, o_addAttributes, o_noRawRefObjPtr, o_remapInclude, o_outputTkbms);

        if(o_profile)
            ProfilePoint::print("clang-extract profile");

        if (!o_disableLog)
        {
            auto handle = fs.open(projectLogPathStr + "/hkReflect_clang_messagelog.txt", Filesystem::RW);
            handle << hkLog.loggedMessages();
        }

        if (o_timeStampFile.length())
        {
            if (ret == 0)
            {
                hkLog.info("Writing timestamp file %s", o_timeStampFile.c_str());
                auto handle = fs.openRW(o_timeStampFile, true);
                handle << "// hkPreBuild completed\n";
            }
            else
            {
                hkLog.info("Removing timestamp file %s as the build failed", o_timeStampFile.c_str());
                llvm::sys::fs::remove(o_timeStampFile);
            }
        }

        if(o_cleanOutputPath)
        {
            fs.cleanOutputPath(projectOutputPathStr);
            fs.cleanOutputPath(projectLogPathStr);
        }

        return ret;
    }
    catch (const FatalError& e)
    {
        hkLog.error("Fatal Error: %s", e.what());
    }
    catch(const std::exception& e)
    {
        hkLog.error("Unhandled std::exception: %s", e.what());
    }
#ifdef WIN32
    catch(...)
    {
        LPVOID lpMsgBuf;
        DWORD dw = GetLastError();

        FormatMessage(
            FORMAT_MESSAGE_ALLOCATE_BUFFER |
            FORMAT_MESSAGE_FROM_SYSTEM |
            FORMAT_MESSAGE_IGNORE_INSERTS,
            NULL,
            dw,
            MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
            (LPTSTR) &lpMsgBuf,
            0, NULL );
        hkLog.error("Unhandled Windows error: %s", (char*) lpMsgBuf);
    }

#else
    catch(...)
    {
        hkLog.error("Unknown exception caught.");
    }
#endif

    return 1;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
