// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
#include "filter/ExtraAttributesFilter.h"
#include "utils/Database.h"
#include "utils/StlUtils.h"
#include "utils/ElementUtils.h"

namespace
{
    inline clang::SourceLocation getLoc(const hkStringView& token, const llvm::MemoryBuffer* file, clang::SourceLocation startOfFile)
    {
        if (token.begin() >= file->getBufferStart() && token.end() <= file->getBufferEnd())
        {
            int offset = static_cast<int>(token.begin() - file->getBufferStart());
            return startOfFile.getLocWithOffset(offset);
        }
        return clang::SourceLocation();
    }

    inline clang::SourceRange getRange(const hkStringView& token, const llvm::MemoryBuffer* file, clang::SourceLocation startOfFile)
    {
        clang::SourceLocation begin = getLoc(token, file, startOfFile);

        if (begin.isValid())
        {
            return clang::SourceRange(begin, begin.getLocWithOffset(token.getSize()));
        }
        return clang::SourceRange();
    }
}

ExtraAttributesFilter::ExtraAttributesFilter(const std::vector<std::string>& addAttributes) : m_addAttributes(addAttributes) {}

ExtraAttributesFilter::~ExtraAttributesFilter()
{
}

void ExtraAttributesFilter::run(Database& db)
{
    clang::FileManager& fileMgr = db.getSourceManager().getFileManager();

    // For every file in the database.
    for(Database::FileDeclMapType::const_iterator fit = db.inputFilesDecls.begin();
        fit != db.inputFilesDecls.end();
        ++fit)
    {
        // Temp, add the attributes to all decls
        if (m_addAttributes.size())
        {
            const Database::DeclListType& declList = fit->second.decls;

            // For every named declaration.
            for (Database::DeclListType::const_iterator dit = declList.begin();
                dit != declList.end();
                ++dit)
            {
                if (clang::CXXRecordDecl* recDecl = clang::dyn_cast<clang::CXXRecordDecl>(*dit))
                {
                    if (db.isReflected(recDecl))
                    {
                        for (const std::string& attr : m_addAttributes)
                        {
                            db.addAttribute(recDecl, attr);
                        }
                    }
                }
            }
        }

        // Look for a "<header_name>.hkReflect" file beside the header.
        std::string fileName = fit->first;
        auto dot = fileName.rfind('.');
        fileName.replace(dot, std::string::npos, ".hkReflect");

        if (const clang::FileEntry* reflectFile = fileMgr.getFile(fileName, true))
        {
            // Read the file.
            llvm::MemoryBuffer* contents;
            {
                std::string error;
                contents = fileMgr.getBufferForFile(reflectFile, &error);

                if (contents == nullptr)
                {
                    hkLog.error("Cannot open %s: %s", fileName.c_str(), error.c_str());
                    continue;
                }
            }

            // Add the file to the SourceManager
            clang::FileID reflectFileId =
                db.getSourceManager().createFileID(reflectFile, clang::SourceLocation(), clang::SrcMgr::C_User);
            clang::SourceLocation startOfFile = db.getSourceManager().getLocForStartOfFile(reflectFileId);

            // Parse the file.
            hkReflectFileParser parser(hkStringView(contents->getBufferStart(), contents->getBufferEnd()), fileName.c_str());
            hkReflectFileParser::Error error;
            if (parser.parse(error).isFailure())
            {
                hkLog.errorWithoutDatabase(error.m_line, error.m_fileName, "%s", error.m_message.cString());
                continue;
            }

            // Extract the attributes from the file.
            auto& decls = parser.getClassDeclarations();
            for (auto& decl : decls)
            {
                // Get a class decl with a matching name.
                clang::TypeDecl* annotatedDecl = nullptr;
                for (auto& typeDecl : fit->second.decls)
                {
                    if (typeDecl->getQualifiedNameAsString() == decl.m_name)
                    {
                        annotatedDecl = typeDecl;
                        break;
                    }
                }

                if (annotatedDecl == nullptr)
                {
                    hkLog.error(db.getSourceManager(), getLoc(decl.m_name, contents, startOfFile),
                        "'%s' is not a class declared in %s", hkStringBuf(decl.m_name).cString(), fit->first.c_str());
                    continue;
                }

                if (auto recDecl = clang::dyn_cast<clang::CXXRecordDecl>(annotatedDecl))
                {
                    // Add attributes to the class.
                    for ( auto attr : decl.m_attributes )
                    {
                        std::string attrString( attr.m_wholeString.begin(), attr.m_wholeString.getSize() );
                        clang::SourceRange attrRange = getRange( attr.m_wholeString, contents, startOfFile );

                        try
                        {
                            Attribute attrObj( db, attrString, attrRange.getBegin() );
                            db.addAttribute( recDecl, attrString, attrRange );
                        }
                        catch ( RuntimeError e )
                        {
                            hkLog.errorWithoutDatabase( e.getLine(), e.getFileName(), e.getMessage() );
                        }
                    }

                    // Add attributes to fields.
                    std::map<std::string, clang::NamedDecl*> fields;
                    std::map<const char*, bool> attributes;

                    // Add fields to the map
                    const std::string& fieldPrefix = db.getAttributeValue<std::string>(recDecl, "hk::FieldPrefix");
                    for (auto it = recDecl->field_begin(); it != recDecl->field_end(); ++it)
                    {
                        fields.insert(std::make_pair(fieldNameWithoutPrefix(fieldPrefix, it->getNameAsString()), *it));
                    }

                    // Add properties to the map
                    for (auto it = clang::CXXRecordDecl::specific_decl_iterator<clang::CXXRecordDecl>(recDecl->decls_begin());
                    it != clang::CXXRecordDecl::specific_decl_iterator<clang::CXXRecordDecl>(recDecl->decls_end()); ++it)
                    {
                        if (auto propDecl = const_cast<clang::CXXRecordDecl*>(isPropertyDecl(*it)))
                        {
                            fields.insert(std::make_pair(getPropertyDeclName(propDecl), propDecl));
                        }
                    }

                    for (auto& property : decl.m_properties)
                    {
                        auto field = fields.find(llvm::StringRef(property.m_name.begin(), property.m_name.getSize()));
                        if (field == fields.end())
                        {
                            hkLog.error(db.getSourceManager(), getLoc(property.m_name, contents, startOfFile),
                                "'%s' is not a field/property of '%s'",
                                hkStringBuf(property.m_name).cString(), recDecl->getNameAsString().c_str());
                            continue;
                        }

                        for (auto& attr : property.m_attributes)
                        {
                            std::string attrString(attr.m_wholeString.begin(), attr.m_wholeString.getSize());
                            clang::SourceRange attrRange = getRange(attr.m_wholeString, contents, startOfFile);
                            if (attrRange.isInvalid())
                            {
                                // Automatically generated attribute, use range of the property
                                attrRange = getRange(property.m_name, contents, startOfFile);
                            }

                            // The same attribute might be applied to many fields. We keep a map in order to reject it
                            // (and print an error) only once.
                            auto result = attributes.insert(std::make_pair(attr.m_wholeString.begin(), false));
                            auto& iterator = result.first;
                            if (result.second)
                            {
                                try
                                {
                                    Attribute attrObj(db, attrString, attrRange.getBegin());
                                    iterator->second = true;
                                }
                                catch (RuntimeError e)
                                {
                                    hkLog.errorWithoutDatabase(e.getLine(), e.getFileName(), e.getMessage());
                                }
                            }
                            if (iterator->second)
                            {
                                db.addAttribute(field->second, attrString, attrRange);
                            }
                        }
                    }
                }
            }
        }

    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
